/*
    opcode
    copyright (c) 1998-2012 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "opcode.h"
#include "peimage.h"


/******************************************************************************
*                                                                             *
* ja:オペコード関数                                                           *
*                                                                             *
******************************************************************************/
typedef struct _ModRM
{
  gint mod, regop, regmem;
  gchar regop_s[5];
  gchar regmem_s[24];
} ModRM;


const gchar *regname[8][8] = {
  {"al",   "cl",   "dl",   "bl",   "ah",   "ch",   "dh",   "bh"},
  {"ax",   "cx",   "dx",   "bx",   "sp",   "bp",   "si",   "di"},
  {"eax",  "ecx",  "edx",  "ebx",  "esp",  "ebp",  "esi",  "edi"},
  {"mm0",  "mm1",  "mm2",  "mm3",  "mm4",  "mm5",  "mm6",  "mm7"},
  {"xmm0", "xmm1", "xmm2", "xmm3", "xmm4", "xmm5", "xmm6", "xmm7"},
  {"es",   "cs",   "ss",   "ds",   "fs",   "gs",   "hs",   "is"},
  {"cr0",  "cr1",  "cr2",  "cr3",  "cr4",  "cr5",  "cr6",  "cr7"},
  {"dr0",  "dr1",  "dr2",  "dr3",  "dr4",  "dr5",  "dr6",  "dr7"}};
const static gchar *cc[16] = {"o", "no", "b", "ae", "z", "nz", "be", "a",
                              "s", "ns", "p", "np", "l", "ge", "le", "g"};


/*  ja:プリフィックスからセグメントレジスタ文字列を取得する
    pseg,セグメントレジスタプリフィックス
     RET,セグメントレジスタ文字列                                           */
static const gchar *
segment (const guint8 pseg)
{
  switch (pseg)
    {
      case 0x2e: return "cs:";
      case 0x36: return "ss:";
      case 0x3e: return "ds:";
      case 0x26: return "es:";
      case 0x64: return "fs:";
      case 0x65: return "gs:";
    }
  return "";
}


/*  ja:ModR/MおよびSIBビット解析
     image,PEイメージ
      size,サイズ
        op,オペコード構造体
     regop,レジスタオペコードのビット数(8,16,32,64,128,256)
    regmem,R/Mでレジスタを指定するときのビット数(8,16,32,64,128,256)
      padr,アドレスサイズプレフィックス(TRUE:16ビット,FALSE:32ビット)
      pseg,セグメントレジスタプリフィックス
         m,ModR/M構造体
       RET,TRUE:正常終了,FALSE:エラー                                       */
static gboolean
opcode_modrm (const guint8   *image,
              const gsize     size,
              Opcode         *op,
              const gint      regop,
              const gint      regmem,
              const gboolean  padr,
              const guint8    pseg,
              ModRM          *m)
{
  m->mod = m->regop = m->regmem = 0;
  m->regop_s[0] = m->regmem_s[0] = '\0';
  if (op->index < 0 || size <= op->index)
    return FALSE;
  m->mod = (image[op->index] >> 6) & 3;
  m->regop = (image[op->index] >> 3) & 7;
  m->regmem = image[op->index] & 7;
  op->index++;
  switch (regop)
    {
      case   8: g_strcpy (m->regop_s, regname[0][m->regop]); break;
      case  16: g_strcpy (m->regop_s, regname[1][m->regop]); break;
      case  32: g_strcpy (m->regop_s, regname[2][m->regop]); break;
      case  64: g_strcpy (m->regop_s, regname[3][m->regop]); break;
      case 128: g_strcpy (m->regop_s, regname[4][m->regop]); break;
      default:  g_strcpy (m->regop_s, regname[5][m->regop]);
    }
  if (m->mod == 3)
    {
      if (pseg != 0)
        return FALSE;
      switch (regmem)
        {
          case   8: g_strcpy (m->regmem_s, regname[0][m->regmem]); break;
          case  16: g_strcpy (m->regmem_s, regname[1][m->regmem]); break;
          case  32: g_strcpy (m->regmem_s, regname[2][m->regmem]); break;
          case  64: g_strcpy (m->regmem_s, regname[3][m->regmem]); break;
          case 128: g_strcpy (m->regmem_s, regname[4][m->regmem]); break;
          default:  g_strcpy (m->regmem_s, regname[5][m->regmem]);
        }
      return TRUE;
    }
  if (padr)
    {
      /* ja:16ビット */
      const gchar *r = NULL;

      if (!(m->mod == 0 && m->regmem == 6))
        switch (m->regmem)
          {
            case 0: r = "bx+si"; break;
            case 1: r = "bx+di"; break;
            case 2: r = "bp+si"; break;
            case 3: r = "bp+di"; break;
            case 4: r = "si";    break;
            case 5: r = "di";    break;
            case 6: r = "bp";    break;
            case 7: r = "bx";
          }
      if (m->mod == 1)
        {
          if (op->index + 1 >= size)
            return FALSE;
          if ((gint8)image[op->index] < 0)
            g_sprintf (m->regmem_s, "%s[%s-%02x]",
                                segment (pseg), r, -(gint8)image[op->index]);
          else
            g_sprintf (m->regmem_s, "%s[%s+%02x]",
                                        segment (pseg), r, image[op->index]);
          op->index++;
        }
      else if ((m->mod == 0 && m->regmem == 6) || m->mod == 2)
        {
          gint d;

          if (op->index + 2 >= size)
            return FALSE;
          d = GUINT16_FROM_LE (*(guint16 *)(image + op->index));
          if (r)
            g_sprintf (m->regmem_s, "%s[%s+%04x]", segment (pseg), r, d);
          else
            g_sprintf (m->regmem_s, "%s[%04x]", segment (pseg), d);
          op->index += 2;
        }
      else
        {
          g_sprintf (m->regmem_s, "%s[%s]", segment (pseg), r);
        }
    }
  else
    {
      /* ja:32ビット */
      gchar d[9], r[10];
      guint8 ss = 0, index = 0, base = 0;

      d[0] = r[0] = '\0';
      if (!((m->mod == 0 && m->regmem == 5) || m->regmem == 4))
        g_strcpy (r, regname[2][m->regmem]);
      if (m->regmem == 4)
        {
          if (op->index + 1 >= size)
            return FALSE;
          ss = (image[op->index] >> 6) & 3;
          index = (image[op->index] >> 3) & 7;
          base = image[op->index] & 7;
          op->index++;
        }
      if (m->mod == 1)
        {
          if (op->index + 1 >= size)
            return FALSE;
          if ((gint8)image[op->index] < 0)
            g_sprintf (d, "-%02x", -(gint8)image[op->index]);
          else
            g_sprintf (d, "%02x", image[op->index]);
          op->index++;
        }
      else if ((m->mod == 0 && m->regmem == 4 && base == 5)
                            || (m->mod == 0 && m->regmem == 5) || m->mod == 2)
        {
          if (op->index + 4 >= size)
            return FALSE;
          op->offset = GUINT32_FROM_LE (*(guint32 *)(image + op->index));
          g_sprintf (d, "%08x", op->offset);
          op->index += 4;
        }
      if (m->regmem == 4)
        {
          if (index != 4)
            {
              g_strcpy (r, regname[2][index]);
              switch (ss)
                {
                  case 1: g_strcat (r, "*2"); break;
                  case 2: g_strcat (r, "*4"); break;
                  case 3: g_strcat (r, "*8");
                }
            }
          if (!(base ==5 && m->mod == 0))
            {
              if (index != 4)
                g_strcat (r, "+");
              g_strcat (r, regname[2][base]);
            }
        }
      if (r[0] != '\0')
        {
          op->array = op->offset;
          op->offset = (guint32)-1;
        }
      g_strcpy (m->regmem_s, segment (pseg));
      g_strcat (m->regmem_s, "[");
      g_strcat (m->regmem_s, r);
      if (d[0] != '\0' && d[0] != '-' && r[0] != '\0')
        g_strcat (m->regmem_s, "+");
      g_strcat (m->regmem_s, d);
      g_strcat (m->regmem_s, "]");
    }
  return TRUE;
}


/*  ja:拡張命令解析
    image,PEイメージ
     size,サイズ
     base,ベースアドレス
       op,オペコード構造体
     pope,オペランドプレフィックス(TRUE:16ビット,FALSE:32ビット)
     padr,アドレスサイズプレフィックス(TRUE:16ビット,FALSE:32ビット)
      RET,TRUE:正常終了,FALSE:エラー                                        */
static gboolean
opcode_extend (const guint8   *image,
               const gsize     size,
               const guint32   base,
               Opcode         *op,
               const gboolean  pope,
               const gboolean  padr,
               const guint8    pseg,
               const guint8    prep)
{
  guint8 code;

  while (0 <= op->index && op->index < size)
    switch ((code = image[op->index++]))
      {
        case 0x00: /* sldt/str/lldt/ltr/verr/verw */
        case 0x01: /* sgdt/sidt/lgdt/lidt/smsw/lmsw/invlpg/monitor/mwait */
          {
            gint i;
            ModRM m;
            const static gchar *mnemonic[] = {
                                            "sldt", "str",  "lldt", "ltr",
                                            "verr", "verw", NULL,   NULL,
                                            "sgdt", "sidt", "lgdt", "lidt",
                                            "smsw", NULL,   "lmsw", "invlpg"};
            const static gboolean mod[] = {FALSE, FALSE, FALSE, FALSE,
                                           FALSE, FALSE, FALSE, FALSE,
                                           TRUE,  TRUE,  TRUE,  TRUE,
                                           FALSE, FALSE, FALSE, TRUE};

            op->bit = pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            i = m.regop + code * 8;
            if (mnemonic[i] && (m.mod != 3 || !mod[i]))
              g_sprintf (op->mnemonic, "%s %s", mnemonic[i], m.regmem_s);
            else if (m.regop == 1 && m.regmem == 0)
              g_strcpy (op->mnemonic, "monitor");
            else if (m.regop == 1 && m.regmem == 1)
              g_strcpy (op->mnemonic, "mwait");
            else
              return FALSE;
          }
          return TRUE;
        case 0x02: /* lar */
        case 0x03: /* lsl */
          {
            ModRM m;

            op->bit = pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "%s %s,%s",
                        code == 0x02 ? "lar" : "lsl", m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0x05: /* loadall */
        case 0x07:
          g_strcpy (op->mnemonic, "loadall");
          return TRUE;
        case 0x06: /* clts */
          g_strcpy (op->mnemonic, "clts");
          return TRUE;
        case 0x08: /* invd */
          g_strcpy (op->mnemonic, "invd");
          return TRUE;
        case 0x09: /* wbinvd */
          g_strcpy (op->mnemonic, "wbinvd");
          return TRUE;
        case 0x0b: /* ub2 */
          g_strcpy (op->mnemonic, "ub2");
          return TRUE;
        case 0x10: /* movsd/movss/movupd/movups */
        case 0x11:
        case 0x12: /* movddup/movhlps/movlpd/movlps/movsldup */
        case 0x13:
        case 0x16: /* movhpd/movhps/movlhps/movshdup */
        case 0x17:
        case 0x28: /* movapd/movaps */
        case 0x29:
          {
            const gchar *mnemonic;
            ModRM m;

            if ((pope && prep != 0) || prep == 0xf0)
              return FALSE;
            op->bit = 128;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            if (code == 0x10 || code == 0x11)
              {
                if (prep == 0xf2)
                  mnemonic = "movsd";
                else if (prep == 0xf3)
                  mnemonic = "movss";
                else if (pope)
                  mnemonic = "movupd";
                else
                  mnemonic = "movups";
              }
            else if (code == 0x12 || code == 0x13)
              {
                if (prep == 0xf2)
                  mnemonic = "movddup";
                else if (prep == 0xf3)
                  mnemonic = "movsldup";
                else if (pope)
                  mnemonic = "movlpd";
                else if (m.mod == 3)
                  mnemonic = "movhlps";
                else
                  mnemonic = "movlps";
              }
            else if (code == 0x16 || code == 0x17)
              {
                if (prep == 0xf3)
                  mnemonic = "movshdup";
                else if (pope)
                  mnemonic = "movhpd";
                else if (m.mod == 3)
                  mnemonic = "movlhps";
                else
                  mnemonic = "movhps";
              }
            else
              {
                if (prep != 0 && prep != 0xf0)
                  return FALSE;
                else if (pope)
                  mnemonic = "movapd";
                else
                  mnemonic = "movaps";
              }
            if ((code & 1) == 0)
              g_sprintf (op->mnemonic, "%s %s,%s",
                                            mnemonic, m.regop_s, m.regmem_s);
            else
              g_sprintf (op->mnemonic, "%s %s,%s",
                                            mnemonic, m.regmem_s, m.regop_s);
          }
          return TRUE;
        case 0x14: /* unpcklpd/unpcklps */
        case 0x15: /* unpckhpd/unpckhps */
          {
            ModRM m;

            op->bit = 128;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "%s%s %s,%s",
                                    (code & 1) == 0 ? "unpcklp" : "unpckhp",
                                    pope ? "d" : "s", m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0x18: /* prefetcht0/prefetcht1/prefetcht2/prefetchnta */
          {
            ModRM m;
            const static gchar *mnemonic[] = {"prefetchnta", "prefetcht0",
                                              "prefetcht1",  "prefetcht2"};

            op->bit = 128;
            if (!opcode_modrm (image, size, op,
                            op->bit, op->bit, padr, pseg, &m) || m.regop > 3)
              return FALSE;
            g_sprintf (op->mnemonic, "%s %s", mnemonic[m.regop], m.regmem_s);
          }
          return TRUE;
        case 0x20: /* mov */
        case 0x21:
        case 0x22:
        case 0x23:
          {
            ModRM m;

            op->bit = 32;
            if (!opcode_modrm (image, size, op,
                            op->bit, op->bit, padr, pseg, &m) || m.mod != 3)
              return FALSE;
            if ((code & 2) == 0)
              g_sprintf (op->mnemonic, "mov %s,%s",
                        m.regmem_s, regname[(code & 1) == 0 ? 6 : 7][m.regop]);
            else
              g_sprintf (op->mnemonic, "mov %s,%s",
                        regname[(code & 1) == 0 ? 6 : 7][m.regop], m.regmem_s);
          }
          return TRUE;
        case 0x2a: /* cvtpi2pd/cvtpi2ps/cvtsi2sd/cvtsi2ss */
          {
            ModRM m;

            op->bit = prep == 0xf2 || prep == 0xf3 ? 32 : 64;
            if (!opcode_modrm (image, size, op, 128, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "cvt%si2%s%s %s,%s",
                    prep == 0xf2 || prep == 0xf3 ? "s" : "d",
                    prep == 0xf2 || prep == 0xf3 ? "s" : "p",
                    prep == 0xf2 || pope ? "d" : "s", m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0x2b: /* movntpd/movntps */
        case 0x2e: /* ucomisd/ucomiss */
        case 0x2f: /* comisd/comiss */
          {
            ModRM m;
            const static gchar *mnemonic[] = {
                                    "movntp", NULL, NULL, "ucomis", "comis"};

            op->bit = 128;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "%s%s %s,%s", mnemonic[code - 0x2b],
                                    pope ? "d" : "s", m.regmem_s, m.regop_s);
          }
        case 0x2c: /* cvttpd2pi/cvttps2pi/cvttsd2si/cvttss2si */
        case 0x2d: /* cvtpd2pi/cvtps2pi/cvtsd2si/cvtss2si */
          {
            ModRM m;

            if ((pope && prep != 0) || prep == 0xf0)
              return FALSE;
            op->bit = 128;
            if (!opcode_modrm (image, size, op,
                                        prep == 0xf2 || prep == 0xf3 ? 32 : 64,
                                                    op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "cvt%s%s%s2%si %s,%s",
                                    (code & 1) == 0 ? "t" : "",
                                    prep == 0xf2 || prep == 0xf3 ? "s" : "p",
                                    prep == 0xf2 || pope ? "d" : "s",
                                    prep == 0xf2 || prep == 0xf3 ? "s" : "p",
                                    m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0x30: /* wrmsr */
          g_strcpy (op->mnemonic, "wrmsr");
          return TRUE;
        case 0x31: /* rdtsc */
          g_strcpy (op->mnemonic, "rdtsc");
          return TRUE;
        case 0x32: /* rdmsr */
          g_strcpy (op->mnemonic, "rdmsr");
          return TRUE;
        case 0x33: /* rdpmc */
          g_strcpy (op->mnemonic, "rdpmc");
          return TRUE;
        case 0x34: /* sysenter */
          g_strcpy (op->mnemonic, "sysenter");
          return TRUE;
        case 0x35: /* sysexit */
          g_strcpy (op->mnemonic, "sysexit");
          return TRUE;
        case 0x40: case 0x41: case 0x42: case 0x43: /* cmovcc */
        case 0x44: case 0x45: case 0x46: case 0x47:
        case 0x48: case 0x49: case 0x4a: case 0x4b:
        case 0x4c: case 0x4d: case 0x4e: case 0x4f:
          {
            ModRM m;

            op->bit = pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "cmov%s %s,%s",
                                    cc[code - 0x40], m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0x50: /* movmskpd/movmskpd */
          {
            ModRM m;

            if (!opcode_modrm (image, size, op, 32, 128, padr, pseg, &m)
                                                                || m.mod != 3)
              return FALSE;
            g_sprintf (op->mnemonic, "movmskp%s %s,%s",
                                    pope ? "d" : "s", m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0x51: /* sqrtpd/sqrtps/sqrtsd/sqrtss */
        case 0x52: /* rsqrtps/rsqrtss */
        case 0x53: /* rcpps/rcpss */
        case 0x54: /* andpd/andps */
        case 0x55: /* andnpd/andnps */
        case 0x56: /* orpd/orps */
        case 0x57: /* xorpd/xornps */
        case 0x58: /* addpd/addps/addsd/addss */
        case 0x59: /* mulpd/mulps/mulsd/mulss */
        case 0x5c: /* subpd/subps/subsd/subss */
        case 0x5d: /* minpd/minps/minsd/minss */
        case 0x5e: /* divpd/divps/divsd/divss */
        case 0x5f: /* maxpd/maxps/maxsd/maxss */
          {
            ModRM m;
            const gchar *mnemonic[] = {       "sqrt", "rsqrt", "rcp",  /* 51 */
                                       "and", "andn", "or",    "xor",  /* 54 */
                                       "add", "mul",  NULL,    NULL,   /* 58 */
                                       "sub", "min",  "div",   "max"}; /* 5c */

            if ((pope && prep != 0) || prep == 0xf0)
              return FALSE;
            op->bit = 128;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "%s%s%s %s,%s",
                                    mnemonic[code - 0x51],
                                    prep == 0xf2 || prep == 0xf3 ? "s" : "p",
                                    prep == 0xf2 || pope ? "d" : "s",
                                    m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0x5a: /* cvtpd2ps/cvtps2pd/cvtsd2ss/cvtss2sd */
          {
            ModRM m;

            if ((pope && prep != 0) || prep == 0xf0)
              return FALSE;
            op->bit = 128;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "cvt%s%s2%s%s %s,%s",
                                    prep == 0xf2 || prep == 0xf3 ? "s" : "p",
                                    prep == 0xf2 || pope ? "d" : "s",
                                    prep == 0xf2 || prep == 0xf3 ? "s" : "p",
                                    prep == 0xf2 || pope ? "s" : "d",
                                    m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0x5b: /* cvtdq2ps/cvttps2dq/cvtps2dq */
          {
            ModRM m;

            if ((pope && prep != 0) || prep == 0xf0)
              return FALSE;
            op->bit = 128;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "cvt%s%s %s,%s",
                                    prep == 0xf2 || prep == 0xf3 ? "t" : "",
                                    prep == 0xf2 || !pope ? "dq2ps" : "ps2dq",
                                                        m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0x60: /* punpcklbw */
        case 0x61: /* punpcklwd */
        case 0x62: /* punpckldq */
        case 0x63: /* packsswb */
        case 0x64: /* pcmpgtb */
        case 0x65: /* pcmpgtw */
        case 0x66: /* pcmpgtd */
        case 0x67: /* packuswb */
        case 0x68: /* punpckhbw */
        case 0x69: /* punpckhwd */
        case 0x6a: /* punpckhdq */
        case 0x6b: /* packssdw */
        case 0x6c: /* punpcklqdq */
        case 0x6d: /* punpckhqdq */
        case 0x74: /* pcmpeqb */
        case 0x75: /* pcmpeqw */
        case 0x76: /* pcmpeqd */
          {
            ModRM m;
            const gchar *mnemonic[] = {
                "punpcklbw",  "punpcklwd",  "punpckldq", "packsswb", /* 60 */
                "pcmpgtb",    "pcmpgtw",    "pcmpgtd",   "packuswb", /* 64 */
                "punpckhbw",  "punpckhwd",  "punpckhdq", "packssdw", /* 68 */
                "punpcklqdq", "punpckhqdq", NULL,        NULL,       /* 6c */
                NULL,         NULL,         NULL,        NULL,       /* 70 */
                "pcmpeqb",    "pcmpeqw",    "pcmpeqd"};              /* 74 */

            op->bit = pope ? 128 : 64;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "%s %s,%s",
                                mnemonic[code - 0x60], m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0x6e: /* movd/movq */
        case 0x7e:
          {
            ModRM m;

            if ((pope && prep != 0) || prep == 0xf0 || prep == 0xf2)
              return FALSE;
            op->bit = prep == 0xf3 ? 128 : 32;
            if (!opcode_modrm (image, size, op,
                    prep == 0xf3 || pope ? 128 : 64, op->bit, padr, pseg, &m))
              return FALSE;
            if ((code & 16) == 0)
              g_sprintf (op->mnemonic, "%s %s,%s",
                        prep == 0xf3 ? "movq" : "movd", m.regop_s, m.regmem_s);
            else
              g_sprintf (op->mnemonic, "%s %s,%s",
                        prep == 0xf3 ? "movq" : "movd", m.regmem_s, m.regop_s);
          }
          return TRUE;
        case 0x6f: /* movdqa/movdqu/movq */
        case 0x7f:
          {
            ModRM m;

            if ((pope && prep != 0) || prep == 0xf0 || prep == 0xf2)
              return FALSE;
            op->bit = prep == 0xf3 || pope ? 128 : 64;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            if ((code & 16) == 0)
              g_sprintf (op->mnemonic, "%s %s,%s",
                            prep == 0xf3 ? "movdqu" : pope ? "movdqa" : "movq",
                                                        m.regop_s, m.regmem_s);
            else
              g_sprintf (op->mnemonic, "%s %s,%s",
                            prep == 0xf3 ? "movdqu" : pope ? "movdqa" : "movq",
                                                        m.regmem_s, m.regop_s);
          }
          return TRUE;
        case 0x70: /* pshufd/pshufhw/pshuflw/pshufw */
          {
            ModRM m;
            const gchar *mnemonic;

            if ((pope && prep != 0) || prep == 0xf0)
              return FALSE;
            op->bit = prep == 0xf2 || prep == 0xf3 || pope ? 128 : 64;
            if (!opcode_modrm (image, size, op,
                    op->bit, op->bit, padr, pseg, &m) || op->index + 1 >= size)
              return FALSE;
            if (prep == 0xf2)
              mnemonic = "pshuflw";
            else if (prep == 0xf3)
              mnemonic = "pshufhw";
            else if (pope)
              mnemonic = "pshufd";
            else
              mnemonic = "pshufw";
            g_sprintf (op->mnemonic, "%s %s,%s,%02x",
                            mnemonic, m.regop_s, m.regmem_s, image[op->index]);
            op->index++;
          }
          return TRUE;
        case 0x71: /* psrlw/psraw/psllw */
        case 0x72: /* psrld/psrad/pslld */
        case 0x73: /* psrlq/psraq/psllq */
          {
            ModRM m;
            const gchar *mnemonic[] = {NULL,   NULL, "psrl", NULL,
                                       "psra", NULL, "psll", NULL};
            const gchar *wdq[] = {"w", "d", "q"};

            op->bit = pope ? 128 : 64;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m)
                                || !mnemonic[m.regop] || op->index + 1 >= size)
              return FALSE;
            g_sprintf (op->mnemonic, "%s%s %s,%02x", mnemonic[m.regop],
                            wdq[code - 0x71], m.regmem_s, image[op->index]);
            op->index++;
          }
          return TRUE;
        case 0x77: /* emms */
          g_strcpy (op->mnemonic, "emms");
          return TRUE;
        case 0x7c: /* haddpd/haddps */
        case 0x7d: /* hsubpd/hsubps */
          {
            ModRM m;

            if ((pope && prep != 0) || (!pope && prep != 0xf2))
              return FALSE;
            op->bit = 128;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "%s%s %s,%s", 
                                            (code & 1) == 0 ? "haddp" : "hsub",
                                            prep == 0xf2 ? "s" : "d",
                                            m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0x80: case 0x81: case 0x82: case 0x83: /* jcc */
        case 0x84: case 0x85: case 0x86: case 0x87:
        case 0x88: case 0x89: case 0x8a: case 0x8b:
        case 0x8c: case 0x8d: case 0x8e: case 0x8f:
          if (pope)
            {
              if (op->index + 2 >= size)
                return FALSE;
              op->relative = GINT16_FROM_LE (*(gint16 *)(image + op->index));
              op->index += 2;
            }
          else
            {
              if (op->index + 4 >= size)
                return FALSE;
              op->relative = GINT32_FROM_LE (*(gint32 *)(image + op->index));
              op->index += 4;
            }
          op->relative += op->index + base;
          g_sprintf (op->mnemonic, "j%s %08x", cc[code - 0x80], op->relative);
          op->type = OPCODE_TYPE_JMP;
          return TRUE;
        case 0x90: case 0x91: case 0x92: case 0x93: /* setcc */
        case 0x94: case 0x95: case 0x96: case 0x97:
        case 0x98: case 0x99: case 0x9a: case 0x9b:
        case 0x9c: case 0x9d: case 0x9e: case 0x9f:
          {
            ModRM m;

            op->bit = 8;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "set%s %s", cc[code - 0x90], m.regmem_s);
          }
          return TRUE;
        case 0xa0: /* push fs */
          g_strcpy (op->mnemonic, "push fs");
          return TRUE;
        case 0xa1: /* pop fs */
          g_strcpy (op->mnemonic, "pop fs");
          return TRUE;
        case 0xa2: /* cpuid */
          g_strcpy (op->mnemonic, "cpuid");
          return TRUE;
        case 0xa3: /* bt */
        case 0xab: /* bts */
        case 0xb3: /* btr */
        case 0xba: /* bt/bts/btr/btc */
        case 0xbb: /* btc */
          {
            ModRM m;
            const gchar *mnemonic[] = {"bt", "bts", "btr", "btc"};

            op->bit = pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            if (code == 0xba)
              {
                if (m.regop < 4 || op->index + 1 >= size)
                  return FALSE;
                g_sprintf (op->mnemonic, "%s %s,%02x",
                        mnemonic[m.regop - 4], m.regmem_s, image[op->index]);
                op->index++;
              }
            else
              {
                g_sprintf (op->mnemonic, "%s %s,%s",
                            mnemonic[(code >> 3) & 3], m.regmem_s, m.regop_s);
              }
          }
          return TRUE;
        case 0xa4: /* shld */
        case 0xa5:
        case 0xac: /* shrd */
        case 0xad:
          {
            ModRM m;

            op->bit = pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            if ((code & 1) == 0)
              {
                if (op->index + 1 >= size)
                  return FALSE;
                g_sprintf (op->mnemonic, "%s %s,%s,%02x",
                                    (code & 8) == 0 ? "shld" : "shrd",
                                    m.regmem_s, m.regop_s, image[op->index]);
                op->index++;
              }
            else
              {
                g_sprintf (op->mnemonic, "%s %s,%s,cl",
                    (code & 8) == 0 ? "shld" : "shrd", m.regmem_s, m.regop_s);
              }
          }
          return TRUE;
        case 0xa8: /* push gs */
          g_strcpy (op->mnemonic, "push gs");
          return TRUE;
        case 0xa9: /* pop gs */
          g_strcpy (op->mnemonic, "pop gs");
          return TRUE;
        case 0xaa: /* rsm */
          g_strcpy (op->mnemonic, "rsm");
          return TRUE;
        case 0xae: /* fxsave/fxrstor/ldmxcsr/stmxcsr
                      clflush/lfence/mfence/sfence */
          {
            ModRM m;

            op->bit = pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            if (m.mod != 3)
              {
                const gchar *mnemonic[] = {"fxsave",  "fxrstor",
                                           "ldmxcsr", "stmxcsr",
                                           NULL,      NULL,
                                           NULL,      "clflush"};

                if (mnemonic[m.regop])
                  return FALSE;
                g_sprintf (op->mnemonic, "%s %s",
                                                mnemonic[m.regop], m.regmem_s);
              }
            else if (m.regop == 5)
              {
                g_strcpy (op->mnemonic, "lfence");
              }
            else if (m.regop == 6)
              {
                g_strcpy (op->mnemonic, "mfence");
              }
            else if (m.regop == 7)
              {
                g_strcpy (op->mnemonic, "sfence");
              }
            else
              {
                return FALSE;
              }
          }
          return TRUE;
        case 0xaf: /* imul */
          {
            ModRM m;

            op->bit = pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "imul %s,%s", m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0xb0: /* cmpxchg */
        case 0xb1:
        case 0xc0: /* xadd */
        case 0xc1:
          {
            ModRM m;

            op->bit = (code & 1) == 0 ? 8 : pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "%s %s,%s",
                (code & 1) == 0 ? "xadd" : "cmpxchg", m.regmem_s, m.regop_s);
          }
          return TRUE;
        case 0xb2: /* lss */
        case 0xb4: /* lfs */
        case 0xb5: /* lgs */
          if (pope)
            {
              if (op->index + 4 >= size)
                return FALSE;
              g_sprintf (op->mnemonic, "%s %04x:%04x",
                        code == 0xb2 ? "lss" : code == 0xb4 ? "lfs" : "lgs",
                        GUINT16_FROM_LE (*(guint16 *)(image + op->index + 2)),
                        GUINT16_FROM_LE (*(guint16 *)(image + op->index)));
              op->index += 4;
            }
          else
            {
              if (op->index + 6 >= size)
                return FALSE;
              g_sprintf (op->mnemonic, "%s %04x:%08x",
                        code == 0xb2 ? "lss" : code == 0xb4 ? "lfs" : "lgs",
                        GUINT16_FROM_LE (*(guint16 *)(image + op->index + 4)),
                        GUINT32_FROM_LE (*(guint32 *)(image + op->index)));
              op->index += 6;
            }
          return TRUE;
        case 0xb6: /* movzx */
        case 0xb7:
        case 0xbe: /* movsx */
        case 0xbf:
          {
            ModRM m;

            op->bit = (code & 1) == 0 ? 8 : 16;
            if (!opcode_modrm (image, size, op,
                                    pope ? 16 : 32, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "%s %s,%s",
                (code & 8) == 0 ? "movzx" : "movsx", m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0xbc: /* bsf */
        case 0xbd: /* bsr */
          {
            ModRM m;

            op->bit = pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "%s %s,%s",
                    (code & 1) == 0 ? "bsf" : "bsr", m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0xc2: /* cmppd/cmpps/cmpsd/cmpss */
        case 0xc6: /* stufpd/stufps/stufsd/stufss */
          {
            ModRM m;

            if ((pope && prep != 0) || prep == 0xf0)
              return FALSE;
            op->bit = 128;
            if (!opcode_modrm (image, size, op,
                    op->bit, op->bit, padr, pseg, &m) || op->index + 1 >= size)
              return FALSE;
            g_sprintf (op->mnemonic, "%s%s%s %s,%s,%02x",
                                    code == 0xc2 ? "cmp" : "stuf",
                                    prep == 0xf2 || prep == 0xf3 ? "s" : "p",
                                    prep == 0xf2 || pope ? "d" : "s",
                                    m.regmem_s, m.regop_s, image[op->index]);
            op->index++;
          }
          return TRUE;
        case 0xc3: /* movnti */
          {
            ModRM m;

            op->bit = 32;
            if (!opcode_modrm (image, size, op,
                            op->bit, op->bit, padr, pseg, &m) || m.mod == 3)
              return FALSE;
            g_sprintf (op->mnemonic, "movnti %s,%s", m.regmem_s, m.regop_s);
          }
          return TRUE;
        case 0xc4: /* pinsrw */
          {
            ModRM m;

            if (!opcode_modrm (image, size, op,
                pope ? 128 : 64, 32, padr, pseg, &m) || op->index + 1 >= size)
              return FALSE;
            g_sprintf (op->mnemonic, "pinsrw %s,%s,%02x",
                                    m.regop_s, m.regmem_s, image[op->index]);
            op->index++;
          }
          return TRUE;
        case 0xc5: /* pextrw */
          {
            ModRM m;

            if (!opcode_modrm (image, size, op,
                                        32, pope ? 128 : 64, padr, pseg, &m)
                                        || m.mod != 3 || op->index + 1 >= size)
              return FALSE;
            g_sprintf (op->mnemonic, "pextrw %s,%s,%02x",
                                    m.regop_s, m.regmem_s, image[op->index]);
            op->index++;
          }
          return TRUE;
        case 0xc7: /* cmpxchg8b */
          {
            ModRM m;

            op->bit = 64;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m)
                                                || m.mod == 3 || m.regop != 1)
              return FALSE;
            g_sprintf (op->mnemonic, "cmpxchg8b %s", m.regmem_s);
            op->index++;
          }
          return TRUE;
        case 0xc8: case 0xc9: case 0xca: case 0xcb: /* bswap */
        case 0xcc: case 0xcd: case 0xce: case 0xcf:
          g_sprintf (op->mnemonic, "bswap %s", regname[2][code & 7]);
          return TRUE;
        case 0xd0: /* addsubpd/addsubps */
          {
            ModRM m;

            if ((pope && prep != 0) || prep == 0xf0)
              return FALSE;
            op->bit = 128;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            if (prep == 0xf2)
              g_sprintf (op->mnemonic, "addsubps %s,%s",
                                                        m.regop_s, m.regmem_s);
            else if (pope)
              g_sprintf (op->mnemonic, "addsubpd %s,%s",
                                                        m.regop_s, m.regmem_s);
            else
              return FALSE;
          }
          return TRUE;
        case 0xd1: /* psrlw */
        case 0xd2: /* psrld */
        case 0xd3: /* psrlq */
        case 0xd4: /* paddq */
        case 0xd5: /* pmullw */
        case 0xd8: /* psubusb */
        case 0xd9: /* psubusw */
        case 0xda: /* pminub */
        case 0xdb: /* pand */
        case 0xdc: /* paddusb */
        case 0xdd: /* paddusw */
        case 0xde: /* pmaxub */
        case 0xdf: /* pandn */
        case 0xe0: /* pavgb */
        case 0xe1: /* psraw */
        case 0xe2: /* psrad */
        case 0xe3: /* pavgw */
        case 0xe4: /* pmulhuw */
        case 0xe5: /* pmulhw */
        case 0xe8: /* psubsb */
        case 0xe9: /* psubsw */
        case 0xea: /* pminsw */
        case 0xeb: /* por */
        case 0xec: /* paddsb */
        case 0xed: /* paddsw */
        case 0xee: /* pmaxsw */
        case 0xef: /* pxor */
        case 0xf1: /* psllw */
        case 0xf2: /* pslld */
        case 0xf3: /* psllq */
        case 0xf4: /* pmuludq */
        case 0xf5: /* pmaddwd */
        case 0xf6: /* psadbw */
        case 0xf8: /* psubb */
        case 0xf9: /* psubw */
        case 0xfa: /* psubd */
        case 0xfb: /* psubq */
        case 0xfc: /* paddb */
        case 0xfd: /* paddw */
        case 0xfe: /* paddd */
          {
            ModRM m;
            const gchar *mnemonic[] = { "psrlw",   "psrld",  "psrlq", /* d1 */
                             "paddq",   "pmullw",  NULL,     NULL,    /* d4 */
                             "psubusb", "psubusw", "pminub", "pand",  /* d8 */
                             "paddusb", "paddusw", "pmaxub", "pandn", /* dc */
                             "pavgb",   "psraw",   "psrad",  "pavgw", /* e0 */
                             "pmulhuw", "pmulhw",  NULL,     NULL,    /* e4 */
                             "psubsb",  "psubsw",  "pminsw", "por",   /* e8 */
                             "paddsb",  "paddsw",  "pmaxsw", "pxor",  /* ec */
                             NULL,      "psllw",   "pslld",  "psllq", /* f0 */
                             "pmuludq", "pmaddwd", "psadbw", NULL,    /* f4 */
                             "psubb",   "psubw",   "psubd",  "psubq", /* f8 */
                             "paddb",   "paddw",   "paddd"};          /* fc */

            op->bit = pope ? 128 : 64;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "%s %s,%s",
                                mnemonic[code - 0xd1], m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0xd6: /* movdq2q/movq/movq2dq */
          {
            ModRM m;

            if ((pope && prep != 0) || prep == 0xf0)
              return FALSE;
            op->bit = prep == 0xf3 ? 64 : 128;
            if (!opcode_modrm (image, size, op,
                                prep == 0xf2 ? 64 : 128, 128, padr, pseg, &m))
              return FALSE;
            if (prep == 0xf2)
              g_sprintf (op->mnemonic, "movdq2q %s,%s", m.regop_s, m.regmem_s);
            else if (prep == 0xf3)
              g_sprintf (op->mnemonic, "movq2dq %s,%s", m.regop_s, m.regmem_s);
            else if (pope)
              g_sprintf (op->mnemonic, "movq %s,%s", m.regmem_s, m.regop_s);
            else
              return FALSE;
          }
          return TRUE;
        case 0xd7: /* pmovmskb */
          {
            ModRM m;

            op->bit = pope ? 128 : 64;
            if (!opcode_modrm (image, size, op,
                                    32, op->bit, padr, pseg, &m) || m.mod != 3)
              return FALSE;
            g_sprintf (op->mnemonic, "pmovmskb %s,%s", m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0xe6: /* cvtdq2pd/cvtpd2dq/cvttpd2dq */
          {
            ModRM m;

            if ((pope && prep != 0) || prep == 0xf0)
              return FALSE;
            op->bit = 128;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "cvt%s%s %s,%s",
                                    prep == 0xf2 || prep == 0xf3 ? "" : "t",
                                    prep == 0xf2 || pope ? "pd2dq" : "dq2pd",
                                                        m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0xe7: /* movntdq/movntq */
        case 0xf7: /* maskmovdqu/maskmovq */
          {
            ModRM m;

            op->bit = pope ? 128 : 64;
            if (!opcode_modrm (image, size, op,
                            op->bit, op->bit, padr, pseg, &m) || m.mod == 3)
              return FALSE;
            g_sprintf (op->mnemonic, "%s %s,%s",
                            (code & 1) == 0 ? pope ? "movntdq"    : "movntq"
                                            : pope ? "maskmovdqu" : "maskmovq",
                                                        m.regmem_s, m.regop_s);
          }
          return TRUE;
        case 0xf0: /* lddqu */
          {
            ModRM m;

            op->bit = 128;
            if (!opcode_modrm (image, size, op,
                            op->bit, op->bit, padr, pseg, &m) || m.mod == 3)
              return FALSE;
            g_sprintf (op->mnemonic, "lddqu %s,%s", m.regop_s, m.regmem_s);
          }
          return TRUE;
        default:
          return FALSE;
      }
  return FALSE;
}


/*  ja:命令解析
    image,PEイメージ
     size,サイズ(サイズが0ならば自動計算)
     base,ベースアドレス(サイズが0ならば自動計算)
      RET,TRUE:正常終了,FALSE:エラー                                        */
gboolean
opcode_parse (const guint8  *image,
              const gsize    size_arg,
              const guint32  base_arg,
              Opcode        *op)
{
  gboolean pope = FALSE, padr = FALSE;
  guint8 code, pseg = 0, prep = 0;
  gsize size;
  guint32 base;

  if (!image || !op)
    return FALSE;
  if (size_arg <= 0)
    {
      size = pe_ioh_get_size_of_image (image);
      base = pe_ioh_get_image_base (image);
    }
  else
    {
      size = size_arg;
      base = base_arg;
    }
  op->mnemonic[0] = '\0';
  op->bit = 0;
  op->type = 0;
  op->relative = op->immediate = op->offset = op->array = (guint32)-1;
  while (0 <= op->index && op->index < size)
    switch ((code = image[op->index++]))
      {
        case 0x00: case 0x01: case 0x02: /* add */
        case 0x03: case 0x04: case 0x05:
        case 0x08: case 0x09: case 0x0a: /* or */
        case 0x0b: case 0x0c: case 0x0d:
        case 0x10: case 0x11: case 0x12: /* adc */
        case 0x13: case 0x14: case 0x15:
        case 0x18: case 0x19: case 0x1a: /* sbb */
        case 0x1b: case 0x1c: case 0x1d:
        case 0x20: case 0x21: case 0x22: /* and */
        case 0x23: case 0x24: case 0x25:
        case 0x28: case 0x29: case 0x2a: /* sub */
        case 0x2b: case 0x2c: case 0x2d:
        case 0x30: case 0x31: case 0x32: /* xor */
        case 0x33: case 0x34: case 0x35:
        case 0x38: case 0x39: case 0x3a: /* cmp */
        case 0x3b: case 0x3c: case 0x3d:
          {
            const static gchar *mnemonic[] = {"add", "or",  "adc", "sbb",
                                              "and", "sub", "xor", "cmp"};
            guint8 type;

            type = (code >> 3) & 7;
            if ((code & 7) == 4)
              {
                if (op->index + 1 >= size)
                  return FALSE;
                g_sprintf (op->mnemonic, "%s al,%02x",
                                            mnemonic[type], image[op->index]);
                op->index++;
              }
            else if ((code & 7) == 5)
              {
                if (pope)
                  {
                    if (op->index + 2 >= size)
                      return FALSE;
                    g_sprintf (op->mnemonic, "%s ax,%04x", mnemonic[type],
                            GUINT16_FROM_LE (*(guint16 *)(image + op->index)));
                    op->index += 2;
                  }
                else
                  {
                    if (op->index + 4 >= size)
                      return FALSE;
                    op->immediate = GUINT32_FROM_LE (*(guint32 *)
                                                        (image + op->index));
                    g_sprintf (op->mnemonic, "%s eax,%08x",
                                                mnemonic[type], op->immediate);
                    op->index += 4;
                  }
              }
            else
              {
                ModRM m;

                op->bit = (code & 1) == 0 ? 8 : pope ? 16 : 32;
                if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
                  return FALSE;
                if ((code & 2) == 0)
                  g_sprintf (op->mnemonic, "%s %s,%s",
                                        mnemonic[type], m.regmem_s, m.regop_s);
                else
                  g_sprintf (op->mnemonic, "%s %s,%s",
                                        mnemonic[type], m.regop_s, m.regmem_s);
              }
          }
          return TRUE;
        case 0x06: /* push es */
          g_strcpy (op->mnemonic, "push es");
          return TRUE;
        case 0x07: /* pop es */
          g_strcpy (op->mnemonic, "pop es");
          return TRUE;
        case 0x0e: /* push cs */
          g_strcpy (op->mnemonic, "push cs");
          return TRUE;
        case 0x0f:
          return opcode_extend (image, size, base, op, pope, padr, pseg, prep);
        case 0x16: /* push ss */
          g_strcpy (op->mnemonic, "push ss");
          return TRUE;
        case 0x17: /* pop ss */
          g_strcpy (op->mnemonic, "pop ss");
          return TRUE;
        case 0x1e: /* push ds */
          g_strcpy (op->mnemonic, "push ds");
          return TRUE;
        case 0x1f: /* pop ds */
          g_strcpy (op->mnemonic, "pop ds");
          return TRUE;
        case 0x26: /* es */
        case 0x2e: /* cs */
        case 0x36: /* ss */
        case 0x3e: /* ds */
        case 0x64: /* fs */
        case 0x65: /* gs */
          if (pseg != 0)
            return FALSE;
          pseg = code;
          break;
        case 0x27: /* daa */
          g_strcpy (op->mnemonic, "daa");
          return TRUE;
        case 0x2f: /* das */
          g_strcpy (op->mnemonic, "das");
          return TRUE;
        case 0x37: /* aaa */
          g_strcpy (op->mnemonic, "aaa");
          return TRUE;
        case 0x3f: /* aas */
          g_strcpy (op->mnemonic, "aas");
          return TRUE;
        case 0x40: case 0x41: case 0x42: case 0x43: /* inc */
        case 0x44: case 0x45: case 0x46: case 0x47:
        case 0x48: case 0x49: case 0x4a: case 0x4b: /* dec */
        case 0x4c: case 0x4d: case 0x4e: case 0x4f:
        case 0x50: case 0x51: case 0x52: case 0x53: /* push */
        case 0x54: case 0x55: case 0x56: case 0x57:
        case 0x58: case 0x59: case 0x5a: case 0x5b: /* pop */
        case 0x5c: case 0x5d: case 0x5e: case 0x5f:
          {
            const static gchar *mnemonic[] = {"inc", "dec", "push", "pop"};

            g_sprintf (op->mnemonic, "%s %s", mnemonic[(code >> 3) - 8],
                                            regname[pope ? 1 : 2][code & 7]);
          }
          return TRUE;
        case 0x60: /* pusha/pushad */
          g_strcpy (op->mnemonic, pope ? "pusha" : "pushad");
          return TRUE;
        case 0x61: /* popa/popad */
          g_strcpy (op->mnemonic, pope ? "popa" : "popad");
          return TRUE;
        case 0x62: /* bound */
          {
            ModRM m;

            if (!opcode_modrm (image, size, op,
                            pope ? 16 : 32, pope ? 32 : 64, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "bound %s,%s", m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0x63: /* arpl */
          {
            ModRM m;

            op->bit = 16;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "arpl %s,%s", m.regmem_s, m.regop_s);
          }
          return TRUE;
        case 0x66:
          if (pope)
            return FALSE;
          pope = TRUE;
          break;
        case 0x67:
          if (padr)
            return FALSE;
          padr = TRUE;
          break;
        case 0x68: /* push */
          if (pope)
            {
              if (op->index + 2 >= size)
                return FALSE;
              g_sprintf (op->mnemonic, "push %04x",
                            GUINT16_FROM_LE (*(guint16 *)(image + op->index)));
              op->index += 2;
            }
          else
            {
              if (op->index + 4 >= size)
                return FALSE;
              op->immediate = GUINT32_FROM_LE (*(guint32 *)
                                                        (image + op->index));
              g_sprintf (op->mnemonic, "push %08x", op->immediate);
              op->index += 4;
            }
          return TRUE;
        case 0x69: /* imul */
        case 0x6b:
          {
            ModRM m;

            op->bit = pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            if (code == 0x6b)
              {
                if (op->index + 1 >= size)
                  return FALSE;
                g_sprintf (op->mnemonic, "imul %s,%s,%02x",
                                    m.regop_s, m.regmem_s, image[op->index]);
                op->index++;
              }
            else if (pope)
              {
                if (op->index + 2 >= size)
                  return FALSE;
                g_sprintf (op->mnemonic, "imul %s,%s,%04x",
                            m.regop_s, m.regmem_s,
                            GUINT16_FROM_LE (*(guint16 *)(image + op->index)));
                op->index += 2;
              }
            else
              {
                if (op->index + 4 >= size)
                  return FALSE;
                g_sprintf (op->mnemonic, "imul %s,%s,%08x",
                            m.regop_s, m.regmem_s,
                            GUINT32_FROM_LE (*(guint32 *)(image + op->index)));
                op->index += 4;
              }
          }
          return TRUE;
        case 0x6a: /* push */
          if (op->index + 1 >= size)
            return FALSE;
          g_sprintf (op->mnemonic, "push %02x", image[op->index]);
          op->index++;
          return TRUE;
        case 0x6c: /* insb */
        case 0x6d: /* insw/insd */
          g_sprintf (op->mnemonic, "%s%sins%s",
                        segment (pseg),
                        prep == 0xf2 ? "repnz " : prep == 0xf3 ? "rep " : "",
                        (code & 1) == 0 ? "b" : pope ? "w" : "d");
          return TRUE;
        case 0x6e: /* outsb */
        case 0x6f: /* outsw/outsd */
          g_sprintf (op->mnemonic, "%s%souts%s",
                        segment (pseg),
                        prep == 0xf2 ? "repnz " : prep == 0xf3 ? "rep " : "",
                        (code & 1) == 0 ? "b" : pope ? "w" : "d");
          return TRUE;
        case 0x70: case 0x71: case 0x72: case 0x73: /* jcc */
        case 0x74: case 0x75: case 0x76: case 0x77:
        case 0x78: case 0x79: case 0x7a: case 0x7b:
        case 0x7c: case 0x7d: case 0x7e: case 0x7f:
          if (op->index + 1 >= size)
            return FALSE;
          op->relative = op->index + 1 + (gint8)image[op->index] + base;
          g_sprintf (op->mnemonic, "j%s %08x", cc[code - 0x70], op->relative);
          op->index++;
          op->type = OPCODE_TYPE_JMP;
          return TRUE;
        case 0x80: /* add/or/adc/sbb/and/sub/xor/cmp */
        case 0x81:
        case 0x82:
        case 0x83:
          {
            gchar d[9];
            ModRM m;
            const static gchar *mnemonic[] = {"add", "or",  "adc", "sbb",
                                              "and", "sub", "xor", "cmp"};

            op->bit = (code & 1) == 0 ? 8 : pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            if (code != 0x81)
              {
                if (op->index + 1 >= size)
                  return FALSE;
                g_sprintf (d, "%02x", image[op->index]);
                op->index++;
              }
            else if (pope)
              {
                if (op->index + 2 >= size)
                  return FALSE;
                g_sprintf (d, "%04x", GUINT16_FROM_LE (*(guint16 *)
                                                        (image + op->index)));
                op->index += 2;
              }
            else
              {
                if (op->index + 4 >= size)
                  return FALSE;
                g_sprintf (d, "%08x", GUINT32_FROM_LE (*(guint32 *)
                                                        (image + op->index)));
                op->index += 4;
              }
            g_sprintf (op->mnemonic, "%s %s,%s",
                                            mnemonic[m.regop], m.regmem_s, d);
          }
          return TRUE;
        case 0x84: /* test */
        case 0x85:
        case 0x86: /* xchg */
        case 0x87:
          {
            ModRM m;

            op->bit = (code & 1) == 0 ? 8 : pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "%s %s,%s",
                    (code & 2) == 0 ? "test" : "xchg", m.regmem_s, m.regop_s);
          }
          return TRUE;
        case 0x88: /* mov */
        case 0x89:
        case 0x8a:
        case 0x8b:
          {
            ModRM m;

            op->bit = (code & 1) == 0 ? 8 : pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            if ((code & 2) == 0)
              g_sprintf (op->mnemonic, "mov %s,%s", m.regmem_s, m.regop_s);
            else
              g_sprintf (op->mnemonic, "mov %s,%s", m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0x8c: /* mov */
        case 0x8e:
          {
            ModRM m;

            op->bit = 16;
            if (!opcode_modrm (image, size, op, 16, 16, padr, pseg, &m))
              return FALSE;
            if (code == 0x8c)
              g_sprintf (op->mnemonic, "mov %s,%s",
                                            m.regmem_s, regname[5][m.regop]);
            else
              g_sprintf (op->mnemonic, "mov %s,%s",
                                            regname[5][m.regop], m.regmem_s);
          }
          return TRUE;
        case 0x8d: /* lea */
          {
            ModRM m;

            op->bit = pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                            op->bit, op->bit, padr, pseg, &m) || m.mod == 3)
              return FALSE;
            g_sprintf (op->mnemonic, "lea %s,%s", m.regop_s, m.regmem_s);
          }
          return TRUE;
        case 0x8f: /* pop */
          {
            ModRM m;

            op->bit = pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            g_sprintf (op->mnemonic, "pop %s", m.regmem_s);
          }
          return TRUE;
        case 0x90: /* nop */
          g_strcpy (op->mnemonic, prep == 0xf3 ? "pause" : "nop");
          return TRUE;
        case 0x91: case 0x92: case 0x93: /* xchg */
        case 0x94: case 0x95: case 0x96: case 0x97:
          g_sprintf (op->mnemonic, "xchg %s,%s",
                        pope ? "ax" : "eax", regname[pope ? 1 : 2][code & 7]);
          return TRUE;
        case 0x98: /* cbw/cwde */
          g_strcpy (op->mnemonic, pope ? "cbw" : "cwde");
          return TRUE;
        case 0x99: /* cwd/cdq */
          g_strcpy (op->mnemonic, pope ? "cwd" : "cdq");
          return TRUE;
        case 0x9a: /* call/callf */
          if (pope)
            {
              if (op->index + 4 >= size)
                return FALSE;
              g_sprintf (op->mnemonic, "callf %04x:%04x",
                        GUINT16_FROM_LE (*(guint16 *)(image + op->index + 2)),
                        GUINT16_FROM_LE (*(guint16 *)(image + op->index)));
              op->index += 4;
            }
          else
            {
              if (op->index + 6 >= size)
                return FALSE;
              g_sprintf (op->mnemonic, "callf %04x:%08x",
                        GUINT16_FROM_LE (*(guint16 *)(image + op->index + 4)),
                        GUINT32_FROM_LE (*(guint32 *)(image + op->index)));
              op->index += 6;
            }
          op->type = OPCODE_TYPE_CALL;
          return TRUE;
        case 0x9b: /* wait */
          g_strcpy (op->mnemonic, "wait");
          return TRUE;
        case 0x9c: /* pushf/pushfd */
          g_strcpy (op->mnemonic, pope ? "pushf" : "pushfd");
          return TRUE;
        case 0x9d: /* popf/popfd */
          g_strcpy (op->mnemonic, pope ? "popf" : "popfd");
          return TRUE;
        case 0x9e: /* sahf */
          g_strcpy (op->mnemonic, "sahf");
          return TRUE;
        case 0x9f: /* lahf */
          g_strcpy (op->mnemonic, "lahf");
          return TRUE;
        case 0xa0: /* mov */
        case 0xa1:
        case 0xa2:
        case 0xa3:
          {
            gchar d[9];
            const gchar *r;

            if (padr)
              {
                if (op->index + 2 >= size)
                  return FALSE;
                g_sprintf (d, "%04x",
                            GUINT16_FROM_LE (*(guint16 *)(image + op->index)));
                op->index += 2;
              }
            else
              {
                if (op->index + 4 >= size)
                  return FALSE;
                op->offset = GUINT32_FROM_LE (*(guint32 *)(image + op->index));
                g_sprintf (d, "%08x", op->offset);
                op->index += 4;
              }
            r = (code & 1) == 0 ? "al" : pope ? "ax" : "eax";
            if ((code & 2) == 0)
              g_sprintf (op->mnemonic, "mov %s,%s[%s]", r, segment (pseg), d);
            else
              g_sprintf (op->mnemonic, "mov %s[%s],%s", segment (pseg), d, r);
          }
          return TRUE;
        case 0xa4: /* movsb */
        case 0xa5: /* movsw/movsd */
          g_sprintf (op->mnemonic, "%s%smovs%s",
                        segment (pseg),
                        prep == 0xf2 ? "repnz " : prep == 0xf3 ? "rep " : "",
                        (code & 1) == 0 ? "b" : pope ? "w" : "d");
          return TRUE;
        case 0xa6: /* cmpsb */
        case 0xa7: /* cmpsw/cmpsd */
          g_sprintf (op->mnemonic, "%s%scmps%s",
                        segment (pseg),
                        prep == 0xf2 ? "repne " : prep == 0xf3 ? "repe " : "",
                        (code & 1) == 0 ? "b" : pope ? "w" : "d");
          return TRUE;
        case 0xa8: /* test */
        case 0xa9:
          if ((code & 1) == 0)
            {
              if (op->index + 1 >= size)
                return FALSE;
              g_sprintf (op->mnemonic, "test ax,%02x", image[op->index]);
              op->index++;
            }
          else if (pope)
            {
              if (op->index + 2 >= size)
                return FALSE;
              g_sprintf (op->mnemonic, "test ax,%04x",
                            GUINT16_FROM_LE (*(guint16 *)(image + op->index)));
              op->index += 2;
            }
          else
            {
              if (op->index + 4 >= size)
                return FALSE;
              g_sprintf (op->mnemonic, "test eax,%08x",
                            GUINT32_FROM_LE (*(guint32 *)(image + op->index)));
              op->index += 4;
            }
          return TRUE;
        case 0xaa: /* stosb */
        case 0xab: /* stosw/stosd */
          g_sprintf (op->mnemonic, "%s%sstos%s",
                        segment (pseg),
                        prep == 0xf2 ? "repnz " : prep == 0xf3 ? "rep " : "",
                        (code & 1) == 0 ? "b" : pope ? "w" : "d");
          return TRUE;
        case 0xac: /* lodsb */
        case 0xad: /* lodsw/lodsd */
          g_sprintf (op->mnemonic, "%s%slods%s",
                        segment (pseg),
                        prep == 0xf2 ? "repnz " : prep == 0xf3 ? "rep " : "",
                        (code & 1) == 0 ? "b" : pope ? "w" : "d");
          return TRUE;
        case 0xae: /* scasb */
        case 0xaf: /* scasw/scasd */
          g_sprintf (op->mnemonic, "%s%sscas%s",
                        segment (pseg),
                        prep == 0xf2 ? "repne " : prep == 0xf3 ? "repe " : "",
                        (code & 1) == 0 ? "b" : pope ? "w" : "d");
          return TRUE;
        case 0xb0: case 0xb1: case 0xb2: case 0xb3: /* mov */
        case 0xb4: case 0xb5: case 0xb6: case 0xb7:
        case 0xb8: case 0xb9: case 0xba: case 0xbb:
        case 0xbc: case 0xbd: case 0xbe: case 0xbf:
          if ((code & 8) == 0)
            {
              if (op->index + 1 >= size)
                return FALSE;
              g_sprintf (op->mnemonic, "mov %s,%02x",
                                    regname[0][code & 7], image[op->index]);
              op->index++;
            }
          else if (pope)
            {
              if (op->index + 2 >= size)
                return FALSE;
              g_sprintf (op->mnemonic, "mov %s,%04x",
                            regname[1][code & 7],
                            GUINT16_FROM_LE (*(guint16 *)(image + op->index)));
              op->index += 2;
            }
          else
            {
              if (op->index + 4 >= size)
                return FALSE;
              op->immediate = GUINT32_FROM_LE (*(guint32 *)
                                                        (image + op->index));
              g_sprintf (op->mnemonic, "mov %s,%08x",
                                        regname[2][code & 7], op->immediate);
              op->index += 4;
            }
          return TRUE;
        case 0xc0: case 0xc1: /* rol/ror/rcl/rcr/shl/shr/sal/sar */
        case 0xd0: case 0xd1: case 0xd2: case 0xd3:
          {
            gchar d[3];
            ModRM m;
            const static gchar *mnemonic[] = {"rol", "ror", "rcl", "rcr",
                                              "shl", "shr", "sal", "sar"};

            op->bit = (code & 1) == 0 ? 8 : pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            if (code == 0xc0 || code == 0xc1)
              {
                if (op->index + 1 >= size)
                    return FALSE;
                g_sprintf (d, "%02x", image[op->index]);
                op->index++;
              }
            else if (code == 0xd0 || code == 0xd1)
              {
                g_strcpy (d, "1");
              }
            else
              {
                g_strcpy (d, "cl");
              }
            g_sprintf (op->mnemonic, "%s %s,%s",
                                            mnemonic[m.regop], m.regmem_s, d);
          }
          return TRUE;
        case 0xc2: /* retn */
          if (op->index + 2 >= size)
            return FALSE;
          g_sprintf (op->mnemonic, "retn %04x",
                            GUINT16_FROM_LE (*(guint16 *)(image + op->index)));
          op->index += 2;
          op->type = OPCODE_TYPE_EXIT;
          return TRUE;
        case 0xc3: /* ret */
          g_strcpy (op->mnemonic, "ret");
          op->type = OPCODE_TYPE_EXIT;
          return TRUE;
        case 0xc4: /* les */
        case 0xc5: /* lds */
          if (pope)
            {
              if (op->index + 4 >= size)
                return FALSE;
              g_sprintf (op->mnemonic, "%s %04x:%04x",
                        code == 0xc4 ? "les" : "lds",
                        GUINT16_FROM_LE (*(guint16 *)(image + op->index + 2)),
                        GUINT16_FROM_LE (*(guint16 *)(image + op->index)));
              op->index += 4;
            }
          else
            {
              if (op->index + 6 >= size)
                return FALSE;
              g_sprintf (op->mnemonic, "%s %04x:%08x",
                        code == 0xc4 ? "les" : "lds",
                        GUINT16_FROM_LE (*(guint16 *)(image + op->index + 4)),
                        GUINT32_FROM_LE (*(guint32 *)(image + op->index)));
              op->index += 6;
            }
          return TRUE;
        case 0xc6: /* mov */
        case 0xc7:
          {
            ModRM m;

            op->bit = (code & 1) == 0 ? 8 : pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            if ((code & 1) == 0)
              {
                if (op->index + 1 >= size)
                  return FALSE;
                g_sprintf (op->mnemonic, "mov %s,%02x",
                                                m.regmem_s, image[op->index]);
                op->index++;
              }
            else if (pope)
              {
                if (op->index + 2 >= size)
                  return FALSE;
                g_sprintf (op->mnemonic, "mov %s,%04x", m.regmem_s,
                            GUINT16_FROM_LE (*(guint16 *)(image + op->index)));
                op->index += 2;
              }
            else
              {
                if (op->index + 4 >= size)
                  return FALSE;
                op->immediate = GUINT32_FROM_LE (*(guint32 *)
                                                        (image + op->index));
                g_sprintf (op->mnemonic, "mov %s,%08x",
                                                    m.regmem_s, op->immediate);
                op->index += 4;
              }
          }
          return TRUE;
        case 0xc8: /* enter */
          if (op->index + 3 >= size)
            return FALSE;
          g_sprintf (op->mnemonic, "enter %04x,%02x",
                            GUINT16_FROM_LE (*(guint16 *)(image + op->index)),
                            image[op->index + 2]);
          op->index += 3;
          return TRUE;
        case 0xc9: /* leave */
          g_strcpy (op->mnemonic, "leave");
          return TRUE;
        case 0xca: /* retnf */
          if (op->index + 2 >= size)
            return FALSE;
          g_sprintf (op->mnemonic, "retnf %04x",
                            GUINT16_FROM_LE (*(guint16 *)(image + op->index)));
          op->index += 2;
          op->type = OPCODE_TYPE_EXIT;
          return TRUE;
        case 0xcb: /* retf */
          g_strcpy (op->mnemonic, "retf");
          op->type = OPCODE_TYPE_EXIT;
          return TRUE;
        case 0xcc: /* int 3 */
          g_strcpy (op->mnemonic, "int 3");
          op->type = OPCODE_TYPE_EXIT;
          return TRUE;
        case 0xcd: /* int */
          if (op->index + 1 >= size)
            return FALSE;
          g_sprintf (op->mnemonic, "int %02x", image[op->index]);
          op->index++;
          return TRUE;
        case 0xce: /* into */
          g_strcpy (op->mnemonic, "into");
          return TRUE;
        case 0xcf: /* iret/iretd */
          g_strcpy (op->mnemonic, pope ? "iret" : "iretd");
          op->type = OPCODE_TYPE_EXIT;
          return TRUE;
        case 0xd4: /* aam */
          if (op->index + 1 >= size)
            return FALSE;
          g_sprintf (op->mnemonic, "aam %02x", image[op->index]);
          op->index++;
          return TRUE;
        case 0xd5: /* aad */
          if (op->index + 1 >= size)
            return FALSE;
          g_sprintf (op->mnemonic, "aad %02x", image[op->index]);
          op->index++;
          return TRUE;
        case 0xd6: /* setalc */
          g_strcpy (op->mnemonic, "setalc");
          return TRUE;
        case 0xd7: /* xlat */
          g_strcpy (op->mnemonic, "xlat");
          return TRUE;
        case 0xd8: /* fadd/fmul/fcom/fcomp/fsub/fsubr/fdiv/fdivr */
          {
            ModRM m;
            const gchar *mnemonic[] = {"fadd", "fmul",  "fcom", "fcomp",
                                       "fsub", "fsubr", "fdiv", "fdivr"};

            if (!opcode_modrm (image, size, op, 0, 0, padr, pseg, &m))
              return FALSE;
            if (m.mod < 3)
              g_sprintf (op->mnemonic, "%s %s", mnemonic[m.regop], m.regmem_s);
            else
              g_sprintf (op->mnemonic, "%s st(0),st(%d)",
                                                mnemonic[m.regop], m.regmem);
          }
          return TRUE;
        case 0xd9: /* fld/fst/fstp/fldenv/fldcw/fnstenv/fnstcw
                      fxch/fnop/fchs/fabs/ftst/fxam
                      fld1/fldl2t/fldl2e/fldpi/fldlg2/fldln2/fldz
                      f2xm1/fyl2x/fptan/fpatan/fxtract/fprem1/fdecstp/fincstp
                      fprem/fyl2xp1/fsqrt/fsincos/frndint/fscale/fsin/fcos */
          {
            ModRM m;

            if (!opcode_modrm (image, size, op, 0, 0, padr, pseg, &m))
              return FALSE;
            if (m.mod < 3)
              {
                const gchar *mnemonic[] = {
                                    "fld",    NULL,    "fst",    "fstp",
                                    "fldenv", "fldcw", "fnstenv", "fnstcw"};

                if (!mnemonic[m.regop])
                  return FALSE;
                g_sprintf (op->mnemonic, "%s %s",
                                                mnemonic[m.regop], m.regmem_s);
              }
            else if (m.regop == 0)
              {
                g_sprintf (op->mnemonic, "fld st(0),st(%d)", m.regmem);
              }
            else if (m.regop == 1)
              {
                g_sprintf (op->mnemonic, "fxch st(0),st(%d)", m.regmem);
              }
            else
              {
                const gchar *mnemonic[] = {
                                    "fnop",    NULL,      NULL,      NULL,
                                    NULL,      NULL,      NULL,      NULL,
                                    NULL,      NULL,      NULL,      NULL,
                                    NULL,      NULL,      NULL,      NULL,
                                    "fchs",    "fabs",    NULL,      NULL,
                                    "ftst",    "fxam",    NULL,      NULL,
                                    "fld1",    "fldl2t",  "fldl2e",  "fldpi",
                                    "fldlg2",  "fldln2",  "fldz",    NULL,
                                    "f2xm1",   "fyl2x",   "fptan",   "fpatan",
                                    "fxtract", "fprem1",  "fdecstp", "fincstp",
                                    "fprem",   "fyl2xp1", "fsqrt",   "fsincos",
                                    "frndint", "fscale",  "fsin",    "fcos"};

                if (!mnemonic[(m.regop - 2) * 8 + m.regmem])
                  return FALSE;
                g_strcpy (op->mnemonic,
                                    mnemonic[(m.regop - 2) * 8 + m.regmem]);
              }
          }
          return TRUE;
        case 0xda: /* fiadd/fimul/ficom/ficomp/fisub/fisubr/fidiv/fidivr
                      fcmovb/fcmove/fcmovbe/fcmovu/fucompp */
          {
            ModRM m;

            if (!opcode_modrm (image, size, op, 0, 0, padr, pseg, &m))
              return FALSE;
            if (m.mod < 3)
              {
                const gchar *mnemonic[] = {
                                        "fiadd", "fimul",  "ficom", "ficomp",
                                        "fisub", "fisubr", "fidiv", "fidivr"};

                g_sprintf (op->mnemonic, "%s %s",
                                                mnemonic[m.regop], m.regmem_s);
              }
            else if (m.regop < 4)
              {
                const gchar *mnemonic[] = {
                                    "fcmovb" ,"fcmove", "fcmovbe", "fcmovu"};

                g_sprintf (op->mnemonic, "%s st(0),st(%d)",
                                                mnemonic[m.regop], m.regmem);
              }
            else if (m.regop == 5 && m.regmem == 0)
              {
                g_strcpy (op->mnemonic, "fucompp");
              }
            else
              {
                return FALSE;
              }
          }
          return TRUE;
        case 0xdb: /* fild/fist/fistp/fld/fstp/fnclex/fninit
                      fcmovnb/fcmovne/fcmovnbe/fcmovnu/fucomi/fcomi */
          {
            ModRM m;

            if (!opcode_modrm (image, size, op, 0, 0, padr, pseg, &m))
              return FALSE;
            if (m.mod < 3)
              {
                const gchar *mnemonic[] = {"fild", NULL,  "fist", "fistp",
                                           NULL,   "fld", NULL,   "fstp"};

                if (!mnemonic[m.regop])
                  return FALSE;
                g_sprintf (op->mnemonic, "%s %s",
                                                mnemonic[m.regop], m.regmem_s);
              }
            else if (m.regop == 4 && m.regmem == 2)
              {
                g_strcpy (op->mnemonic, "fnclex");
              }
            else if (m.regop == 4 && m.regmem == 3)
              {
                g_strcpy (op->mnemonic, "fninit");
              }
            else
              {
                const gchar *mnemonic[] = {
                                "fcmovnb", "fcmovne", "fcmovnbe", "fcmovnu",
                                NULL,      "fucomi",  "fcomi",    NULL};

                if (!mnemonic[m.regop])
                  return FALSE;
                g_sprintf (op->mnemonic, "%s st(0),st(%d)",
                                                mnemonic[m.regop], m.regmem);
              }
          }
          return TRUE;
        case 0xdc: /* fadd/fmul/fcom/fcomp/fsub/fsubr/fdiv/fdivr */
          {
            ModRM m;

            if (!opcode_modrm (image, size, op, 0, 0, padr, pseg, &m))
              return FALSE;
            if (m.mod < 3)
              {
                const gchar *mnemonic[] = {"fadd", "fmul",  "fcom", "fcomp",
                                           "fsub", "fsubr", "fdiv", "fdivr"};

                g_sprintf (op->mnemonic, "%s %s",
                                                mnemonic[m.regop], m.regmem_s);
              }
            else
              {
                const gchar *mnemonic[] = {"fadd",  "fmul", NULL,    NULL,
                                           "fsubr", "fsub", "fdivr", "fdiv"};

                if (!mnemonic[m.regop])
                  return FALSE;
                g_sprintf (op->mnemonic, "%s st(%d),st(0)",
                                                mnemonic[m.regop], m.regmem);
              }
          }
          return TRUE;
        case 0xdd: /* fld/fst/fstp/frstor/fnsave/fnstsw/ffree/fucomp */
          {
            ModRM m;

            if (!opcode_modrm (image, size, op, 0, 0, padr, pseg, &m))
              return FALSE;
            if (m.mod < 3)
              {
                const gchar *mnemonic[] = {"fld",    NULL, "fst",    "fstp",
                                           "frstor", NULL, "fnsave", "fnstsw"};

                if (!mnemonic[m.regop])
                  return FALSE;
                g_sprintf (op->mnemonic, "%s %s",
                                                mnemonic[m.regop], m.regmem_s);
              }
            else
              {
                const gchar *mnemonic[] = {"ffree", NULL,     "fst", "fstp",
                                           NULL,    "fucomp", NULL,  NULL,};

                if (mnemonic[m.regop])
                  g_sprintf (op->mnemonic, "%s st(%d)",
                                                mnemonic[m.regop], m.regmem);
                else if (m.regop == 4)
                  g_sprintf (op->mnemonic, "fucom st(%d),st(0)", m.regmem);
                else
                  return FALSE;
              }
          }
          return TRUE;
        case 0xde: /* fiadd/fimul/ficom/ficomp/fisub/fisubr/fidiv/fidivr
                      faddp/fmulp/fsubrp/fsubp/fdivrp/fdivp */
          {
            ModRM m;

            if (!opcode_modrm (image, size, op, 0, 0, padr, pseg, &m))
              return FALSE;
            if (m.mod < 3)
              {
                const gchar *mnemonic[] = {
                                        "fiadd", "fimul",  "ficom", "ficomp",
                                        "fisub", "fisubr", "fidiv", "fidivr"};

                g_sprintf (op->mnemonic, "%s %s",
                                                mnemonic[m.regop], m.regmem_s);
              }
            else if (m.regop == 3 && m.regmem == 1)
              {
                g_strcpy (op->mnemonic, "fcompp");
              }
            else
              {
                const gchar *mnemonic[] = {
                                        "faddp",  "fmulp", NULL,     NULL,
                                        "fsubrp", "fsubp", "fdivrp", "fdivp"};

                if (!mnemonic[m.regop])
                  return FALSE;
                g_sprintf (op->mnemonic, "%s st(%d),st(0)",
                                                mnemonic[m.regop], m.regmem);
              }
          }
          return TRUE;
        case 0xdf: /* fild/fist/fistp/fbld/fild/fbstp/fistp */
          {
            ModRM m;

            if (!opcode_modrm (image, size, op, 0, 0, padr, pseg, &m))
              return FALSE;
            if (m.mod < 3)
              {
                const gchar *mnemonic[] = {"fild", NULL,  "fist",   "fistp",
                                           "fbld", "fild", "fbstp", "fistp"};

                if (!mnemonic[m.regop])
                  return FALSE;
                g_sprintf (op->mnemonic, "%s %s",
                                                mnemonic[m.regop], m.regmem_s);
              }
            else if (m.regop == 4 && m.regmem == 0)
              {
                g_strcpy (op->mnemonic, "fstsw ax");
              }
            else if (m.regop == 5 || m.regop == 6)
              {
                g_sprintf (op->mnemonic, "%s st(0),st(%d)",
                                m.regop == 5 ? "fucomip" : "fcomip", m.regmem);
              }
            else
              {
                return FALSE;
              }
          }
          return TRUE;
        case 0xe0: /* loopnz */
        case 0xe1: /* loopz */
        case 0xe2: /* loop */
          {
            const gchar *mnemonic[] = {"loopnz", "loopz", "loop"};

            if (op->index + 1 >= size)
              return FALSE;
            op->relative = op->index + 1 + (gint8)image[op->index] + base;
            g_sprintf (op->mnemonic, "%s %08x",
                                        mnemonic[code - 0xe0], op->relative);
            op->type = OPCODE_TYPE_JMP;
            op->index++;
          }
          return TRUE;
        case 0xe3: /* jcxz/jecxz */
          if (op->index + 1 >= size)
            return FALSE;
          op->relative = op->index + 1 + (gint8)image[op->index] + base;
          g_sprintf (op->mnemonic, "%s %08x",
                                        pope ? "jcxz" : "jecxz", op->relative);
          op->type = OPCODE_TYPE_JMP;
          op->index++;
          return TRUE;
        case 0xe4: /* in */
        case 0xe5:
          if (op->index + 1 >= size)
            return FALSE;
          g_sprintf (op->mnemonic, "in %s,%02x",
                                (code & 1) == 0 ? "al" : pope ? "ax" : "eax",
                                                            image[op->index]);
          op->index++;
          return TRUE;
        case 0xe6: /* out */
        case 0xe7:
          if (op->index + 1 >= size)
            return FALSE;
          g_sprintf (op->mnemonic, "out %02x,%s", image[op->index],
                                (code & 1) == 0 ? "al" : pope ? "ax" : "eax");
          op->index++;
          return TRUE;
        case 0xe8: /* call */
        case 0xe9: /* jmp */
          if (pope)
            {
              if (op->index + 2 >= size)
                return FALSE;
              op->relative = GINT16_FROM_LE (*(gint16 *)(image + op->index));
              op->index += 2;
            }
          else
            {
              if (op->index + 4 >= size)
                return FALSE;
              op->relative = GINT32_FROM_LE (*(gint32 *)(image + op->index));
              op->index += 4;
            }
          op->relative += op->index + base;
          if (code == 0xe8)
            {
              g_sprintf (op->mnemonic, "call %08x", op->relative);
              op->type = OPCODE_TYPE_CALL;
            }
          else
            {
              g_sprintf (op->mnemonic, "jmp %08x", op->relative);
              op->type = OPCODE_TYPE_JMP | OPCODE_TYPE_EXIT;
            }
          return TRUE;
        case 0xea: /* jmp */
          if (pope)
            {
              if (op->index + 4 >= size)
                return FALSE;
              g_sprintf (op->mnemonic, "jmp %04x:%04x",
                        GUINT16_FROM_LE (*(guint16 *)(image + op->index + 2)),
                        GUINT16_FROM_LE (*(guint16 *)(image + op->index)));
              op->index += 4;
            }
          else
            {
              if (op->index + 6 >= size)
                return FALSE;
              g_sprintf (op->mnemonic, "jmp %04x:%08x",
                        GUINT16_FROM_LE (*(guint16 *)(image + op->index + 4)),
                        GUINT32_FROM_LE (*(guint32 *)(image + op->index)));
              op->index += 6;
            }
          op->type = OPCODE_TYPE_JMP | OPCODE_TYPE_EXIT;
          return TRUE;
        case 0xeb: /* jmp */
          op->relative = op->index + 1 + (gint8)image[op->index] + base;
          g_sprintf (op->mnemonic, "jmp %08x", op->relative);
          op->index++;
          op->type = OPCODE_TYPE_JMP | OPCODE_TYPE_EXIT;
          return TRUE;
        case 0xec: /* in */
        case 0xed:
          g_sprintf (op->mnemonic, "in %s,dx",
                                (code & 1) == 0 ? "al" : pope ? "ax" : "eax");
          return TRUE;
        case 0xee: /* out */
        case 0xef:
          g_sprintf (op->mnemonic, "out dx,%s",
                                (code & 1) == 0 ? "al" : pope ? "ax" : "eax");
          return TRUE;
        case 0xf0: /* lock */
        case 0xf2: /* repnz */
        case 0xf3: /* rep/repz */
          if (prep != 0)
            return FALSE;
          prep = code;
          break;
        case 0xf1: /* icebp */
          g_strcpy (op->mnemonic, "icebp");
          return TRUE;
        case 0xf4: /* hlt */
          g_strcpy (op->mnemonic, "hlt");
          return TRUE;
        case 0xf5: /* cmc */
          g_strcpy (op->mnemonic, "cmc");
          return TRUE;
        case 0xf6: /* test/not/neg/mul/imul/div/idiv */
        case 0xf7:
          {
            ModRM m;

            op->bit = (code & 1) == 0 ? 8 : pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            if (m.regop == 0)
              {
                if ((code & 1) == 0)
                  {
                    if (op->index + 1 >= size)
                      return FALSE;
                    g_sprintf (op->mnemonic, "test %s,%02x",
                                                m.regmem_s, image[op->index]);
                    op->index++;
                  }
                else if (pope)
                  {
                    if (op->index + 2 >= size)
                      return FALSE;
                    g_sprintf (op->mnemonic, "test %s,%04x", m.regmem_s,
                            GUINT16_FROM_LE (*(guint16 *)(image + op->index)));
                    op->index += 2;
                  }
                else
                  {
                    if (op->index + 4 >= size)
                      return FALSE;
                    g_sprintf (op->mnemonic, "test %s,%08x", m.regmem_s,
                            GUINT32_FROM_LE (*(guint32 *)(image + op->index)));
                    op->index += 4;
                  }
              }
            else if (m.regop == 1)
              {
                return FALSE;
              }
            else
              {
                const gchar *mnemonic[] = {"not", "neg",
                                           "mul", "imul", "div", "idiv"};

                g_sprintf (op->mnemonic, "%s %s",
                                            mnemonic[m.regop - 2], m.regmem_s);
              }
          }
          return TRUE;
        case 0xf8: /* clc */
          g_strcpy (op->mnemonic, "clc");
          return TRUE;
        case 0xf9: /* stc */
          g_strcpy (op->mnemonic, "stc");
          return TRUE;
        case 0xfa: /* cli */
          g_strcpy (op->mnemonic, "cli");
          return TRUE;
        case 0xfb: /* sti */
          g_strcpy (op->mnemonic, "sti");
          return TRUE;
        case 0xfc: /* cld */
          g_strcpy (op->mnemonic, "cld");
          return TRUE;
        case 0xfd: /* std */
          g_strcpy (op->mnemonic, "std");
          return TRUE;
        case 0xfe: /* inc/dec */
          {
            ModRM m;

            op->bit = 8;
            if (!opcode_modrm (image, size, op, 8, 8, padr, pseg, &m))
              return FALSE;
            if (m.regop != 0 && m.regop != 1)
              return FALSE;
            g_sprintf (op->mnemonic, "%s %s",
                                    m.regop == 0 ? "inc" : "dec", m.regmem_s);
          }
          return TRUE;
        case 0xff: /* inc/dec/call/callf/jmp/push */
          {
            ModRM m;

            op->bit = pope ? 16 : 32;
            if (!opcode_modrm (image, size, op,
                                            op->bit, op->bit, padr, pseg, &m))
              return FALSE;
            switch (m.regop)
              {
                case 0: /* inc */
                  g_sprintf (op->mnemonic, "inc %s", m.regmem_s);
                  break;
                case 1: /* dec */
                  g_sprintf (op->mnemonic, "dec %s", m.regmem_s);
                  break;
                case 2: /* call */
                  g_sprintf (op->mnemonic, "call %s", m.regmem_s);
                  op->type = OPCODE_TYPE_CALL;
                  break;
                case 3: /* callf */
                  op->bit = pope ? 32 : 48;
                  g_sprintf (op->mnemonic, "callf %s", m.regmem_s);
                  op->type = OPCODE_TYPE_CALL;
                  break;
                case 4: /* jmp */
                  g_sprintf (op->mnemonic, "jmp %s", m.regmem_s);
                  op->type = OPCODE_TYPE_JMP | OPCODE_TYPE_EXIT;
                  break;
                case 5: /* jmpf */
                 op->bit = pope ? 32 : 48;
                  g_sprintf (op->mnemonic, "jmpf %s", m.regmem_s);
                  op->type = OPCODE_TYPE_JMP | OPCODE_TYPE_EXIT;
                  break;
                case 6: /* push */
                  g_sprintf (op->mnemonic, "push %s", m.regmem_s);
                  break;
                default:
                  return FALSE;
              }
          }
          return TRUE;
        default:
          return FALSE;
      }
  return FALSE;
}
