/*
    IA32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "save.h"
#include "memory.h"
#include "misc/peimage.h"


/******************************************************************************
*                                                                             *
* ja:保存関数群                                                               *
*                                                                             *
******************************************************************************/
/*  ja:プロセス保存
    process,プロセス構造体
    dynamic,TRUE:すべてのメモリを保存,FALSE:プロセス生成モジュールのみ
     import,TRUE:インポートテーブル作成,FALSE:なし
      bytes,ファイルサイズ(NULL:取得しない)
        RET,PEイメージ,NULL:エラー                                          */
guint8 *
ia32_save_process (Ia32Process    *process,
                   const gboolean  dynamic,
                   const gboolean  import,
                   gsize          *bytes)
{
  gsize length;
  gint i;
  guint8 *image;
  guint16 section;
  guint32 code = 0, initialized = 0, uninitialized = 0, total = 0;
  ImageOptionalHeader *ioh;
  ImageSectionHeader *ish;

  /* ja:保存 */
  if (!process || process->address == IA32_MEMORY_NULL)
    return NULL;
  /* ja:本体をコピー */
  image = ia32_memory_real_address (process, process->address);
  length = ia32_memory_real_size (process, process->address);
  if (!image || length <= 0)
    return NULL;
  image = g_memdup (image, length);
  if (!peimage_file_is_valid (image, length))
    {
      gsize len;

      if (process->image
            && (len = peimage_file_header_bytes (process->image)) <= length)
        {
          g_memmove (image, process->image, len);
        }
      else
        {
          g_free (image);
          return NULL;
        }
    }
  ioh = pe_image_optional_header (image);
  if (ioh_get_size_of_image (ioh) > length)
    ioh_set_size_of_image (ioh, length);
  if (ioh_get_section_alignment (ioh) > IA32_SYSTEM_PAGESIZE)
    ioh_set_section_alignment (ioh, IA32_SYSTEM_PAGESIZE);
  if (ioh_get_file_alignment (ioh) > IA32_SYSTEM_PAGESIZE)
    ioh_set_file_alignment (ioh, IA32_SYSTEM_PAGESIZE);
  /* ja:セクション */
  section = pe_ifh_get_number_of_sections (image);
  ish = pe_image_section_header (image);
  for (i = 0; i < section; i++)
    {
      guint32 characteristics;

      ish_set_pointer_to_raw_data (ish, ish_get_virtual_address (ish));
      ish_set_size_of_raw_data (ish, ish_get_virtual_size (ish));
      characteristics = ish_get_characteristics (ish)
                                        | PEIMAGE_SCN_CNT_CODE
                                        | PEIMAGE_SCN_CNT_INITIALIZED_DATA
                                        | PEIMAGE_SCN_CNT_UNINITIALIZED_DATA
                                        | PEIMAGE_SCN_MEM_EXECUTE
                                        | PEIMAGE_SCN_MEM_READ
                                        | PEIMAGE_SCN_MEM_WRITE;
      ish_set_characteristics (ish, characteristics);
      ish++;
    }
  /* ja:各メモリの内容をコピー */
  if (dynamic)
    {
      gint n;
      guint16 number_of_sections;
      guint32 header = G_MAXUINT32;
      GList *glist;
      ImageSectionHeader *image_section_header;

      /* ja:セクションのコピー */
      number_of_sections = section;
      for (glist = g_list_first (process->memory);
                                            glist; glist = g_list_next (glist))
        {
          Ia32Memory *memory;

          memory = glist->data;
          if (memory->address != process->address
                                && memory->attribute == IA32_MEMORY_ATTR_USER)
            number_of_sections++;
        }
      image_section_header
                = g_malloc0 (number_of_sections * sizeof (ImageSectionHeader));
      g_memmove (image_section_header, pe_image_section_header (image),
                                        section * sizeof (ImageSectionHeader));
      ish = image_section_header + section;
      for (glist = g_list_first (process->memory);
                                            glist; glist = g_list_next (glist))
        {
          Ia32Memory *memory;

          memory = glist->data;
          if (memory->address != process->address
                                && memory->attribute == IA32_MEMORY_ATTR_USER)
            {
              gint delta;
              guint32 address, size;

              address = memory->address - process->address;
              address = (address / IA32_SYSTEM_PAGESIZE)
                                                        * IA32_SYSTEM_PAGESIZE;
              delta = memory->address - process->address - address;
              size = delta + memory->length;
              size = ((size + IA32_SYSTEM_PAGESIZE - 1) / IA32_SYSTEM_PAGESIZE)
                                                    * IA32_SYSTEM_PAGESIZE;
              image = g_realloc (image, length + size);
              g_memset (image + length, 0, size);
              g_memmove (image + length + delta, memory->data, memory->length);
              ish_set_virtual_size (ish, size);
              ish_set_virtual_address (ish, address);
              ish_set_size_of_raw_data (ish, size);
              ish_set_pointer_to_raw_data (ish, length);
              ish_set_characteristics (ish, PEIMAGE_SCN_CNT_CODE
                                          | PEIMAGE_SCN_CNT_INITIALIZED_DATA
                                          | PEIMAGE_SCN_CNT_UNINITIALIZED_DATA
                                          | PEIMAGE_SCN_MEM_EXECUTE
                                          | PEIMAGE_SCN_MEM_READ
                                          | PEIMAGE_SCN_MEM_WRITE);
              ish++;
              length += size;
            }
        }
      /* ja:セクション数 */
      ish = image_section_header;
      for (i = 0; i < section; i++)
        {
          if (ish_get_virtual_address (ish) < header)
            header = ish_get_virtual_address (ish);
          ish++;
        }
      n = (header - (idh_get_lfanew (image) + sizeof (guint32)
                                + sizeof (ImageFileHeader)
                                + pe_ifh_get_size_of_optional_header (image)))
                                                / sizeof (ImageSectionHeader);
      if (import)
        n--;
      if (number_of_sections > n)
        {
          /* ja:セクション統合 */
          for (i = number_of_sections - 1; i > 0; i--)
            {
              if (i == section && number_of_sections <= n)
                break;
              if (ish_get_virtual_address (ish + i - 1)
                                        + ish_get_virtual_size (ish + i - 1)
                                        == ish_get_virtual_address (ish + i))
                {
                  ish_set_virtual_size (ish + i - 1,
                                            ish_get_virtual_size (ish + i - 1)
                                            + ish_get_virtual_size (ish + i));
                  g_memmove (ish + i, ish + i + 1, (number_of_sections - i - 1)
                                                * sizeof (ImageSectionHeader));
                  number_of_sections--;
                }
            }
          if (number_of_sections > n)
            {
              g_free (image_section_header);
              g_free (image);
              g_free (process->table);
              process->table = NULL;
              return NULL;
            }
        }
      /* ja:ヘッダ再設定 */
      section = number_of_sections;
      pe_ifh_set_number_of_sections (image, section);
      ish = pe_image_section_header (image);
      g_memmove (ish, image_section_header,
                                        section * sizeof (ImageSectionHeader));
      g_free (image_section_header);
    }
  /* ja:インポートテーブル追加 */
  if (import)
    {
      gint count = 0;
      Ia32NameTable *table = NULL;

      for (i = 0; i < length - 4; i++)
        {
          Ia32Api *api;

          api = g_hash_table_lookup (process->export,
                                GUINT_TO_POINTER (*(guint32 *)(image + i)));
          if (api && api->file)
            {
              table = g_realloc (table, (count + 1) * sizeof (Ia32NameTable));
              table[count].address = i;
              table[count].file = api->file;
              table[count].name = api->name;
              table[count].ordinal = api->ordinal;
              count++;
            }
        }
      if (count > 0)
        {
          gint n = 0;

          /* ja:インポートディレクトリテーブルの数 */
          i = 0;
          while (i < count)
            {
              gint j;

              for (j = i + 1; j < count; j++)
                if (g_ascii_strcasecmp (table[i].file, table[j].file) != 0
                            || table[j - 1].address + 4 != table[j].address)
                  break;
              if (*(guint32 *)(image + table[j - 1].address + 4) == 0)
                {
                  while (++i < j)
                    table[i].file = NULL;
                  n++;
                }
              else
                {
                  g_memmove (table + i, table + j,
                                        (count - j) * sizeof (Ia32NameTable));
                  count -= j - i;
                }
            }
          if (n > 0)
            {
              gsize leng;
              gchar *p;
              guint32 address;
              ImageImportDescriptor *iid;

              leng = ++n * sizeof (ImageImportDescriptor);
              for (i = 0; i < count; i++)
                {
                  if (table[i].file)
                    leng += ((g_strlen (table[i].file) + 2) / 2) * 2;
                  if (table[i].name)
                    leng += ((g_strlen (table[i].name) + 2) / 2) * 2 + 2;
                }
              leng = ((leng + IA32_SYSTEM_PAGESIZE - 1) / IA32_SYSTEM_PAGESIZE)
                                                        * IA32_SYSTEM_PAGESIZE;
              image = g_realloc (image, length + leng);
              g_memset (image + length, 0, leng);
              /* ja:ヘッダ */
              ish = pe_image_section_header_nth (image, section - 1);
              address = ((ish_get_virtual_size (ish)
                    + ish_get_virtual_address (ish) + IA32_SYSTEM_PAGESIZE - 1)
                                / IA32_SYSTEM_PAGESIZE) * IA32_SYSTEM_PAGESIZE;
              pe_ioh_set_data_directory_virtual_address (image,
                                    PEIMAGE_DIRECTORY_ENTRY_IMPORT, address);
              pe_ioh_set_data_directory_size (image,
                                    PEIMAGE_DIRECTORY_ENTRY_IMPORT,
                                    n * sizeof (ImageImportDescriptor));
              ish++;
              ish_set_virtual_size (ish, leng);
              ish_set_virtual_address (ish, address);
              ish_set_size_of_raw_data (ish, leng);
              ish_set_pointer_to_raw_data (ish, length);
              ish_set_characteristics (ish, PEIMAGE_SCN_CNT_INITIALIZED_DATA
                                          | PEIMAGE_SCN_MEM_READ);
              section++;
              pe_ifh_set_number_of_sections (image, section);
              /* ja:インポートディレクトリテーブル */
              iid = (ImageImportDescriptor *)(image + length);
              p = (gchar *)(iid + n);
              address += n * sizeof (ImageImportDescriptor);
              length += leng;
              for (i = 0; i < count; i++)
                {
                  if (table[i].file)
                    {
                      guint32 thunk;

                      thunk = peimage_file_offset_to_address (image,
                                                            table[i].address);
                      iid_set_original_first_thunk (iid, thunk);
                      iid_set_name (iid, address);
                      iid_set_first_thunk (iid, thunk);
                      iid++;
                      g_strcpy (p, table[i].file);
                      leng = ((g_strlen (table[i].file) + 2) / 2) * 2;
                      p += leng;
                      address += leng;
                    }
                  if (table[i].name)
                    {
                      itd_set_address_of_data (image + table[i].address,
                                                                    address);
                      iibn_set_name (p, table[i].name);
                      leng = ((g_strlen (table[i].name) + 2) / 2) * 2 + 2;
                      p += leng;
                      address += leng;
                    }
                  else
                    {
                      itd_set_ordinal (image + table[i].address,
                                    table[i].ordinal | PEIMAGE_ORDINAL_FLAG);
                    }
                }
              table = g_realloc (table, (count + 1) * sizeof (Ia32NameTable));
              table[count].address = 0;
              table[count].file = NULL;
              table[count].name = NULL;
              table[count].ordinal = 0;
              process->table = table;
            }
          else
            {
              g_free (table);
            }
        }
    }
  /* ja:ヘッダ再設定 */
  ish = pe_image_section_header (image);
  for (i = 0; i < section; i++)
    {
      guint32 address;

      address = ish_get_virtual_address (ish);
      if ((gint32)address > 0)
        {
          guint32 size, characteristics;

          size = ish_get_virtual_size (ish);
          characteristics = ish_get_characteristics (ish);
          if (characteristics | PEIMAGE_SCN_CNT_CODE)
            code += size;
          if (characteristics | PEIMAGE_SCN_CNT_INITIALIZED_DATA)
            initialized += size;
          if (characteristics | PEIMAGE_SCN_CNT_UNINITIALIZED_DATA)
            uninitialized += size;
          if (total < size + address)
            total = size + address;
        }
      ish++;
    }
  ioh = pe_image_optional_header (image);
  ioh_set_size_of_code (ioh, code);
  ioh_set_size_of_initialized_data (ioh, initialized);
  ioh_set_size_of_uninitialized_data (ioh, uninitialized);
  ioh_set_size_of_image (ioh, total);
  if (peimage_file_header_bytes (image) > ioh_get_size_of_headers (ioh))
    ioh_set_size_of_headers (ioh, peimage_file_header_bytes (image));
  if (process->winmain != IA32_MEMORY_NULL)
    ioh_set_address_of_entry_point (ioh, process->winmain - process->address);

  if (bytes)
    *bytes = length;
  return image;
}
