/*
    disw32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "info.h"
#include "misc/peimage.h"


/******************************************************************************
*                                                                             *
* ja:情報関数                                                                 *
*                                                                             *
******************************************************************************/
/*  ja:イメージ情報構造体を作成する
    image,PEイメージ
      RET,イメージ情報構造体                                                */
Dw32Info *
dw32_info_create (const guint8 *image)
{
  gint i;
  Dw32Info *info;

  info = g_malloc (sizeof (Dw32Info));
  info->image = image;
  info->import = NULL;
  info->analyse = NULL;
  info->subroutine = NULL;
  info->base = pe_ioh_get_image_base (info->image);
  info->size = pe_ioh_get_size_of_image (info->image);
  info->separate = g_malloc0 ((info->size + 1) * sizeof (gchar *));
  info->comment = g_malloc0 (info->size *sizeof (gchar *));
  info->mnemonic = g_malloc0 (info->size * sizeof (gchar *));
  info->status = g_malloc0 (info->size * sizeof (guint8));
  info->belong = g_malloc (info->size * sizeof (gint));
  for (i = 0; i < info->size; i++)
    info->belong[i] = -1;
  return info;
}


/*  ja:イメージ情報構造体の整理
        info,イメージ情報構造体
      export,エクスポート(TRUE:存続,FALSE:削除)
      import,インポート(TRUE:存続,FALSE:削除)
    resource,リソース(TRUE:存続,FALSE:削除)
       reloc,再配置(TRUE:存続,FALSE:削除)                                   */
void
dw32_info_organize (Dw32Info       *info,
                    const gboolean  export,
                    const gboolean  import,
                    const gboolean  resource,
                    const gboolean  reloc)
{
  gint i, belong;

  /* ja:情報削除 */
  g_fprintf (stderr, "Organize\n");
  if (!export)
    for (i = 0; i < info->size; i++)
      if (info->status[i] == DW32_STAT_EXPORT)
        {
          g_free (info->mnemonic[i]);
          info->mnemonic[i] = NULL;
        }
  if (!import)
    for (i = 0; i < info->size; i++)
      if (info->status[i] == DW32_STAT_IMPORT)
        {
          g_free (info->mnemonic[i]);
          info->mnemonic[i] = NULL;
        }
  if (!resource)
    for (i = 0; i < info->size; i++)
      if (info->status[i] == DW32_STAT_RESOURCE)
        {
          g_free (info->mnemonic[i]);
          info->mnemonic[i] = NULL;
        }
  if (!reloc)
    for (i = 0; i < info->size; i++)
      if (info->status[i] == DW32_STAT_RELOC)
        {
          g_free (info->mnemonic[i]);
          info->mnemonic[i] = NULL;
        }
  /* ja:関数の境界 */
  g_fprintf (stderr, "Border\n");
  belong = G_MAXINT;
  for (i = 1; i < info->size; i++)
    if (info->mnemonic[i] && info->belong[i] != belong)
      {
        if (!info->separate[i])
          {
            info->separate[i] = g_strdup ("");
          }
        else if (*info->separate[i] != '\0' && *info->separate[i] != '\n')
          {
            gchar *tmp;

            tmp = g_strconcat ("\n", info->separate[i], NULL);
            g_free (info->separate[i]);
            info->separate[i] = tmp;
          }
        belong = info->belong[i];
      }
}


/*  ja:イメージ情報構造体を表示する
      fp,ファイルポインタ
    info,イメージ情報構造体                                                 */
void
dw32_info_disp (FILE           *fp,
                const Dw32Info *info)
{
  gint i;

  g_fprintf (fp, "; "PACKAGE" "VERSION" ("BUILD_ENVIRONMENT")\n");
  for (i = 0; i < info->size && !info->separate[i]; i++)
    if (info->mnemonic[i])
      {
        g_fprintf (fp, ";\n");
        break;
      }
  for (i = 0; i < info->size; i++)
    if (info->mnemonic[i])
      {
        gchar *p, *mnemonic;
        gint j;
        gsize len;

        if (info->separate[i])
          {
            if (*info->separate[i] == '\0')
              {
                g_fprintf (fp, ";\n");
              }
            else
              {
                gchar **str;

                str = g_strsplit (info->separate[i], "\n", 0);
                if (str)
                  {
                    for (j = 0; str[j]; j++)
                      {
                        g_strstrip (str[j]);
                        if (g_strlen (str[j]) > 0)
                          g_fprintf (fp, "; %s\n", str[j]);
                        else
                          g_fprintf (fp, ";\n");
                      }
                    g_strfreev (str);
                  }
              }
          }
        g_fprintf (fp, "%08x", i + info->base);
        for (len = 1; i + len < info->size && !info->mnemonic[i + len]
                        && (info->status[i + len] == DW32_STAT_INTER
                        || info->status[i] == info->status[i + len]); len++);
        for (j = 0; j < MIN (len, 10); j++)
          g_fprintf (fp, " %02x", info->image[i + j]);
        if (len == 11)
          g_fprintf (fp, " %02x", info->image[i + j]);
        else if (len > 11)
          g_fprintf (fp, "...");
        else
          while (j++ < 11)
            g_fprintf (fp, "   ");
        mnemonic = g_strdup (info->mnemonic[i]);
        g_strstrip (mnemonic);
        p = g_strchr (mnemonic, ' ');
        if (p)
          {
            *p = '\0';
            p++;
            g_fprintf (fp, " %-7s %s", mnemonic, p);
          }
        else
          {
            g_fprintf (fp, " %s", mnemonic);
          }
        if (info->comment[i] && g_strlen (info->comment[i]) > 0)
          g_fprintf (fp, " ; %s", info->comment[i]);
        g_fprintf (fp, "\n");
      }
}


/*  ja:イメージ情報構造体を解放する
    info,イメージ情報構造体                                                 */
void
dw32_info_free (Dw32Info *info)
{
  gint i;

  while (info->analyse)
    {
      Dw32Label *label;

      label = info->analyse->data;
      g_free (label->name);
      g_free (label);
      info->analyse = g_list_delete_link (info->analyse, info->analyse);
    }
  while (info->import)
    {
      Dw32Label *label;

      label = info->import->data;
      g_free (label->name);
      g_free (label);
      info->import = g_list_delete_link (info->import, info->import);
    }
  while (info->subroutine)
    {
      Dw32Label *label;

      label = info->subroutine->data;
      g_free (label->name);
      g_free (label);
      info->subroutine = g_list_delete_link (info->subroutine,
                                                            info->subroutine);
    }
  for (i = 0; i < info->size; i++)
    {
      g_free (info->separate[i]);
      g_free (info->comment[i]);
      g_free (info->mnemonic[i]);
    }
  g_free (info->separate[info->size]);
  g_free (info->separate);
  g_free (info->comment);
  g_free (info->mnemonic);
  g_free (info->status);
  g_free (info->belong);
  g_free (info);
}
