/*
    findep
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "ia32/ia32.h"
#include "ia32/cpu.h"
#include "ia32/memory.h"
#include "ia32/message.h"
#include "ia32/process.h"
#include "misc/opcode.h"
#include "misc/peimage.h"


#define WIDTH 79
#define PROGRESS 50
#define MAXSTEP 24
#define MINSTEP 8
#define ALIGNMENT 0x00001000
#define IMAGEBASE 0x00400000


/*  ja:ヘルプを表示する
      fp,ファイルポインタ
    file,ファイル名                                                         */
static void
help (FILE        *fp,
      const gchar *file)
{
  g_fprintf (fp,
"findep "VERSION" ("BUILD_ENVIRONMENT")\n"
"copyright (c) 1998-2012 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org\n"
"\n"
"Usage: %s command [option...] [file|string...]\n"
"\n"
"   c    check            check entry point\n"
"   d    display          display registered information\n"
"   e    entry            register entry point\n"
"   f    find             find entry point\n"
"   i    ignore           ignore entry point\n"
"   r    remove           remove registered information\n"
"\n"
"  -a, --align=BYTE       section alignment\n"
"  -b, --base=ADDRESS     image base\n"
"  -d, --dump             memory dump\n"
"  -r, --raw              raw binary\n"
"      --dynamic          dynamic analyse (emulation)\n"
"      --static           static analyse (disassemble)\n"
"\n", file);
}


static gchar *
trace (GList *glist)
{
  gsize len = 0;
  gchar *mnemonic = NULL;

  for (glist = g_list_first (glist); glist; glist = g_list_next (glist))
    {
      gsize l;
      gchar *p, *q;

      p = glist->data;
      if (g_strtoul (p, &q, 16) >= IA32_MEMORY_BASE_SYSTEM || q - p != 8)
        break;
      for (p = q; *p != '\0' && !g_ascii_isgraph (*p); p++);
      for (q = p; *q != '\0' && g_ascii_isgraph (*q); q++);
      l = q - p;
      if (l > 0)
        {
          mnemonic = g_realloc (mnemonic, (len + l + 1) * sizeof (gchar));
          g_memmove (mnemonic + len, p, l * sizeof (gchar));
          len += l;
          mnemonic[len++] = ',';
        }
    }
  if (mnemonic)
    mnemonic[len - 1] = '\0';
  return mnemonic;
}


static gchar *
analyse (const guint8  *image,
         const gint     index,
         const gsize    length,
         const guint32  base)
{
  gsize len = 0;
  gchar *mnemonic = NULL;
  gint i;
  Opcode op;

  op.index = index;
  for (i = 0; i < MAXSTEP; i++)
    {
      gsize l;
      gchar *p;

      if (!opcode_parse (image, length, base, &op))
        break;
      for (p = op.mnemonic; *p != '\0' && g_ascii_isgraph (*p); p++);
      l = p - op.mnemonic;
      mnemonic = g_realloc (mnemonic, (len + l + 1) * sizeof (gchar));
      g_memmove (mnemonic + len, op.mnemonic, l * sizeof (gchar));
      len += l;
      mnemonic[len++] = ',';
      if (op.type & OPCODE_TYPE_EXIT)
        {
          if (base <= op.relative && op.relative < base + length)
            op.index = op.relative - base;
          else
            break;
        }
    }
  if (len > 0)
    mnemonic[len - 1] = '\0';
  return mnemonic;
}


static gint
steps (const gchar *mnemonic)
{
  const gchar *p;
  gint step = 0;

  if (mnemonic)
    for (p = mnemonic; *p != '\0'; p++)
      if (*p == ',')
        step++;
  return step;
}


static gchar *
chomp (gchar       *str,
       const gsize  len)
{
  if (g_strlen (str) > len)
    {
      gint i;

      for (i = len - 4; i >= 0; i--)
        if (str[i] == ',')
          {
            str[i + 1] = str[i + 2] = str[i + 3] = '.';
            str[i + 4] = '\0';
            break;
          }
    }
  return str;
}


typedef struct _Entry
{
  gchar *mnemonic;
  guint count;
} Entry;


static gint
compare (gconstpointer a,
         gconstpointer b)
{
  const Entry *aa, *bb;

  aa = a;
  bb = b;
  return bb->count - aa->count;
}


static void
hash_func (gpointer key,
           gpointer value,
           gpointer user_data)
{
  GList **glist;
  Entry *e;

  glist = user_data;
  e = g_malloc (sizeof (Entry));
  e->mnemonic = g_strdup (key);
  e->count = GPOINTER_TO_UINT (value);
  *glist = g_list_insert_sorted (*glist, e, compare);
}


int
main (int   argc,
      char *argv[])
{
  gboolean dump = FALSE, raw = FALSE, dynamic = FALSE;
  gboolean edited = FALSE, result = TRUE;
  gchar cmd = '\0', *p, *file, *name, *basename;
  gint i, index = 0, n_ignore = 0;
  guint32 align = ALIGNMENT, base = (guint32)-1;
  GKeyFile *key_file;
  GHashTable *ghash_entry, *ghash_ignore;

  /* ja:初期化 */
#if ! GTK_CHECK_VERSION (2,24,0)
  gtk_set_locale ();
#endif /* not GTK_CHECK_VERSION (2,24,0) */
  gtk_init (&argc, &argv);

  /* ja:引数 */
  for (i = 1; i < argc; i++)
    if (cmd == '\0' && (g_ascii_strcasecmp (argv[i], "c") == 0
                            || g_ascii_strcasecmp (argv[i], "check") == 0))
      {
        cmd = 'c';
      }
    else if (cmd == '\0' && (g_ascii_strcasecmp (argv[i], "d") == 0
                            || g_ascii_strcasecmp (argv[i], "display") == 0))
      {
        cmd = 'd';
      }
    else if (cmd == '\0' && (g_ascii_strcasecmp (argv[i], "e") == 0
                            || g_ascii_strcasecmp (argv[i], "entry") == 0))
      {
        cmd = 'e';
      }
    else if (cmd == '\0' && (g_ascii_strcasecmp (argv[i], "f") == 0
                            || g_ascii_strcasecmp (argv[i], "find") == 0))
      {
        cmd = 'f';
      }
    else if (cmd == '\0' && (g_ascii_strcasecmp (argv[i], "i") == 0
                            || g_ascii_strcasecmp (argv[i], "ignore") == 0))
      {
        cmd = 'i';
      }
    else if (cmd == '\0' && (g_ascii_strcasecmp (argv[i], "r") == 0
                            || g_ascii_strcasecmp (argv[i], "remove") == 0))
      {
        cmd = 'r';
      }
    else if (g_ascii_strcasecmp (argv[i], "-a") == 0
                            || g_ascii_strcasecmp (argv[i], "--align") == 0)
      {
        align = g_strtol (argv[++i], NULL, 16);
      }
    else if (g_ascii_strncasecmp (argv[i], "--align=", 8) == 0)
      {
        align = g_strtol (argv[i] + 8, NULL, 16);
      }
    else if (g_ascii_strcasecmp (argv[i], "-b") == 0
                            || g_ascii_strcasecmp (argv[i], "--base") == 0)
      {
        base = g_strtol (argv[++i], NULL, 16);
      }
    else if (g_ascii_strncasecmp (argv[i], "--base=", 7) == 0)
      {
        base = g_strtol (argv[i] + 7, NULL, 16);
      }
    else if (g_ascii_strcasecmp (argv[i], "-d") == 0
                            || g_ascii_strcasecmp (argv[i], "--dump") == 0)
      {
        dump = TRUE;
      }
    else if (g_ascii_strcasecmp (argv[i], "-r") == 0
                            || g_ascii_strcasecmp (argv[i], "--raw") == 0)
      {
        raw = TRUE;
      }
    else if (g_ascii_strcasecmp (argv[i], "-d") == 0
                            || g_ascii_strcasecmp (argv[i], "--dynamic") == 0)
      {
        dynamic = TRUE;
      }
    else if (g_ascii_strcasecmp (argv[i], "--static") == 0)
      {
        dynamic = FALSE;
      }
    else if (g_ascii_strcasecmp (argv[i], "-h") == 0
                                || g_ascii_strcasecmp (argv[i], "-?") == 0
                                || g_ascii_strcasecmp (argv[i], "--help") == 0)
      {
        help (stdout, argv[0]);
        return 0;
      }
    else
      {
        index = i;
        break;
      }
  if (cmd =='\0' || ((cmd == 'd') ^ (index <= 0)) || align <= 0)
    {
      help (stderr, argv[0]);
      return -1;
    }
  /* ja:ファイル読み込み */
  basename = g_path_get_basename (argv[0]);
  p = g_strrchr (basename, '.');
  if (p)
    *p = '\0';
  name = g_strconcat (basename, ".ini", NULL);
  g_free (basename);
#ifdef G_OS_WIN32
  file = g_build_filename (g_get_user_config_dir (),
                                        "maid.org", "iwmutlis", name, NULL);
#else /* not G_OS_WIN32 */
  file = g_build_filename (g_get_home_dir (),
                                        ".maid.org", "iwmutlis", name, NULL);
#endif /* not G_OS_WIN32 */
  g_free (name);
  ghash_entry = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  ghash_ignore = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  key_file = g_key_file_new ();
  if (g_key_file_load_from_file (key_file,
                                        file, G_KEY_FILE_KEEP_COMMENTS, NULL))
    {
      gchar **keys;

      keys = g_key_file_get_keys (key_file, "entry", NULL, NULL);
      if (keys)
        {
          for (i = 0; keys[i]; i++)
            {
              gchar *str;

              str = g_key_file_get_string (key_file, "entry", keys[i], NULL);
              if (str)
                g_hash_table_insert (ghash_entry, str,
                                GUINT_TO_POINTER (GPOINTER_TO_UINT (
                                g_hash_table_lookup (ghash_entry, str)) + 1));
            }
          g_strfreev (keys);
        }
      for (n_ignore = 0; n_ignore < 10000; n_ignore++)
        {
          gchar *key, *str;

          key = g_strdup_printf ("n%04d", n_ignore);
          str = g_key_file_get_string (key_file, "ignore", key, NULL);
          g_free (key);
          if (!str)
            break;
          g_hash_table_insert (ghash_ignore, str, GUINT_TO_POINTER (1));
        }
    }
  /* ja:コマンド */
  g_fprintf (stderr, "findep "VERSION" ("BUILD_ENVIRONMENT")\n\n");
  if (cmd == 'e' || cmd == 'i')
    {
      /* ja:登録 */
      for (i = index; i < argc; i++)
        {
          gchar *mnemonic;

          if (dynamic)
            {
              /* ja:動的解析 */
              gchar *a[2];
              Ia32Process *process;

              a[0] = argv[i];
              a[1] = NULL;
              process = ia32_create_process_from_argv (a);
              if (!process)
                {
                  g_fprintf (stderr, "%s : Create process error.\n", argv[i]);
                  result = FALSE;
                  break;
                }
              ia32_cpu_loop (process, NULL,
                                MAXSTEP, 0, MAXSTEP, IA32_MESSAGE_MNEMONIC);
              mnemonic = trace (process->current->trace);
              if (!mnemonic)
                {
                  GList *glist;

                  glist = g_list_first (process->current->err);
                  if (glist)
                    g_fprintf (stdout, "%s : %s\n",
                                                argv[i], (gchar *)glist->data);
                }
              if (!ia32_destroy_process (process))
                {
                  g_fprintf (stderr, "%s : Destroy process error.\n", argv[i]);
                  result = FALSE;
                  break;
                }
            }
          else
            {
              /* ja:静的解析 */
              guint8 *image;

              image = peimage_file_load (argv[i]);
              if (!image)
                {
                  g_fprintf (stderr, "\'%s\' load error.\n", argv[i]);
                  result = FALSE;
                  break;
                }
              mnemonic = analyse (image,
                                    pe_ioh_get_address_of_entry_point (image),
                                    pe_ioh_get_size_of_image (image),
                                    pe_ioh_get_image_base (image));
              g_free (image);
              if (!mnemonic)
                g_fprintf (stderr, "\'%s\' entry error.\n", argv[i]);
            }
          if (mnemonic)
            {
              if (steps (mnemonic) >= MINSTEP)
                {
                  gsize length;
                  gchar *sha1 = NULL;
                  guchar *data;

                  if (g_file_get_contents (argv[i],
                                            (gchar **)&data, &length, NULL))
                    {
                      sha1 = g_compute_checksum_for_data (G_CHECKSUM_SHA1,
                                                                data, length);
                      g_free (data);
                    }
                  if (!sha1)
                    {
                      g_fprintf (stderr, "%s : SHA-1 error.\n", argv[i]);
                      result = FALSE;
                      break;
                    }
                  else if (g_key_file_has_key (key_file, "entry", sha1, NULL))
                    {
                      g_fprintf (stdout, "%s : Already registered in entry.\n",
                                                                    argv[i]);
                    }
                  else if (g_hash_table_lookup (ghash_ignore, mnemonic))
                    {
                      g_fprintf (stdout,
                                        "%s : Already registered in ignore.\n",
                                                                    argv[i]);
                    }
                  else
                    {
                      if (cmd == 'e')
                        {
                          g_key_file_set_string (key_file, "entry", sha1,
                                                                    mnemonic);
                          g_hash_table_insert (ghash_entry,
                            g_strdup (mnemonic),
                            GUINT_TO_POINTER (GPOINTER_TO_UINT (
                            g_hash_table_lookup (ghash_entry, mnemonic)) + 1));
                        }
                      else
                        {
                          gchar *key;

                          key = g_strdup_printf ("n%04d", n_ignore++);
                          g_key_file_set_string (key_file, "ignore", key,
                                                                    mnemonic);
                          g_free (key);
                          g_hash_table_insert (ghash_ignore,
                                    g_strdup (mnemonic), GUINT_TO_POINTER (1));
                        }
                      g_fprintf (stdout, "%s : Append.\n", argv[i]);
                      edited = TRUE;
                    }
                  g_free (sha1);
                }
              else
                {
                  g_fprintf (stdout, "%s : Under minimum steps.\n", argv[i]);
                }
              g_free (mnemonic);
            }
        }
      g_fprintf (stderr, "\nentry  : %d\nignore : %d\n\n",
                                    g_hash_table_size (ghash_entry), n_ignore);
    }
  if (cmd == 'r')
    {
      /* ja:削除 */
      for (i = index; i < argc; i++)
        {
          gsize leng;
          gchar **keys;
          gint j;
          GList *gl, *glist = NULL;

          leng = g_strlen (argv[i]);
          keys = g_key_file_get_keys (key_file, "entry", NULL, NULL);
          if (keys)
            {
              guint len;

              len = g_checksum_type_get_length (G_CHECKSUM_SHA1) * 2;
              for (j = 0; keys[j]; j++)
                {
                  gchar *str;

                  str = g_key_file_get_string (key_file,
                                                    "entry", keys[j], NULL);
                  if (str && (g_strcmp (argv[i], "-") == 0
                        || g_ascii_strncasecmp (argv[i], keys[j], leng) == 0
                            || (g_ascii_strncasecmp (argv[i], str, leng) == 0
                                && (str[leng] == '\0' || str[leng] == ','))))
                    {
                      g_key_file_remove_key (key_file, "entry", keys[j], NULL);
                      g_hash_table_remove (ghash_entry, str);
                      chomp (str, WIDTH - len - 10);
                      g_fprintf (stderr, "Delete : %s %s\n", keys[j], str);
                      edited = TRUE;
                    }
                  g_free (str);
                }
              g_strfreev (keys);
            }
          for (j = 0; j < n_ignore; j++)
            {
              gchar *key, *str;

              key = g_strdup_printf ("n%04d", j);
              str = g_key_file_get_string (key_file, "ignore", key, NULL);
              if (str)
                {
                  if (g_strcmp (argv[i], "-") == 0
                            || g_ascii_strncasecmp (argv[i], key, leng) == 0
                            || (g_ascii_strncasecmp (argv[i], str, leng) == 0
                                && (str[leng] == '\0' || str[leng] == ',')))
                    {
                      g_key_file_remove_key (key_file, "ignore", key, NULL);
                      g_hash_table_remove (ghash_ignore, str);
                      chomp (str, WIDTH - 15);
                      g_fprintf (stdout, "Delete : %s %s\n", key, str);
                      g_free (str);
                      edited = TRUE;
                    }
                  else
                    {
                      glist = g_list_append (glist, str);
                    }
                }
              g_free (key);
            }
          j = 0;
          for (gl = g_list_first (glist); gl; gl = g_list_next (gl))
            {
              gchar *key;

              key = g_strdup_printf ("n%04d", j++);
              g_key_file_set_string (key_file, "ignore", key, gl->data);
              g_free (gl->data);
              g_free (key);
            }
          while (j < n_ignore)
            {
              gchar *key;

              key = g_strdup_printf ("n%04d", j++);
              g_key_file_remove_key (key_file, "ignore", key, NULL);
              g_free (key);
            }
          n_ignore = g_list_length (glist);
          g_list_free (glist);
        }
      g_fprintf (stderr, "\nentry  : %d\nignore : %d\n\n",
                                    g_hash_table_size (ghash_entry), n_ignore);
    }
  /* ja:ファイル書き込み */
  if (edited)
    {
      gsize length;
      gchar *data;

      data = g_key_file_to_data (key_file, &length, NULL);
      if (!data || !g_file_set_contents (file, data, length, NULL))
        {
          g_fprintf (stderr, "\'%s\' save error.\n", file);
          result = FALSE;
        }
    }
  g_free (file);
  if (cmd == 'd')
    {
      /* ja:表示 */
      if (g_hash_table_size (ghash_entry) > 0)
        {
          gchar **keys;
          GList *glist = NULL;

          keys = g_key_file_get_keys (key_file, "entry", NULL, NULL);
          if (keys)
            {
              guint len;

              len = g_checksum_type_get_length (G_CHECKSUM_SHA1) * 2;
              g_fprintf (stdout, "Entry point files : %d\n",
                                            g_hash_table_size (ghash_entry));
              for (i = 0; keys[i]; i++)
                {
                  gchar *str;

                  str = g_key_file_get_string (key_file,
                                                    "entry", keys[i], NULL);
                  chomp (str, WIDTH - len - 1);
                  g_fprintf (stdout, "%s %s\n", keys[i], str);
                  g_free (str);
                }
              g_fprintf (stdout, "\n");
              g_strfreev (keys);
            }
          glist = NULL;
          g_hash_table_foreach (ghash_entry, hash_func, &glist);
          while ((glist = g_list_first (glist)))
            {
              Entry *e;

              e = glist->data;
              chomp (e->mnemonic, WIDTH - 5);
              g_fprintf (stdout, "%d %s\n", e->count, e->mnemonic);
              g_free (e->mnemonic);
              g_free (e);
              glist = g_list_delete_link (glist, glist);
            }
          g_fprintf (stdout, "\n");
        }
      if (n_ignore > 0)
        {
          g_fprintf (stdout, "Ignore : %d\n", n_ignore);
          for (i = 0; i < n_ignore; i++)
            {
              gchar *key, *str;

              key = g_strdup_printf ("n%04d", i);
              str = g_key_file_get_string (key_file, "ignore", key, NULL);
              g_free (key);
              chomp (str, WIDTH - 6);
              g_fprintf (stdout, "n%04d %s\n", i, str);
              g_free (str);
            }
          g_fprintf (stdout, "\n");
        }
    }
  g_key_file_free (key_file);
  if (cmd == 'c')
    {
      /* ja:チェック */
      g_fprintf (stderr, "entry  : %d\nignore : %d\n\n",
                                    g_hash_table_size (ghash_entry), n_ignore);
      for (i = index; i < argc; i++)
        {
          gchar *mnemonic = NULL;

          if (dynamic)
            {
              /* ja:動的解析 */
              gchar *a[2];
              Ia32Process *process;

              a[0] = argv[i];
              a[1] = NULL;
              process = ia32_create_process_from_argv (a);
              if (!process)
                {
                  g_fprintf (stdout, "%s : Create process error.\n", argv[i]);
                  result = FALSE;
                  continue;
                }
              ia32_cpu_loop (process, NULL,
                                MAXSTEP, 0, MAXSTEP, IA32_MESSAGE_MNEMONIC);
              mnemonic = trace (process->current->trace);
              if (!ia32_destroy_process (process))
                {
                  g_fprintf (stdout, "%s : Destroy process error.\n", argv[i]);
                  result = FALSE;
                  continue;
                }
            }
          else
            {
              /* ja:静的解析 */
              guint8 *image;

              image = peimage_file_load (argv[i]);
              if (!image)
                {
                  g_fprintf (stdout, "%s : Load error.\n", argv[i]);
                  result = FALSE;
                  continue;
                }
              mnemonic = analyse (image,
                                    pe_ioh_get_address_of_entry_point (image),
                                    pe_ioh_get_size_of_image (image),
                                    pe_ioh_get_image_base (image));
              g_free (image);
            }
          if (mnemonic && g_hash_table_lookup (ghash_entry, mnemonic))
            g_fprintf (stdout, "%s : Known entry point.\n", argv[i]);
          else if (mnemonic && g_hash_table_lookup (ghash_ignore, mnemonic))
            g_fprintf (stdout, "%s : Ignore entry point.\n", argv[i]);
          else
            g_fprintf (stdout, "%s : Unknown entry point.\n", argv[i]);
          g_free (mnemonic);
        }
      g_fprintf (stderr, "\n");
    }
  g_hash_table_destroy (ghash_ignore);
  if (cmd == 'f')
    {
      /* ja:探査 */
      gsize length;
      guint8 *image;

      g_fprintf (stderr, "entry  : %d\nignore : %d\n\n",
                                    g_hash_table_size (ghash_entry), n_ignore);
      if (g_file_get_contents (argv[index], (gchar **)&image, &length, NULL))
        {
          guint32 header;
          GList *glist = NULL;

          if (dump && !raw && !peimage_file_is_memorydump (image, length))
            {
              g_fprintf (stderr, "\'%s\' is not memory dump.\n", argv[index]);
              dump = FALSE;
              result = FALSE;
            }
          else if (dump == raw)
            {
              dump = peimage_file_is_memorydump (image, length);
            }
          header = (sizeof (ImageDosHeader)
                    + sizeof (guint32)
                    + sizeof (ImageFileHeader)
                    + sizeof (ImageOptionalHeader)
                    + sizeof (ImageSectionHeader)
                    + align - 1) / align * align;
          if (base == (guint32)-1)
            base = dump ? pe_ioh_get_image_base (image) : IMAGEBASE + header;
  /* ja:プログレスバー */
#ifndef G_OS_WIN32
          g_fprintf (stderr, "*");
          for (i = 0; i < PROGRESS - 1; i++)
            g_fprintf (stderr, "-");
          g_fprintf (stderr, "\n");
#endif /* not G_OS_WIN32 */
          for (i = 0; i < length && result; i++)
            {
              gchar *mnemonic;

              /* ja:プログレスバー */
#ifdef G_OS_WIN32
              if ((glonglong)i * MIN (PROGRESS, 100) / length
                        != (glonglong)(i + 1) * MIN (PROGRESS, 100) / length)
                g_fprintf (stderr, "%d/%d (%d%%)\n",
                                        i, (gint)length,
                                        (gint)((glonglong)i * 100 / length));
#else /* not G_OS_WIN32 */
              if ((glonglong)i * PROGRESS / length
                                    != (glonglong)(i + 1) * PROGRESS / length)
                g_fprintf (stderr, "\x1b[1A\x1b[%dC*\n",
                                    (gint)((glonglong)i * PROGRESS / length));
#endif /* not G_OS_WIN32 */
              if (dynamic)
                {
                  /* ja:動的解析 */
                  Ia32Process *process;

                  process = ia32_create_process_from_image (image,
                                                            length, base, i);
                  if (!process)
                    {
                      g_fprintf (stderr, "Create process error.\n");
                      result = FALSE;
                      break;
                    }
                  ia32_cpu_loop (process, NULL,
                                MAXSTEP, 0, MAXSTEP, IA32_MESSAGE_MNEMONIC);
                  mnemonic = trace (process->current->trace);
                  if (!ia32_destroy_process (process))
                    {
                      g_free (mnemonic);
                      g_fprintf (stderr, "Destroy process error.\n");
                      result = FALSE;
                      break;
                    }
                }
              else
                {
                  /* ja:静的解析 */
                  mnemonic = analyse (image, i, length, base);
                }
              if (mnemonic)
                {
                  if (g_hash_table_lookup (ghash_entry, mnemonic))
                    {
                      glist = g_list_append (glist,
                                    g_strdup_printf ("Entry Point : %08x %s\n",
                                            i, chomp (mnemonic, WIDTH - 23 )));
                      if (argv[index + 1])
                        {
                          if (dump)
                            {
                              peimage_file_canonicalize_memorydump (image,
                                                                    length);
                              pe_ioh_set_image_base (image, base);
                              pe_ioh_set_address_of_entry_point (image, i);
                            }
                          else
                            {
                              guint8 *data;

                              data = image;
                              image = peimage_file_image_from_binary (data,
                                            length, base - header, i, align);
                              g_free (data);
                              length = pe_ioh_get_size_of_image (image);
                            }
                          if (!g_file_set_contents (argv[index + 1],
                                        (const gchar *)image, length, NULL))
                            {
                              g_fprintf (stderr, "\'%s\' save error.\n",
                                                            argv[index + 1]);
                              result = FALSE;
                            }
                          break;
                        }
                    }
                  g_free (mnemonic);
                }
            }
          g_free (image);
          if (glist)
            while ((glist = g_list_first (glist)))
              {
                g_fprintf (stdout, "%s", (gchar *)glist->data);
                g_free (glist->data);
                glist = g_list_delete_link (glist, glist);
              }
          else
            g_fprintf (stderr, "Entry point not found.\n");
        }
      else
        {
          g_fprintf (stderr, "\'%s\' load error.\n", argv[index]);
          result = FALSE;
        }
      g_fprintf (stderr, "\n");
    }
  g_hash_table_destroy (ghash_entry);
  return result ? 0 : -1;
}
