/*
    findsh
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "ia32/ia32.h"
#include "ia32/cpu.h"
#include "ia32/memory.h"
#include "ia32/message.h"
#include "ia32/process.h"
#include "misc/opcode.h"
#include "misc/peimage.h"


#define PROGRESS 50
#define MAXSTEP 256
#define TOP 10
#define ALIGNMENT 0x00001000
#define IMAGEBASE 0x00400000


/*  ja:ヘルプを表示する
      fp,ファイルポインタ
    file,ファイル名                                                         */
static void
help (FILE        *fp,
      const gchar *file)
{
  g_fprintf (fp,
"findsh "VERSION" ("BUILD_ENVIRONMENT")\n"
"copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org\n"
"\n"
"Usage: %s [option...] infile [outfile]\n"
"\n"
"  -a, --align=BYTE       section alignment\n"
"  -b, --base=ADDRESS     image base\n"
"  -d, --dynamic          dynamic analyse (emulation)\n"
"      --static           static analyse (disassemble)\n"
"  -s, --step=STEP        max step\n"
"  -t, --top=N            ranking\n"
"\n", file);
}


static gint
trace (GList         *glist,
       const gsize    length,
       const guint32  base,
       gboolean      *checked)
{
  gint depth = 0;

  for (glist = g_list_first (glist); glist; glist = g_list_next (glist))
    {
      gchar *p, *q;
      guint32 address;

      p = glist->data;
      address = g_strtoul (p, &q, 16) - base;
      if (address >= length || q - p != 8)
        break;
      checked[address] = TRUE;
      depth++;
    }
  return depth;
}


typedef struct _Cache
{
  gboolean checked, result;
  gint index, relative;
  guint type;
} Cache;


static gint
recursive (const guint8  *data,
           const gint     index,
           const gsize    length,
           const guint32  base,
           gboolean      *checked,
           Cache         *cache)
{
  gint depth = 0;

  if (0 <= index && index < length && !checked[index])
    {
      if (!cache[index].checked)
        {
          Opcode op;

          cache[index].checked = TRUE;
          op.index = index;
          cache[index].result = opcode_parse (data, length, base, &op);
          if (cache[index].result)
            {
              cache[index].index = op.index;
              if (base <= op.relative && op.relative < base + length)
                cache[index].relative = op.relative - base;
              cache[index].type = op.type;
            }
        }
      if (cache[index].result)
        {
          gint i;

          for (i = index; i < cache[index].index; i++)
            checked[i] = TRUE;
          depth++;
          if (!(cache[index].type & OPCODE_TYPE_EXIT))
            depth += recursive (data,
                        cache[index].index, length, base, checked, cache);
          if (cache[index].relative >= 0)
            depth += recursive (data,
                        cache[index].relative, length, base, checked, cache);
        }
    }
  return depth;
}


typedef struct _Ranking
{
  gboolean *checked;
  gint index, depth;
} Ranking;


int
main (int   argc,
      char *argv[])
{
  gboolean dynamic = FALSE, result = TRUE;
  gsize header, length;
  gchar *infile = NULL, *outfile = NULL;
  gint i, step = MAXSTEP, top = TOP;
  guint8 *data;
  guint32 align = ALIGNMENT, base = (guint32)-1;
  Cache *cache;
  Ranking *rank;

  /* ja:初期化 */
#if ! GTK_CHECK_VERSION (2,24,0)
  gtk_set_locale ();
#endif /* not GTK_CHECK_VERSION (2,24,0) */
  gtk_init (&argc, &argv);

  /* ja:引数 */
  for (i = 1; i < argc; i++)
    if (g_ascii_strcasecmp (argv[i], "-a") == 0
                            || g_ascii_strcasecmp (argv[i], "--align") == 0)
      {
        align = g_strtol (argv[++i], NULL, 16);
      }
    else if (g_ascii_strncasecmp (argv[i], "--align=", 8) == 0)
      {
        align = g_strtol (argv[i] + 8, NULL, 16);
      }
    else if (g_ascii_strcasecmp (argv[i], "-b") == 0
                            || g_ascii_strcasecmp (argv[i], "--base") == 0)
      {
        base = g_strtol (argv[++i], NULL, 16);
      }
    else if (g_ascii_strncasecmp (argv[i], "--base=", 7) == 0)
      {
        base = g_strtol (argv[i] + 7, NULL, 16);
      }
    else if (g_ascii_strcasecmp (argv[i], "-d") == 0
                            || g_ascii_strcasecmp (argv[i], "--dynamic") == 0)
      {
        dynamic = TRUE;
      }
    else if (g_ascii_strcasecmp (argv[i], "--static") == 0)
      {
        dynamic = FALSE;
      }
    else if (g_ascii_strcasecmp (argv[i], "-s") == 0
                            || g_ascii_strcasecmp (argv[i], "--step") == 0)
      {
        step = g_strtol (argv[++i], NULL, 10);
      }
    else if (g_ascii_strncasecmp (argv[i], "--step=", 7) == 0)
      {
        step = g_strtol (argv[i] + 7, NULL, 10);
      }
    else if (g_ascii_strcasecmp (argv[i], "-t") == 0
                            || g_ascii_strcasecmp (argv[i], "--top") == 0)
      {
        top = g_strtol (argv[++i], NULL, 10);
      }
    else if (g_ascii_strncasecmp (argv[i], "--top=", 6) == 0)
      {
        top = g_strtol (argv[i] + 6, NULL, 10);
      }
    else if (g_ascii_strcasecmp (argv[i], "-h") == 0
                                || g_ascii_strcasecmp (argv[i], "-?") == 0
                                || g_ascii_strcasecmp (argv[i], "--help") == 0)
      {
        help (stdout, argv[0]);
        return 0;
      }
    else if (!infile)
      {
        infile = argv[i];
      }
    else if (!outfile)
      {
        outfile = argv[i];
      }
    else
      break;
  if (!infile || i < argc)
    {
      help (stderr, argv[0]);
      return -1;
    }

  g_fprintf (stderr, "findsh "VERSION" ("BUILD_ENVIRONMENT")\n\n");
  /* ja:初期化 */
  rank = g_malloc ((top + 1) * sizeof (Ranking));
  for (i = 0; i <= top; i++)
    {
      rank[i].checked = NULL;
      rank[i].index = -1;
      rank[i].depth = 0;
    }
  header = (sizeof (ImageDosHeader)
            + sizeof (guint32)
            + sizeof (ImageFileHeader)
            + sizeof (ImageOptionalHeader)
            + sizeof (ImageSectionHeader)
            + align - 1) / align * align;
  if (base == (guint32)-1)
    base = IMAGEBASE + header;
  /* ja:ファイル読み込み */
  if (!g_file_get_contents (infile, (gchar **)&data, &length, NULL))
    {
      g_fprintf (stderr, "\'%s\' load error.\n", infile);
      return -1;
    }
  /* ja:キャッシュ */
  cache = g_malloc (length * sizeof (Cache));
  for (i = 0; i < length; i++)
    {
      cache[i].checked = cache[i].result = FALSE;
      cache[i].index = cache[i].relative = -1;
      cache[i].type = 0;
    }
  /* ja:プログレスバー */
#ifndef G_OS_WIN32
  g_fprintf (stderr, "*");
  for (i = 0; i < PROGRESS - 1; i++)
    g_fprintf (stderr, "-");
  g_fprintf (stderr, "\n");
#endif /* not G_OS_WIN32 */
  for (i = 0; i < length; i++)
    {
      gint j;

      /* ja:プログレスバー */
#ifdef G_OS_WIN32
      if ((glonglong)i * MIN (PROGRESS, 100) / length
                        != (glonglong)(i + 1) * MIN (PROGRESS, 100) / length)
        g_fprintf (stderr, "%d/%d (%d%%)\n",
                        i, (gint)length, (gint)((glonglong)i * 100 / length));
#else /* not G_OS_WIN32 */
      if (i * PROGRESS / length != (i + 1) * PROGRESS / length)
        g_fprintf (stderr, "\x1b[1A\x1b[%dC*\n",
                                    (gint)((glonglong)i * PROGRESS / length));
#endif /* not G_OS_WIN32 */
      rank[top].checked = g_malloc0 (length * sizeof (gint));
      rank[top].index = i;
      if (dynamic)
        {
          /* ja:動的解析 */
          Ia32Process *process;

          process = ia32_create_process_from_image (data, length, base, i);
          if (!process)
            {
              g_fprintf (stderr, "Create process error.\n");
              result = FALSE;
              break;
            }
          ia32_cpu_loop (process, NULL, step, 0, step, IA32_MESSAGE_MNEMONIC);
          rank[top].depth = trace (process->current->trace, length, base,
                                                            rank[top].checked);
          if (!ia32_destroy_process (process))
            {
              g_fprintf (stderr, "Destroy process error.\n");
              result = FALSE;
              break;
            }
        }
      else
        {
          /* ja:静的解析 */
          rank[top].depth = recursive (data, i, length, base,
                                                    rank[top].checked, cache);
        }
      /* ja:新規が既存を含むかチェック */
      for (j = top - 1; j >= 0; j--)
        if (rank[j].checked && rank[j].depth < rank[top].depth)
          {
            gint k;

            for (k = 0; k < length; k++)
              if (rank[j].checked[k] && !rank[top].checked[k])
                break;
            if (k >= length)
              {
                g_free (rank[j].checked);
                g_memmove (rank + j, rank + j + 1,
                                            (top - j - 1) * sizeof (Ranking));
                rank[top - 1].checked = NULL;
                rank[top - 1].index = -1;
                rank[top - 1].depth = 0;
              }
          }
      /* ja:新規を既存が含むかチェック */
      for (j = 0; j < top; j++)
        if (rank[j].checked)
          {
            gint k;

            for (k = 0; k < length; k++)
              if (!rank[j].checked[k] && rank[top].checked[k])
                break;
            if (k >= length)
              break;
          }
      /* ja:新規と既存を比較 */
      if (j >= top)
        for (j = top; j > 0; j--)
          if (rank[j - 1].depth < rank[j].depth)
            {
              gboolean *checked;
              gint tmp;

              tmp = rank[j - 1].index;
              rank[j - 1].index = rank[j].index;
              rank[j].index = tmp;
              tmp = rank[j - 1].depth;
              rank[j - 1].depth = rank[j].depth;
              rank[j].depth = tmp;
              checked = rank[j - 1].checked;
              rank[j - 1].checked = rank[j].checked;
              rank[j].checked = checked;
            }
      g_free (rank[top].checked);
      rank[top].checked = NULL;
    }
  g_free (cache);
  for (i = 0; i <= top; i++)
    g_free (rank[i].checked);
  if (result)
    {
      for (i = 0; i < top; i++)
        if (rank[i].index >= 0)
          g_fprintf (stderr, "%2d : %08x (%d)\n",
                                        i + 1, rank[i].index, rank[i].depth);
      g_fprintf (stderr, "\n");
      if (outfile)
        {
          guint8 *image;

          image = peimage_file_image_from_binary (data, length,
                                        base - header, rank[0].index, align);
          if (!image)
            {
              g_fprintf (stderr, "Image construction error.\n");
              result = FALSE;
            }
          else if (!g_file_set_contents (outfile,
                (const gchar *)image, pe_ioh_get_size_of_image (image), NULL))
            {
              g_fprintf (stderr, "\'%s\' save error.\n", outfile);
              result = FALSE;
            }
          g_free (image);
        }
    }
  g_free (rank);
  g_free (data);
  return result ? 0 : -1;
}
