/*
    gcommon
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "gcommon.h"
#ifdef G_OS_WIN32
# include <windows.h>
# include <tchar.h>
# include <shlwapi.h>
#endif /* G_OS_WIN32 */


/******************************************************************************
* Byte Order Macros                                                           *
******************************************************************************/
guint16
gcommon_swap_16 (const guint16 val)
{
  return (val >> 8) | (val << 8);
}


guint32
gcommon_swap_32 (const guint32 val)
{
  return ((val & 0x000000ff) << 24) | ((val & 0x0000ff00) << 8)
                    | ((val & 0x00ff0000) >> 8) | ((val & 0xff000000) >> 24);
}


guint64
gcommon_swap_64 (const guint64 val)
{
  guint8 *p, temp;
  guint64 v;

  v = val;
  p = (guint8 *)&v;
  temp = p[0]; p[0] = p[7]; p[7] = temp;
  temp = p[1]; p[1] = p[6]; p[6] = temp;
  temp = p[2]; p[2] = p[5]; p[5] = temp;
  temp = p[3]; p[3] = p[4]; p[4] = temp;
  return v;
}


/******************************************************************************
* Extended                                                                    *
******************************************************************************/
gchar *
g_strcpy (gchar       *dest,
          const gchar *src)
{
  return dest && src ? strcpy (dest, src) : dest;
}


gchar *
g_strncpy (gchar       *dest,
           const gchar *src,
           const gsize  n)
{
  return dest && src ? strncpy (dest, src, n) : dest;
}


gchar *
g_strcat (gchar       *dest,
          const gchar *src)
{
  return dest && src ? strcat (dest, src) : dest;
}


gchar *
g_strncat (gchar       *dest,
           const gchar *src,
           const gsize  n)
{
  return dest && src ? strncat (dest, src, n) : dest;
}


gint
g_strcmp (const gchar *s1,
          const gchar *s2)
{
  if (!s1 && !s2)
    return 0;
  else if (s1 && !s2)
    return G_MININT;
  else if (!s1 && s2)
    return G_MAXINT;
  return strcmp (s1, s2);
}


gint
g_strncmp (const gchar *s1,
           const gchar *s2,
           const gsize  n)
{
  if (!s1 && !s2)
    return 0;
  else if (s1 && !s2)
    return G_MININT;
  else if (!s1 && s2)
    return G_MAXINT;
  return strncmp (s1, s2, n);
}


gchar *
g_strchr (const gchar *s,
          const gint   c)
{
  return s ? strchr (s, c) : NULL;
}


gchar *
g_strrchr (const gchar *s,
           const gint   c)
{
  return s ? strrchr (s, c) : NULL;
}


gsize
g_strspn (const gchar *s,
          const gchar *accept)
{
  return s && accept ? strspn (s, accept) : 0;
}


gsize
g_strcspn (const gchar *s,
           const gchar *reject)
{
  return !s ? 0 : !reject ? g_strlen (s) : strcspn (s, reject);
}


gchar *
g_strpbrk (const gchar *s,
           const gchar *accept)
{
  return s && accept ? strpbrk (s, accept) : NULL;
}


gchar *
g_strstr (const gchar *haystack,
          const gchar *needle)
{
  return g_strstr_len (haystack, -1, needle);
}


gsize
g_strlen (const gchar *s)
{
  return s ? strlen (s) : 0;
}


gchar *
g_strtok (gchar       *str,
          const gchar *delim)
{
  return delim ? strtok (str, delim) : NULL;
}


glong
g_strtol (const gchar  *nptr,
          gchar       **endptr,
          gint          base)
{
  glong ret = 0;

  if (nptr)
    {
      gchar *p;

      ret = strtol (nptr, &p, base);
      if (endptr)
        *endptr = p;
    }
  return ret;
}


gulong
g_strtoul (const gchar  *nptr,
           gchar       **endptr,
           gint          base)
{
  gulong ret = 0;

  if (nptr)
    {
      gchar *p;

      ret = strtoul (nptr, &p, base);
      if (endptr)
        *endptr = p;
    }
  return ret;
}


gint
g_memcmp (gconstpointer s1,
          gconstpointer s2,
          const gsize   n)
{
  if (!s1 && !s2)
    return 0;
  else if (s1 && !s2)
    return G_MININT;
  else if (!s1 && s2)
    return G_MAXINT;
  return memcmp (s1, s2, n);
}


gpointer
g_memchr (gconstpointer s,
          const gint    c,
          const gsize   n)
{
  return s ? memchr (s, c, n) : NULL;
}


gpointer
g_memset (gpointer    s,
          const gint  c,
          const gsize n)
{
  return s ? memset (s, c, n) : s;
}


gint
g_asprintf (gchar       **string,
            const gchar  *format,
            ...)
{
  gsize len;
  va_list args;

  va_start (args, format);
  len = g_vasprintf (string, format, args);
  va_end (args);
  return len;
}


/*  ja:ファイルの絶対パスを取得する
    file,ファイル名
    path,パス名
     RET,フルパス                                                           */
gchar *
g_path_get_absolute (const gchar *file,
                     const gchar *path)
{
#ifdef G_OS_WIN32
  gchar *utf8str, *filename;
  gint i = 0, j;
  DWORD dwSize;             /* ja:フルパスの長さ */
  HANDLE hFind;             /* ja:検索ハンドル */
  LPTSTR p, lpszFile, lpszTemp, lpszLongName;

  if (!file)
    return NULL;
# ifdef UNICODE
  utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
  if (!utf8str)
    return g_strdup (file);
  lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
# else /* not UNICODE */
  lpszFile = g_win32_locale_filename_from_utf8 (file);
# endif /* not UNICODE */
  if (PathIsRelative (lpszFile))
    {
      int nLeng;
      LPTSTR lpszPath = NULL;

      if (path)
        {
# ifdef UNICODE
          utf8str = g_filename_to_utf8 (path, -1, NULL, NULL, NULL);
          if (utf8str)
            {
              lpszPath = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
              g_free (utf8str);
            }
# else /* not UNICODE */
          lpszPath = g_win32_locale_filename_from_utf8 (path);
# endif /* not UNICODE */
        }
      if (!lpszPath || PathIsRelative (lpszPath))
        {
          g_free (lpszPath);
          dwSize = GetCurrentDirectory (0, NULL);
          lpszPath = g_malloc (dwSize * sizeof (TCHAR));
          if (GetCurrentDirectory (dwSize, lpszPath) != dwSize - 1)
            {
              g_free (lpszPath);
              g_free (lpszFile);
              return g_strdup (file);
            }
        }
      PathRemoveBackslash (lpszPath);
      nLeng = lstrlen (lpszPath);
      lpszTemp = g_malloc ((nLeng + lstrlen (lpszFile) + 2) * sizeof (TCHAR));
      lstrcpy (lpszTemp, lpszPath);
      lpszTemp[nLeng] = G_DIR_SEPARATOR;
      lpszTemp[nLeng + 1] = '\0';
      lstrcat (lpszTemp, lpszFile);
      g_free (lpszPath);
      g_free (lpszFile);
      lpszFile = lpszTemp;
    }
  dwSize = GetFullPathName (lpszFile, 0, NULL, &p);
  if (dwSize == 0)
    {
      g_free (lpszFile);
      return g_strdup (file);
    }
  lpszTemp = g_malloc (dwSize * sizeof (TCHAR));
  if (GetFullPathName (lpszFile, dwSize, lpszTemp, &p) != dwSize - 1)
    {
      g_free (lpszTemp);
      g_free (lpszFile);
      return g_strdup (file);
    }
  dwSize--;
  g_free (lpszFile);
  lpszFile = lpszTemp;
  if (dwSize >= 2)
    {
      if (lpszFile[1] == ':')
        {
          lpszLongName = g_malloc (sizeof (TCHAR) * 4);
          lpszLongName[0] = (TCHAR)CharUpper ((LPTSTR)lpszFile[0]);
          lpszLongName[1] = ':';
          if (G_IS_DIR_SEPARATOR (lpszFile[2]))
            {
              lpszLongName[2] = G_DIR_SEPARATOR;
              i = 3;
            }
          else
            {
              i = 2;
            }
        }
      else if (G_IS_DIR_SEPARATOR (lpszFile[0])
                                        && G_IS_DIR_SEPARATOR (lpszFile[1]))
        {
          i = 2;
          while (i < dwSize)
            {
              if (G_IS_DIR_SEPARATOR (lpszFile[i]))
                break;
# ifdef UNICODE
              i++;
# else /* not UNICODE */
              i += IsDBCSLeadByteEx (CP_ACP, lpszFile[i]) ? 2 : 1;
# endif /* not UNICODE */
            }
          i++;
          lpszLongName = g_malloc ((i + 1) * sizeof (TCHAR));
          g_memmove (lpszLongName, lpszFile, i * sizeof (TCHAR));
        }
    }
  lpszLongName[i] = '\0';
  j = i;
  while (i <= dwSize)
    {
      if (lpszFile[i] == '\0' || G_IS_DIR_SEPARATOR (lpszFile[i]))
        {
          int nLeng;
          WIN32_FIND_DATA rcf;

          nLeng = lstrlen (lpszLongName);
          lpszLongName = g_realloc (lpszLongName,
                                        (nLeng + i - j + 2) * sizeof (TCHAR));
          g_memmove (lpszLongName + nLeng, lpszFile + j,
                                                    (i - j) * sizeof (TCHAR));
          lpszLongName[nLeng + i - j] = '\0';
          hFind = FindFirstFile (lpszLongName, &rcf);
          if (hFind != INVALID_HANDLE_VALUE)
            FindClose (hFind);
          else
            lstrcpy (rcf.cFileName, _T("."));
          if (lstrcmp (rcf.cFileName, _T(".")) != 0)
            {
              int nLen;

              nLen = lstrlen (rcf.cFileName);
              lpszLongName = g_realloc (lpszLongName,
                                        (nLeng + nLen + 2) * sizeof (TCHAR));
              g_memmove (lpszLongName + nLeng, rcf.cFileName,
                                                        nLen * sizeof (TCHAR));
              nLeng += nLen;
            }
          else
            {
              nLeng += i - j;
            }
          if (G_IS_DIR_SEPARATOR (lpszFile[i]))
            lpszLongName[nLeng++] = G_DIR_SEPARATOR;
          lpszLongName[nLeng] = '\0';
          j = i + 1;
        }
# ifdef UNICODE
      i++;
# else /* not UNICODE */
      i += IsDBCSLeadByteEx (CP_ACP, lpszFile[i]) ? 2 : 1;
# endif /* not UNICODE */
    }
  g_free (lpszFile);
# ifdef UNICODE
  utf8str = g_utf16_to_utf8 (lpszLongName, -1, NULL, NULL, NULL);
# else /* not UNICODE */
  utf8str = g_locale_to_utf8 (lpszLongName, -1, NULL, NULL, NULL);
# endif /* not UNICODE */
  g_free (lpszLongName);
  if (!utf8str)
    return g_strdup (file);
  filename = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
  return filename ? filename : g_strdup (file);
#else /* not G_OS_WIN32 */
  gchar *absolate;
  gint i, leng;

  if (!file)
    return NULL;
  if (g_path_is_absolute (file))
    {
      absolate = g_strdup (file);
    }
  else
    {
      if (path && g_path_is_absolute (path))
        {
          leng = g_strlen (path);
          absolate = leng > 0 && G_IS_DIR_SEPARATOR (path[leng - 1])
                        ? g_strconcat (path, file, NULL)
                        : g_strconcat (path, G_DIR_SEPARATOR_S, file, NULL);
        }
      else
        {
          gchar *dir;

          dir = g_get_current_dir ();
          absolate = g_strconcat (dir, G_DIR_SEPARATOR_S, file, NULL);
          g_free (dir);
        }
    }
  leng = g_strlen (absolate);
  i = 0;
  while (i < leng)
    if (G_IS_DIR_SEPARATOR (absolate[i])
                                    && G_IS_DIR_SEPARATOR (absolate[i + 1]))
      {
        leng--;
        g_memmove (absolate + i + 1, absolate + i + 2, leng - i);
      }
    else
      {
        i++;
      }
  i = 0;
  while (i < leng)
    if (G_IS_DIR_SEPARATOR (absolate[i]) && absolate[i + 1] == '.'
        && (G_IS_DIR_SEPARATOR (absolate[i + 2]) || absolate[i + 2] == '\0'))
      {
        leng -= 2;
        g_memmove (absolate + i, absolate + i + 2, leng - i + 1);
      }
    else
      {
        i++;
      }
  i = 0;
  while (i < leng)
    if (G_IS_DIR_SEPARATOR (absolate[i]) && absolate[i + 1] == '.'
        && absolate[i + 2] == '.'
        && (G_IS_DIR_SEPARATOR (absolate[i + 3]) || absolate[i + 3] == '\0'))
      {
        gint j;

        leng -= 3;
        g_memmove (absolate + i, absolate + i + 3, leng - i + 1);
        for (j = i - 1; j >= 0; j--)
          if (G_IS_DIR_SEPARATOR (absolate[j]))
            break;
        if (j >= 0)
          {
            g_memmove (absolate + j, absolate + i, leng - i + 1);
            leng -= i - j;
            i = j;
          }
      }
    else
      {
        i++;
      }
  for (i = 0; i < leng; i++)
    if (G_IS_DIR_SEPARATOR (absolate[i]))
      absolate[i] = G_DIR_SEPARATOR;
  if (leng <= 0)
    g_strcpy (absolate, G_DIR_SEPARATOR_S);
  return absolate;
#endif /* not G_OS_WIN32 */
}


/*  ja:ファイルの相対パスを取得する
    file,ファイル名(終点)
    path,パス名(起点)
     RET,フルパス                                                           */
gchar *
g_path_get_relative (const gchar *file,
                     const gchar *path)
{
  gchar *absolate, *relative = NULL, *name;
  gchar *path_to, *path_from, **ary_to, **ary_from;
  gint i, j;

  if (!file)
    return NULL;
  /* ja:終点 */
  absolate = g_path_get_absolute (file, NULL);
  path_to = g_path_get_dirname (absolate);
  g_free (absolate);
  ary_to = g_strsplit (path_to, G_DIR_SEPARATOR_S, 0);
  /* ja:起点 */
  path_from = path ? g_path_get_absolute (path, NULL) : g_get_current_dir ();
  ary_from = g_strsplit (path_from, G_DIR_SEPARATOR_S, 0);
  if (!ary_to || !ary_from || !ary_to[0] || !ary_from[0]
#ifdef G_OS_WIN32
                              || g_strfilecmp (ary_to[0], ary_from[0]) != 0
    || (G_IS_DIR_SEPARATOR (path_to[0]) && G_IS_DIR_SEPARATOR (path_to[1])
    && G_IS_DIR_SEPARATOR (path_from[0]) && G_IS_DIR_SEPARATOR (path_from[1])
                                && g_strfilecmp (ary_to[2], ary_from[2]) != 0)
#endif /* G_OS_WIN32 */
                                                                            )
    {
      g_free (path_to);
      g_free (path_from);
      g_strfreev (ary_to);
      g_strfreev (ary_from);
      return NULL;
    }
  g_free (path_to);
  g_free (path_from);
  for (i = 0; ary_to[i] && ary_from[i]; i++)
    if (g_strfilecmp (ary_to[i], ary_from[i]) != 0)
      break;
  for (j = i; ary_from[j]; j++)
    if (*(ary_from[j]) != '\0')
      {
        gchar *tmp;

        tmp = relative ? g_strconcat (relative, ".."G_DIR_SEPARATOR_S, NULL)
                       : g_strdup (".."G_DIR_SEPARATOR_S);
        g_free (relative);
        relative = tmp;
      }
  for (j = i; ary_to[j]; j++)
    {
      gchar *tmp;

      tmp = relative
                ? g_strconcat (relative, ary_to[j], G_DIR_SEPARATOR_S, NULL)
                : g_strconcat (ary_to[j], G_DIR_SEPARATOR_S, NULL);
      g_free (relative);
      relative = tmp;
    }
  g_strfreev (ary_to);
  g_strfreev (ary_from);
  name = g_path_get_basename (file);
  if (name)
    {
      gchar *tmp;

      tmp = relative ? g_strconcat (relative, name, NULL) : g_strdup (name);
      g_free (relative);
      relative = tmp;
      g_free (name);
    }
  return relative;
}
