/*
    gcommon
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "gcommon.h"


/******************************************************************************
* Data Checksums                                                              *
******************************************************************************/
#if ! GLIB_CHECK_VERSION(2,16,0)
# define SHA1_DATASIZE   64
# define SHA1_DIGEST_LEN 20


struct _GChecksum
{
  gchar *digest_str;
  guint32 buf[5];
  guint32 bits[2];
  guint32 data[16];
  guchar digest[SHA1_DIGEST_LEN];
};


gssize
g_checksum_type_get_length (GChecksumType checksum_type)
{
  return checksum_type == G_CHECKSUM_SHA1 ? SHA1_DIGEST_LEN : -1;
}


GChecksum *
g_checksum_new (GChecksumType checksum_type)
{
  GChecksum *checksum = NULL;

  if (checksum_type == G_CHECKSUM_SHA1)
    {
      checksum = g_malloc0 (sizeof (GChecksum));
      g_checksum_reset (checksum);
    }
  return checksum;
}


GChecksum *g_checksum_copy (const GChecksum *checksum)
{
  GChecksum *copy = NULL;

  if (checksum)
    {
      copy = g_malloc (sizeof (GChecksum));
      *copy = *checksum;
      copy->digest_str = g_strdup (checksum->digest_str);
    }
  return copy;
}


void
g_checksum_free (GChecksum *checksum)
{
  if (checksum)
    {
      g_free (checksum->digest_str);
      g_free (checksum);
    }
}


void
g_checksum_reset (GChecksum *checksum)
{
  if (checksum)
    {
      g_free (checksum->digest_str);
      checksum->digest_str = NULL;
      checksum->buf[0] = 0x67452301;
      checksum->buf[1] = 0xefcdab89;
      checksum->buf[2] = 0x98badcfe;
      checksum->buf[3] = 0x10325476;
      checksum->buf[4] = 0xc3d2e1f0;
      checksum->bits[0] = checksum->bits[1] = 0;
    }
}


static void
sha1_byte_reverse (guint32    *buf,
                   const gint  len)
{
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
  gint i;

  for (i = len / sizeof (guint32); i > 0; i--)
    {
      *buf = GUINT32_SWAP_LE_BE (*buf);
      buf++;
    }
#endif /* G_BYTE_ORDER == G_LITTLE_ENDIAN */
}


# define f1(x,y,z) ((z)^((x)&((y)^(z))))
# define f2(x,y,z) ((x)^(y)^(z))
# define f3(x,y,z) (((x)&(y))|((z)&((x)|(y))))
# define f4(x,y,z) ((x)^(y)^(z))
# define k1 0x5a827999
# define k2 0x6ed9eba1
# define k3 0x8f1bbcdc
# define k4 0xca62c1d6
# define rotl(n,x) (((x)<<(n))|((x)>>(32-(n))))
# define expand(w,i) ((w)[(i)&15]=rotl(1,((w)[(i)&15]^(w)[((i)-14)&15]^(w)[((i)-8)&15]^(w)[((i)-3)&15])))
# define subround(a,b,c,d,e,f,k,data) ((e)+=rotl(5,(a))+f((b),(c),(d))+(k)+(data),(b)=rotl(30,(b)))


static void
sha1_transform (guint32 buf[5],
                guint32 in[16])
{
  guint32 a, b, c, d, e;

  a = buf[0];
  b = buf[1];
  c = buf[2];
  d = buf[3];
  e = buf[4];

  subround (a, b, c, d, e, f1, k1, in[0]);
  subround (e, a, b, c, d, f1, k1, in[1]);
  subround (d, e, a, b, c, f1, k1, in[2]);
  subround (c, d, e, a, b, f1, k1, in[3]);
  subround (b, c, d, e, a, f1, k1, in[4]);
  subround (a, b, c, d, e, f1, k1, in[5]);
  subround (e, a, b, c, d, f1, k1, in[6]);
  subround (d, e, a, b, c, f1, k1, in[7]);
  subround (c, d, e, a, b, f1, k1, in[8]);
  subround (b, c, d, e, a, f1, k1, in[9]);
  subround (a, b, c, d, e, f1, k1, in[10]);
  subround (e, a, b, c, d, f1, k1, in[11]);
  subround (d, e, a, b, c, f1, k1, in[12]);
  subround (c, d, e, a, b, f1, k1, in[13]);
  subround (b, c, d, e, a, f1, k1, in[14]);
  subround (a, b, c, d, e, f1, k1, in[15]);
  subround (e, a, b, c, d, f1, k1, expand (in, 16));
  subround (d, e, a, b, c, f1, k1, expand (in, 17));
  subround (c, d, e, a, b, f1, k1, expand (in, 18));
  subround (b, c, d, e, a, f1, k1, expand (in, 19));

  subround (a, b, c, d, e, f2, k2, expand (in, 20));
  subround (e, a, b, c, d, f2, k2, expand (in, 21));
  subround (d, e, a, b, c, f2, k2, expand (in, 22));
  subround (c, d, e, a, b, f2, k2, expand (in, 23));
  subround (b, c, d, e, a, f2, k2, expand (in, 24));
  subround (a, b, c, d, e, f2, k2, expand (in, 25));
  subround (e, a, b, c, d, f2, k2, expand (in, 26));
  subround (d, e, a, b, c, f2, k2, expand (in, 27));
  subround (c, d, e, a, b, f2, k2, expand (in, 28));
  subround (b, c, d, e, a, f2, k2, expand (in, 29));
  subround (a, b, c, d, e, f2, k2, expand (in, 30));
  subround (e, a, b, c, d, f2, k2, expand (in, 31));
  subround (d, e, a, b, c, f2, k2, expand (in, 32));
  subround (c, d, e, a, b, f2, k2, expand (in, 33));
  subround (b, c, d, e, a, f2, k2, expand (in, 34));
  subround (a, b, c, d, e, f2, k2, expand (in, 35));
  subround (e, a, b, c, d, f2, k2, expand (in, 36));
  subround (d, e, a, b, c, f2, k2, expand (in, 37));
  subround (c, d, e, a, b, f2, k2, expand (in, 38));
  subround (b, c, d, e, a, f2, k2, expand (in, 39));

  subround (a, b, c, d, e, f3, k3, expand (in, 40));
  subround (e, a, b, c, d, f3, k3, expand (in, 41));
  subround (d, e, a, b, c, f3, k3, expand (in, 42));
  subround (c, d, e, a, b, f3, k3, expand (in, 43));
  subround (b, c, d, e, a, f3, k3, expand (in, 44));
  subround (a, b, c, d, e, f3, k3, expand (in, 45));
  subround (e, a, b, c, d, f3, k3, expand (in, 46));
  subround (d, e, a, b, c, f3, k3, expand (in, 47));
  subround (c, d, e, a, b, f3, k3, expand (in, 48));
  subround (b, c, d, e, a, f3, k3, expand (in, 49));
  subround (a, b, c, d, e, f3, k3, expand (in, 50));
  subround (e, a, b, c, d, f3, k3, expand (in, 51));
  subround (d, e, a, b, c, f3, k3, expand (in, 52));
  subround (c, d, e, a, b, f3, k3, expand (in, 53));
  subround (b, c, d, e, a, f3, k3, expand (in, 54));
  subround (a, b, c, d, e, f3, k3, expand (in, 55));
  subround (e, a, b, c, d, f3, k3, expand (in, 56));
  subround (d, e, a, b, c, f3, k3, expand (in, 57));
  subround (c, d, e, a, b, f3, k3, expand (in, 58));
  subround (b, c, d, e, a, f3, k3, expand (in, 59));

  subround (a, b, c, d, e, f4, k4, expand (in, 60));
  subround (e, a, b, c, d, f4, k4, expand (in, 61));
  subround (d, e, a, b, c, f4, k4, expand (in, 62));
  subround (c, d, e, a, b, f4, k4, expand (in, 63));
  subround (b, c, d, e, a, f4, k4, expand (in, 64));
  subround (a, b, c, d, e, f4, k4, expand (in, 65));
  subround (e, a, b, c, d, f4, k4, expand (in, 66));
  subround (d, e, a, b, c, f4, k4, expand (in, 67));
  subround (c, d, e, a, b, f4, k4, expand (in, 68));
  subround (b, c, d, e, a, f4, k4, expand (in, 69));
  subround (a, b, c, d, e, f4, k4, expand (in, 70));
  subround (e, a, b, c, d, f4, k4, expand (in, 71));
  subround (d, e, a, b, c, f4, k4, expand (in, 72));
  subround (c, d, e, a, b, f4, k4, expand (in, 73));
  subround (b, c, d, e, a, f4, k4, expand (in, 74));
  subround (a, b, c, d, e, f4, k4, expand (in, 75));
  subround (e, a, b, c, d, f4, k4, expand (in, 76));
  subround (d, e, a, b, c, f4, k4, expand (in, 77));
  subround (c, d, e, a, b, f4, k4, expand (in, 78));
  subround (b, c, d, e, a, f4, k4, expand (in, 79));

  buf[0] += a;
  buf[1] += b;
  buf[2] += c;
  buf[3] += d;
  buf[4] += e;
}


void
g_checksum_update (GChecksum    *checksum,
                   const guchar *data,
                   gssize        length)
{
  if (checksum && data && length != 0 && !checksum->digest_str)
    {
      guint32 tmp;
      guint count;

      if (length < 0)
        length = g_strlen ((const gchar *)data);

      tmp = checksum->bits[0];
      checksum->bits[0] = tmp + ((guint32)length << 3);
      if (checksum->bits[0] < tmp)
        checksum->bits[1]++;
      checksum->bits[1] += length >> 29;

      count = (tmp >> 3) & 0x3f;
      if (count)
        {
          guchar *p;

          p = (guchar *)checksum->data + count;
          count = SHA1_DATASIZE - count;
          if (length < count)
            {
              g_memmove (p, data, length);
              return;
            }

          g_memmove (p, data, count);

          sha1_byte_reverse (checksum->data, SHA1_DATASIZE);
          sha1_transform (checksum->buf, checksum->data);

          data += count;
          length -= count;
        }

      while (length >= SHA1_DATASIZE)
        {
          g_memmove (checksum->data, data, SHA1_DATASIZE);

          sha1_byte_reverse (checksum->data, SHA1_DATASIZE);
          sha1_transform (checksum->buf, checksum->data);

          data += SHA1_DATASIZE;
          length -= SHA1_DATASIZE;
        }

      g_memmove (checksum->data, data, length);
    }
}


const gchar *
g_checksum_get_string (GChecksum *checksum)
{
  gint i, count;
  guchar *p;

  if (!checksum)
    return NULL;
  if (checksum->digest_str)
    return checksum->digest_str;

  count = (checksum->bits[0] >> 3) & 0x3f;
  p = (guchar *)checksum->data + count;
  *p++ = 0x80;
  count = SHA1_DATASIZE - 1 - count;
  if (count < 8)
    {
      g_memset (p, 0, count);
      sha1_byte_reverse (checksum->data, SHA1_DATASIZE);
      sha1_transform (checksum->buf, checksum->data);
      g_memset (checksum->data, 0, SHA1_DATASIZE - 8);
    }
  else
    {
      g_memset (p, 0, count - 8);
    }
  checksum->data[14] = checksum->bits[1];
  checksum->data[15] = checksum->bits[0];
  sha1_byte_reverse (checksum->data, SHA1_DATASIZE - 8);
  sha1_transform (checksum->buf, checksum->data);
  sha1_byte_reverse (checksum->buf, SHA1_DIGEST_LEN);
  g_memmove (checksum->digest, checksum->buf, SHA1_DIGEST_LEN);
  g_memset (checksum->buf, 0, sizeof (checksum->buf));
  g_memset (checksum->data, 0, sizeof (checksum->data));

  checksum->digest_str = g_malloc ((SHA1_DIGEST_LEN * 2 + 1) * sizeof (gchar));
  for (i = 0; i < SHA1_DIGEST_LEN; i++)
    {
      const static gchar hex[] = "0123456789abcdef";
      guint8 byte;

      byte = checksum->digest[i];
      checksum->digest_str[i * 2] = hex[byte >> 4];
      checksum->digest_str[i * 2 + 1] = hex[byte & 0xf];
    }
  checksum->digest_str[SHA1_DIGEST_LEN * 2] = 0;

  return checksum->digest_str;
}


void
g_checksum_get_digest (GChecksum *checksum,
                       guint8    *buffer,
                       gsize     *digest_len)
{
  if (checksum && buffer && digest_len
                && *digest_len >= g_checksum_type_get_length (G_CHECKSUM_SHA1)
                && g_checksum_get_string (checksum))
    {
      g_memmove (buffer, checksum->digest, SHA1_DIGEST_LEN);
      *digest_len = SHA1_DIGEST_LEN;
    }
}


gchar *
g_compute_checksum_for_data (GChecksumType  checksum_type,
                             const guchar  *data,
                             gsize          length)
{
  gchar *str = NULL;

  if (checksum_type == G_CHECKSUM_SHA1 && (data || length == 0))
    {
      GChecksum *checksum;

      checksum = g_checksum_new (checksum_type);
      g_checksum_update (checksum, data, length);
      str = g_strdup (g_checksum_get_string (checksum));
      g_checksum_free (checksum);
    }
  return str;
}


gchar *
g_compute_checksum_for_string (GChecksumType  checksum_type,
                               const gchar   *str,
                               gssize         length)
{
  if (!str && length != 0)
    return NULL;
  if (length < 0)
    length = g_strlen (str);
  return g_compute_checksum_for_data (checksum_type,
                                                (const guchar *)str, length);
}
#endif /* not GLIB_CHECK_VERSION(2,16,0) */
