/*
    IA32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef __MEMORY_H__
#define __MEMORY_H__


#include "ia32.h"


G_BEGIN_DECLS


#define IA32_MEMORY_BASE_IMAGE  0x00400000
#define IA32_MEMORY_BASE_SYSTEM 0x70000000
#define IA32_MEMORY_ATTR_USER   1
#define IA32_MEMORY_ATTR_SYSTEM 2
#define IA32_MEMORY_ATTR_STACK  3


typedef struct _Ia32Memory
{
  gpointer data;
  gpointer level;
  guint32 address;
  guint32 length;
  guint attribute;
} Ia32Memory;


/******************************************************************************
*                                                                             *
* ja:メモリ関数群                                                             *
*                                                                             *
******************************************************************************/
/*  ja:仮想メモリ登録
      process,プロセス構造体
      address,仮想アドレス
       length,サイズ
    attribute,属性
         data,実アドレス
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
ia32_memory_register (Ia32Process   *process,
                      const guint32  address,
                      const gsize    length,
                      const guint    attribute,
                      gpointer       data);


/*  ja:仮想メモリ確保
      process,プロセス構造体
      address,仮想アドレス
       length,サイズ
    attribute,属性
          RET,仮想アドレス,IA32_MEMORY_NULL:エラー                          */
guint32
ia32_memory_alloc (Ia32Process   *process,
                   const guint32  address,
                   const gsize    length,
                   const guint    attribute);


/*  ja:仮想メモリ解放
    process,プロセス構造体
    address,仮想アドレス
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
ia32_memory_free (Ia32Process   *process,
                  const guint32  address);


/*  ja:実アドレスから仮想アドレスを求める
    process,プロセス構造体
       data,実アドレス
        RET,仮想アドレス,IA32_MEMORY_NULL:エラー                            */
guint32
ia32_memory_virtual_address (Ia32Process   *process,
                             gconstpointer  data);


/*  ja:仮想アドレスから実アドレスを求める
    process,プロセス構造体
    address,仮想アドレス
        RET,実アドレス,NULL:エラー                                          */
gpointer
ia32_memory_real_address (Ia32Process   *process,
                          const guint32  address);


/*  ja:仮想アドレスから実サイズを求める
    process,プロセス構造体
    address,仮想アドレス
        RET,実サイズ,0:エラー                                               */
gsize
ia32_memory_real_size (Ia32Process   *process,
                       const guint32  address);


/******************************************************************************
*                                                                             *
* ja:メモリ文字列関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:仮想アドレスが文字列か判定する
    process,プロセス構造体
    address,仮想アドレス
        RET,TRUE:文字列,FALSE:エラー                                        */
gboolean
ia32_memory_is_string (Ia32Process   *process,
                       const guint32  address);


/*  ja:仮想アドレスがワイド文字列か判定する
    process,プロセス構造体
    address,仮想アドレス
        RET,TRUE:ワイド文字列,FALSE:エラー                                  */
gboolean
ia32_memory_is_stringw (Ia32Process   *process,
                        const guint32  address);


/******************************************************************************
*                                                                             *
* ja:メモリI/O関数群                                                          *
*                                                                             *
******************************************************************************/
/*  ja:仮想アドレスからデータを読み込む
    process,プロセス構造体
    address,仮想アドレス
     buffer,データ
     length,サイズ
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_read (Ia32Process   *process,
                  const guint32  address,
                  gpointer       buffer,
                  const gsize    length);


/*  ja:仮想アドレスから1バイトデータを読み込む
    process,プロセス構造体
    address,仮想アドレス
      value,値
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_read_byte (Ia32Process   *process,
                       const guint32  address,
                       guint8        *value);


/*  ja:仮想アドレスから2バイトデータを読み込む
    process,プロセス構造体
    address,仮想アドレス
      value,値
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_read_word (Ia32Process   *process,
                       const guint32  address,
                       guint16       *value);


/*  ja:仮想アドレスから2バイトデータを読み込む
    process,プロセス構造体
    address,仮想アドレス
      value,値
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_read_dword (Ia32Process   *process,
                        const guint32  address,
                        guint32       *value);


/*  ja:仮想アドレスにデータを書き込む
    process,プロセス構造体
    address,仮想アドレス
     buffer,データ
     length,サイズ
      level,レベル(-1:変更なし)
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_write (Ia32Process   *process,
                   const guint32  address,
                   gconstpointer  buffer,
                   const gsize    length,
                   const gint8    level);


/*  ja:仮想アドレスに1バイトデータを書き込む
    process,プロセス構造体
    address,仮想アドレス
      value,値
      level,レベル(-1:変更なし)
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_write_byte (Ia32Process   *process,
                        const guint32  address,
                        const guint8   value,
                        const gint8    level);


/*  ja:仮想アドレスに2バイトデータを書き込む
    process,プロセス構造体
    address,仮想アドレス
      value,値
      level,レベル(-1:変更なし)
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_write_word (Ia32Process   *process,
                        const guint32  address,
                        const guint16  value,
                        const gint8    level);


/*  ja:仮想アドレスに4バイトデータを書き込む
    process,プロセス構造体
    address,仮想アドレス
      value,値
      level,レベル(-1:変更なし)
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_write_dword (Ia32Process   *process,
                         const guint32  address,
                         const guint32  value,
                         const gint8    level);


/*  ja:仮想アドレスのレベルを求める
    process,プロセス構造体
    address,仮想アドレス
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_get_level (Ia32Process   *process,
                       const guint32  address);


G_END_DECLS


#endif /* __MEMORY_H__ */
