/*
    IA32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "message.h"
#include "memory.h"
#include "module.h"
#include "stack.h"
#include "misc/opcode.h"


static gchar *
ia32_message_escape (const gchar *str)
{
  gchar *html = NULL;

  if (str)
    {
      gsize length = 0;
      gchar *p, *q, *esc;

      esc = g_strescape (str, NULL);
      for (p = esc; *p != '\0'; p++)
        switch (*p)
          {
            case '\"' : length++;
            case '&'  : length++;
            case '<'  :
            case '>'  : length += 3;
            default   : length++;
          }
      q = html = g_malloc ((length + 1) * sizeof (gchar));
      for (p = esc; *p != '\0'; p++)
        switch (*p)
          {
            case '\"' : g_strcpy (q, "&quot;"); q += 6; break;
            case '&'  : g_strcpy (q, "&amp;");  q += 5; break;
            case '<'  : g_strcpy (q, "&lt;");   q += 4; break;
            case '>'  : g_strcpy (q, "&gt;");   q += 4; break;
            default   : *q = *p; q++;
          }
      *q = '\0';
      g_free (esc);
    }
  return html;
}


/******************************************************************************
*                                                                             *
* ja:メッセージ記録関数群                                                     *
*                                                                             *
******************************************************************************/
/*  ja:エラーメッセージを記録する
    thread,スレッド構造体
       str,メッセージ                                                       */
void
ia32_message_record_error (Ia32Thread  *thread,
                           const gchar *str)
{
  if (thread && str
            && !g_list_find_custom (thread->err, str, (GCompareFunc)strcmp))
    thread->err = g_list_append (thread->err, g_strdup (str));
}


/*  ja:例外エラーを記録する
    thread,スレッド構造体
    vector,割り込みベクタ                                                   */
void
ia32_message_record_exception (Ia32Thread *thread,
                               const gint  vector)
{
  const static gchar *str[] = {/*  0 */ "Divide Error",
                               /*  1 */ "Debug",
                               /*  2 */ "NMI Interrupt",
                               /*  3 */ "Breakpoint",
                               /*  4 */ "Overflow",
                               /*  5 */ "Bound Range Exceeded",
                               /*  6 */ "Invalid Opcode",
                               /*  7 */ "Device Not Available",
                               /*  8 */ "Double Fault",
                               /*  9 */ "CoProcessor Segment Overrun",
                               /* 10 */ "Invalid TSS",
                               /* 11 */ "Segment Not Present",
                               /* 12 */ "Stack Segment Fault",
                               /* 13 */ "General Protection",
                               /* 14 */ "Page Fault",
                               /* 15 */ NULL,
                               /* 16 */ "Floating-Point Error",
                               /* 17 */ "Alignment Check",
                               /* 18 */ "Machine Check",
                               /* 19 */ "SIMD Floating-Point Exception"};

  if (vector < G_N_ELEMENTS (str) && str[vector])
    ia32_message_record_error (thread, str[vector]);
}


/*  ja:API呼び出しを記録する
    thread,スレッド構造体
       str,API呼び出し                                                      */
void
ia32_message_record_api (Ia32Thread  *thread,
                         const gchar *str)
{
  if (thread && str)
    thread->api = g_list_append (thread->api, g_strdup (str));
}


/*  ja:API呼び出しを記録する
    thread,スレッド構造体
      name,APIの名前                                                        */
void
ia32_message_record_api_with_argument (Ia32Thread  *thread,
                                       const gchar *name,
                                       ...)
{
  gchar *arg = NULL, *argv, *ret, *str;
  gint argc;
  va_list ap;

  va_start (ap, name);
  ret = va_arg (ap, gint) == IA32_MESSAGE_DEC
            ? g_strdup_printf ("%08X (%d)", thread->reg.eax, thread->reg.eax)
            : g_strdup_printf ("%08X", thread->reg.eax);
  for (argc = 0; (argv = ia32_message_escape (va_arg (ap, gchar *))); argc++)
    {
      gchar *tag;

      switch (va_arg (ap, gint))
        {
          case IA32_MESSAGE_HEX:
            tag = g_strdup_printf ("<td colspan=\"2\">%08X</td>",
                                        ia32_stack_value (thread, argc + 1));
            break;
          case IA32_MESSAGE_DEC:
            {
              guint32 value;

              value = ia32_stack_value (thread, argc + 1);
              tag = g_strdup_printf ("<td colspan=\"2\">%08X (%d)</td>",
                                                        value, (gint)value);
            }
            break;
          case IA32_MESSAGE_STRING:
            {
              guint32 address;

              address = ia32_stack_value (thread, argc + 1);
              if (ia32_memory_is_string (thread->process, address))
                {
                  gchar *esc;

                  esc = ia32_message_escape (ia32_memory_real_address
                                                (thread->process, address));
                  tag = g_strdup_printf ("<td>%08X</td><td>%s</td>",
                                                                address, esc);
                  g_free (esc);
                }
              else
                {
                  tag = g_strdup_printf ("<td colspan=\"2\">%08X</td>",
                                                                    address);
                }
            }
            break;
          case IA32_MESSAGE_STRINGW:
            {
              guint32 address;

              address = ia32_stack_value (thread, argc + 1);
              if (ia32_memory_is_stringw (thread->process, address))
                {
                  gchar *esc, *utf8str;

                  utf8str = g_utf16_to_utf8 (ia32_memory_real_address
                            (thread->process, address), -1, NULL, NULL, NULL);
                  esc = ia32_message_escape (utf8str);
                  g_free (utf8str);
                  tag = g_strdup_printf ("<td>%08X</td><td>%s</td>",
                                                                address, esc);
                  g_free (esc);
                }
              else
                {
                  tag = g_strdup_printf ("<td colspan=\"2\">%08X</td>",
                                                                    address);
                }
            }
            break;
          default:
            tag = g_strdup ("<td colspan=\"2\"></td>");
        }
      if (arg)
        {
          gchar *tmp;

          tmp = g_strconcat (arg, "<tr>",
                                    "<td>", argv, "</td>", tag, "</tr>", NULL);
          g_free (arg);
          arg = tmp;
        }
      else
        {
          arg = g_strconcat ("<td>", argv, "</td>", tag, "</tr>", NULL);
        }
      g_free (tag);
      g_free (argv);
    }
  va_end (ap);
  if (!arg)
    arg = g_strdup ("<td colspan=\"3\"></td>");
  if (argc > 1)
    str = g_strdup_printf ("<tr>"
                           "<td rowspan=\"%d\">%s</td>"
                           "<td rowspan=\"%d\">%08X</td>"
                           "<td rowspan=\"%d\">%s</td>%s",
            argc, name, argc, ia32_stack_value (thread, 0), argc, ret, arg);
  else
    str = g_strdup_printf ("<tr><td>%s</td><td>%08X</td><td>%s</td>%s",
                                name, ia32_stack_value (thread, 0), ret, arg);
  g_free (arg);
  g_free (ret);
  ia32_message_record_api (thread, str);
  g_free (str);
}


/*  ja:レジスタの内容を記録/表示する
    thread,スレッド構造体
      tail,記録するステップ数(0:随時出力)
     flags,フラグ                                                           */
void
ia32_message_record_trace (Ia32Thread  *thread,
                           const gint   tail,
                           const guint  flags)
{
  if (thread)
    {
      gchar *str;

      str = flags & IA32_MESSAGE_REGISTER ? g_strdup_printf (
          "EAX = %08X EBX = %08X ECX = %08X EDX = %08X ESI = %08X EDI = %08X\n"
          "EIP = %08X ESP = %08X EBP = %08X EFL = %08X "
          "OV=%d UP=%d EI=%d PL=%d ZR=%d AC=%d PE=%d CY=%d\n\n",
          thread->reg.eax,
          thread->reg.ebx,
          thread->reg.ecx,
          thread->reg.edx,
          thread->reg.esi,
          thread->reg.edi,
          thread->reg.eip,
          thread->reg.esp,
          thread->reg.ebp,
          thread->reg.eflags,
          (thread->reg.eflags >> 11) & 1,
          (thread->reg.eflags >> 10) & 1,
          (thread->reg.eflags >> 9) & 1,
          (thread->reg.eflags >> 7) & 1,
          (thread->reg.eflags >> 6) & 1,
          (thread->reg.eflags >> 4) & 1,
          (thread->reg.eflags >> 2) & 1,
          thread->reg.eflags & 1) : NULL;
      if (flags & IA32_MESSAGE_MNEMONIC)
        {
          Opcode op;

          op.index = 0;
          op.mnemonic[0] = '\0';
          if (opcode_parse (
                ia32_memory_real_address (thread->process, thread->reg.eip),
                ia32_memory_real_size (thread->process, thread->reg.eip),
                thread->reg.eip, &op))
            {
              if (str)
                {
                  gchar *tmp;

                  tmp = g_strdup_printf ("%s\n%s", op.mnemonic, str);
                  g_free (str);
                  str = tmp;
                }
              else
                {
                  str = g_strdup_printf ("%08x %s\n",
                                                thread->reg.eip, op.mnemonic);
                }
            }
        }
      if (str)
        {
          if (tail > 0)
            {
              thread->trace = g_list_append (thread->trace, str);
              if (g_list_length (thread->trace) > tail)
                {
                  GList *glist;

                  glist = g_list_first (thread->trace);
                  g_free (glist->data);
                  thread->trace = g_list_delete_link (thread->trace, glist);
                }
            }
          else
            {
              g_fprintf (stdout, "%s", str);
              g_free (str);
            }
        }
    }
}


/******************************************************************************
*                                                                             *
* ja:メッセージ表示関数群                                                     *
*                                                                             *
******************************************************************************/
/*  ja:レジスタの内容を表示する
        fp,ファイルポインタ
    thread,スレッド構造体                                                   */
void
ia32_message_disp_trace (FILE        *fp,
                         Ia32Thread *thread)
{
  if (thread)
    {
      GList *glist;

      for (glist = g_list_first (thread->trace);
                                            glist; glist = g_list_next (glist))
        g_fprintf (fp, "%s", (gchar *)glist->data);
    }
}


/*  ja:レジスタの内容を表示する
        fp,ファイルポインタ
    thread,スレッド構造体                                                   */
static void
ia32_message_disp_register (FILE       *fp,
                            Ia32Thread *thread)
{
  if (thread)
    g_fprintf (fp,
      "EAX = %08X EBX = %08X ECX = %08X EDX = %08X ESI = %08X EDI = %08X\n"
      "EIP = %08X ESP = %08X EBP = %08X EFL = %08X "
      "OV=%d UP=%d EI=%d PL=%d ZR=%d AC=%d PE=%d CY=%d\n\n",
      thread->reg.eax,
      thread->reg.ebx,
      thread->reg.ecx,
      thread->reg.edx,
      thread->reg.esi,
      thread->reg.edi,
      thread->reg.eip,
      thread->reg.esp,
      thread->reg.ebp,
      thread->reg.eflags,
      (thread->reg.eflags >> 11) & 1,
      (thread->reg.eflags >> 10) & 1,
      (thread->reg.eflags >> 9) & 1,
      (thread->reg.eflags >> 7) & 1,
      (thread->reg.eflags >> 6) & 1,
      (thread->reg.eflags >> 4) & 1,
      (thread->reg.eflags >> 2) & 1,
      thread->reg.eflags & 1);
}


/*  ja:情報を表示する
         fp,ファイルポインタ
    process,プロセス構造体
    verbose,TRUE:詳細表示,FALSE:簡易表示
    appname,アプリケーション名                                              */
void
ia32_message_disp_info (FILE           *fp,
                        Ia32Process    *process,
                        const gboolean  verbose,
                        const gchar    *appname)
{
  if (process)
    {
      GList *glist;

      /* ja:プロセス */
      g_fprintf (fp,
        "%s "VERSION" ("BUILD_ENVIRONMENT")\n"
        "\n"
        "Process ID : %u\n"
        "Exit Code : %08X\n",
        appname,
        process->id,
        process->exitcode);
      if (process->address != IA32_MEMORY_NULL)
        g_fprintf (fp,
          "Executable File : %s\n"
          "Base Address : %08X\n",
          ia32_module_get_filename (process, process->address),
          process->address);
      if (process->winmain != IA32_MEMORY_NULL)
        g_fprintf (fp, "Entry Point : %08X (%d)\n",
                                            process->winmain, process->level);
      g_fprintf (fp, "\n");
      /* ja:モジュール */
      if (verbose)
        {
          glist = ia32_module_get_list (process);
          if (g_list_length (glist) > 1)
            {
              GList *gl;

              g_fprintf (fp, "Module\n");
              for (gl = g_list_first (glist); gl; gl = g_list_next (gl))
                {
                  guint32 address;

                  address = GPOINTER_TO_UINT (gl->data);
                  if (process->address != address)
                    {
                      gchar *name;

                      name = g_path_get_basename (ia32_module_get_filename
                                                        (process, address));
                      g_fprintf (fp, "DLL : %s (%08X)\n", name, address);
                      g_free (name);
                    }
                }
              g_fprintf (fp, "\n");
            }
          g_list_free (glist);
        }
      /* ja:インポートテーブル */
      if (verbose && process->table && process->table[0].address != 0)
        {
          gsize l_file = 0, l_name = 0;
          gchar *fmt;
          gint i;

          for (i = 0; process->table[i].address != 0; i++)
            {
              gsize leng;

              if (process->table[i].file)
                {
                  leng = g_strlen (process->table[i].file);
                  if (l_file < leng)
                    l_file = leng;
                }
              if (process->table[i].name)
                {
                  leng = g_strlen (process->table[i].name);
                  if (l_name < leng)
                    l_name = leng;
                }
            }
          g_fprintf (fp, "Import Table\n");
          fmt = g_strdup_printf ("%%-%ds %%-%ds %%4d %%08X\n", l_file, l_name);
          for (i = 0; process->table[i].address != 0; i++)
            g_fprintf (fp, fmt,
                        process->table[i].file ? process->table[i].file : "",
                        process->table[i].name,
                        process->table[i].ordinal,
                        process->table[i].address);
          g_free (fmt);
          g_fprintf (fp, "\n");
        }
      /* ja:スレッド */
      for (glist = g_list_first (process->thread);
                                            glist; glist = g_list_next (glist))
        {
          GList *err;
          Ia32Thread *thread;

          thread = glist->data;
          g_fprintf (fp, "Thread ID : %u\nExit Code : %08X\n",
                                                thread->id, thread->exitcode);
          for (err = g_list_first (thread->err); err; err = g_list_next (err))
            g_fprintf (fp, "%s\n", (gchar *)err->data);
          if (verbose)
            ia32_message_disp_register (fp, thread);
          else
            g_fprintf (fp, "\n");
        }
    }
}


/*  ja:情報をHTMLで表示する
         fp,ファイルポインタ
    process,プロセス構造体
    appname,アプリケーション名                                              */
void
ia32_message_disp_html (FILE        *fp,
                        Ia32Process *process,
                        const gchar *appname)
{
  if (process)
    {
      gchar *name;
      const gchar *file;
      GList *glist;

      file = process->address != IA32_MEMORY_NULL
                ? ia32_module_get_filename (process, process->address) : NULL;
      name = file ? g_path_get_basename (file) : g_malloc0 (sizeof (gchar));
      g_fprintf (fp,
"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\">\n"
"<html lang=\"en\">\n"
"<head>\n"
"<meta http-equiv=\"content-type\" content=\"text/html;charset=ISO-8859-1\">\n"
"<meta http-equiv=\"content-style-type\" content=\"text/css\">\n"
"<style type=\"text/css\"><!--\n"
"table {\n"
"  border-spacing: 0\n"
"}\n"
"\n"
"table, th, td {\n"
"  border: thin solid\n"
"}\n"
"--></style>\n"
"<title>%s - %s "VERSION" ("BUILD_ENVIRONMENT")</title>\n"
"</head>\n"
"<body>\n"
"<h1>%s</h1>\n"
"<h2>Process</h2>\n"
"<dl>\n"
"<dt>Process ID</dt>\n"
"<dd>%u</dd>\n"
"<dt>Exit Code</dt>\n"
"<dd>%08X</dd>\n",      name, appname, name, process->id, process->exitcode);
      if (process->address != IA32_MEMORY_NULL)
        g_fprintf (fp,
          "<dt>Executable File</dt>\n"
          "<dd>%s</dd>\n"
          "<dt>Base Address</dt>\n"
          "<dd>%08X</dd>\n",
          file, process->address);
      if (process->winmain != IA32_MEMORY_NULL)
        g_fprintf (fp, "<dt>Entry Point</dt>\n<dd>%08X (%d)</dd>\n",
                                            process->winmain, process->level);
      g_fprintf (fp, "</dl>\n");
      /* ja:モジュール */
      glist = ia32_module_get_list (process);
      if (g_list_length (glist) > 1)
        {
          GList *gl;

          g_fprintf (fp, 
            "<h2>Module</h2>\n"
            "<table summary=\"Name &amp; Address\">\n"
            "<thead>\n"
            "<tr><th>Name</th><th>Address</th></tr>\n"
            "</thead>\n"
            "<tbody>\n");
          for (gl = g_list_first (glist); gl; gl = g_list_next (gl))
            {
              guint32 address;

              address = GPOINTER_TO_UINT (gl->data);
              if (process->address != address)
                {
                  gchar *name;

                  name = g_path_get_basename (ia32_module_get_filename
                                                        (process, address));
                  g_fprintf (fp, "<tr><td>%s</td><td>%08X</td></tr>\n",
                                                                name, address);
                  g_free (name);
                }
            }
          g_fprintf (fp, "</tbody>\n</table>\n");
        }
      g_list_free (glist);
      /* ja:インポートテーブル */
      if (process->table && process->table[0].address != 0)
        {
          gint i;

          g_fprintf (fp,
"<h2>Import Table</h2>\n"
"<table summary=\"Import Table\">\n"
"<thead>\n"
"<tr><th>File</th><th>Name</th><th>Ordinal</th><th>Address</th></tr>\n"
"</thead>\n"
"<tbody>\n");
          for (i = 0; process->table[i].address != 0; i++)
            {
              g_fprintf (fp, "<tr>");
              if (process->table[i].file)
                {
                  gint j, rowspan;

                  for (j = i + 1; process->table[j].address != 0; j++)
                    if (process->table[j].file)
                      break;
                  rowspan = j - i;
                  g_fprintf (fp, "<td");
                  if (rowspan > 1)
                    g_fprintf (fp, " rowspan=\"%d\"", rowspan);
                  g_fprintf (fp, ">%s</td>", process->table[i].file);
                }
              g_fprintf (fp, "<td>%s</td><td>%d</td><td>%08X</td></tr>\n",
                        process->table[i].name,
                        process->table[i].ordinal,
                        process->table[i].address);
            }
          g_fprintf (fp, "</tbody>\n</table>\n");
        }
      /* ja:スレッド */
      for (glist = g_list_first (process->thread);
                                            glist; glist = g_list_next (glist))
        {
          Ia32Thread *thread;

          thread = glist->data;
          g_fprintf (fp,
            "<h2>Thread #%u</h2>\n"
            "<dl>\n"
            "<dt>Thread ID</dt>\n"
            "<dd>%u</dd>\n"
            "<dt>Exit Code</dt>\n"
            "<dd>%08X</dd>\n"
            "</dl>\n",              thread->id, thread->id, thread->exitcode);
          if (thread->api)
            {
              GList *api;

              g_fprintf (fp,
"<h3>API</h3>\n"
"<table summary=\"API\">\n"
"<thead>\n"
"<tr><th>Name</th><th>Return Address</th><th>Return Value</th><th colspan=\"3\">Argument</th></tr>\n"
"</thead>\n"
"<tbody>\n");
              for (api = g_list_first (thread->api);
                                                api; api = g_list_next (api))
                g_fprintf (fp, "%s\n", (gchar *)api->data);
              g_fprintf (fp, "</tbody>\n</table>\n");
            }
          if (thread->err)
            {
              GList *err;

              g_fprintf (fp, "<h3>Error</h3>\n<ul>\n");
              for (err = g_list_first (thread->err);
                                                err; err = g_list_next (err))
                g_fprintf (fp, "<li>%s</li>\n", (gchar *)err->data);
              g_fprintf (fp, "</ul>\n");
            }
          g_fprintf (fp,
"<h3>Register</h3>\n"
"<table summary=\"Register\">\n"
"<tbody>\n"
"<tr><th>EAX</th><td>%08X</td><th>EBX</th><td>%08X</td><th>ECX</th><td>%08X</td><th>EDX</th><td>%08X</td><th>ESI</th><td>%08X</td><th>EDI</th><td>%08X</td></tr>\n"
"<tr><th>EIP</th><td>%08X</td><th>ESP</th><td>%08X</td><th>EBP</th><td>%08X</td><th>EFL</th><td>%08X</td><td colspan=\"4\"></td></tr>\n"
"<tr><td colspan=\"12\">OV=%d UP=%d EI=%d PL=%d ZR=%d AC=%d PE=%d CY=%d</td></tr>\n"
"</tbody>\n"
"</table>\n",
            thread->reg.eax,
            thread->reg.ebx,
            thread->reg.ecx,
            thread->reg.edx,
            thread->reg.esi,
            thread->reg.edi,
            thread->reg.eip,
            thread->reg.esp,
            thread->reg.ebp,
            thread->reg.eflags,
            (thread->reg.eflags >> 11) & 1,
            (thread->reg.eflags >> 10) & 1,
            (thread->reg.eflags >> 9) & 1,
            (thread->reg.eflags >> 7) & 1,
            (thread->reg.eflags >> 6) & 1,
            (thread->reg.eflags >> 4) & 1,
            (thread->reg.eflags >> 2) & 1,
            thread->reg.eflags & 1);
        }
      g_fprintf (fp, "</body>\n</html>\n");
    }
}
