/*
    snap
    copyright (c) 2000-2017 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <windows.h>
#include <shlwapi.h>
#include <tchar.h>
#include "common.h"
#include "peimage.h"
#include "snap.h"


static VOID WINAPI
SnapShotLoadApi (PSNAP_INFO psi,
                 LPCBYTE    lpcbBuffer,
                 DWORD      modBaseAddr,
                 LPCTSTR    lpszModule)
{
  PIMAGE_NT_HEADERS pinth;

  pinth = (PIMAGE_NT_HEADERS)(lpcbBuffer
                                + ((PIMAGE_DOS_HEADER)lpcbBuffer)->e_lfanew);
  if (pinth->OptionalHeader.
            DataDirectory[IMAGE_DIRECTORY_ENTRY_EXPORT].VirtualAddress != 0)
    {
      PDLL_LIST pdl;
      int i;
      LPDWORD lpAddressOfFunctions;
      LPDWORD lpAddressOfNames;
      LPWORD lpAddressOfNameOrdinals;
      PIMAGE_EXPORT_DIRECTORY pied;

      pied = (PIMAGE_EXPORT_DIRECTORY)(lpcbBuffer + pinth->OptionalHeader.
                DataDirectory[IMAGE_DIRECTORY_ENTRY_EXPORT].VirtualAddress);
      lpAddressOfFunctions = (LPDWORD)(lpcbBuffer + pied->AddressOfFunctions);
      lpAddressOfNames = (LPDWORD)(lpcbBuffer + pied->AddressOfNames);
      lpAddressOfNameOrdinals = (LPWORD)(lpcbBuffer
                                                + pied->AddressOfNameOrdinals);
      psi->dl = MemoryReAlloc (psi->dl, (psi->nDll + 1) * sizeof (DLL_LIST));
      pdl = psi->dl + psi->nDll;
      pdl->al = MemoryAlloc (pied->NumberOfNames * sizeof(API_LIST));
#ifdef UNICODE
      WideCharToMultiByte (CP_ACP, 0, lpszModule, -1,
                            pdl->szModule, MAX_MODULE_NAME32 + 1, NULL, NULL);
#else /* not UNICODE */
      lstrcpy (pdl->szModule, lpszModule);
#endif /* not UNICODE */
      for (i = 0; i < pied->NumberOfNames; i++)
        {
          pdl->al[pdl->nApi].dwAddress
                            = lpAddressOfFunctions[lpAddressOfNameOrdinals[i]];
          if (pdl->al[pdl->nApi].dwAddress != 0)
            {
              pdl->al[pdl->nApi].dwAddress += modBaseAddr;
              pdl->al[pdl->nApi].lpszName
                    = MemoryStringA ((LPCSTR)lpcbBuffer + lpAddressOfNames[i]);
              pdl->nApi++;
            }
        }
      pdl->dwMin = UINT_MAX;
      for (i = 0; i < pdl->nApi; i++)
        {
          if (pdl->dwMin > pdl->al[i].dwAddress)
            pdl->dwMin = pdl->al[i].dwAddress;
          if (pdl->dwMax < pdl->al[i].dwAddress)
            pdl->dwMax = pdl->al[i].dwAddress;
        }
      psi->nDll++;
    }
}


/*  スナップショット情報を取得する
      dwProcessId,プロセスID
       lpszModule,モジュール名,NULL:アドレスとサイズが必要
    lpBaseAddress,アドレス,dwBytesが0ならばNULL
          dwBytes,サイズ,0ならばlpszModuleが有効
              RET,スナップショット情報                                      */
PSNAP_INFO WINAPI
SnapShotLoad (DWORD   dwProcessId,
              LPCTSTR lpszModule,
              LPCVOID lpBaseAddress,
              DWORD   dwBytes)
{
  PSNAP_INFO psi = NULL;
  HANDLE hSnap;

  hSnap = CreateToolhelp32Snapshot (TH32CS_SNAPMODULE, dwProcessId);
  if (hSnap != INVALID_HANDLE_VALUE)
    {
      BOOL fResult;
      MODULEENTRY32 me;

      psi = MemoryAlloc (sizeof (SNAP_INFO));
      me.dwSize = sizeof (MODULEENTRY32);
      for (fResult = Module32First (hSnap, &me); fResult;
                                        fResult = Module32Next (hSnap, &me))
        {
          DWORD dwRead;
          LPBYTE lpbImage;

          PrintConsole (STD_OUTPUT_HANDLE, _T("%s %08X %08X\n"),
                                me.szExePath, me.modBaseAddr, me.modBaseSize);
          lpbImage = MemoryAlloc (me.modBaseSize);
          if (Toolhelp32ReadProcessMemory (dwProcessId,
                            me.modBaseAddr, lpbImage, me.modBaseSize, &dwRead)
                                && dwRead == me.modBaseSize
                                && PeCheckImage32 (lpbImage, me.modBaseSize))
            {
              SnapShotLoadApi (psi, lpbImage,
                                        (DWORD)me.modBaseAddr, me.szModule);
              if (lpszModule && !psi->lpExe
                                    && lstrcmpi (lpszModule, me.szModule) == 0)
                {
                  int i;
                  PIMAGE_NT_HEADERS pinth;
                  PIMAGE_SECTION_HEADER pish;

                  pinth = (PIMAGE_NT_HEADERS)(lpbImage
                                    + ((PIMAGE_DOS_HEADER)lpbImage)->e_lfanew);
                  pish = (PIMAGE_SECTION_HEADER)(pinth + 1);
                  pinth->OptionalHeader.ImageBase = (DWORD)me.modBaseAddr;
                  for (i = 0; i < pinth->FileHeader.NumberOfSections; i++)
                    {
                      pish[i].SizeOfRawData = pish[i].Misc.VirtualSize;
                      pish[i].PointerToRawData = pish[i].VirtualAddress;
                    }
                  psi->dwSize = dwRead;
                  psi->lpExe = lpbImage;
                  lpbImage = NULL;
                }
            }
          else
            {
              MemoryFree (lpbImage);
              lpbImage = PeLoad32 (me.szExePath, &dwRead);
              if (lpbImage)
                SnapShotLoadApi (psi, lpbImage,
                                        (DWORD)me.modBaseAddr, me.szModule);
              else
                PrintConsole (STD_ERROR_HANDLE,
                            _T("Error : PeLoad32 (\"%s\")\n"), me.szExePath);
            }
          MemoryFree (lpbImage);
        }
      CloseHandle (hSnap);
      if (!psi->lpExe)
        {
          if (lpszModule)
            PrintConsole (STD_ERROR_HANDLE,
                        _T("Error : Module not Found \"%s\"\n"), lpszModule);
          if (dwBytes > 0)
            {
              psi->lpExe = MemoryAlloc (dwBytes);
              if (!Toolhelp32ReadProcessMemory (dwProcessId,
                            lpBaseAddress, psi->lpExe, dwBytes, &psi->dwSize))
                {
                  PrintConsole (STD_ERROR_HANDLE,
                                _T("Error : Toolhelp32ReadProcessMemory\n"));
                  MemoryFree (psi->lpExe);
                  psi->lpExe = NULL;
                }
            }
        }
    }
  else
    {
      PrintConsole (STD_ERROR_HANDLE,
                                    _T("Error : CreateToolhelp32Snapshot\n"));
    }
  return psi;
}


/*  スナップショット情報を保存する
         psi,スナップショット情報
    lpszFile,ファイル名
         RET,TRUE:正常終了,FALSE:エラー                                     */
BOOL WINAPI
SnapShotSave (PSNAP_INFO psi,
              LPCTSTR    lpszFile)
{
  int i;
  BOOL fResult = FALSE;
  DWORD dwWrite;
  HANDLE hFile;
  TCHAR szFile[MAX_PATH];

  if (!psi || !psi->lpExe)
    return FALSE;

  lstrcpy (szFile, lpszFile);
  PathRemoveExtension (szFile);
  lstrcat (szFile, _T(".dmp"));
  if (!SaveFile (szFile, psi->lpExe, psi->dwSize))
    {
      PrintConsole (STD_ERROR_HANDLE,
                                    _T("Error : SaveFile (\"%s\")\n"), szFile);
      return FALSE;
    }

  lstrcpy (szFile, lpszFile);
  PathRemoveExtension (szFile);
  lstrcat (szFile, _T(".imp"));
  hFile = CreateFile (szFile, GENERIC_WRITE, FILE_SHARE_READ,
                                NULL, CREATE_NEW, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile == INVALID_HANDLE_VALUE)
    {
      PrintConsole (STD_ERROR_HANDLE,
                                _T("Error : CreateFile (\"%s\")\n"), szFile);
      return FALSE;
    }
  for (i = 0; i < psi->dwSize - 4; i++)
    {
      int j;
      DWORD dwAddress;

      dwAddress = *(LPDWORD)((LPBYTE)psi->lpExe + i);
      for (j = 0; j < psi->nDll; j++)
        if (psi->dl[j].dwMin <= dwAddress && dwAddress <= psi->dl[j].dwMax)
          {
            int k;

            for (k = 0; k < psi->dl[j].nApi; k++)
              if (psi->dl[j].al[k].dwAddress == dwAddress)
                {
                  CHAR szText[19];

                  wsprintfA (szText, "%08X,%08X,", i, dwAddress);
                  WriteFile (hFile, szText, lstrlenA (szText), &dwWrite, NULL);
                  WriteFile (hFile, psi->dl[j].szModule,
                            lstrlenA (psi->dl[j].szModule), &dwWrite, NULL);
                  WriteFile (hFile, ",", sizeof (CHAR), &dwWrite, NULL);
                  WriteFile (hFile, psi->dl[j].al[k].lpszName,
                                        lstrlenA (psi->dl[j].al[k].lpszName),
                                                            &dwWrite, NULL);
                  WriteFile (hFile, "\r\n", sizeof (CHAR) * 2, &dwWrite, NULL);
                  break;
                }
          }
    }
  return CloseHandle(hFile);
}


/*  スナップショット情報を解放する
    psi,スナップショット情報                                                */
VOID WINAPI
SnapShotFree (PSNAP_INFO psi)
{
  if (psi)
    {
      int i;

      for (i = 0; i < psi->nDll; i++)
        {
          int j;

          for (j = 0; j < psi->dl[i].nApi; j++)
            MemoryFree (psi->dl[i].al[j].lpszName);
          MemoryFree (psi->dl[i].al);
        }
      MemoryFree (psi->dl);
      MemoryFree (psi->lpExe);
      MemoryFree (psi);
   }
}


/*  メモリイメージをファイルに保存する
      dwProcessId,プロセスID
         lpszFile,ファイル名
       lpszModule,モジュール名,NULL:アドレスとサイズが必要
    lpBaseAddress,アドレス,dwBytesが0ならばNULL
          dwBytes,サイズ,0ならばlpszModuleが有効                            */
BOOL WINAPI
SnapShot (DWORD   dwProcessId,
          LPCTSTR lpszFile,
          LPCTSTR lpszModule,
          LPCVOID lpBaseAddress,
          DWORD   dwBytes)
{
  BOOL fResult = FALSE;
  PSNAP_INFO psi;

  psi = SnapShotLoad (dwProcessId, dwBytes <= 0 ? lpszModule : NULL,
                                                    lpBaseAddress, dwBytes);
  if (psi)
    {
      fResult = SnapShotSave (psi, lpszFile);
      SnapShotFree (psi);
    }
  return fResult;
}
