/*
    gcommon
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "gcommon.h"


/******************************************************************************
* Doubly-Linked Lists                                                         *
******************************************************************************/
#ifdef USE_GTK_EMULATE
GList *
g_list_append (GList    *glist,
               gpointer  data)
{
  GList *gl;

  gl = g_malloc (sizeof (GList));
  gl->data = data;
  gl->next = NULL;
  if (glist)
    {
      glist = g_list_last (glist);
      glist->next = gl;
      gl->prev = glist;
      gl = g_list_first (glist);
    }
  else
    {
      gl->prev = NULL;
    }
  return gl;
}


GList *
g_list_prepend (GList    *glist,
                gpointer  data)
{
  GList *gl;

  gl = g_malloc (sizeof (GList));
  gl->data = data;
  gl->prev = NULL;
  if (glist)
    {
      glist = g_list_first (glist);
      glist->prev = gl;
      gl->next = glist;
    }
  else
    {
      gl->next = NULL;
    }
  return gl;
}


GList *
g_list_insert (GList    *glist,
               gpointer  data,
               gint      position)
{
  GList *gl;

  if (position < 0 || !(gl = g_list_nth (glist, position)))
    return g_list_append (glist, data);
  glist = g_malloc (sizeof (GList));
  glist->data = data;
  glist->prev = gl->prev;
  glist->next = gl;
  if (gl->prev)
    gl->prev->next = glist;
  gl->prev = glist;
  return g_list_first (glist);
}


GList *
g_list_insert_before (GList    *glist,
                      GList    *sibling,
                      gpointer  data)
{
  if (sibling)
    {
      glist = g_malloc (sizeof (GList));
      glist->data = data;
      glist->prev = sibling->prev;
      glist->next = sibling;
      if (sibling->prev)
        sibling->prev->next = glist;
      sibling->prev = glist;
      return g_list_first (glist);
    }
  return g_list_append (glist, data);
}


GList *
g_list_insert_sorted (GList        *glist,
                      gpointer      data,
                      GCompareFunc  compare)
{
  if (compare)
    {
      GList *gl;

      for (gl = g_list_last (glist); gl; gl = g_list_previous (gl))
        if (compare (gl->data, data) <= 0)
          break;
      if (gl)
        {
          glist = gl;
          gl = g_malloc (sizeof (GList));
          gl->data = data;
          gl->prev = glist;
          gl->next = glist->next;
          if (glist->next)
            glist->next->prev = gl;
          glist->next = gl;
          glist = g_list_first (glist);
        }
      else
        {
          glist = g_list_prepend (glist, data);
        }
    }
  return glist;
}


GList *
g_list_remove (GList         *glist,
               gconstpointer  data)
{
  GList *gl;

  glist = g_list_first (glist);
  for (gl = glist; gl; gl = g_list_next (gl))
    if (gl->data == data)
      return g_list_delete_link (glist, gl);
  return glist;
}


GList *
g_list_remove_link (GList *glist,
                    GList *glink)
{
  if (glink)
    {
      if (glink->prev)
        glink->prev->next = glink->next;
      if (glink->next)
        glink->next->prev = glink->prev;
      glist = glink->prev ? glink->prev : glink->next;
      glink->prev = glink->next = NULL;
    }
  return g_list_first (glist);
}


GList *
g_list_delete_link (GList *glist,
                    GList *glink)
{
  glist = g_list_remove_link (glist, glink);
  g_free (glink);
  return glist;
}


GList *
g_list_remove_all (GList         *glist,
                   gconstpointer  data)
{
  GList *gl;

  gl = glist = g_list_first (glist);
  while (gl)
    {
      GList *next;

      next = g_list_next (gl);
      if (gl->data == data)
        glist = g_list_delete_link (glist, gl);
      gl = next;
    }
  return glist;

}


void
g_list_free (GList *glist)
{
  while (glist)
    glist = g_list_delete_link (glist, glist);
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,28,0)
void g_list_free_full (GList          *glist,
                       GDestroyNotify  destroy)
{
  g_list_foreach (glist, (GFunc)destroy, NULL);
  g_list_free (glist);
}
#endif /* not GLIB_CHECK_VERSION(2,28,0) */


#ifdef USE_GTK_EMULATE
GList *
g_list_alloc (void)
{
  return g_malloc0 (sizeof (GList));
}


void
g_list_free_1 (GList *glist)
{
  g_free (glist);
}


guint
g_list_length (GList *glist)
{
  guint n = 0;

  for (glist = g_list_first (glist); glist; glist = g_list_next (glist))
    n++;
  return n;
}


GList *
g_list_copy (GList *glist)
{
  GList *gl = NULL;

  for (glist = g_list_first (glist); glist; glist = g_list_next (glist))
    gl = g_list_append (gl, glist->data);
  return gl;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,34,0)
GList *
g_list_copy_deep (GList     *glist,
                  GCopyFunc  func,
                  gpointer   user_data)
{
  if (func)
    {
      GList *gl = NULL;

      for (glist = g_list_first (glist); glist; glist = g_list_next (glist))
        gl = g_list_append (gl, func (glist->data, user_data));
      return gl;
    }
  return g_list_copy (glist);
}
#endif /* not GLIB_CHECK_VERSION(2,34,0) */


#ifdef USE_GTK_EMULATE
GList *
g_list_reverse (GList *glist)
{
  GList *gl = NULL;

  while (glist)
    {
      gl = glist;
      glist = glist->next;
      gl->next = gl->prev;
      gl->prev = glist;
    }
  return gl;
}


GList *
g_list_sort (GList        *glist,
             GCompareFunc  compare_func)
{
  if (glist && compare_func)
    {
      glist = g_list_first (glist);
      while (glist->next)
        {
          GList *gl, *glist_small;

          gl = glist_small = glist;
          while ((gl = g_list_next (gl)))
            if (compare_func (gl->data, glist_small->data) < 0)
              glist_small = gl;
          if (glist != glist_small)
            {
              GList *prev, *next;

              prev = glist->prev;
              next = glist->next;
              glist->prev = glist_small->prev;
              glist->next = glist_small->next;
              glist_small->prev = prev;
              glist_small->next = next;
              glist = glist_small;
            }
          glist = glist->next;
        }
      glist = g_list_first (glist);
    }
  return glist;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,10,0)
GList *
g_list_insert_sorted_with_data (GList            *glist,
                                gpointer          data,
                                GCompareDataFunc  func,
                                gpointer          user_data)
{
  if (func)
    {
      GList *gl;

      for (gl = g_list_last (glist); gl; gl = g_list_previous (gl))
        if (func (gl->data, data, user_data) <= 0)
          break;
      if (gl)
        {
          glist = gl;
          gl = g_malloc (sizeof (GList));
          gl->data = data;
          gl->prev = glist;
          gl->next = glist->next;
          if (glist->next)
            glist->next->prev = gl;
          glist->next = gl;
          glist = g_list_first (glist);
        }
      else
        {
          glist = g_list_prepend (glist, data);
        }
    }
  return glist;
}
#endif /* not GLIB_CHECK_VERSION(2,10,0) */


#ifdef USE_GTK_EMULATE
GList *
g_list_sort_with_data (GList            *glist,
                       GCompareDataFunc  compare_func,
                       gpointer          user_data)
{
  if (glist && compare_func)
    {
      glist = g_list_first (glist);
      while (glist->next)
        {
          GList *gl, *glist_small;

          gl = glist_small = glist;
          while ((gl = g_list_next (gl)))
            if (compare_func (gl->data, glist_small->data, user_data) < 0)
              glist_small = gl;
          if (glist != glist_small)
            {
              GList *prev, *next;

              prev = glist->prev;
              next = glist->next;
              glist->prev = glist_small->prev;
              glist->next = glist_small->next;
              glist_small->prev = prev;
              glist_small->next = next;
              glist = glist_small;
            }
          glist = glist->next;
        }
      glist = g_list_first (glist);
    }
  return glist;
}


GList *
g_list_concat (GList *glist1,
               GList *glist2)
{
  if (glist1 && glist2)
    {
      glist1 = g_list_last (glist1);
      glist2 = g_list_first (glist2);
      glist1->next = glist2;
      glist2->prev = glist1;
    }
  return g_list_first (glist1 ? glist1 : glist2);
}


void
g_list_foreach (GList    *glist,
                GFunc     func,
                gpointer  user_data)
{
  if (func)
    {
      glist =  g_list_first (glist);
      while (glist)
        {
          gpointer data;

          data = glist->data;
          glist = g_list_next (glist);
          func (data, user_data);
        }
    }
}


GList *
g_list_first (GList *glist)
{
  if (glist)
    while (glist->prev)
      glist = glist->prev;
  return glist;
}


GList *
g_list_last (GList *glist)
{
  if (glist)
    while (glist->next)
      glist = glist->next;
  return glist;
}


GList *
g_list_nth (GList *glist,
            guint  n)
{
  while (glist && n-- > 0)
    glist = g_list_next (glist);
  return glist;
}


gpointer
g_list_nth_data (GList *glist,
                 guint  n)
{
  glist = g_list_nth (glist, n);
  return glist ? glist->data : NULL;
}


GList *
g_list_nth_prev (GList *glist,
                 guint  n)
{
  while (glist && n-- > 0)
    glist = g_list_previous (glist);
  return glist;
}


GList *
g_list_find (GList         *glist,
             gconstpointer  data)
{
  while (glist)
    {
      if (glist->data == data)
        return glist;
      glist = g_list_next (glist);
    }
  return NULL;
}


GList *
g_list_find_custom (GList         *glist,
                    gconstpointer  data,
                    GCompareFunc   compare)
{
  if (compare)
    while (glist)
      {
        if (compare (glist->data, data) == 0)
          return glist;
        glist = g_list_next (glist);
      }
  return NULL;
}


gint
g_list_position (GList *glist,
                 GList *llink)
{
  gint n = 0;

  while (glist)
    {
      if (glist == llink)
        return n;
      n++;
      glist = g_list_next (glist);
    }
  return -1;
}


gint
g_list_index (GList         *glist,
              gconstpointer  data)
{
  gint n = 0;

  while (glist)
    {
      if (glist->data == data)
        return n;
      n++;
      glist = g_list_next (glist);
    }
  return -1;
}
#endif /* USE_GTK_EMULATE */
