/*
    common
    copyright (c) 1998-2017 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <windows.h>
#include <shlobj.h>
#include <shlwapi.h>
#include <tchar.h>
#include "common.h"


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  10進数文字列を数値に変換する
    lpszText,文字列
         RET,数値                                                           */
DWORD WINAPI
DecW (LPCWSTR lpszText)
{
  DWORD dwResult = 0;

  if (lpszText)
    {
      LPCWSTR p;

      for (p = lpszText; *p != '\0'; p++)
        {
          int i;
          WCHAR c;
          const static WCHAR cDec[] = {'0', '1', '2', '3', '4',
                                       '5', '6', '7', '8', '9'};

          c = (WCHAR)CharLowerW ((LPWSTR)*p);
          for (i = 0; i < n_elements (cDec); i++)
            if (c == cDec[i])
              {
                dwResult *= 10;
                dwResult += i;
                break;
              }
          if (i >= n_elements (cDec))
            return (DWORD)-1;
        }
    }
  return dwResult;
}


/*  10進数文字列を数値に変換する
    lpszText,文字列
         RET,数値                                                           */
DWORD WINAPI
DecA (LPCSTR lpszText)
{
  DWORD dwResult = 0;

  if (lpszText)
    {
      LPCSTR p;

      for (p = lpszText; *p != '\0'; p++)
        {
          int i;
          CHAR c;
          const static CHAR cDec[] = {'0', '1', '2', '3', '4',
                                      '5', '6', '7', '8', '9'};

          c = (CHAR)CharLowerA ((LPSTR)*p);
          for (i = 0; i < n_elements (cDec); i++)
            if (c == cDec[i])
              {
                dwResult *= 10;
                dwResult += i;
                break;
              }
          if (i >= n_elements (cDec))
            return (DWORD)-1;
        }
    }
  return dwResult;
}


/*  16進数文字列を数値に変換する
    lpszText,文字列
         RET,数値                                                           */
DWORD WINAPI
HexW (LPCWSTR lpszText)
{
  DWORD dwResult = 0;

  if (lpszText)
    {
      LPCWSTR p;

      for (p = lpszText; *p != '\0'; p++)
        {
          int i;
          WCHAR c;
          const static WCHAR cHex[] = {'0', '1', '2', '3', '4', '5', '6', '7',
                                       '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'};

          c = (WCHAR)CharLowerW ((LPWSTR)*p);
          for (i = 0; i < n_elements (cHex); i++)
            if (c == cHex[i])
              {
                dwResult <<= 4;
                dwResult += i;
                break;
              }
          if (i >= n_elements (cHex))
            return (DWORD)-1;
        }
    }
  return dwResult;
}


/*  16進数文字列を数値に変換する
    lpszText,文字列
         RET,数値                                                           */
DWORD WINAPI
HexA (LPCSTR lpszText)
{
  DWORD dwResult = 0;

  if (lpszText)
    {
      LPCSTR p;

      for (p = lpszText; *p != '\0'; p++)
        {
          int i;
          CHAR c;
          const static CHAR cHex[] = {'0', '1', '2', '3', '4', '5', '6', '7',
                                      '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'};

          c = (CHAR)CharLowerA ((LPSTR)*p);
          for (i = 0; i < n_elements (cHex); i++)
            if (c == cHex[i])
              {
                dwResult <<= 4;
                dwResult += i;
                break;
              }
          if (i >= n_elements (cHex))
            return (DWORD)-1;
        }
    }
  return dwResult;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  メモリの内容を埋める
    lpvDest,メモリ
      bByte,内容
     uBytes,バイト数                                                        */
VOID WINAPI
MemorySet (LPVOID lpvDest,
           BYTE   bByte,
           SIZE_T uBytes)
{
  if (lpvDest)
    {
      LPBYTE p, q;

      p = lpvDest;
      q = p + uBytes;
      while (p < q)
        *p++ = bByte;
    }
}


/*  メモリの内容をコピーする
    lpvDest,コピー先
     lpvSrc,コピー元
     uBytes,バイト数                                                        */
VOID WINAPI
MemoryCopy (LPVOID  lpvDest,
            LPCVOID lpvSrc,
            SIZE_T  uBytes)
{
  if (lpvDest && lpvSrc && uBytes > 0)
    {
      LPBYTE p;
      LPCBYTE q;

      p = lpvDest;
      q = lpvSrc;
      if ((UINT_PTR)lpvDest < (UINT_PTR)lpvSrc)
        {
          LPBYTE r;

          r = p + uBytes;
          while (p < r)
            *p++ = *q++;
        }
      else if ((UINT_PTR)lpvDest > (UINT_PTR)lpvSrc)
        {
          LPBYTE r;

          r = p;
          uBytes--;
          p += uBytes;
          q += uBytes;
          while (p >= r)
            *p-- = *q--;
        }
    }
}


/*  メモリの内容を比較する
    lpvBuf1,バッファ1
    lpvBuf1,バッファ2
     uBytes,バイト数
        RET,0:等しい,負:バッファ1は小さい,正:バッファ1は大きい              */
int WINAPI
MemoryCompare (LPCVOID lpvBuf1,
               LPCVOID lpvBuf2,
               SIZE_T  uBytes)
{
  if (lpvBuf1 && lpvBuf2)
    {
      LPCBYTE p, q, r;

      p = lpvBuf1;
      q = lpvBuf2;
      r = p + uBytes;
      while (p < r)
        {
          int n;

          n = *p - *q;
          if (n != 0)
            return n;
          p++;
          q++;
        }
    }
  else if (lpvBuf1)
    return INT_MAX;
  else if (lpvBuf2)
    return INT_MIN;
  return 0;
}


/*  メモリを確保する
    uBytes,サイズ
       RET,確保したメモリ,NULL:エラー                                       */
LPVOID WINAPI
MemoryAlloc (SIZE_T uBytes)
{
  return uBytes > 0 ? HeapAlloc (GetProcessHeap (), HEAP_ZERO_MEMORY, uBytes)
                    : NULL;
}


/*  メモリを解放する
    lpMemory,確保したメモリ
         RET,TRUE:正常終了,FALSE:エラー                                     */
BOOL WINAPI
MemoryFree (LPVOID lpMemory)
{
  return !lpMemory || HeapFree (GetProcessHeap (), 0, lpMemory);
}


/*  メモリのサイズを変更する
    lpMemory,確保したメモリ
      uBytes,サイズ
         RET,確保したメモリ,NULL:エラー                                     */
LPVOID WINAPI
MemoryReAlloc (LPVOID lpMemory,
               SIZE_T uBytes)
{
  return uBytes > 0 ? lpMemory ? HeapReAlloc (GetProcessHeap (),
                                            HEAP_ZERO_MEMORY, lpMemory, uBytes)
                               : MemoryAlloc (uBytes)
                    : (MemoryFree (lpMemory), NULL);
}


/*  メモリを複製する
    lpMemory,複製するメモリ
      uBytes,サイズ
         RET,確保したメモリ,NULL:エラー                                     */
LPVOID WINAPI
MemoryDuplicate (LPCVOID lpMemory,
                 SIZE_T  uBytes)
{
  LPVOID lpResult;

  if (lpMemory)
    {
      lpResult = MemoryAlloc (uBytes);
      MemoryCopy (lpResult, lpMemory, uBytes);
    }
  else
    {
      lpResult = NULL;
    }
  return lpResult;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  文字列を書式化する
    ppszString,文字列
    lpszFormat,書式文字列
       arglist,引数
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wvasprintfW (LPWSTR  *ppszString,
             LPCWSTR  lpszFormat,
             va_list  arglist)
{
  SSIZE_T nResult = -1;

  if (lpszFormat && arglist)
    {
      LPWSTR lpszText = NULL;
      SSIZE_T nSize = 0;

      do
        {
          nSize += 4096;
          lpszText = MemoryReAlloc (lpszText, nSize * sizeof (WCHAR));
          nResult = wvnsprintfW (lpszText, nSize, lpszFormat, arglist);
        }
      while (nResult < 0 || nResult >= nSize - 1);
      if (ppszString)
        *ppszString = MemoryReAlloc (lpszText, (nResult + 1) * sizeof (WCHAR));
      else
        MemoryFree (lpszText);
    }
  return nResult;
}


/*  文字列を書式化する
    ppszString,文字列
    lpszFormat,書式文字列
       arglist,引数
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wvasprintfA (LPSTR  *ppszString,
             LPCSTR  lpszFormat,
             va_list arglist)
{
  SSIZE_T nResult = -1;

  if (lpszFormat && arglist)
    {
      LPSTR lpszText = NULL;
      SSIZE_T nSize = 0;

      do
        {
          nSize += 4096;
          lpszText = MemoryReAlloc (lpszText, nSize * sizeof (CHAR));
          nResult = wvnsprintfA (lpszText, nSize, lpszFormat, arglist);
        }
      while (nResult < 0 || nResult >= nSize - 1);
      if (ppszString)
        *ppszString = MemoryReAlloc (lpszText, (nResult + 1) * sizeof (CHAR));
      else
        MemoryFree (lpszText);
    }
  return nResult;
}


/*  文字列を書式化する
    ppszString,文字列
      lpFormat,書式文字列
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wasprintfW (LPWSTR  *ppszString,
            LPCWSTR  lpszFormat, ...)
{
  int nResult;
  va_list arglist;

  va_start (arglist, lpszFormat);
  nResult = wvasprintfW (ppszString, lpszFormat, arglist);
  va_end (arglist);
  return nResult;
}


/*  文字列を書式化する
    ppszString,文字列
      lpFormat,書式文字列
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wasprintfA (LPSTR  *ppszString,
            LPCSTR  lpszFormat, ...)
{
  int nResult;
  va_list arglist;

  va_start (arglist, lpszFormat);
  nResult = wvasprintfA (ppszString, lpszFormat, arglist);
  va_end (arglist);
  return nResult;
}


/*  文字列を複製する
    lpszText,文字列
       nSize,文字数(負:NULL終端文字列)
         RET,複製したメモリ,NULL:エラー                                     */
LPWSTR WINAPI
StringDuplicateExW (LPCWSTR lpszText,
                    SSIZE_T nSize)
{
  LPWSTR lpszResult;

  if (nSize > 0)
    {
      if (lpszText && lpszText[nSize - 1] != '\0')
        nSize++;
    }
  else if (lpszText && nSize < 0)
    {
      nSize = lstrlenW (lpszText) + 1;
    }
  if (nSize > 0)
    {
      lpszResult = MemoryAlloc (nSize * sizeof (WCHAR));
      MemoryCopy (lpszResult, lpszText, (nSize - 1) * sizeof (WCHAR));
    }
  else
    {
      lpszResult = NULL;
    }
  return lpszResult;
}


/*  文字列を複製する
    lpszText,文字列
       nSize,文字数(負:NULL終端文字列)
         RET,複製したメモリ,NULL:エラー                                     */
LPSTR WINAPI
StringDuplicateExA (LPCSTR  lpszText,
                    SSIZE_T nSize)
{
  LPSTR lpszResult;

  if (nSize > 0)
    {
      if (lpszText && lpszText[nSize - 1] != '\0')
        nSize++;
    }
  else if (lpszText && nSize < 0)
    {
      nSize = lstrlenA (lpszText) + 1;
    }
  if (nSize > 0)
    {
      lpszResult = MemoryAlloc (nSize * sizeof (CHAR));
      MemoryCopy (lpszResult, lpszText, (nSize - 1) * sizeof (CHAR));
    }
  else
    {
      lpszResult = NULL;
    }
  return lpszResult;
}


/*  文字列を複製する
    lpszText,文字列
         RET,複製したメモリ,NULL:エラー                                     */
LPWSTR WINAPI
StringDuplicateW (LPCWSTR lpszText)
{
  return StringDuplicateExW (lpszText, -1);
}


/*  文字列を複製する
    lpszText,文字列
         RET,複製したメモリ,NULL:エラー                                     */
LPSTR WINAPI
StringDuplicateA (LPCSTR lpszText)
{
  return StringDuplicateExA (lpszText, -1);
}


/*  マルチバイト文字列をワイド文字列に変換する
    lpszMultiByte,マルチバイト文字列
     cchMultiByte,文字数(負:NULL終端文字列)
              RET,ワイド文字列                                              */
LPWSTR WINAPI
StringMultiByteToWideCharEx (LPCSTR  lpszMultiByte,
                             SSIZE_T cchMultiByte)
{
  LPWSTR lpszWideChar = NULL;

  if (lpszMultiByte && cchMultiByte != 0)
    {
      int cchWideChar;

      if (cchMultiByte < 0)
        cchMultiByte = -1;
      cchWideChar = MultiByteToWideChar (CP_ACP, 0,
                                        lpszMultiByte, cchMultiByte, NULL, 0);
      if (cchWideChar > 0)
        {
          lpszWideChar = MemoryAlloc (cchWideChar * sizeof (WCHAR));
          if (MultiByteToWideChar (CP_ACP, 0, lpszMultiByte, cchMultiByte,
                                            lpszWideChar, cchWideChar) <= 0)
            {
              MemoryFree (lpszWideChar);
              lpszWideChar = NULL;
            }
          else if (lpszWideChar[cchWideChar - 1] != '\0')
            {
              lpszWideChar = MemoryReAlloc (lpszWideChar,
                                        (cchWideChar + 1) * sizeof (WCHAR));
            }
        }
    }
  return lpszWideChar;
}


/*  ワイド文字列をマルチバイト文字列に変換する
    lpszWideChar,ワイド文字列
     cchWideChar,文字数(負:NULL終端文字列)
             RET,マルチバイト文字列                                         */
LPSTR WINAPI
StringWideCharToMultiByteEx (LPCWSTR lpszWideChar,
                             SSIZE_T cchWideChar)
{
  LPSTR lpszMultiByte = NULL;

  if (lpszWideChar && cchWideChar != 0)
    {
      int cchMultiByte;

      if (cchWideChar < 0)
        cchWideChar = -1;
      cchMultiByte = WideCharToMultiByte (CP_ACP, 0, lpszWideChar, cchWideChar,
                                                        NULL, 0, NULL, NULL);
      if (cchMultiByte > 0)
        {
          lpszMultiByte = MemoryAlloc (cchMultiByte * sizeof (CHAR));
          if (WideCharToMultiByte (CP_ACP, 0, lpszWideChar, cchWideChar,
                                lpszMultiByte, cchMultiByte, NULL, NULL) <= 0)
            {
              MemoryFree (lpszMultiByte);
              lpszMultiByte = NULL;
            }
          else if (lpszMultiByte[cchMultiByte - 1] != '\0')
            {
              lpszMultiByte = MemoryReAlloc (lpszMultiByte,
                                        (cchMultiByte + 1) * sizeof (CHAR));
            }
        }
    }
  return lpszMultiByte;
}


/*  マルチバイト文字列をワイド文字列に変換する
    lpszMultiByte,マルチバイト文字列
              RET,ワイド文字列                                              */
LPWSTR WINAPI
StringMultiByteToWideChar (LPCSTR lpszMultiByte)
{
  return StringMultiByteToWideCharEx (lpszMultiByte, -1);
}


/*  ワイド文字列をマルチバイト文字列に変換する
    lpszWideChar,ワイド文字列
             RET,マルチバイト文字列                                         */
LPSTR WINAPI
StringWideCharToMultiByte (LPCWSTR lpszWideChar)
{
  return StringWideCharToMultiByteEx (lpszWideChar, -1);
}


/*  文字列を正規化する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
           RET,正規化された文字列                                           */
LPWSTR WINAPI
StringCanonicalizeExW (LPCWSTR lpszString,
                       SSIZE_T nSize)
{
  LPWSTR lpszResult;

  if (lpszString)
    {
      int nLen;
      LPCWSTR p, r;
      LPWSTR q;

      nLen = nSize >= 0 ? nSize : lstrlenW (lpszString);
      p = lpszString;
      r = p + nLen;
      q = lpszResult = MemoryAlloc ((nLen * 6 + 3) * sizeof (WCHAR));
      *q++ = '\"';
      while (p < r)
        {
          switch (*p)
            {
              case '\a': *q++ = '\\'; *q++ = 'a';  break;
              case '\b': *q++ = '\\'; *q++ = 'b';  break;
              case '\n': *q++ = '\\'; *q++ = 'n';  break;
              case '\r': *q++ = '\\'; *q++ = 'r';  break;
              case '\f': *q++ = '\\'; *q++ = 'f';  break;
              case '\t': *q++ = '\\'; *q++ = 't';  break;
              case '\v': *q++ = '\\'; *q++ = 'v';  break;
              case '\\': *q++ = '\\'; *q++ = '\\'; break;
              case '\?': *q++ = '\\'; *q++ = '\?'; break;
              case '\'': *q++ = '\\'; *q++ = '\''; break;
              case '\"': *q++ = '\\'; *q++ = '\"'; break;
              default:
                if (0x20 <= *p && *p <= 0x7e)
                  {
                    *q++ = *p;
                  }
                else
                  {
                    wsprintfW (q, L"\\x%04X", *p);
                    q += 6;
                  }
            }
          p++;
        }
      *q++ = '\"';
      *q = '\0';
    }
  else
    {
      lpszResult = StringDuplicateW (L"NULL");
    }
  return lpszResult;
}


/*  文字列を正規化する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
           RET,正規化された文字列                                           */
LPSTR WINAPI
StringCanonicalizeExA (LPCSTR  lpszString,
                       SSIZE_T nSize)
{
  LPSTR lpszResult;

  if (lpszString)
    {
      int nLen;
      LPCSTR p, r;
      LPSTR q;

      nLen = nSize >= 0 ? nSize : lstrlenA (lpszString);
      p = lpszString;
      r = p + nLen;
      q = lpszResult = MemoryAlloc ((nLen * 6 + 3) * sizeof (CHAR));
      *q++ = '\"';
      while (p < r)
        {
          switch (*p)
            {
              case '\a': *q++ = '\\'; *q++ = 'a';  break;
              case '\b': *q++ = '\\'; *q++ = 'b';  break;
              case '\n': *q++ = '\\'; *q++ = 'n';  break;
              case '\r': *q++ = '\\'; *q++ = 'r';  break;
              case '\f': *q++ = '\\'; *q++ = 'f';  break;
              case '\t': *q++ = '\\'; *q++ = 't';  break;
              case '\v': *q++ = '\\'; *q++ = 'v';  break;
              case '\\': *q++ = '\\'; *q++ = '\\'; break;
              case '\?': *q++ = '\\'; *q++ = '\?'; break;
              case '\'': *q++ = '\\'; *q++ = '\''; break;
              case '\"': *q++ = '\\'; *q++ = '\"'; break;
              default:
                if (0x20 <= *p && *p <= 0x7e)
                  {
                    *q++ = *p;
                  }
                else
                  {
                    wsprintfA (q, "\\x%04X", (BYTE)*p);
                    q += 6;
                  }
            }
          p++;
        }
      *q++ = '\"';
      *q = '\0';
    }
  else
    {
      lpszResult = StringDuplicateA ("NULL");
    }
  return lpszResult;
}


/*  文字列を正規化する
    lpszString,文字列
           RET,正規化された文字列                                           */
LPWSTR WINAPI
StringCanonicalizeW (LPCWSTR lpszString)
{
  return StringCanonicalizeExW (lpszString, -1);
}


/*  文字列を正規化する
    lpszString,文字列
           RET,正規化された文字列                                           */
LPSTR WINAPI
StringCanonicalizeA (LPCSTR lpszString)
{
  return StringCanonicalizeExA (lpszString, -1);
}


/*  文字列を分割する
    lpszString,文字列
    cDelimiter,文字
           RET,分割された文字列                                             */
LPWSTR * WINAPI
StringSplitW (LPCWSTR lpszString,
              WCHAR   cDelimiter)
{
  LPWSTR *plpszArray = NULL;

  if (lpszString)
    {
      int nCount = 2;
      LPCWSTR p;
      LPWSTR q, *r;

      for (p = lpszString; *p != '\0'; p++)
        if (*p == cDelimiter)
          nCount++;
      plpszArray = MemoryAlloc (nCount * sizeof (LPWSTR)
                                    + (p - lpszString + 1) * sizeof (WCHAR));
      q = (LPWSTR)(plpszArray + nCount);
      lstrcpyW (q, lpszString);
      r = plpszArray;
      *r++ = q;
      while (*q != '\0')
        {
          if (*q == cDelimiter)
            {
              *r++ = q + 1;
              *q = '\0';
            }
          q++;
        }
    }
  return plpszArray;
}


/*  文字列を分割する
    lpszString,文字列
    cDelimiter,文字
           RET,分割された文字列                                             */
LPSTR * WINAPI
StringSplitA (LPCSTR lpszString,
              CHAR   cDelimiter)
{
  LPSTR *plpszArray = NULL;

  if (lpszString)
    {
      int nCount = 2;
      LPCSTR p;
      LPSTR q, *r;

      for (p = lpszString; *p != '\0';
                                    p += IsDBCSLeadByteEx (CP_ACP, *p) ? 2 : 1)
        if (*p == cDelimiter)
          nCount++;
      plpszArray = MemoryAlloc (nCount * sizeof (LPSTR)
                                    + (p - lpszString + 1) * sizeof (CHAR));
      q = (LPSTR)(plpszArray + nCount);
      lstrcpyA (q, lpszString);
      r = plpszArray;
      *r++ = q;
      while (*q != '\0')
        {
          if (*q == cDelimiter)
            {
              *r++ = q + 1;
              *q = '\0';
            }
          q += IsDBCSLeadByteEx (CP_ACP, *q) ? 2 : 1;
        }
    }
  return plpszArray;
}


/*  文字列を結合する
    lpszSeparator,区切り文字列
       plpszArray,分割された文字列
              RET,文字列                                                    */
LPWSTR WINAPI
StringJoinVW (LPCWSTR  lpszSeparator,
              LPCWSTR *plpszArray)
{
  LPWSTR lpszResult = NULL;

  if (plpszArray)
    {
      int i, nLen = 1;

      for (i = 0; plpszArray[i]; i++)
        nLen += lstrlenW (plpszArray[i]);
      if (lpszSeparator && i > 1)
        nLen += lstrlenW (lpszSeparator) * (i - 1);
      if (nLen > 1)
        {
          lpszResult = MemoryAlloc (nLen * sizeof (WCHAR));
          while (*plpszArray)
            {
              lstrcatW (lpszResult, *plpszArray);
              plpszArray++;
              if (lpszSeparator && *plpszArray)
                lstrcatW (lpszResult, lpszSeparator);
            }
        }
    }
  return lpszResult;
}


/*  文字列を結合する
    lpszSeparator,区切り文字列
       plpszArray,分割された文字列
              RET,文字列                                                    */
LPSTR WINAPI
StringJoinVA (LPCSTR  lpszSeparator,
              LPCSTR *plpszArray)
{
  LPSTR lpszResult = NULL;

  if (plpszArray)
    {
      int i, nLen = 1;

      for (i = 0; plpszArray[i]; i++)
        nLen += lstrlenA (plpszArray[i]);
      if (lpszSeparator && i > 1)
        nLen += lstrlenA (lpszSeparator) * (i - 1);
      if (nLen > 1)
        {
          lpszResult = MemoryAlloc (nLen * sizeof (CHAR));
          while (*plpszArray)
            {
              lstrcatA (lpszResult, *plpszArray);
              plpszArray++;
              if (lpszSeparator && *plpszArray)
                lstrcatA (lpszResult, lpszSeparator);
            }
        }
    }
  return lpszResult;
}


/*  文字列を結合する
    lpszSeparator,区切り文字列
              RET,文字列                                                    */
LPWSTR WINAPI
StringJoinW (LPCWSTR lpszSeparator, ...)
{
  LPWSTR lpszResult = NULL;
  LPCWSTR lpszString;
  int i = 0, nLen = 1;
  va_list arglist;

  va_start (arglist, lpszSeparator);
  while (lpszString = va_arg (arglist, LPCWSTR))
    {
      nLen += lstrlenW (lpszString);
      i++;
    }
  va_end (arglist);
  if (lpszSeparator && i > 1)
    nLen += lstrlenW (lpszSeparator) * (i - 1);
  if (nLen > 1)
    {
      lpszResult = MemoryAlloc (nLen * sizeof (WCHAR));
      va_start (arglist, lpszSeparator);
      lpszString = va_arg (arglist, LPCWSTR);
      while (lpszString)
        {
          lstrcatW (lpszResult, lpszString);
          lpszString = va_arg (arglist, LPCWSTR);
          if (lpszSeparator && lpszString)
            lstrcatW (lpszResult, lpszSeparator);
        }
      va_end (arglist);
    }
  return lpszResult;
}


/*  文字列を結合する
    lpszSeparator,区切り文字列
              RET,文字列                                                    */
LPSTR WINAPI
StringJoinA (LPCSTR lpszSeparator, ...)
{
  LPSTR lpszResult = NULL;
  LPCSTR lpszString;
  int i = 0, nLen = 1;
  va_list arglist;

  va_start (arglist, lpszSeparator);
  while (lpszString = va_arg (arglist, LPCSTR))
    {
      nLen += lstrlenA (lpszString);
      i++;
    }
  va_end (arglist);
  if (lpszSeparator && i > 1)
    nLen += lstrlenA (lpszSeparator) * (i - 1);
  if (nLen > 1)
    {
      lpszResult = MemoryAlloc (nLen * sizeof (CHAR));
      va_start (arglist, lpszSeparator);
      lpszString = va_arg (arglist, LPCSTR);
      while (lpszString)
        {
          lstrcatA (lpszResult, lpszString);
          lpszString = va_arg (arglist, LPCSTR);
          if (lpszSeparator && lpszString)
            lstrcatA (lpszResult, lpszSeparator);
        }
      va_end (arglist);
    }
  return lpszResult;
}


/*  分割された文字列の数を求める
    plpszArray,分割された文字列
           RET,分割された文字列の数                                         */
int WINAPI
StringArrayLengthW (LPCWSTR *plpszArray)
{
  int nResult = 0;

  if (plpszArray)
    while (plpszArray[nResult])
      nResult++;
  return nResult;
}


/*  分割された文字列の数を求める
    plpszArray,分割された文字列
           RET,分割された文字列の数                                         */
int WINAPI
StringArrayLengthA (LPCSTR *plpszArray)
{
  int nResult = 0;

  if (plpszArray)
    while (plpszArray[nResult])
      nResult++;
  return nResult;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ファイルをメモリに読み込む
    lpszFile,ファイル名
    lpdwSize,ファイルサイズ
         RET,メモリ                                                         */
LPVOID WINAPI
LoadFileW (LPCWSTR lpszFile,
           LPDWORD lpdwSize)
{
  LPVOID lpBuffer = NULL;

  if (lpszFile)
    {
      BOOL fResult = FALSE;
      DWORD dwSize;
      HANDLE hFile;

      hFile = CreateFileW (lpszFile, GENERIC_READ, FILE_SHARE_READ, NULL,
                                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
      if (hFile != INVALID_HANDLE_VALUE)
        {
          dwSize = GetFileSize (hFile, NULL);
          if (dwSize != INVALID_FILE_SIZE)
            {
              DWORD dwRead;

              lpBuffer = MemoryAlloc (((dwSize + 1) / sizeof (WCHAR) + 1)
                                                            * sizeof (WCHAR));
              if (lpBuffer)
                fResult = ReadFile (hFile, lpBuffer, dwSize, &dwRead, NULL)
                                                        && dwSize == dwRead;
            }
          if (!CloseHandle (hFile))
            fResult = FALSE;
        }
      if (!fResult)
        {
          MemoryFree (lpBuffer);
          lpBuffer = NULL;
        }
      else if (lpdwSize)
        {
          *lpdwSize = dwSize;
        }
    }
  return lpBuffer;
}


/*  ファイルをメモリに読み込む
    lpszFile,ファイル名
    lpdwSize,ファイルサイズ
         RET,メモリ                                                         */
LPVOID WINAPI
LoadFileA (LPCSTR  lpszFile,
           LPDWORD lpdwSize)
{
  LPVOID lpBuffer = NULL;

  if (lpszFile)
    {
      BOOL fResult = FALSE;
      DWORD dwSize;
      HANDLE hFile;

      hFile = CreateFileA (lpszFile, GENERIC_READ, FILE_SHARE_READ, NULL,
                                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
      if (hFile != INVALID_HANDLE_VALUE)
        {
          dwSize = GetFileSize (hFile, NULL);
          if (dwSize != INVALID_FILE_SIZE)
            {
              DWORD dwRead;

              lpBuffer = MemoryAlloc (((dwSize + 1) / sizeof (WCHAR) + 1)
                                                            * sizeof (WCHAR));
              if (lpBuffer)
                fResult = ReadFile (hFile, lpBuffer, dwSize, &dwRead, NULL)
                                                        && dwSize == dwRead;
            }
          if (!CloseHandle (hFile))
            fResult = FALSE;
        }
      if (!fResult)
        {
          MemoryFree (lpBuffer);
          lpBuffer = NULL;
        }
      else if (lpdwSize)
        {
          *lpdwSize = dwSize;
        }
    }
  return lpBuffer;
}


/*  メモリをファイルに書き込む
    lpszFile,ファイル名
    lpBuffer,メモリ
      nSize,バイト数(負:NULL終端文字列)
         RET,TRUE:正常終了,FALSE:エラー                                     */
BOOL WINAPI
SaveFileW (LPCWSTR lpszFile,
           LPCVOID lpBuffer,
           SSIZE_T nSize)
{
  BOOL fResult = FALSE;

  if (lpszFile)
    {
      HANDLE hFile;
      WCHAR szDir[MAX_PATH];

      lstrcpyW (szDir, lpszFile);
      PathRemoveFileSpecW (szDir);
      SHCreateDirectoryExW (NULL, szDir, NULL);
      hFile = CreateFileW (lpszFile, GENERIC_WRITE, FILE_SHARE_READ, NULL,
                                CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
      if (hFile != INVALID_HANDLE_VALUE)
        {
          DWORD dwSize, dwWrite;

          dwSize = nSize >= 0 ? nSize: lstrlenW (lpBuffer) * sizeof (WCHAR);
          fResult = WriteFile (hFile, lpBuffer, dwSize, &dwWrite, NULL)
                                                        && dwSize == dwWrite;
          if (!CloseHandle (hFile))
            fResult = FALSE;
        }
    }
  return fResult;
}


/*  メモリをファイルに書き込む
    lpszFile,ファイル名
    lpBuffer,メモリ
      nSize,バイト数(負:NULL終端文字列)
         RET,TRUE:正常終了,FALSE:エラー                                     */
BOOL WINAPI
SaveFileA (LPCSTR  lpszFile,
           LPCVOID lpBuffer,
           SSIZE_T nSize)
{
  BOOL fResult = FALSE;

  if (lpszFile)
    {
      HANDLE hFile;
      CHAR szDir[MAX_PATH];

      lstrcpyA (szDir, lpszFile);
      PathRemoveFileSpecA (szDir);
      SHCreateDirectoryExA (NULL, szDir, NULL);
      hFile = CreateFileA (lpszFile, GENERIC_WRITE, FILE_SHARE_READ, NULL,
                                CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
      if (hFile != INVALID_HANDLE_VALUE)
        {
          DWORD dwSize, dwWrite;

          dwSize = nSize >= 0 ? nSize: lstrlenA (lpBuffer) * sizeof (CHAR);
          fResult = WriteFile (hFile, lpBuffer, dwSize, &dwWrite, NULL)
                                                        && dwSize == dwWrite;
          if (!CloseHandle (hFile))
            fResult = FALSE;
        }
    }
  return fResult;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
typedef struct _LIST {
  struct _LIST *lpNext;
  LPVOID lpValue;
} LIST, *LPLIST;


/*  値を直接比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareDirect (LPVOID lpValue1,
                   LPVOID lpValue2,
                   LPVOID lpData)
{
  return (UINT_PTR)lpValue1 - (UINT_PTR)lpValue2;
}


/*  値を文字列として比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareStringW (LPVOID lpValue1,
                    LPVOID lpValue2,
                    LPVOID lpData)
{
  return lstrcmpW (lpValue1, lpValue2);
}


/*  値を文字列として比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareStringA (LPVOID lpValue1,
                    LPVOID lpValue2,
                    LPVOID lpData)
{
  return lstrcmpA (lpValue1, lpValue2);
}


/*  リストの値を解放する
    lpValue,値
     lpData,ユーザデータ                                                    */
VOID WINAPI
ListFree (LPVOID lpValue,
          LPVOID lpData)
{
  MemoryFree (lpValue);
}


/*  線形リストの末尾に追加する
    lpStart,線形リスト(NULL:新規作成)
    lpValue,値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListAppend (LPLIST lpStart,
            LPVOID lpValue)
{
  LPLIST lpNew, lpList;

  lpNew = MemoryAlloc (sizeof (LIST));
  lpNew->lpValue = lpValue;
  if (!lpStart)
    return lpNew;
  for (lpList = lpStart; lpList->lpNext; lpList = lpList->lpNext);
  lpList->lpNext = lpNew;
  return lpStart;
}


/*  線形リストの先頭に追加する
    lpStart,線形リスト(NULL:新規作成)
    lpValue,値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListPrepend (LPLIST lpStart,
             LPVOID lpValue)
{
  LPLIST lpNew;

  lpNew = MemoryAlloc (sizeof (LIST));
  lpNew->lpValue = lpValue;
  lpNew->lpNext = lpStart;
  return lpNew;
}


/*  線形リストに追加する
      lpStart,線形リスト(NULL:新規作成)
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListInsert (LPLIST        lpStart,
            LPVOID        lpValue,
            ListCompare_t lpCompare,
            LPVOID        lpData)
{
  LPLIST lpNew, lpList, lpPrev = NULL;

  lpNew = MemoryAlloc (sizeof (LIST));
  lpNew->lpValue = lpValue;
  if (!lpStart)
    return lpNew;
  if (!lpCompare)
    lpCompare = ListCompareDirect;
  for (lpList = lpStart; lpList->lpNext; lpList = lpList->lpNext)
    {
      if (lpCompare (lpList->lpValue, lpValue, lpData) > 0)
        {
          lpNew->lpNext = lpList;
          if (lpPrev)
            {
              lpPrev->lpNext = lpNew;
              return lpStart;
            }
          return lpNew;
        }
      lpPrev = lpList;
    }
  lpList->lpNext = lpNew;
  return lpStart;
}


/*  線形リストをソートする
      lpStart,線形リスト
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListSort (LPLIST        lpStart,
          ListCompare_t lpCompare,
          LPVOID        lpData)
{
  LPLIST lpFirst = NULL, lpLast = NULL;

  if (!lpCompare)
    lpCompare = ListCompareDirect;
  while (lpStart)
    {
      LPLIST lpList, lpPrev = NULL, lpSmall, lpSmallPrev = NULL;

      lpSmall = lpStart;
      for (lpList = lpStart->lpNext; lpList; lpList = lpList->lpNext)
        {
          if (lpCompare (lpList->lpValue, lpSmall->lpValue, lpData) < 0)
            {
              lpSmall = lpList;
              lpSmallPrev = lpPrev;
            }
          lpPrev = lpList;
        }
      if (lpSmallPrev)
        lpSmallPrev->lpNext = lpSmall->lpNext;
      else
        lpStart = lpSmall->lpNext;
      if (lpLast)
        lpLast->lpNext = lpSmall;
      else
        lpFirst = lpSmall;
      lpLast = lpSmall;
    }
  if (lpLast)
    lpLast->lpNext = NULL;
  return lpFirst;
}


/*  線形リストの先頭を求める
    lpStart,線形リスト
        RET,値(NULL:要素なし)                                               */
LPVOID WINAPI
ListFirst (LPLIST lpStart)
{
  return lpStart ? lpStart->lpValue : NULL;
}


/*  線形リストの末尾を求める
    lpStart,線形リスト
        RET,値(NULL:要素なし)                                               */
LPVOID WINAPI
ListLast (LPLIST lpStart)
{
  if (lpStart)
    {
      while (lpStart->lpNext)
        lpStart = lpStart->lpNext;
      return lpStart->lpValue;
    }
  return NULL;
}


/*  線形リストのn番目を求める
    lpStart,線形リスト
     nIndex,n番(負:後方から,正or0:前方から)
        RET,値(NULL:要素なし)                                               */
LPVOID WINAPI
ListIndex (LPLIST lpStart,
           int    nIndex)
{
  int nCount = 0;

  if (nIndex < 0)
    nIndex += ListLength (lpStart);
  while (lpStart)
    {
      if (nCount == nIndex)
        return lpStart->lpValue;
      nCount++;
      lpStart = lpStart->lpNext;
    }
  return NULL;
}


/*  線形リストの要素数を求める
    lpStart,線形リスト
        RET,要素数                                                          */
int WINAPI
ListLength (LPLIST lpStart)
{
  int nCount = 0;

  while (lpStart)
    {
      nCount++;
      lpStart = lpStart->lpNext;
    }
  return nCount;
}


/*  線形リストの一致する要素の数を求める
      lpStart,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,一致数                                                        */
int WINAPI
ListHasValue (LPLIST        lpStart,
              LPVOID        lpValue,
              ListCompare_t lpCompare,
              LPVOID        lpData)
{
  int nCount = 0;

  while (lpStart)
    {
      if (lpCompare ? lpCompare (lpStart->lpValue, lpValue, lpData) == 0
                    : lpStart->lpValue == lpValue)
        nCount++;
      lpStart = lpStart->lpNext;
    }
  return nCount;
}


/*  線形リストの一致する要素を削除する
      lpStart,線形リスト
      lpValue,値
         fAll,TRUE:すべて削除,FALSE:最初の要素のみ削除
    lpCompare,比較関数
       lpFree,解放関数
       lpData,ユーザデータ
          RET,線形リスト(NULL:すべて削除)                                   */
LPLIST WINAPI
ListRemove (LPLIST        lpStart,
            LPVOID        lpValue,
            BOOL          fAll,
            ListCompare_t lpCompare,
            ListFree_t    lpFree,
            LPVOID        lpData)
{
  LPLIST lpList, lpPrev = NULL;

  lpList = lpStart;
  while (lpList)
    if (lpCompare ? lpCompare (lpList->lpValue, lpValue, lpData) == 0
                  : lpList->lpValue == lpValue)
      {
        LPLIST lpNext;

        lpNext = lpList->lpNext;
        if (lpPrev)
          lpPrev->lpNext = lpNext;
        else
          lpStart = lpNext;
        if (lpFree)
          lpFree (lpList->lpValue, lpData);
        MemoryFree (lpList);
        if (!fAll)
          break;
        lpList = lpNext;
      }
    else
      {
        lpPrev = lpList;
        lpList = lpList->lpNext;
      }
  return lpStart;
}


/*  線形リストを解放する
    lpStart,線形リスト
     lpFree,解放関数
     lpData,ユーザデータ                                                    */
VOID WINAPI
ListRelease (LPLIST     lpStart,
             ListFree_t lpFree,
             LPVOID     lpData)
{
  while (lpStart)
    {
      LPLIST lpNext;

      lpNext = lpStart->lpNext;
      if (lpFree)
        lpFree (lpStart->lpValue, lpData);
      MemoryFree (lpStart);
      lpStart = lpNext;
    }
}


/*  線形リストの要素を列挙する
    lpStart,線形リスト
    lpFree,列挙関数
    lpData,ユーザデータ
       RET,TRUE:すべて列挙,FALSE:中断                                       */
BOOL WINAPI
ListEnum (LPLIST         lpStart,
          ListEnumProc_t lpProc,
          LPVOID         lpData)
{
  if (lpProc)
    while (lpStart)
      {
        if (!lpProc (lpStart->lpValue, lpData))
          return FALSE;
        lpStart = lpStart->lpNext;
      }
  return lpProc != NULL;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
typedef struct _HASHNODE {
  struct _HASHNODE *lpNext;
  LPVOID lpKey, lpValue;
} HASHNODE, *PHASHNODE, *LPHASHNODE;
typedef struct _HASH {
  HASHNODE *lpNode;
  HashCompare_t lpCompare;
  HashFree_t lpFreeKey, lpFreeValue;
  LPVOID lpData;
} HASH, *LPHASH;


/*  鍵を直接比較する
    lpKey1,鍵1
    lpKey2,鍵2
    lpData,ユーザデータ
       RET,TRUE:等しい,FALSE:異なる                                         */
BOOL WINAPI
HashCompareDirect (LPVOID lpKey1,
                   LPVOID lpKey2,
                   LPVOID lpData)
{
  return lpKey1 == lpKey2;
}


/*  鍵を文字列として比較する
    lpKey1,鍵1
    lpKey2,鍵2
    lpData,ユーザデータ
       RET,TRUE:等しい,FALSE:異なる                                         */
BOOL WINAPI
HashCompareStringA (LPVOID lpKey1,
                    LPVOID lpKey2,
                    LPVOID lpData)
{
  return lstrcmpA (lpKey1, lpKey2) == 0;
}


/*  鍵を文字列として比較する
    lpKey1,鍵1
    lpKey2,鍵2
    lpData,ユーザデータ
       RET,TRUE:等しい,FALSE:異なる                                         */
BOOL WINAPI
HashCompareStringW (LPVOID lpKey1,
                    LPVOID lpKey2,
                    LPVOID lpData)
{
  return lstrcmpW (lpKey1, lpKey2) == 0;
}


/*  ハッシュの鍵または値を解放する
    lpObject,鍵または値
      lpData,ユーザデータ                                                   */
VOID WINAPI
HashFree (LPVOID lpObject,
          LPVOID lpData)
{
  MemoryFree (lpObject);
}


/*  ハッシュテーブルを作成する
      lpHashCompare,比較関数
      lpHashFreeKey,鍵解放関数
    lpHashFreeValue,値解放関数
             lpData,ユーザデータ
                RET,ハッシュテーブル                                        */
LPHASH WINAPI
HashCreate (HashCompare_t lpHashCompare,
            HashFree_t    lpHashFreeKey,
            HashFree_t    lpHashFreeValue,
            LPVOID        lpData)
{
  LPHASH lpHash;

  lpHash = MemoryAlloc (sizeof (HASH));
  lpHash->lpCompare = lpHashCompare ? lpHashCompare : HashCompareDirect;
  lpHash->lpFreeKey = lpHashFreeKey;
  lpHash->lpFreeValue = lpHashFreeValue;
  lpHash->lpData = lpData;
  return lpHash;
}


/*  ハッシュテーブルを解放する
    lpHash,ハッシュテーブル                                                 */
VOID WINAPI
HashRelease (LPHASH lpHash)
{
  if (lpHash)
    while (lpHash->lpNode)
      {
        LPHASHNODE lpNode;

        lpNode = lpHash->lpNode;
        lpHash->lpNode = lpHash->lpNode->lpNext;
        if (lpHash->lpFreeKey)
          lpHash->lpFreeKey (lpNode->lpKey, lpHash->lpData);
        if (lpHash->lpFreeValue)
          lpHash->lpFreeValue (lpNode->lpValue, lpHash->lpData);
        MemoryFree (lpNode);
      }
  MemoryFree (lpHash);
}


/*  ハッシュテーブルに要素を挿入する
     lpHash,ハッシュテーブル
      lpKey,鍵
    lpValue,値
        RET,ハッシュテーブル                                                */
LPHASH WINAPI
HashInsert (LPHASH lpHash,
            LPVOID lpKey,
            LPVOID lpValue)
{
  LPHASHNODE lpNode;

  if (!lpHash)
    lpHash = HashCreate (NULL, NULL, NULL, NULL);
  if (lpHash->lpNode)
    {
      LPHASHNODE lpPrev = NULL;

      for (lpNode = lpHash->lpNode;
        lpNode && !lpHash->lpCompare (lpNode->lpKey, lpKey, lpHash->lpData);
                                                    lpNode = lpNode->lpNext)
        lpPrev = lpNode;
      if (lpNode)
        {
          if (lpHash->lpFreeKey)
            lpHash->lpFreeKey (lpNode->lpKey, lpHash->lpData);
          if (lpHash->lpFreeValue)
            lpHash->lpFreeValue (lpNode->lpValue, lpHash->lpData);
        }
      else
        {
          lpNode = MemoryAlloc (sizeof (HASHNODE));
          lpPrev->lpNext = lpNode;
        }
    }
  else
    {
      lpNode = lpHash->lpNode = MemoryAlloc (sizeof (HASHNODE));
    }
  lpNode->lpKey = lpKey;
  lpNode->lpValue = lpValue;
  return lpHash;
}


/*  ハッシュテーブルの値を求める
    lpHash,ハッシュテーブル
     lpKey,鍵
       RET,値(NULL:値なし)                                                  */
LPVOID WINAPI
HashLookup (LPHASH lpHash,
            LPVOID lpKey)
{
  LPHASHNODE lpNode;

  if (lpHash)
    for (lpNode = lpHash->lpNode; lpNode; lpNode = lpNode->lpNext)
      if (lpHash->lpCompare (lpNode->lpKey, lpKey, lpHash->lpData))
        return lpNode->lpValue;
  return NULL;
}


/*  ハッシュテーブルの要素を削除する
    lpHash,ハッシュテーブル
     lpKey,鍵
       RET,TRUE:削除できた,FALSE:削除できない                               */
BOOL WINAPI
HashRemove (LPHASH lpHash,
            LPVOID lpKey)
{
  if (lpHash)
    {
      LPHASHNODE lpNode, lpPrev = NULL;

      for (lpNode = lpHash->lpNode; lpNode; lpNode = lpNode->lpNext)
        {
          if (lpHash->lpCompare (lpNode->lpKey, lpKey, lpHash->lpData))
            {
              if (lpPrev)
                lpPrev->lpNext = lpNode->lpNext;
              else
                lpHash->lpNode = lpNode->lpNext;
              if (lpHash->lpFreeKey)
                lpHash->lpFreeKey (lpNode->lpKey, lpHash->lpData);
              if (lpHash->lpFreeValue)
                lpHash->lpFreeValue (lpNode->lpValue, lpHash->lpData);
              MemoryFree (lpNode);
              return TRUE;
            }
          lpPrev = lpNode;
        }
    }
  return FALSE;
}


/*  ハッシュテーブルの要素を列挙する
    lpHash,ハッシュテーブル
    lpProc,列挙関数
    lpData,ユーザデータ
       RET,TRUE:すべて列挙,FALSE:中断                                       */
BOOL WINAPI
HashEnum (LPHASH         lpHash,
          HashEnumProc_t lpProc,
          LPVOID         lpData)
{
  if (lpHash)
    {
      LPHASHNODE lpNode;

      for (lpNode = lpHash->lpNode; lpNode; lpNode = lpNode->lpNext)
        if (!lpProc (lpNode->lpKey, lpNode->lpValue, lpData))
          return FALSE;
    }
  return TRUE;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:コマンドラインの解析
    lpszCmdLine,コマンドライン文字列
            RET,引数へのポインタ,NULL:エラー                                */
LPWSTR * WINAPI
MakeArgumentW (LPCWSTR lpszCmdLine)
{
  LPWSTR *plpszArgv = NULL;

  if (lpszCmdLine)
    {
      int nArgc = 1, nLength = 0;
      LPCWSTR p;
      LPWSTR q;

      p = lpszCmdLine;
      while (*p != '\0')
        if (*p != ' ')
          {
            if (*p == '\"')
              while (*p != '\0')
                if (*p == '\"')
                  {
                    p++;
                    if (*p == ' ')
                      break;
                  }
                else if (*p == '\\' && *(p + 1) == '\"')
                  {
                    nLength++;
                    p += 2;
                  }
                else
                  {
                    nLength++;
                    p++;
                  }
            else
              while (*p != '\0' && *p != ' ')
                if (*p == '\"')
                  {
                    p++;
                  }
                else if (*p == '\\' && *(p + 1) == '\"')
                  {
                    nLength++;
                    p += 2;
                  }
                else
                  {
                    nLength++;
                    p++;
                  }
            nLength++;
            nArgc++;
          }
        else
          {
            p++;
          }
      plpszArgv = MemoryAlloc (nArgc * sizeof (LPWSTR)
                                                + nLength * sizeof (WCHAR));
      q = (LPWSTR)(plpszArgv + nArgc);
      nArgc = 0;
      p = lpszCmdLine;
      while (*p != '\0')
        if (*p != ' ')
          {
            plpszArgv[nArgc++] = q;
            if (*p == '\"')
              while (*p != '\0')
                if (*p == '\"')
                  {
                    p++;
                    if (*p == ' ')
                      break;
                  }
                else if (*p == '\\' && *(p + 1) == '\"')
                  {
                    *q++ = '\"';
                    p += 2;
                  }
                else
                  {
                    *q++ = *p++;
                  }
            else
              while (*p != '\0' && *p != ' ')
                if (*p == '\"')
                  {
                    p++;
                  }
                else if (*p == '\\' && *(p + 1) == '\"')
                  {
                    *q++ = '\"';
                    p += 2;
                  }
                else
                  {
                    *q++ = *p++;
                  }
            q++;
          }
        else
          {
            p++;
          }
    }
  return plpszArgv;
}


/*  ja:コマンドラインの解析
    lpszCmdLine,コマンドライン文字列
            RET,引数へのポインタ,NULL:エラー                                */
LPSTR * WINAPI
MakeArgumentA (LPCSTR lpszCmdLine)
{
  LPSTR *plpszArgv = NULL;

  if (lpszCmdLine)
    {
      int nArgc = 1, nLength = 0;
      LPCSTR p;
      LPSTR q;

      p = lpszCmdLine;
      while (*p != '\0')
        if (*p != ' ')
          {
            if (*p == '\"')
              while (*p != '\0')
                if (*p == '\"')
                  {
                    p++;
                    if (*p == ' ')
                      break;
                  }
                else if (*p == '\\' && *(p + 1) == '\"')
                  {
                    nLength++;
                    p += 2;
                  }
                else if (IsDBCSLeadByteEx (CP_ACP, *p))
                  {
                    nLength += 2;
                    p += 2;
                  }
                else
                  {
                    nLength++;
                    p++;
                  }
            else
              while (*p != '\0' && *p != ' ')
                if (*p == '\"')
                  {
                    p++;
                  }
                else if (*p == '\\' && *p == '\"')
                  {
                    nLength++;
                    p += 2;
                  }
                else if (IsDBCSLeadByteEx (CP_ACP, *p))
                  {
                    nLength += 2;
                    p += 2;
                  }
                else
                  {
                    nLength++;
                    p++;
                  }
            nLength++;
            nArgc++;
          }
        else
          {
            p++;
          }
      plpszArgv = MemoryAlloc (nArgc * sizeof (LPSTR)
                                                    + nLength * sizeof (CHAR));
      q = (LPSTR)(plpszArgv + nArgc);
      nArgc = 0;
      p = lpszCmdLine;
      while (*p != '\0')
        if (*p != ' ')
          {
            plpszArgv[nArgc++] = q;
            if (*p == '\"')
              while (*p != '\0')
                if (*p == '\"')
                  {
                    p++;
                    if (*p == ' ')
                      break;
                  }
                else if (*p == '\\' && *(p + 1) == '\"')
                  {
                    *q++ = '\"';
                    p += 2;
                  }
                else if (IsDBCSLeadByteEx (CP_ACP, *p))
                  {
                    *q++ = *p++;
                    *q++ = *p++;
                  }
                else
                  {
                    *q++ = *p++;
                  }
            else
              while (*p != '\0' && *p != ' ')
                if (*p == '\"')
                  {
                    p++;
                  }
                else if (*p == '\\' && *(p + 1) == '\"')
                  {
                    *q++ = '\"';
                    p += 2;
                  }
                else if (IsDBCSLeadByteEx (CP_ACP, *p))
                  {
                    *q++ = *p++;
                    *q++ = *p++;
                  }
                else
                  {
                    *q++ = *p++;
                  }
            q++;
          }
        else
          {
            p++;
          }
    }
  return plpszArgv;
}


/*  文字列をデバイスに出力する
    nStdHandle,デバイス
    lpszFormat,書式文字列                                                   */
VOID WINAPI
PrintConsoleW (DWORD   nStdHandle,
               LPCWSTR lpszFormat, ...)
{
  if (lpszFormat)
    {
      va_list arglist;
      DWORD dwWrite;
      LPWSTR lpszText = NULL;

      va_start (arglist, lpszFormat);
      wvasprintfW (&lpszText, lpszFormat, arglist);
      va_end (arglist);
      WriteConsoleW (GetStdHandle (nStdHandle), lpszText,
                                        lstrlenW (lpszText), &dwWrite, NULL);
      MemoryFree (lpszText);
    }
}


/*  文字列をデバイスに出力する
    nStdHandle,デバイス
    lpszFormat,書式文字列                                                   */
VOID WINAPI
PrintConsoleA (DWORD  nStdHandle,
               LPCSTR lpszFormat, ...)
{
  if (lpszFormat)
    {
      va_list arglist;
      DWORD dwWrite;
      LPSTR lpszText;

      va_start (arglist, lpszFormat);
      wvasprintfA (&lpszText, lpszFormat, arglist);
      va_end (arglist);
      WriteConsoleA (GetStdHandle (nStdHandle), lpszText,
                                        lstrlenA (lpszText), &dwWrite, NULL);
      MemoryFree (lpszText);
    }
}
