/*
    peimage
    copyright (c) 1998-2017 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <windows.h>
#include "common.h"


#define PECHECK_HEADER 0
#define PECHECK_FILE   1
#define PECHECK_IMAGE  2


/*  ヘッダをチェックする
    lpcvHeader,ヘッダ
        dwFile,サイズ
         nMode,モード(PECHECK_HEADER,PECHECK_FILE,PECHECK_IMAGE)
           RET,TRUE:正常終了,NULL:エラー                                    */
static BOOL WINAPI
PeCheck32 (LPCVOID lpcvHeader,
           DWORD   dwSize,
           int     nMode)
{
  int i;
  DWORD dwRaw = 0, dwVirtual = 0;
  PIMAGE_NT_HEADERS32 pinth;
  PIMAGE_SECTION_HEADER pish;

  if (dwSize <= 0)
    dwSize = UINT_MAX;
  if (!lpcvHeader || sizeof (IMAGE_DOS_HEADER) > dwSize
    || ((PIMAGE_DOS_HEADER)lpcvHeader)->e_magic != IMAGE_DOS_SIGNATURE
    || ((PIMAGE_DOS_HEADER)lpcvHeader)->e_lfanew
                                        + sizeof (IMAGE_NT_HEADERS32) > dwSize)
    return FALSE;
  pinth = (PIMAGE_NT_HEADERS32)((LPBYTE)lpcvHeader
                                + ((PIMAGE_DOS_HEADER)lpcvHeader)->e_lfanew);
  pish = (PIMAGE_SECTION_HEADER)(pinth + 1);
  if (pinth->Signature != IMAGE_NT_SIGNATURE
        || pinth->FileHeader.Machine != IMAGE_FILE_MACHINE_I386
        || pinth->FileHeader.SizeOfOptionalHeader
                                            != sizeof (IMAGE_OPTIONAL_HEADER32)
        || pinth->OptionalHeader.Magic != IMAGE_NT_OPTIONAL_HDR32_MAGIC
        || (pinth->OptionalHeader.Subsystem != IMAGE_SUBSYSTEM_NATIVE
         && pinth->OptionalHeader.Subsystem != IMAGE_SUBSYSTEM_WINDOWS_GUI
         && pinth->OptionalHeader.Subsystem != IMAGE_SUBSYSTEM_WINDOWS_CUI
         && pinth->OptionalHeader.Subsystem != IMAGE_SUBSYSTEM_NATIVE_WINDOWS
         && pinth->OptionalHeader.Subsystem
                                != IMAGE_SUBSYSTEM_WINDOWS_BOOT_APPLICATION)
        || pinth->OptionalHeader.NumberOfRvaAndSizes
                                            != IMAGE_NUMBEROF_DIRECTORY_ENTRIES
        || ((PIMAGE_DOS_HEADER)lpcvHeader)->e_lfanew
                                    + sizeof (IMAGE_NT_HEADERS32)
                                    + pinth->FileHeader.NumberOfSections
                                    * sizeof (IMAGE_SECTION_HEADER) > dwSize)
    return FALSE;
  for (i = 0; i < pinth->FileHeader.NumberOfSections; i++)
    {
      if (pish[i].PointerToRawData + pish[i].SizeOfRawData > dwRaw)
        dwRaw = pish[i].PointerToRawData + pish[i].SizeOfRawData;
      if (pish[i].VirtualAddress + pish[i].Misc.VirtualSize > dwVirtual)
        dwVirtual = pish[i].VirtualAddress + pish[i].Misc.VirtualSize;
    }
  for (i = 0; i < IMAGE_NUMBEROF_DIRECTORY_ENTRIES; i++)
    if (i != IMAGE_DIRECTORY_ENTRY_SECURITY
                    && pinth->OptionalHeader.DataDirectory[i].VirtualAddress
                                + pinth->OptionalHeader.DataDirectory[i].Size
                                                                > dwVirtual)
      return FALSE;
  return pinth->OptionalHeader.AddressOfEntryPoint < dwVirtual
                            && (nMode != PECHECK_FILE  || dwRaw     <= dwSize)
                            && (nMode != PECHECK_IMAGE || dwVirtual <= dwSize);
}


/*  ヘッダをチェックする
    lpcvHeader,ヘッダ
        dwSize,サイズ
           RET,TRUE:正常終了,NULL:エラー                                    */
PeCheckHeader32 (LPCVOID lpcvHeader,
                 DWORD   dwSize)
{
  return PeCheck32 (lpcvHeader, dwSize, PECHECK_HEADER);
}


/*  ヘッダをチェックする
    lpcvFile,ファイル
      dwSize,ファイルサイズ
         RET,TRUE:正常終了,NULL:エラー                                      */
PeCheckFile32 (LPCVOID lpcvFile,
               DWORD   dwSize)
{
  return PeCheck32 (lpcvFile, dwSize, PECHECK_FILE);
}


/*  ヘッダをチェックする
    lpcvImage,イメージ
       dwSize,イメージサイズ
          RET,TRUE:正常終了,NULL:エラー                                     */
PeCheckImage32 (LPCVOID lpcvImage,
                DWORD   dwSize)
{
  return PeCheck32 (lpcvImage, dwSize, PECHECK_IMAGE);
}


/*  ファイルをイメージに変換する
    lpcvFile,ファイル
    lpdwSize,イメージサイズ
         RET,イメージ,NULL:エラー                                           */
LPVOID WINAPI
PeFileToImage32 (LPCVOID lpcvFile,
                 LPDWORD lpdwSize)
{
  LPVOID lpvImage = NULL;

  if (lpcvFile)
    {
      int i;
      DWORD dwMin = UINT_MAX, dwMax = 0;
      PIMAGE_NT_HEADERS32 pinth;
      PIMAGE_SECTION_HEADER pish;

      pinth = (PIMAGE_NT_HEADERS32)((LPBYTE)lpcvFile
                                    + ((PIMAGE_DOS_HEADER)lpcvFile)->e_lfanew);
      pish = (PIMAGE_SECTION_HEADER)(pinth + 1);
      for (i = 0; i < pinth->FileHeader.NumberOfSections; i++)
        {
          DWORD dwValue;

          if (pish[i].PointerToRawData != 0
                                        && dwMin > pish[i].PointerToRawData)
            dwMin = pish[i].PointerToRawData;
          dwValue = max (pish[i].SizeOfRawData, pish[i].Misc.VirtualSize)
                                                    + pish[i].VirtualAddress;
          if (dwMax < dwValue)
            dwMax = dwValue;
        }
      if (dwMin < dwMax && (lpvImage = MemoryAlloc (dwMax)))
        {
          MemoryCopy (lpvImage, lpcvFile, dwMin);
          for (i = 0; i < pinth->FileHeader.NumberOfSections; i++)
            MemoryCopy ((LPBYTE)lpvImage + pish[i].VirtualAddress,
                                  (LPBYTE)lpcvFile + pish[i].PointerToRawData,
                                  pish[i].SizeOfRawData);
          if (lpdwSize)
            *lpdwSize = dwMax;
        }
    }
  return lpvImage;
}


static LPVOID WINAPI
PeLoadHandle32 (HANDLE  hFile,
                LPDWORD lpdwSize)
{
  LPVOID lpvImage = NULL;

  if (hFile != INVALID_HANDLE_VALUE)
    {
      DWORD dwRead, dwSize;
      IMAGE_DOS_HEADER idh;
      IMAGE_NT_HEADERS32 inth;

      if ((dwSize = GetFileSize (hFile, NULL)) != INVALID_FILE_SIZE
        && ReadFile (hFile, &idh, sizeof (IMAGE_DOS_HEADER), &dwRead, NULL)
                                && dwRead == sizeof (IMAGE_DOS_HEADER)
        && idh.e_magic == IMAGE_DOS_SIGNATURE
        && idh.e_lfanew + sizeof (IMAGE_NT_HEADERS32) <= dwSize
        && SetFilePointer (hFile, idh.e_lfanew, NULL, FILE_BEGIN)
                                                    != INVALID_SET_FILE_POINTER
        && ReadFile (hFile, &inth, sizeof (IMAGE_NT_HEADERS32), &dwRead, NULL)
                                && dwRead == sizeof (IMAGE_NT_HEADERS32)
                                && inth.Signature == IMAGE_NT_SIGNATURE
        && inth.FileHeader.Machine == IMAGE_FILE_MACHINE_I386
        && inth.FileHeader.SizeOfOptionalHeader
                                            == sizeof (IMAGE_OPTIONAL_HEADER32)
        && inth.OptionalHeader.Magic == IMAGE_NT_OPTIONAL_HDR32_MAGIC
        && (inth.OptionalHeader.Subsystem == IMAGE_SUBSYSTEM_NATIVE
         || inth.OptionalHeader.Subsystem == IMAGE_SUBSYSTEM_WINDOWS_GUI
         || inth.OptionalHeader.Subsystem == IMAGE_SUBSYSTEM_WINDOWS_CUI
         || inth.OptionalHeader.Subsystem == IMAGE_SUBSYSTEM_NATIVE_WINDOWS
         || inth.OptionalHeader.Subsystem
                                == IMAGE_SUBSYSTEM_WINDOWS_BOOT_APPLICATION)
        && inth.OptionalHeader.NumberOfRvaAndSizes
                                        == IMAGE_NUMBEROF_DIRECTORY_ENTRIES)
        {
          DWORD dwSection;
          PIMAGE_SECTION_HEADER pish;

          dwSection = inth.FileHeader.NumberOfSections
                                            * sizeof (IMAGE_SECTION_HEADER);
          pish = MemoryAlloc (dwSection);
          if (pish)
            {
              if (ReadFile (hFile, pish, dwSection, &dwRead, NULL)
                                                        && dwSection == dwRead)
                {
                  int i;
                  BOOL fError = TRUE;
                  DWORD dwMin = UINT_MAX, dwMax = 0, dwRaw = 0, dwVirtual = 0;

                  for (i = 0; i < inth.FileHeader.NumberOfSections; i++)
                    {
                      DWORD dwValue;

                      dwValue = pish[i].PointerToRawData
                                                    + pish[i].SizeOfRawData;
                      if (dwValue > dwRaw)
                        dwRaw = dwValue;
                      dwValue = pish[i].VirtualAddress
                                                    + pish[i].Misc.VirtualSize;
                      if (dwValue > dwVirtual)
                        dwVirtual = dwValue;
                      if (pish[i].PointerToRawData != 0
                                        && dwMin > pish[i].PointerToRawData)
                        dwMin = pish[i].PointerToRawData;
                      dwValue = max (pish[i].SizeOfRawData,
                                                    pish[i].Misc.VirtualSize)
                                                    + pish[i].VirtualAddress;
                      if (dwMax < dwValue)
                        dwMax = dwValue;
                    }
                  for (i = 0; i < IMAGE_NUMBEROF_DIRECTORY_ENTRIES; i++)
                    if (i != IMAGE_DIRECTORY_ENTRY_SECURITY
                        && inth.OptionalHeader.DataDirectory[i].VirtualAddress
                         + inth.OptionalHeader.DataDirectory[i].Size
                                                                > dwVirtual)
                      break;
                  if (dwRaw <= dwSize && i == IMAGE_NUMBEROF_DIRECTORY_ENTRIES
                        && dwMin < dwMax && (lpvImage = MemoryAlloc (dwMax))
                        && SetFilePointer (hFile, 0, NULL, FILE_BEGIN)
                                                    != INVALID_SET_FILE_POINTER
                        && ReadFile (hFile, lpvImage, dwMin, &dwRead, NULL)
                                                            && dwMin == dwRead)
                    {
                      for (i = 0; i < inth.FileHeader.NumberOfSections; i++)
                        if (pish[i].SizeOfRawData != 0
                            && (SetFilePointer (hFile,
                                                    pish[i].PointerToRawData,
                                                            NULL, FILE_BEGIN)
                                                    == INVALID_SET_FILE_POINTER
                             || !ReadFile (hFile,
                                    (LPBYTE)lpvImage + pish[i].VirtualAddress,
                                    pish[i].SizeOfRawData, &dwRead, NULL)
                             || pish[i].SizeOfRawData != dwRead))
                          break;
                      if (i >= inth.FileHeader.NumberOfSections)
                        fError = FALSE;
                    }
                  if (fError)
                    {
                      MemoryFree (lpvImage);
                      lpvImage = NULL;
                    }
                  else if (lpdwSize)
                    {
                      *lpdwSize = dwMax;
                    }
                }
              MemoryFree (pish);
            }
        }
    }
  return lpvImage;
}


/*  ファイルを読み込んでイメージを取得する
    lpszFile,ファイル名
    lpdwSize,イメージサイズ
         RET,イメージ,NULL:エラー                                           */
LPVOID WINAPI
PeLoad32A (LPCSTR  lpszFile,
           LPDWORD lpdwSize)
{
  LPVOID lpvImage = NULL;
  HANDLE hFile;

  hFile = CreateFileA (lpszFile, GENERIC_READ, FILE_SHARE_READ, NULL,
                                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile != INVALID_HANDLE_VALUE)
    {
      lpvImage = PeLoadHandle32 (hFile, lpdwSize);
      if (!CloseHandle (hFile))
        {
          MemoryFree (lpvImage);
          lpvImage = NULL;
        }
    }
  return lpvImage;
}


/*  ファイルを読み込んでイメージを取得する
    lpszFile,ファイル名
    lpdwSize,イメージサイズ
         RET,イメージ,NULL:エラー                                           */
LPVOID WINAPI
PeLoad32W (LPCWSTR lpszFile,
           LPDWORD lpdwSize)
{
  LPVOID lpvImage = NULL;
  HANDLE hFile;

  hFile = CreateFileW (lpszFile, GENERIC_READ, FILE_SHARE_READ, NULL,
                                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile != INVALID_HANDLE_VALUE)
    {
      lpvImage = PeLoadHandle32 (hFile, lpdwSize);
      if (!CloseHandle (hFile))
        {
          MemoryFree (lpvImage);
          lpvImage = NULL;
        }
    }
  return lpvImage;
}


/*  イメージの再配置
    lpvImage,イメージ
      dwBase,ベースアドレス
         RET,TRUE:正常終了,NULL:エラー                                      */
VOID WINAPI
PeRelocateBase32 (LPVOID lpvImage,
                  DWORD  dwBase)
{
  if (lpvImage)
    {
      int nDelta;
      DWORD dwEnd;
      PIMAGE_NT_HEADERS32 pinh;

      pinh = (PIMAGE_NT_HEADERS32)((LPBYTE)lpvImage
                                    + ((PIMAGE_DOS_HEADER)lpvImage)->e_lfanew);
      nDelta = dwBase - pinh->OptionalHeader.ImageBase;
      dwEnd = pinh->OptionalHeader
                .DataDirectory[IMAGE_DIRECTORY_ENTRY_BASERELOC].VirtualAddress
            + pinh->OptionalHeader
                .DataDirectory[IMAGE_DIRECTORY_ENTRY_BASERELOC].Size;
      if (nDelta != 0 && pinh->OptionalHeader
                    .DataDirectory[IMAGE_DIRECTORY_ENTRY_BASERELOC].Size > 0)
        {
          DWORD dwAddress, dwSize;

          dwSize = pinh->OptionalHeader.SizeOfImage;
          dwAddress = pinh->OptionalHeader
                .DataDirectory[IMAGE_DIRECTORY_ENTRY_BASERELOC].VirtualAddress;
          while (dwAddress + sizeof (IMAGE_BASE_RELOCATION) <= dwEnd)
            {
              int i, count;
              DWORD dwBase;
              LPWORD lpwTypeOffset;
              PIMAGE_BASE_RELOCATION pibr;

              pibr = (PIMAGE_BASE_RELOCATION)((LPBYTE)lpvImage + dwAddress);
              dwBase = pibr->VirtualAddress;
              if (dwBase == 0 || dwAddress + pibr->SizeOfBlock > dwEnd)
                break;
              count = (pibr->SizeOfBlock - sizeof (IMAGE_BASE_RELOCATION))
                                                            / sizeof (WORD);
              lpwTypeOffset = (LPWORD)(pibr + 1);
              for (i = 0; i < count; i++)
                {
                  WORD wTypeOffset;

                  wTypeOffset = lpwTypeOffset[i];
                  if (((wTypeOffset >> 12) & 0xf) == IMAGE_REL_BASED_HIGHLOW
                                && dwBase + (wTypeOffset & 0xfff) <= dwSize)
                    {
                      LPDWORD lpAddress;

                      lpAddress = (LPDWORD)((LPBYTE)lpvImage + dwBase
                                                    + (wTypeOffset & 0xfff));
                      *lpAddress += nDelta;
                    }
                }
              dwAddress += pibr->SizeOfBlock;
            }
        }
    }
}


/*  イメージの再配置
    lpvImage,イメージ
         RET,TRUE:正常終了,NULL:エラー                                      */
VOID WINAPI
PeRelocate32 (LPVOID lpvImage)
{
  PeRelocateBase32 (lpvImage, (DWORD)lpvImage);
}
