/*
 * Copyright [yyyy] [name of copyright owner]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


// Anonymous function start
//
(function( window, undefined )
{

var Config			= h5glib.Config;
var Debug			= h5glib.Debug;
var ListElement		= h5glib.ListElement;
var Task			= h5glib.Task;
var Message			= h5glib.Message;
var MessageHandler	= h5glib.MessageHandler;
var Scene			= h5glib.Scene;

var PI	= Math.PI;
var PI2	= Math.PI * 2;

var SCENE_NAME			= "stage";
var BATTLE_SCENE_NAME	= h5glib.BATTLE_SCENE_NAME;

var USAGE_HTML			= "<p>[←][→]: 回転 [↑][↓]: 移動 [Z]: 発砲</p><p>キャラに接近するとイベント発動（馬に接近すると戦闘）</p>";

/**
 * Priority
 */
var Priority =
{
	NONE	: 0,
	STAGE	: 1,
	ACTOR	: 2,
	EVENT	: 3,
	FRONT	: 4,
	DEBUG	: 5
};

/**
 * RenderInfo
 */
var RenderInfo =
{
	fov			: ( 60 * PI / 180 ),
	rayPixels	: 2,	//  8,   4,   2,   1
	samples		: 200,	// 50, 100, 200, 400
	gridPixels	: 64
};

/**
 * MessageType
 */
var MessageType =
{
	NONE			: 0,
	RET_PREV_SCENE	: 1,
	START_BATTLE	: 2
};

/**
 * StageTask
 */
var StageTask = function( scene )
{
	this.command	= scene.input;

	this.map		= null,
	this.getHeight	= function() { return this.map.length; };
	this.getWidth	= function() { return this.map[0].length; };

	var RaycastingData = function()
	{
		this.dist	= 0;
		this.val	= 0;
		this.tx		= 0;
	};
	this.rd = [];
	for ( var i = 0; i < RenderInfo.samples; i++ )
	{
		this.rd[i] = new RaycastingData();
	}
};
StageTask.prototype = new Task();

(function( pt )
{
	pt.AVAL_BASE	= 10000;

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		this.castRay( scene );

		var canvas		= scene.canvas;
		var context		= scene.context;
		var resource	= scene.resource;
		var viewPoint	= scene.viewPoint;

		var image;
		// sky
		image		= resource.skyImage;
		var skyW2	= image.width - canvas.width;
		context.drawImage(
			image,
			Math.floor( skyW2 - ( viewPoint.dir / PI2 * skyW2 ) ),	// sx
			0,				// sy
			canvas.width,	// sw
			image.height,	// sh
			0,				// dx
			0,				// dy
			canvas.width,	// dw
			image.height	// dh
		);
		// floor
		image = resource.floorImage;
		context.drawImage(
			image,
			0,				// sx
			0,				// sy
			canvas.width,	// sw
			image.height,	// sh
			0,				// dx
			canvas.height - image.height,		// dy
			canvas.width,	// dw
			image.height	// dh
		);

		// wall
		var startTheta	= viewPoint.dir + ( RenderInfo.fov / 2 );

		for ( var i = 0; i < RenderInfo.samples; i++ )
		{
			var theta = startTheta - ( i * RenderInfo.fov / RenderInfo.samples );
			theta %= PI2;

			var sx	= this.rd[i].tx * ( RenderInfo.gridPixels - 1 );
			var sy	= ( this.rd[i].val - 1 ) * RenderInfo.gridPixels;
			var sw	= 1;
			var sh	= RenderInfo.gridPixels;

			var dist= this.rd[i].dist * Math.cos( theta - viewPoint.dir );
			var dh	= canvas.height / dist;
			var dx	= i * RenderInfo.rayPixels;
			var dy	= ( canvas.height / 2 ) - ( dh / 2 );
			var dw	= RenderInfo.rayPixels;

			context.drawImage( resource.wallImage, sx, sy, sw, sh, dx, dy, dw, dh );
		}
	};
	/**
	 * 
	 */
	pt.castRay = function( scene )
	{
		var viewPoint	= scene.viewPoint;
		var startTheta	= viewPoint.dir + ( RenderInfo.fov / 2 );

		for ( var i = 0; i < RenderInfo.samples; i++ )
		{
			var theta = startTheta - ( i * RenderInfo.fov / RenderInfo.samples );
			theta %= PI2;

			var mx = Math.floor( viewPoint.x );
			var my = Math.floor( viewPoint.y );

			var cosTheta = Math.cos( theta );
			var sinTheta = Math.sin( theta );

			var distX, distY;
			var stepX, stepY;

			if ( cosTheta == 0 )
			{
				distX = 100;
			}
			else if ( cosTheta > 0 )
			{
				stepX = 1;
				distX = ( mx + 1 - viewPoint.x ) / cosTheta;
			}
			else
			{
				stepX = -1;
				cosTheta *= -1.0;
				distX = ( viewPoint.x - mx ) / cosTheta;
			}

			if ( sinTheta == 0 )
			{
				distY = 100;
			}
			else if ( sinTheta > 0 )
			{
				stepY = -1;
				distY = ( viewPoint.y - my ) / sinTheta;
			}
			else
			{
				stepY = 1;
				sinTheta *= -1.0;
				distY = ( my + 1 - viewPoint.y ) / sinTheta;
			}

			while ( true )
			{
				if ( distX < distY )
				{
					mx += stepX;
					var mapVal = ( 0 <= mx || mx < this.getWidth() ) ? this.map[ my ][ mx ] : 1;
					if ( mx < 0 || this.getWidth() <= mx || mapVal )
					{
						// not actor
						if ( mapVal < this.AVAL_BASE )
						{
							this.rd[i].dist	= distX;
							this.rd[i].val	= mapVal;
							this.rd[i].tx	= 1 - ( viewPoint.y + distX * sinTheta * stepY ) % 1;
							break;
						}
					}
					distX += ( 1 / cosTheta );	// + dist per grid
				}
				else
				{
					my += stepY;
					var mapVal = this.map[ my ][ mx ];
					var mapVal = ( 0 <= my || my < this.getHeight() ) ? this.map[ my ][ mx ] : 1;
					if ( my < 0 || this.getHeight() <= my || mapVal )
					{
						// not actor
						if ( mapVal < this.AVAL_BASE )
						{
							this.rd[i].dist	= distY;
							this.rd[i].val	= mapVal;
							this.rd[i].tx	= ( viewPoint.x + distY * cosTheta * stepX ) % 1;
							break;
						}
					}
					distY += ( 1 / sinTheta );	// + dist per grid
				}
			}
		}
	};
})( StageTask.prototype );

/**
 * ActorListTask
 */
var ActorListTask = function( scene )
{
	this.command	= scene.input;

	this.child		= new ListElement();	// dummy element
	this.visible	= [];
};
ActorListTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		for ( var actor = this.child.next; actor != null; actor = actor.next )
		{
			if ( actor.update( scene ) )
			{
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		this.pickVisible( scene );

		var canvas		= scene.canvas;
		var context		= scene.context;
		var resource	= scene.resource;
		var stage		= scene.stage;
		var viewPoint	= scene.viewPoint;

		for ( var i = 0; i < this.visible.length; i++ )
		{
			var actor = this.visible[i];
			// if behind wall
			if ( stage.rd[ actor.rd.index ].dist <= actor.rd.dist )
			{
				continue;
			}

			var offSx = 0;
			var offSy = 0;
			if ( actor.status != ActorStatus.STATIC )
			{
				if ( actor.status == ActorStatus.ACTIVE )
				{
					// animation
					offSx = Math.floor( ( scene.ticks % 20 ) / 10 ) * RenderInfo.gridPixels;
				}
				else if ( actor.status == ActorStatus.MORTAL )
				{
					offSx = ( ( actor.lagM == 0 ) ? 3 : 2 ) * RenderInfo.gridPixels;
				}
				// angle
				var sdir = viewPoint.dir - actor.dir;
				if ( sdir < 0 ) { sdir = PI2 + sdir; }
				sdir = sdir / ( PI / 4 );
				if ( sdir < 1 || sdir >= 7 )
				{
					sdir = 0;
				}
				else
				{
					sdir = Math.floor( ( sdir + 1 ) / 2 );
				}
				offSy = sdir * RenderInfo.gridPixels;
			}
			var dh	= canvas.height / actor.rd.dist;
			var dw	= dh;
			var dx	= actor.rd.index * RenderInfo.rayPixels - dw / 2;
			var dy	= canvas.height / 2 - dh / 2;

			var rx	= dx / RenderInfo.rayPixels;
			var rw	= dw / RenderInfo.rayPixels;

			for ( var n = 0; n < rw; n++ )
			{
				var rid = Math.floor( rx + n );
				if ( rid < 0 ) { continue; }
				if ( rid >= RenderInfo.samples ) { break; }
				// draw
				if ( stage.rd[ rid ].dist > actor.rd.dist )
				{
					var image = resource.actorImages[ actor.type ];
					var sx = offSx + ( n / rw ) * ( RenderInfo.gridPixels - 1 );
					var sy = offSy;
					var sw = RenderInfo.gridPixels / rw;
					var sh = RenderInfo.gridPixels;
					context.drawImage(
						image, sx, sy, sw, sh, dx + ( n * RenderInfo.rayPixels ), dy, RenderInfo.rayPixels, dh );
				}
			}
		}
	};
	/**
	 * 
	 */
	pt.pickVisible = function( scene )
	{
		var viewPoint	= scene.viewPoint;
		var stage		= scene.stage;

		var startTheta	= viewPoint.dir + ( RenderInfo.fov / 2 );

		// search visible actor
		this.visible.length = 0;
		for ( var actor = this.child.next.next; actor != null; actor = actor.next )
		{
			var x		= actor.x - viewPoint.x;
			var y		= viewPoint.y - actor.y;
			var thetaV	= Math.atan2( y, x );	// PI ... PI * -1
			if ( thetaV < 0 ) { thetaV += PI2; }

			var diff	= startTheta - thetaV;
			if ( 0 <= diff && diff <= RenderInfo.fov )
			{
				actor.rd.index	= Math.floor( diff / ( RenderInfo.fov / RenderInfo.samples ) );
				actor.rd.dist	= Math.sqrt( x * x + y * y );
				this.visible.push( actor );
			}
		}
		this.visible.sort( function( a, b ) { return b.rd.dist - a.rd.dist;  } );
	};
})( ActorListTask.prototype );

/**
 * ActorStatus
 */
var ActorStatus =
{
	STATIC	: 0,
	ACTIVE	: 1,
	MORTAL	: 2
};

/**
 * ActorTask
 */
var ActorTask = function()
{
	this.updateFunc	=
	[
		this.update,
		this.updateActive,
		this.updateMortal
	];
	this.lagM	= 0;

	this.id			= 0;
	this.type		= 0;
	this.prop		=
	{
		hp	: 100,
		mp	: 100
	};

	this.x			= 0;
	this.y			= 0;
	this.z			= 0;
	this.dir		= 0;
	this.event		= null;

	this.velocity	= 0;

	this.rd =
	{
		index	: 0,
		dist	: 0
	};
};
ActorTask.prototype = new Task();

(function( pt )
{
	pt.LAGM_VAL		= 2;
	pt.ROTATE_VAL	= 0.07;
	pt.STEP_VAL		= 0.20;

	/**
	 * 
	 */
	pt.init = function( scene, id, adata )
	{
		this.id		= id;
		this.type	= adata.type;
		this.moveTo( scene.stage, adata.x, adata.y );
		this.z		= adata.z;
		this.dir	= adata.dir;
		this.event	= adata.event;
		this.command= ( id == 0 ) ? scene.input : new AICommand();

		this.setStatus( ActorStatus.ACTIVE );	//...default status
	};
	/**
	 * 
	 */
	pt.setStatus = function( status )
	{
		this.status	= status;
		this.update	= this.updateFunc[ this.status ];
		if ( this.status == ActorStatus.MORTAL )
		{
			this.lagM	= this.LAGM_VAL;
		}
	};
	/**
	 * 
	 */
	pt.moveTo = function( stage, x, y )
	{
		// restore map value
		stage.map[ Math.floor( this.y ) ][ Math.floor( this.x ) ] = 0;
		this.x = x;
		this.y = y;
		// update map value
		stage.map[ Math.floor( this.y ) ][ Math.floor( this.x ) ] = stage.AVAL_BASE + this.id;
	};
	/**
	 * 
	 */
	pt.rotate = function( val )
	{
		this.dir	+= val;
		this.dir	%= PI2;
		if ( this.dir < 0 ) { this.dir += PI2; }
	};
	/**
	 * 
	 */
	pt.isInnerWall = function( stage, x, y )
	{
		var arr = [ -0.3, 0.3 ];

		for ( var i in arr )
		{
			var my = Math.floor( y + i );
			if ( my < 0 || stage.getHeight() <= my ) { return true; }

			for ( var j in arr )
			{
				var mx = Math.floor( x + j );
				if ( mx < 0 || stage.getWidth() <= mx ) { return true; }

				var value = stage.map[ my ][ mx ];
				if ( value != 0 && value != stage.AVAL_BASE + this.id )
				{
					return true;
				}
			}
		}
		return false;
	};
	/**
	 * 
	 */
	pt.updateActive = function( scene )
	{
		var upd = false;

		this.command.update();

		// left right
		if ( this.command.left )
		{
			this.rotate( this.ROTATE_VAL );
			upd = true;
		}
		else if ( this.command.right )
		{
			this.rotate( this.ROTATE_VAL * -1 );
			upd = true;
		}

		// go back
		if ( this.command.go )
		{
			this.velocity = this.STEP_VAL;
			upd = true;
		}
		else if ( this.command.back )
		{
			this.velocity = this.STEP_VAL * -1;
			upd = true;
		}
		else
		{
			if ( this.velocity != 0 )
			{
				this.velocity /= 2;
				if ( this.velocity < this.STEP_VAL / 8 )
				{
					this.velocity = 0;
				}
				upd = true;
			}
		}

		if ( this.velocity != 0 )
		{
			var newX	= this.x + Math.cos( this.dir ) * this.velocity;
			var newY	= this.y - Math.sin( this.dir ) * this.velocity;

			if ( !this.isInnerWall( scene.stage, newX, this.y ) )
			{
				this.moveTo( scene.stage, newX, this.y );
				upd = true;
			}
			if ( !this.isInnerWall( scene.stage, this.x, newY ) )
			{
				this.moveTo( scene.stage, this.x, newY );
				upd = true;
			}
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.updateMortal = function( scene )
	{
		var upd = false;
		// fire
		if ( this.lagM )
		{
			this.lagM--;
			upd = true;
		}
		return upd;
	};
})( ActorTask.prototype );

/**
 * EventTask
 */
var EventTask = function( scene )
{
	this.command	= scene.input;
	this.prevDist	= [];
};
EventTask.prototype = new Task();

(function( pt )
{
	pt.EVENT_DIST	= 1;
	pt.FIRE_DIST	= 2;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		var actorList = scene.actorList;

		// check distance
		for ( var i = 0; i < actorList.visible.length; i++ )
		{
			var actor = actorList.visible[i];
			// check event
			if ( actor.rd.dist < this.EVENT_DIST )
			{
				var dist = this.prevDist[ actor.id ];
				if ( dist != undefined && dist > this.EVENT_DIST )
				{
					if ( actor.type == 0 )
					{
						// add task
						var task = new TextBoxTask( scene );
						task.link( this );
						task.eventActor	= actor;
						// setFocus
						scene.setFocus( task );
					}
					else if ( actor.type == 1 )
					{
						var message = new Message( MessageType.START_BATTLE );
						message.data.enemy = actor;
						scene.msgManager.postMessage( message );
					}
					upd = true;
					break;
				}
			}
			// check fire
			if ( this.command.fire )
			{
				if ( actor.rd.dist < this.FIRE_DIST && actor.status == ActorStatus.ACTIVE )
				{
					actor.setStatus( ActorStatus.MORTAL );
					upd = true;
					break;
				}
			}
		}
		// copy dist
		this.prevDist.length = 0;
		for ( var i = 0; i < actorList.visible.length; i++ )
		{
			var actor = actorList.visible[i];
			this.prevDist[ actor.id ] = actor.rd.dist;
		}
		return upd;
	};
})( EventTask.prototype );

/**
 * TextBoxTask
 */
var TextBoxTask = function( scene )
{
	this.command	= scene.input;
	this.eventActor	= null;

	this.lag		= this.LAG_VAL;
};
TextBoxTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 10;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;

		if ( scene.getFocus() == this )
		{
			if ( this.lag > 0 )
			{
				this.lag--;
			}
			else if ( this.command.go || this.command.back || this.command.left || this.command.right )
			{
				scene.setFocus( null );
				this.unlink();
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;
		var resource	= scene.resource;

		resource.noticeSound.play();

		var tx = 10;
		var ty = 10;
		var th = 50;

		context.save();

		context.fillStyle	= "rgba(192, 80, 77, 0.7)";
		context.beginPath();
		context.fillRect( tx, ty, canvas.width - tx * 2, th );

		context.strokeStyle	= "rgb(192, 192, 192)";
		context.beginPath();
		context.strokeRect( tx, ty, canvas.width - tx * 2, th );

		context.font		= "bold 18px 'ＭＳ Ｐゴシック'";
		context.fillStyle	= "rgb(255, 255, 255)";
		context.fillText( this.eventActor.event.text, tx + 10, ty + 25 );

		context.restore();
	};
})( TextBoxTask.prototype );

/**
 * FrontViewTask
 */
var FrontViewTask = function( scene )
{
	this.command	= scene.input;
	this.lagF		= 0;
};
FrontViewTask.prototype = new Task();

(function( pt )
{
	pt.LAGF_VAL	= 4;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		// lagF
		if ( this.lagF )
		{
			this.lagF--;
			upd = true;
		}
		else
		{
			if ( this.command.fire )
			{
				this.lagF	= this.LAGF_VAL;
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var canvas		= scene.canvas;
		var context		= scene.context;
		var resource	= scene.resource;
		var stage		= scene.stage;
		var viewPoint	= scene.viewPoint;
		var image;

		// gun
		var gx = canvas.width  - resource.gunImage.width;
		var gy = canvas.height - resource.gunImage.height;
		gx += 20 + Math.cos( scene.ticks / 6 ) * viewPoint.velocity * 70;
		gy += 20 + Math.cos( scene.ticks / 5 ) * viewPoint.velocity * 70;

		// fire
		if ( this.lagF )
		{
			if ( this.lagF == this.LAGF_VAL )
			{
				resource.shootSound.play();
			}
			image		= resource.fireImage;
			var bsize	= image.width;
			var sy		= bsize * ( this.LAGF_VAL - this.lagF );
			context.drawImage(
				image,
				0,				// sx
				sy,				// sy
				bsize,			// sw
				bsize,			// sh
				gx - bsize / 2,	// dx
				gy - bsize / 2,	// dy
				bsize,			// dw
				bsize			// dh
			);
		}

		image = resource.gunImage;
		context.drawImage(
			image,
			0,				// sx
			0,				// sy
			image.width,	// sw
			image.height,	// sh
			gx,				// dx
			gy,				// dy
			image.width,	// dw
			image.height	// dh
		);
	};
})( FrontViewTask.prototype );

/**
 * DebugTextTask
 */
var DebugTextTask = function( scene )
{
	this.command	= scene.input;
};
DebugTextTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		if ( !Debug.text ) { return; }

		var canvas		= scene.canvas;
		var context		= scene.context;

		context.save();
		context.font		= "18px 'ＭＳ Ｐゴシック'";
		context.fillStyle	= "red";
		context.fillText( Debug.text, 4, 22 );
		context.restore();
	};
})( DebugTextTask.prototype );


/**
 * Command
 */
var Command = function()
{
	this.left	= 0;
	this.right	= 0;
	this.go		= 0;
	this.back	= 0;
	this.fire	= 0;
	this.seek	= 0;
	this.escape	= 0;
};

(function( pt )
{
	/**
	 * 
	 */
	pt.clear = function()
	{
		this.left	= 0;
		this.right	= 0;
		this.go		= 0;
		this.back	= 0;
		this.fire	= 0;
		this.seek	= 0;
		this.escape	= 0;
	};
})( Command.prototype );

/**
 * AICommand
 */
var AICommand = function() {};
AICommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function()
	{
		this.clear();

		var r = Math.floor( Math.random() * 18 );
		if      ( 0 <= r && r <= 3 )	{ this.left		= 1; }
		else if ( 4 <= r && r <= 8 )	{ this.right	= 1; }
		else if ( r == 9 )				{ this.go		= 1; }
		else if ( r == 10 )				{ this.back		= 1; }
	};
})( AICommand.prototype );

/**
 * InputCommand
 */
var InputCommand = function() {};
InputCommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function() {};
	/**
	 * 
	 */
	pt.handleSysEvent = function( event )
	{
		var type = event.type.toLowerCase();
		if ( type.substring( 0, 3 ) == "key" )
		{
			var value = 0;
			if      ( type == "keydown" ) { value = 1; }
			else if ( type == "keyup"   ) { value = 0; }

			switch ( event.keyCode )
			{
				case 37: case 65:	this.left	= value; break;	// ← | A
				case 39: case 68:	this.right	= value; break;	// → | D
				case 38: case 87:	this.go		= value; break;	// ↑ | W
				case 40: case 83:	this.back	= value; break;	// ↓ | S
				case 90:			this.fire	= value; break;	// Z
				case 32:			this.seek	= value; break;	// space;
				case 27:			this.escape	= value; break;	// ESC
			}
		}
	};
})( InputCommand.prototype );


/**
 * StageScene
 */
var StageScene = function( app )
{
	this.app		= app;
	this.name		= SCENE_NAME;
	this.loaded		= false;

	this.resource	= null;
	this.input		= new InputCommand();
	this.viewPoint	= null;

	this.stage		= new StageTask( this );
	this.actorList	= new ActorListTask( this );
	this.event		= new EventTask( this );
	this.frontView	= new FrontViewTask( this );
	this.debugText	= new DebugTextTask( this );

	this.stage		.link( this.head );	// after dummy task.
	this.actorList	.link( this.stage );
	this.event		.link( this.actorList );
	this.frontView	.link( this.event );
	this.debugText	.link( this.frontView );

	// add message handler
	var handler = new MessageHandler( MessageType.START_BATTLE, this, this.handleStartBattle );
	handler.skipAfter = true;
	this.msgManager.addHandler( handler );
};
StageScene.prototype = new Scene();

(function( pt )
{
	/**
	 * 
	 */
	pt.handleStartBattle = function( scene, message )
	{
		this.app.sceneManager.push( BATTLE_SCENE_NAME );
		var scene = this.app.sceneManager.current;
		scene.reset( this.resource, this.viewPoint, message.data.enemy );
		scene.show();
	};

	/**
	 * 
	 */
	pt.show = function()
	{
		if ( !this.loaded ) { Debug.alert( "Scene#loaded=" + this.loaded ); return; }

		this.setUsage( USAGE_HTML );

		this.input.clear();

		this.holdCanvas();
		this.drawTasks();
	};
	/**
	 * 
	 */
	pt.update = function()
	{
		this.ticks++;

		// switch to previous scene
		if ( this.input.escape )
		{
			this.app.sceneManager.pop();
			this.app.sceneManager.current.show();
			return;
		}

		//if ( !this.loaded ) { Debug.alert( "Scene#loaded=" + this.loaded ); return; }

		// update & draw
		var upd = this.updateTasks();
		var hdl = this.msgManager.handleMessages();
		if ( upd && hdl )
		{
			this.drawTasks();
		}
	};
	/**
	 * 
	 */
	pt.loadData = function( json )
	{
		try
		{
			// load stage
			this.stage.map	= json.map;

			// load actorList
			var prev = this.actorList.child;
			for ( var i = 0; i < json.actors.length; i++ )
			{
				var actor	= new ActorTask();
				actor.init( this, i, json.actors[i] );
				// add
				actor.link( prev );
				// for next step
				prev = actor;
			}
			// init viewPoint
			this.viewPoint = this.actorList.child.next;

			// load resource
			this.resource	= json.resource;
			this.resource.load();

			this.loaded = true;
		}
		catch ( e )
		{
			this.app.kill();
			Debug.alert( e );
		}
	};
})( StageScene.prototype );


// Expose
if ( !window.h5glib ) { window.h5glib = {}; }
window.h5glib.StageScene		= StageScene;
window.h5glib.STAGE_SCENE_NAME	= SCENE_NAME;

// Anonymous function end
//
})( window );
