//  vorbis_file.hpp: vorbis_file device adapter

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef HAMIGAKI_AUDIO_VORBIS_FILE_HPP
#define HAMIGAKI_AUDIO_VORBIS_FILE_HPP

#include <hamigaki/iostreams/traits.hpp>
#include <boost/config.hpp>
#include <boost/iostreams/detail/adapter/direct_adapter.hpp>
#include <boost/iostreams/detail/ios.hpp>
#include <boost/iostreams/detail/select.hpp>
#include <boost/iostreams/traits.hpp> 
#include <boost/iostreams/operations.hpp>
#include <boost/type_traits/is_same.hpp>
#include <boost/assert.hpp>
#include <boost/cstdint.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/static_assert.hpp>
#include <cstddef>
#include <cstring>
#include <limits>
#include <utility>

namespace hamigaki { namespace audio {

namespace vorbis
{

// Typedefs
typedef std::size_t (*read_func)(void*, std::size_t, std::size_t, void*);
typedef int (*seek_func)(void*, boost::int64_t, int);
typedef int (*close_func)(void*);
typedef long (*tell_func)(void*);

// byte order
extern const int native_endian;
extern const int little_endian;
extern const int big_endian;

} // namespace vorbis

struct vorbis_sample_params
{
    vorbis_sample_params() :
        endian(vorbis::little_endian), size(2), is_signed(true)
    {
    }

    int endian;
    std::size_t size;
    bool is_signed;
};

struct vorbis_info
{
    int version;
    int channels;
    long rate;

    long bitrate_upper;
    long bitrate_nominal;
    long bitrate_lower;
};

class vorbis_error : public BOOST_IOSTREAMS_FAILURE
{
public:
    explicit vorbis_error(int error);
    int error() const { return error_; }
    static void check(int error);

private:
    int error_;
};

namespace detail
{

void clear_errno();
int get_errno();
void set_errno_eio();

class vorbis_file_base : boost::noncopyable
{
public:
    vorbis_file_base();
    ~vorbis_file_base();

    void open(void* self,
        vorbis::read_func read, vorbis::seek_func seek,
        vorbis::close_func close, vorbis::tell_func tell);

    void close();
    long read(char* buffer, int length, int bigendianp, int word, int sgned);
    void seek(boost::int64_t pos);
    boost::int64_t tell();
    boost::int64_t total();

    std::pair<const char**,const char**> comments() const;
    const char* vendor() const;
    vorbis_info info() const;

private:
    void* file_ptr_;
    bool is_open_;
};

template<typename Source>
struct vorbis_nonseekable_source_traits
{
    static ::size_t read_func(void* ptr, ::size_t size, ::size_t nmemb, void* datasource)
    {
        BOOST_ASSERT(size == 1);

        try
        {
            Source& src = *static_cast<Source*>(datasource);
            clear_errno();
            std::streamsize n =
                boost::iostreams::read(src, static_cast<char*>(ptr), nmemb);
            return (n == -1) ? 0 : n;
        }
        catch (...)
        {
            if (get_errno() == 0)
                set_errno_eio();
            return 0;
        }
    }

    static int seek_func(void* datasource, boost::int64_t offset, int whence)
    {
        return -1;
    }

    static int close_func(void* datasource)
    {
        try
        {
            Source& src = *static_cast<Source*>(datasource);
            boost::iostreams::close(src, BOOST_IOS::in);
            return 0;
        }
        catch (...)
        {
            return -1;
        }
    }

    static long tell_func(void* datasource)
    {
        return -1;
    }
};

template<typename Source>
struct vorbis_seekable_source_traits
    : vorbis_nonseekable_source_traits<Source>
{
    static int seek_func(void* datasource, boost::int64_t offset, int whence)
    {
        try
        {
            Source& src = *static_cast<Source*>(datasource);
            if (whence == SEEK_SET)
            {
                boost::iostreams::seek(
                    src, offset, BOOST_IOS::beg, BOOST_IOS::in);
            }
            else if (whence == SEEK_CUR)
            {
                boost::iostreams::seek(
                    src, offset, BOOST_IOS::cur, BOOST_IOS::in);
            }
            else
            {
                boost::iostreams::seek(
                    src, offset, BOOST_IOS::end, BOOST_IOS::in);
            }
            return 0;
        }
        catch (...)
        {
            return -1;
        }
    }

    static long tell_func(void* datasource)
    {
        try
        {
            Source& src = *static_cast<Source*>(datasource);
            return boost::iostreams::position_to_offset(
                boost::iostreams::seek(src, 0, BOOST_IOS::cur));
        }
        catch (...)
        {
            return -1;
        }
    }
};

template<typename Source>
class vorbis_file_impl : vorbis_file_base
{
private:
    typedef vorbis_file_base base_type;

    typedef typename
        boost::iostreams::select<
            boost::iostreams::is_direct<Source>,
                boost::iostreams::detail::direct_adapter<Source>,
            boost::iostreams::else_,
                Source
        >::type value_type;

    typedef typename
        boost::iostreams::select<
            hamigaki::iostreams::is_input_seekable<value_type>,
                vorbis_seekable_source_traits<value_type>,
            boost::iostreams::else_,
                vorbis_nonseekable_source_traits<value_type>
        >::type source_traits;

public:
    BOOST_STATIC_ASSERT((
        boost::is_same<
            char,
            BOOST_DEDUCED_TYPENAME
                boost::iostreams::char_type_of<Source>::type
        >::value
    ));

    explicit vorbis_file_impl(
            const Source& src, const vorbis_sample_params& params)
        : src_(src), params_(params)
    {
        base_type::open(
            &src_,
            source_traits::read_func,
            source_traits::seek_func,
            source_traits::close_func,
            source_traits::tell_func);
    }

    ~vorbis_file_impl()
    {
        close();
    }

    std::size_t block_size() const
    {
        return params_.size * static_cast<std::size_t>(info().channels);
    }

    std::streamsize read(char* s, std::streamsize n)
    {
        if (n % block_size() != 0)
            throw BOOST_IOSTREAMS_FAILURE("invalid read size");

        std::streamsize total = 0;
        while (n != 0)
        {
            long res = base_type::read(
                s, n,
                params_.endian,
                params_.size,
                params_.is_signed);

            if (res == 0)
                return total != 0 ? total : -1;

            total += res;
            s += res;
            n -= res;
        }

        return total;
    }

    void close()
    {
        base_type::close();
    }

    std::streampos seek(
        boost::iostreams::stream_offset off, std::ios_base::seekdir way)
    {
        std::size_t block = block_size();
        if (off % block != 0)
            throw BOOST_IOSTREAMS_FAILURE("invalid offset");

        if (way == BOOST_IOS::beg)
        {
            base_type::seek(off/block);
            return off;
        }
        else if (way == BOOST_IOS::cur)
        {
            boost::int64_t cur = base_type::tell();
            base_type::seek(cur + off/block);
            return cur * block + off;
        }
        else
        {
            boost::int64_t end = base_type::total();
            base_type::seek(end + off/block);
            return end * block + off;
        }
    }

    boost::int64_t total()
    {
        return base_type::total() * block_size();
    }

    using base_type::comments;
    using base_type::vendor;
    using base_type::info;

private:
    value_type src_;
    vorbis_sample_params params_;
};

} // namespace detail

template<typename Source>
class vorbis_file
{
    typedef detail::vorbis_file_impl<Source> impl_type;

public:
    typedef char char_type;

    struct category :
        boost::iostreams::optimally_buffered_tag,
        boost::iostreams::mode_of<Source>::type,
        boost::iostreams::device_tag,
        boost::iostreams::closable_tag {};

    explicit vorbis_file(const Source& src,
            const vorbis_sample_params& params=vorbis_sample_params())
        : pimpl_(new impl_type(src, params))
    {
    }

    std::size_t block_size() const
    {
        return pimpl_->block_size();
    }

    std::streamsize optimal_buffer_size() const
    {
        return pimpl_->block_size() * (pimpl_->info().rate / 5);
    }

    std::streamsize read(char_type* s, std::streamsize n)
    {
        return pimpl_->read(s, n);
    }

    void close()
    {
        pimpl_->close();
    }

    std::streampos seek(
        boost::iostreams::stream_offset off, BOOST_IOS::seekdir way)
    {
        return pimpl_->seek(off, way);
    }

    std::pair<const char**,const char**> comments() const
    {
        return pimpl_->comments();
    }

    const char* vendor() const
    {
        return pimpl_->vendor();
    }

    vorbis_info info() const
    {
        return pimpl_->info();
    }

    boost::int64_t total()
    {
        return pimpl_->total();
    }

private:
    boost::shared_ptr<impl_type> pimpl_;
};

template<typename Source>
inline vorbis_file<Source> make_vorbis_file(const Source& src)
{
    return vorbis_file<Source>(src);
}

} } // End namespaces audio, hamigaki.

#endif // HAMIGAKI_AUDIO_VORBIS_FILE_HPP
