//  copy_manager.hpp: copy operations by another thread

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef HAMIGAKI_IOSTREAMS_COPY_MANAGER_HPP
#define HAMIGAKI_IOSTREAMS_COPY_MANAGER_HPP

#include <boost/iostreams/detail/buffer.hpp>
#include <boost/iostreams/constants.hpp>
#include <boost/iostreams/positioning.hpp>
#include <boost/iostreams/read.hpp>
#include <boost/iostreams/traits.hpp> 
#include <boost/iostreams/write.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/thread/thread.hpp>
#include <boost/bind.hpp>
#include <boost/noncopyable.hpp>

namespace hamigaki { namespace iostreams {

namespace detail
{

class copy_manager_base
{
public:
    virtual ~copy_manager_base(){}

    void run()
    {
        do_run();
    }

    bool done()
    {
        return do_done();
    }

    std::streampos tell()
    {
        return do_tell();
    }

    void stop()
    {
        return do_stop();
    }

    void reset()
    {
        return do_reset();
    }

private:
    virtual void do_run() = 0;
    virtual bool do_done() = 0;
    virtual std::streampos do_tell() = 0;
    virtual void do_stop() = 0;
    virtual void do_reset() = 0;
};

template<typename Source, typename Sink>
class copy_manager_impl
    : public copy_manager_base
{
    typedef typename boost::iostreams::char_type_of<Source>::type char_type;
    typedef boost::iostreams::detail::buffer<char_type> buffer_type;

public:
    copy_manager_impl(
            const Source& src, const Sink& sink, std::streamsize buffer_size)
        : src_(src), sink_(sink), buffer_(buffer_size), total_(0)
        , done_(false), interrupted_(false)
    {
    }

private:
    boost::mutex mutex_;
    Source src_;
    Sink sink_;
    buffer_type buffer_;
    volatile std::streamsize total_;
    volatile bool done_;
    volatile bool interrupted_;

    void do_run() // virtual
    {
        try
        {
            buffer_type& buf = buffer_;

            while (!interrupted())
            {
                if (buf.ptr() == buf.eptr())
                {
                    std::streamsize amt =
                        boost::iostreams::read(src_, buf.data(), buf.size());

                    if (amt == -1)
                        break;

                    buf.set(0, amt);
                }

                while ((buf.ptr() != buf.eptr()) && !interrupted())
                {
                    std::streamsize amt =
                        boost::iostreams::write(
                            sink_, buf.ptr(), buf.eptr() - buf.ptr());

                    total_ += amt;
                    buf.ptr() += amt;
                }
            }
        }
        catch (...)
        {
        }

        boost::mutex::scoped_lock locking(mutex_);
        done_ = true;
    }

    bool do_done() // virtual
    {
        boost::mutex::scoped_lock locking(mutex_);
        return done_;
    }

    std::streampos do_tell() // virtual
    {
        boost::mutex::scoped_lock locking(mutex_);
        return boost::iostreams::offset_to_position(total_);
    }

    void do_stop() // virtual
    {
        boost::mutex::scoped_lock locking(mutex_);
        interrupted_ = true;
    }

    bool interrupted()
    {
        boost::mutex::scoped_lock locking(mutex_);
        return interrupted_;
    }

    void do_reset() // virtual
    {
        done_ = false;
        interrupted_ = false;
    }
};

} // namespace detail

class copy_manager : boost::noncopyable
{
public:
    template<typename Source, typename Sink>
    void copy(Source src, Sink sink,
        std::streamsize buffer_size = 
            boost::iostreams::default_device_buffer_size)
    {
        stop();

        typedef detail::copy_manager_impl<Source,Sink> impl_type;
        pimpl_.reset(new impl_type(src, sink, buffer_size));
        thread_ptr_.reset(
            new boost::thread(
                boost::bind(&detail::copy_manager_base::run, pimpl_.get())));
    }

    ~copy_manager()
    {
        stop();
    }

    bool operator!() const
    {
        return pimpl_.get() == 0;
    }

    void reset()
    {
        stop();
        pimpl_.reset();
    }

    bool done()
    {
        bool result = pimpl_->done();
        if (result)
        {
            thread_ptr_->join();
            thread_ptr_.reset();
        }
        return result;
    }

    void stop()
    {
        if (thread_ptr_.get())
        {
            pimpl_->stop();
            thread_ptr_->join();

            thread_ptr_.reset();
        }
    }

    void resume()
    {
        pimpl_->reset();
        thread_ptr_.reset(
            new boost::thread(
                boost::bind(&detail::copy_manager_base::run, pimpl_.get())));
    }

    std::streampos tell()
    {
        return pimpl_->tell();
    }

private:
    std::auto_ptr<detail::copy_manager_base> pimpl_;
    std::auto_ptr<boost::thread> thread_ptr_;
};

} } // End namespaces iostreams, hamigaki.

#endif // HAMIGAKI_IOSTREAMS_COPY_MANAGER_HPP
