//  wave_file_test.cpp: test case for wave_source/wave_sink

//  Copyright Takeshi Mouri 2006.
//  Use, modification, and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#include <hamigaki/audio/pcm_device.hpp>
#include <hamigaki/audio/wave_file.hpp>
#include <hamigaki/iostreams/device/tmp_file.hpp>
#include <hamigaki/iostreams/dont_close.hpp>
#include <boost/iostreams/device/array.hpp>
#include <boost/iostreams/copy.hpp>
#include <boost/test/unit_test.hpp>
#include <algorithm>
#include <cmath>
#include <iterator>
#include <vector>

namespace audio = hamigaki::audio;
namespace io_ex = hamigaki::iostreams;
namespace io = boost::iostreams;
namespace ut = boost::unit_test;

double power(double x, double y)
{
    if (y >= 0.0)
        return std::pow(x, y);
    else
        return std::pow(1.0/x, -y);
}

// calculate the frequency from specified MIDI note number
double calc_frequency(unsigned short note)
{
    return 440.0 * power(2.0, (note-69) / 12.0);
}

template<int Bits>
struct sample_traits;

template<>
struct sample_traits<8>
{
    static const boost::int32_t amplitude = 64;
    static const boost::int32_t median = 128;

    template<typename OutputIterator>
    static OutputIterator serialize(OutputIterator out, boost::int32_t n)
    {
        *out = static_cast<char>(static_cast<boost::uint8_t>(n));
        return ++out;
    }
};

template<>
struct sample_traits<16>
{
    static const boost::int32_t amplitude = 16384;
    static const boost::int32_t median = 0;

    template<typename OutputIterator>
    static OutputIterator serialize(OutputIterator out, boost::int32_t n)
    {
        // int32_t may not be two's complement representation
        boost::uint32_t tmp =
            n >= 0
            ? static_cast<boost::uint32_t>(n)
            : 0x10000 - static_cast<boost::uint32_t>(-n);

        // little endian output
        *out = static_cast<char>(static_cast<boost::uint8_t>(tmp & 0xFF));
        *(++out) = static_cast<char>(
            static_cast<boost::uint8_t>((tmp >> 8) & 0xFF));
        return ++out;
    }
};

template<int Bits>
void wave_file_test_aux(unsigned rate, unsigned channels)
{
    typedef sample_traits<Bits> traits;
    const double double_rate = static_cast<double>(rate);

    const double pi = 3.14159265358979323846;
    const double twice_pi = 2.0*pi;

    const double freq = calc_frequency(60);
    const double v = twice_pi * freq / double_rate;

    std::vector<char> data;
    data.reserve(rate*channels);

    double sum = 0.0;
    for (unsigned i = 0; i < rate; ++i)
    {
        double size = std::sin(pi * i / double_rate) * traits::amplitude;
        boost::int32_t val =
            traits::median + static_cast<boost::int32_t>(size * std::sin(sum));

        for (unsigned j = 0; j < channels; ++j)
            traits::serialize(std::back_inserter(data), val);

        sum += v;
        if (sum >= twice_pi)
            sum -= twice_pi;
    }

    audio::pcm_format fmt;
    fmt.rate = rate;
    fmt.bits = Bits;
    fmt.channels = channels;

    io_ex::tmp_file tmp;
    io::copy(
        io::array_source(&data[0], data.size()),
        audio::make_wave_sink(io_ex::dont_close(tmp), fmt)
    );
    io::seek(tmp, 0, BOOST_IOS::beg);
    io::copy(
        audio::make_wave_source(tmp),
        audio::pcm_sink(fmt, rate/5*((Bits+7)/8)*channels)
    );
}

void wave_file_test()
{
    const unsigned rates[] = { 11025, 22050, 44100 };

    for (std::size_t i = 0; i < sizeof(rates)/sizeof(rates[0]); ++i)
    {
        for (unsigned channels = 1; channels <= 2; ++channels)
        {
            wave_file_test_aux<8>(rates[i], channels);
            wave_file_test_aux<16>(rates[i], channels);
        }
    }
}

ut::test_suite* init_unit_test_suite(int, char* [])
{
    ut::test_suite* test = BOOST_TEST_SUITE("wave file test");
    test->add(BOOST_TEST_CASE(&wave_file_test));
    return test;
}
