#include "main.h"
#include <stdio.h>
#include <string.h>
struct tasks *tctl;
struct timer *tts;
unsigned int ptime;

struct task *taskNow(void)
{
	struct tlevel *tl = &tctl->level[tctl->now];
	return tl->task[tl->now];
}

void taskAdd(struct task *task)
{
	struct tlevel *tl = &tctl->level[task->lv];
	tl->task[tl->running] = task;
	tl->running++;
	task->flag = 2;
	return;
}

void taskRemove(struct task *task)
{
	struct tlevel *tl = &tctl->level[task->lv];
	int i;
	for (i = 0; i < tl->running; i++) {
		if (tl->task[i] == task) {
			break;
		}
	}
	tl->running--;
	if (i < tl->now) {
		tl->now--;
	}
	if (tl->now >= tl->running) {
		tl->now = 0;
	}
	task->flag = 1;
	for (; i < tl->running; i++) {
		tl->task[i] = tl->task[i + 1];
	}
	return;
}

void taskSwitch0(void)
{
	int i;
	for (i = 0; i < TLEVELS; i++) {
		if (tctl->level[i].running > 0) {
			break;
		}
	}
	tctl->now = i;
	tctl->chlv = 0;
	return;
}

void taskSettime(struct task *task)
{
	extern struct timers *tms;
	task->time += tms->count - ptime;
	ptime = tms->count;
	return;
}

void taskIdle(void)
{
	for (;;) {
		hlt();
	}
}

struct task *taskInit(void)
{
	extern struct timers *tms;
	struct segdesc *sd = (struct segdesc *) GDT_ADDR;
	struct memory *mem = (struct memory *) MEMORY_ADDR;
	struct task *tsys, *tidle;
	int i;
	tctl = (struct tasks *) memAlloc(mem, sizeof (struct tasks));
	for (i = 0; i < TASKS; i++) {
		tctl->task[i].flag = 0;
		tctl->task[i].sel = (3 + i) * 8;
		tctl->task[i].tss.ldtr = (3 + TASKS + i) * 8;
		segdescSet(sd + 3 + i, (int) &tctl->task[i].tss, 103, TSS);
		segdescSet(sd + 3 + TASKS + i, (int) &tctl->task[i].ldt, 15, LDT);
	}
	for (i = 0; i < TLEVELS; i++) {
		tctl->level[i].running = 0;
		tctl->level[i].now = 0;
	}

	tsys = taskAlloc("System");
	tsys->flag = 2;
	tsys->pri = 2;
	tsys->lv = 1;
	taskAdd(tsys);
	taskSwitch0();
	trLoad(tsys->sel);

	tidle = taskAlloc("Idle");
	tidle->tss.esp = memAlloc(mem, 65536) + 65536;
	tidle->tss.eip = (int) &taskIdle;
	tidle->tss.es = 1 * 8;
	tidle->tss.cs = 2 * 8;
	tidle->tss.ss = 1 * 8;
	tidle->tss.ds = 1 * 8;
	tidle->tss.fs = 1 * 8;
	tidle->tss.gs = 1 * 8;
	taskRun(tidle, TLEVELS - 1, 1);

	ptime = tms->count;
	tts = timerAlloc();
	timerSet(tts, tsys->pri, 0);
	tctl->tfpu = 0;
	return tsys;
}

void taskSwitch(void)
{
	struct tlevel *tl = &tctl->level[tctl->now];
	struct task *nt, *t = tl->task[tl->now];
	taskSettime(tl->task[tl->now]);
	tl->now++;
	if (tl->now == tl->running) {
		tl->now = 0;
	}
	if (tctl->chlv != 0) {
		taskSwitch0();
		tl = &tctl->level[tctl->now];
	}
	nt = tl->task[tl->now];
	timerSet(tts, nt->pri, 0);
	if (nt != t) {
		farjmp(0, nt->sel);
	}
	return;
}

struct task *taskAlloc(char *name)
{
	struct task *t;
	int i;
	for (i = 0; i < TASKS; i++) {
		if (tctl->task[i].flag == 0) {
			t = &tctl->task[i];
			t->flag = 1;
			t->time = 0;
			t->tss.eflags = 0x00000202;	/* IF */
			t->tss.eax = 0;
			t->tss.ecx = 0;
			t->tss.edx = 0;
			t->tss.ebx = 0;
			t->tss.ebp = 0;
			t->tss.esi = 0;
			t->tss.edi = 0;
			t->tss.es = 0;
			t->tss.ds = 0;
			t->tss.fs = 0;
			t->tss.gs = 0;
			t->tss.iomap = 0x40000000;
			t->tss.ss0 = 0;
			t->fpu[0] = 0x037f;
			t->fpu[1] = 0x0000;
			t->fpu[2] = 0xffff;
			for (i = 3; i < 27; i++) {
				t->fpu[i] = 0;
			}
			t->lbyte = 0;
			for (i = 0; i < 24; i++) {
				if (name[i] == 0) {
					break;
				}
				t->name[i] = name[i];
			}
			t->name[i] = 0;
			return t;
		}
	}
	return 0;
}

void taskRun(struct task *task, int lv, int pri)
{
	if (lv < 0) {
		lv = task->lv;
	}
	if (pri > 0) {
		task->pri = pri;
	}
	if (task->flag == 2 && task->lv != lv) {
		taskRemove(task);
	}
	if (task->flag != 2) {
		task->lv = lv;
		taskAdd(task);
	}
	tctl->chlv = 1;
	return;
}

void taskSleep(struct task *task)
{
	struct task *nt;
	if (task->flag == 2) {
		nt = taskNow();
		taskRemove(task);
		if (task == nt) {
			timerCancel(tts);
			taskSwitch0();
			nt = taskNow();
			taskSettime(task);
			timerSet(tts, nt->pri, 0);
			farjmp(0, nt->sel);
		}
	}
	return;
}

void fifoInit(struct fifo *fifo, int *buf, int size, struct task *task)
{
	fifo->size = size;
	fifo->buf = buf;
	fifo->free = size;
	fifo->wp = 0;
	fifo->rp = 0;
	fifo->task = task;
	return;
}

void fifoPut(struct fifo *fifo, int data)
{
	if (fifo->free == 0) {
		return;
	}
	fifo->buf[fifo->wp] = data;
	fifo->wp++;
	if (fifo->wp == fifo->size) {
		fifo->wp = 0;
	}
	fifo->free--;
	if (fifo->task != 0) {
		if (fifo->task->flag != 2) {
			taskRun(fifo->task, -1, 0);
		}
	}
	return;
}

int fifoGet(struct fifo *fifo)
{
	int data;
	if (fifo->free == fifo->size) {
		return -1;
	}
	data = fifo->buf[fifo->rp];
	fifo->rp++;
	if (fifo->rp == fifo->size) {
		fifo->rp = 0;
	}
	fifo->free++;
	return data;
}

int fifoStat(struct fifo *fifo)
{
	return fifo->size - fifo->free;
}

#define PID_POS		7
#define PID_SIZE	4 + 8 * 4
#define NAME_POS	PID_POS + 1 + PID_SIZE
#define NAME_SIZE	4 + 8 * 24
#define LEVEL_POS	NAME_POS + 1 + NAME_SIZE
#define LEVEL_SIZE	4 + 8 * 6
#define PRI_POS		LEVEL_POS + 1 + LEVEL_SIZE
#define PRI_SIZE	4 + 8 * 4
#define TIME_POS	PRI_POS + 1 + PRI_SIZE
#define TIME_SIZE	4 + 8 * 9
#define STAT_POS	TIME_POS + 1 + TIME_SIZE
#define STAT_SIZE	4 + 8 * 9

void tmgrPlist(struct tmgr *tmgr)
{
	unsigned char s[13], bc, c;
	int i, j;
	for (i = 0; i < 8; i++) {
		graphicBox(tmgr->sht, _FFFFFF, 8, 32 + (i + 1) * 16, tmgr->sht->sx - 23, 32 + (i + 1) * 16 + 16);
		if (tmgr->task[i] == 0) {
			break;
		}
		for (j = 0; j < 13; j++) {
			s[j] = tmgr->task[i]->name[j];
		}
		if (tmgr->now - tmgr->start == i && tmgr->act != 0) {
			bc = _000084;
			c = _FFFFFF;
		} else {
			bc = _FFFFFF;
			c = _000000;
		}
		graphicBox(tmgr->sht, bc, 8, 32 + (i + 1) * 16, tmgr->sht->sx - 23, 32 + (i + 1) * 16 + 16);
		graphicPuts(tmgr->sht, c, NAME_POS + 1, 32 + (i + 1) * 16, s);
		sprintf(s, "%4d", tmgr->task[i]->sel / 8 - 3);
		graphicPuts(tmgr->sht, c, PID_POS + 1, 32 + (i + 1) * 16, s);
		sprintf(s, "    %2d", tmgr->task[i]->lv);
		graphicPuts(tmgr->sht, c, LEVEL_POS + 1, 32 + (i + 1) * 16, s);
		sprintf(s, " %3d", tmgr->task[i]->pri);
		graphicPuts(tmgr->sht, c, PRI_POS + 1, 32 + (i + 1) * 16, s);
		sprintf(s, " %5d.%02d", (int) (tmgr->task[i]->time / 100), tmgr->task[i]->time % 100);
		graphicPuts(tmgr->sht, c, TIME_POS + 1, 32 + (i + 1) * 16, s);
		if (tmgr->task[i]->flag == 1) {
			sprintf(s, "Sleeping");
		} else {
			sprintf(s, "Running");
		}
		graphicPuts(tmgr->sht, c, STAT_POS + 1, 32 + (i + 1) * 16, s);
	}
	return;
}

void tmgrPstat(struct tmgr *tmgr, int time)
{
	struct memory *mem = (struct memory *) MEMORY_ADDR;
	char s[63], t[50];
	unsigned int eax, ebx, ecx, edx, max_cpuid, max_cpuid_ex;
	cpuid(0x80000000, 0x0, &eax, &ebx, &ecx, &edx);
	max_cpuid_ex = eax;
	if(max_cpuid_ex >= 0x80000004){
		cpuid(0x80000002, 0x0, &eax, &ebx, &ecx, &edx);
		memcpy(t      , &eax, 4);
		memcpy(t  +  4, &ebx, 4);
		memcpy(t  +  8, &ecx, 4);
		memcpy(t  + 12, &edx, 4);
		cpuid(0x80000003, 0x0, &eax, &ebx, &ecx, &edx);
		memcpy(t  + 16, &eax, 4);
		memcpy(t  + 20, &ebx, 4);
		memcpy(t  + 24, &ecx, 4);
		memcpy(t  + 28, &edx, 4);
		cpuid(0x80000004, 0x0, &eax, &ebx, &ecx, &edx);
		memcpy(t  + 32, &eax, 4);
		memcpy(t  + 36, &ebx, 4);
		memcpy(t  + 40, &ecx, 4);
		memcpy(t  + 44, &edx, 4);
		t[49] = '\0';
	} else {
		cpuid(0x00000000, 0x0, &eax, &ebx, &ecx, &edx);
		max_cpuid = eax;
		memcpy(t    , &ebx, 4);
		memcpy(t + 4, &edx, 4);
		memcpy(t + 8, &ecx, 4);
		t[12] = '\0';
	}
	sprintf(s, "CPU    %3d %% %s", 100 - time, t);
	graphicBox(tmgr->sht, _FFFFFF, 8, 184, tmgr->sht->sx - 23, 184 + 15);
	graphicPuts(tmgr->sht, _000000, 8, 184, s);
	sprintf(s, " %4d / %4d MB", memStat(mem) / (1024 * 1024), mem->allsize / (1024 * 1024));
	graphicBox(tmgr->sht, _FFFFFF, 8, 184 + 16, tmgr->sht->sx - 23, 184 + 31);
	graphicPuts(tmgr->sht, _000000, 8, 184 + 16, s);
	return;
}

void tmgrPrint(struct tmgr *tmgr, int time)
{
	int i, j = 0, k = 0, l;
restart:
	for (i = 0; i < 8; i++) {
		tmgr->task[i] = 0;
	}
	for (i = 0; i < TASKS; i++) {
		if (tctl->task[i].flag != 0) {
			if (j == tmgr->start) {
				break;
			}
			j++;
		}
	}
	for (j = 0; i < TASKS; i++) {
		if (tctl->task[i].flag != 0) {
			tmgr->task[j] = &tctl->task[i];
			j++;
			if (j >= 8) {
				break;
			}
		}
	}
	for (i = 0; i < TASKS; i++) {
		if (tctl->task[i].flag != 0) {
			k++;
		}
	}
	if (tmgr->task[7] == 0) {
		tmgr->start--;
		if (tmgr->start < 0) {
			tmgr->start = 0;
			goto next;
		}
		tmgr->dc = -1;
		goto restart;
	}
next:
	graphicButton(tmgr->sht, PID_POS, 27, PID_SIZE, 20, "PID", 0);
	graphicButton(tmgr->sht, NAME_POS, 27, NAME_SIZE, 20, "Name", 0);
	graphicButton(tmgr->sht, LEVEL_POS, 27, LEVEL_SIZE, 20, "Level", 0);
	graphicButton(tmgr->sht, PRI_POS, 27, PRI_SIZE, 20, "Pri", 0);
	graphicButton(tmgr->sht, TIME_POS, 27, TIME_SIZE, 20, "Time", 0);
	graphicButton(tmgr->sht, STAT_POS, 27, STAT_SIZE, 20, "Status", 0);
	tmgrPlist(tmgr);
	graphicBox(tmgr->sht, _C6C6C6, tmgr->sht->sx - 21, 27, tmgr->sht->sx - 8, tmgr->sht->sy - 49);
	graphicButton(tmgr->sht, tmgr->sht->sx - 20, 27, 12, 20, "\36", 0);
	graphicButton(tmgr->sht, tmgr->sht->sx - 20, tmgr->sht->sy - 69, 12, 20, "\37", 0);
	l = (tmgr->sht->sy - 48 - 70) * j / k;
	if (k <= 8) {
		i = 48;
	} else {
		i = 48 + (tmgr->sht->sy - 48 - 69 - l) / (k - j) * tmgr->start;
	}
	graphicButton(tmgr->sht, tmgr->sht->sx - 20, i, 12, l - 1, "", 0);
	tmgrPstat(tmgr, time);
	shtRefresh(tmgr->sht, 7, 27, 510, 217);
	return;
}

void tmgrRun(struct tmgr *tmgr)
{
	extern struct sheets *shts;
	struct task *task = tmgr->task[tmgr->now];
	struct sheet *sht = 0;
	int i;
	if (task->sel / 8 <= 2 + 4) {
		return;
	}
	for (i = 0; i < SHEETS; i++) {
		if (shts->sheet[i].task == task) {
			sht = &shts->sheet[i];
			break;
		}
	}
	if (i == SHEETS) {
		goto nonesht;
	}
	taskSleep(task);
	if ((sht->flag & SHEET_APP) != 0) {
		cli();
		task->tss.eax = (int) &(task->tss.esp0);
		task->tss.eip = (int) appEnd;
		taskRun(task, -1, 0);
		sti();
	} else if ((sht->flag & SHEET_DIALOGUE) != 0) {
		sht->flag &= ~SHEET_DIALOGUE;
		taskRun(sht->task, -1, 0);
	} else {
		shtUpdown(sht, -1);
nonesht:
		if (task->tss.ss0 != 0) {
			cli();
			task->tss.eax = (int) &(task->tss.esp0);
			task->tss.eip = (int) appEnd;
			sti();
			taskRun(task, -1, 0);
		} else {
			cli();
			fifoPut(&task->fifo, 1);
			sti();
		}
	}
	return;
}

void tmgrTask(struct sheet *sht)
{
	struct task *task = taskNow();
	struct timer *timer, *timer2;
	struct tmgr tmgr;
	int ctime, ptime, i, j;
	tmgr.sht = sht;
	tmgr.start = 0;
	tmgr.now = -1;
	tmgr.dc = -1;
	tmgr.act = 0;
	ptime = tctl->task[1].time - 98;
	ctime = tctl->task[1].time;
	tmgrPrint(&tmgr, ctime - ptime);
	timer = timerAlloc();
	timerInit(timer, &task->fifo);
	timerSet(timer, 100, 4);
	timer2 = timerAlloc();
	timerInit(timer2, &task->fifo);

	for (;;) {
		cli();
		if (fifoStat(&task->fifo) == 0) {
			taskSleep(task);
			sti();
		} else {
			i = fifoGet(&task->fifo);
			sti();
			if (i == 1) {
				timerCancel(timer);
				timerCancel(timer2);
				cli();
				tmgrExitreq(sht);
				sti();
				for (;;) {
					taskSleep(task);
				}
			} else if (i == 2) {
				tmgr.act = 1;
			} else if (i == 3) {
				tmgr.act = 0;
			} else if (i == 4) {
				ptime = ctime;
				ctime = tctl->task[1].time;
				timerSet(timer, 100, 4);
			} else if (i == 5) {
				tmgr.start--;
				if (tmgr.start < 0) {
					tmgr.start = 0;
				}
				tmgr.dc = -1;
				timerCancel(timer);
				timerSet(timer, 100, 4);
			} else if (i == 6) {
				tmgr.start++;
				j = 0;
				for (i = 0; i < TASKS; i++) {
					if (tctl->task[i].flag != 0) {
						j++;
					}
				}
				if (tmgr.start > j - 8) {
					tmgr.start = j - 8;
					if (tmgr.start < 0) {
						tmgr.start = 0;
					}
				}
				tmgr.dc = -1;
				timerCancel(timer);
				timerSet(timer, 100, 4);
			} else if (i == 7) {
				tmgr.dc = -1;
			} else if (i >= 512) {
				tmgr.now = i - 512 + tmgr.start;
				if (tmgr.now == tmgr.dc) {
					tmgrRun(&tmgr);
					tmgr.now = -1;
					tmgr.dc = -1;
					timerCancel(timer2);
				} else {
					tmgr.dc = tmgr.now;
					timerCancel(timer2);
					timerSet(timer2, 50, 7);
				}
				timerCancel(timer);
				timerSet(timer, 100, 4);
			}
			tmgrPrint(&tmgr, ctime - ptime);
		}
	}
}

struct sheet *tmgrOpen(int *fat)
{
	struct memory *mem = (struct memory *) MEMORY_ADDR;
	struct sheet *sht = shtAlloc();
	unsigned char *buf = (unsigned char *) memAlloc(mem, 506 * 225);
	struct task *task = taskAlloc("Task maneger");
	int *fifo = (int *) memAlloc(mem, 128 * 4);
	shtSet(sht, buf, 506, 225, -1);
	winAlloc(sht, "^XN}l[W", 0);
	textbox(sht, _FFFFFF, 8, 28, 490, 148);
	textbox(sht, _FFFFFF, 8, 184, 490, 32);
	task->stack = memAlloc(mem, 64 * 1024);
	task->tss.esp = task->stack + 64 * 1024 - 8;
	task->tss.eip = (int) &tmgrTask;
	task->tss.es = 1 * 8;
	task->tss.cs = 2 * 8;
	task->tss.ss = 1 * 8;
	task->tss.ds = 1 * 8;
	task->tss.fs = 1 * 8;
	task->tss.gs = 1 * 8;
	*((int *) (task->tss.esp + 4)) = (int) sht;
	fifoInit(&task->fifo, fifo, 128, task);
	task->fat = fat;
	taskRun(task, 2, 2);
	sht->task = task;
	sht->flag |= SHEET_TMGR | SHEET_CUR;
	return sht;
}

void tmgrClose(struct sheet *sht)
{
	extern struct tasks *tctl;
	struct memory *mem = (struct memory *) MEMORY_ADDR;
	struct task *task = sht->task;
	memFree(mem, (int) sht->buf, 506 * 225);
	shtFree(sht);
	taskSleep(task);
	memFree(mem, task->stack, 64 * 1024);
	memFree(mem, (int) task->fifo.buf, 128 * 4);
	cli();
	task->flag = 0;
	if (tctl->tfpu == task) {
		tctl->tfpu = 0;
	}
	sti();
	return;
}

void tmgrExitreq(struct sheet *sht)
{
	extern struct sheets *shts;
	struct fifo *fifo = (struct fifo *) *((int *) FIFO_ADDR);
	fifoPut(fifo, 2536 + (sht - shts->sheet));
	return;
}
