/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.calendar.composite;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.calendar.action.GetCalendarListTableDefine;
import com.clustercontrol.calendar.composite.action.CalendarDoubleClickListener;
import com.clustercontrol.calendar.composite.action.CalendarSelectionChangedListener;
import com.clustercontrol.calendar.composite.action.VerticalBarSelectionListener;
import com.clustercontrol.calendar.util.CalendarEndpointWrapper;
import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.calendar.CalendarInfo;
import com.clustercontrol.ws.calendar.InvalidRole_Exception;

/**
 * カレンダ一覧コンポジットクラス<BR>
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class CalendarListComposite extends Composite {

	// ログ
	private static Log m_log = LogFactory.getLog( CalendarListComposite.class );

	protected CommonTableViewer m_viewer = null;

	protected Table m_table = null;
	protected Label m_labelCount = null;

	protected String m_calendarId = null;
	protected Date m_base = null;

	/**
	 * コンストラクタ
	 * 
	 * @param parent
	 * @param style
	 * @since 2.0.0
	 */
	public CalendarListComposite(Composite parent, int style) {
		super(parent, style);
		initialize();
	}

	/**
	 * 初期化処理<BR>
	 * 
	 * @since 2.0.0
	 */
	private void initialize() {
		GridLayout layout = new GridLayout(1, true);
		this.setLayout(layout);
		layout.marginHeight = 0;
		layout.marginWidth = 0;

		//カレンダ一覧テーブル作成
		m_table = new Table(this, SWT.H_SCROLL | SWT.V_SCROLL
				| SWT.FULL_SELECTION | SWT.SINGLE | SWT.BORDER);
		m_table.setHeaderVisible(true);
		m_table.setLinesVisible(true);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 1;
		m_table.setLayoutData(gridData);

		m_labelCount = new Label(this, SWT.RIGHT);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		m_labelCount.setLayoutData(gridData);

		m_viewer = new CommonTableViewer(m_table);
		m_viewer.createTableColumn(GetCalendarListTableDefine.get(),
				GetCalendarListTableDefine.SORT_COLUMN_INDEX,
				GetCalendarListTableDefine.SORT_ORDER);

		m_viewer.addSelectionChangedListener(new CalendarSelectionChangedListener(this));

		// ダブルクリックリスナの追加
		m_viewer.addDoubleClickListener(new CalendarDoubleClickListener(this));

		m_viewer.getTable().getVerticalBar().addSelectionListener(
				new VerticalBarSelectionListener(this));
	}

	/**
	 * 更新処理<BR>
	 * 
	 * @since 2.0.0
	 */
	@Override
	public void update() {
		List<CalendarInfo> list = null;

		//カレンダ一覧情報取得
		try {
			list = CalendarEndpointWrapper.getCalendarList();

		} catch (InvalidRole_Exception e) {
			// 権限なし
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));

		} catch (Exception e) {
			// 上記以外の例外
			m_log.warn("update(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}
		if(list == null){
			list = new ArrayList<CalendarInfo>();
		}

		ArrayList<Object> listInput = new ArrayList<Object>();
		for (CalendarInfo info : list) {
			ArrayList<Object> obj = new ArrayList<Object>();
			obj.add(info.getId());
			obj.add(info.getName());
			obj.add(new Date(info.getValidTimeFrom()));
			obj.add(new Date(info.getValidTimeTo()));
			obj.add(info.getDescription());
			obj.add(info.getRegUser());
			obj.add(new Date(info.getRegDate()));
			obj.add(info.getUpdateUser());
			obj.add(new Date(info.getUpdateDate()));
			listInput.add(obj);
		}
		m_viewer.setInput(listInput);

		Object[] args = { new Integer(list.size()) };
		m_labelCount.setText(Messages.getString("records", args));
	}

	/**
	 * このコンポジットが利用するテーブルビューアを取得します。<BR>
	 * 
	 * @return テーブルビューア
	 */
	public TableViewer getTableViewer() {
		return m_viewer;
	}

	/**
	 * このコンポジットが利用するテーブルを取得します。<BR>
	 * 
	 * @return テーブル
	 */
	public Table getTable() {
		return m_viewer.getTable();
	}

	/**
	 * @return
	 */
	public String getCalendarId() {
		return m_calendarId;
	}

	/**
	 * @param calendarId
	 */
	public void setCalendarId(String calendarId) {
		m_calendarId = calendarId;
	}
}