/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.calendar.dialog;

import java.util.Date;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.calendar.util.CalendarAddPropertyUtil;
import com.clustercontrol.calendar.util.CalendarEndpointWrapper;
import com.clustercontrol.composite.PropertySheet;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.ws.calendar.CalendarAddInfo;
import com.clustercontrol.ws.calendar.CalendarDuplicate_Exception;
import com.clustercontrol.ws.calendar.InvalidRole_Exception;

/**
 * カレンダ追加情報の作成・変更ダイアログクラス<BR>
 * 
 * @version 4.0.0
 * @since 2.0.0
 */
public class AddInfoDialog extends CommonDialog {

	// ログ
	private static Log m_log = LogFactory.getLog( AddInfoDialog.class );

	/** ID */
	private String id = "";
	/** 時間帯(From) */
	private Date from;
	/** 時間帯(To) */
	private Date to;

	/** 変更用ダイアログ判別フラグ */
	private int mode;

	/** カレンダプロパティシート */
	private PropertySheet propertySheet = null;

	/** カレンダテーブル */
	private TableTree table = null;
	/**
	 * 指定した形式のダイアログのインスタンスを取得します。<BR>
	 * 
	 * @param parent
	 *            親のシェルオブジェクト
	 * @param id
	 *            カレンダID
	 * @param mode
	 *            表示モード
	 */
	public AddInfoDialog(Shell parent, String id, Date from, Date to, int mode) {
		super(parent);

		this.id = id;
		this.from = from;
		this.to = to;
		this.mode = mode;
	}

	/**
	 * ダイアログの初期サイズを返します。
	 * 
	 * @return 初期サイズ
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(500, 600);
	}

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent
	 *            親のインスタンス
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();

		// タイトル
		shell.setText(Messages
				.getString("dialog.calendar.add.create.modify"));

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		parent.setLayout(layout);

		/*
		 * 属性プロパティシート
		 */

		// ラベル
		Label calendarId = new Label(parent, SWT.LEFT);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		calendarId.setLayoutData(gridData);
		calendarId.setText(Messages.getString("calendar.id") + " : " + id);

		Label rtn = new Label(parent, SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		rtn.setLayoutData(gridData);
		rtn.setText("");

		// ラベル
		Label attribute = new Label(parent, SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		attribute.setLayoutData(gridData);
		attribute.setText(Messages.getString("attribute") + " : ");

		// プロパティシート
		table = new TableTree(parent, SWT.H_SCROLL | SWT.V_SCROLL
				| SWT.FULL_SELECTION | SWT.MULTI | SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 1;
		table.setLayoutData(gridData);
		table.addSelectionListener(new SelectionAdapter(){
			@Override
			public void widgetSelected(SelectionEvent e) {
				update();
			}
		});

		this.propertySheet = new PropertySheet(table);

		// プロパティ取得及び設定
		Property property = null;
		if (this.mode == PropertyConstant.MODE_MODIFY) {
			CalendarAddInfo info = null;
			try {
				info = CalendarEndpointWrapper.getCalendarAdd(this.id, this.from.getTime(), this.to.getTime());

			} catch (InvalidRole_Exception e) {
				// 権限なし
				MessageDialog.openInformation(null, Messages.getString("message"),
						Messages.getString("message.accesscontrol.16"));

			} catch (Exception e) {
				// 上記以外の例外
				m_log.warn("customizeDialog(), " + e.getMessage(), e);
				MessageDialog.openError(
						null,
						Messages.getString("failed"),
						Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());

			}

			property = CalendarAddPropertyUtil.dto2property(info, this.mode, Locale.getDefault());
		}
		else if (this.mode == PropertyConstant.MODE_ADD
				|| this.mode == PropertyConstant.MODE_SHOW) {
			CalendarAddInfo info = null;
			property = CalendarAddPropertyUtil.dto2property(info, this.mode, Locale.getDefault());

		}
		this.propertySheet.setInput(property);
		this.propertySheet.expandAll();

		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		line.setLayoutData(gridData);

		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);

		// 必須入力項目を可視化
		this.update();
	}

	/**
	 * 更新処理
	 * 
	 */
	public void update(){
		// 必須項目を明示

		// 日時（開始）のインデックス：0
		if("".equals(table.getItem(0).getText(1))){
			table.getItem(0).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			table.getItem(0).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		// 日時（終了）のインデックス：1
		if("".equals(table.getItem(1).getText(1))){
			table.getItem(1).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			table.getItem(1).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	@Override
	protected ValidateResult validate() {
		return null;
	}

	/**
	 * 入力値をマネージャに登録します。
	 * 
	 * @return true：正常、false：異常
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		boolean result = false;

		Property property = this.getInputData();
		if(property != null){
			if (this.mode == PropertyConstant.MODE_MODIFY) {
				// 変更の場合
				Property copy = PropertyUtil.copy(property);
				PropertyUtil.deletePropertyDefine(copy);

				try {
					CalendarAddInfo info = CalendarAddPropertyUtil.property2dto(copy);
					CalendarEndpointWrapper.modifyCalendarAdd(id, from.getTime(), to.getTime(), info);
					result = true;

					// 完了メッセージ
					MessageDialog.openInformation(
							null,
							Messages.getString("successful"),
							Messages.getString("message.calendar.17"));

				} catch (CalendarDuplicate_Exception e) {
					// 実行例外が重複している場合、エラーダイアログを表示する
					MessageDialog.openInformation(
							null,
							Messages.getString("message"),
							Messages.getString("message.calendar.29"));

				} catch (Exception e) {
					String errMessage = "";
					if (e instanceof InvalidRole_Exception) {
						// 権限なし
						MessageDialog.openInformation(null, Messages.getString("message"),
								Messages.getString("message.accesscontrol.16"));
					} else {
						errMessage = ", " + e.getMessage();
					}

					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.calendar.18") + errMessage);

				}

			}
			else if (this.mode == PropertyConstant.MODE_ADD) {
				// 作成の場合
				Property copy = PropertyUtil.copy(property);
				PropertyUtil.deletePropertyDefine(copy);

				try {
					CalendarAddInfo calendarAddInfo = CalendarAddPropertyUtil.property2dto(copy);
					CalendarEndpointWrapper.addCalendarAdd(id, calendarAddInfo);
					result = true;

					// 完了メッセージ
					MessageDialog.openInformation(
							null,
							Messages.getString("successful"),
							Messages.getString("message.calendar.15"));

				} catch (CalendarDuplicate_Exception e) {
					// 実行例外が重複している場合、エラーダイアログを表示する
					MessageDialog.openInformation(
							null,
							Messages.getString("message"),
							Messages.getString("message.calendar.29"));

				} catch (Exception e) {
					String errMessage = "";
					if (e instanceof InvalidRole_Exception) {
						// 権限なし
						MessageDialog.openInformation(null, Messages.getString("message"),
								Messages.getString("message.accesscontrol.16"));
					} else {
						errMessage = ", " + e.getMessage();
					}

					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.calendar.16") + errMessage);

				}
			}
			else {
				// 表示の場合
				result = true;
			}
		}

		return result;
	}

	/**
	 * 入力値を保持したデータモデルを生成します。
	 * 
	 * @return データモデル
	 */
	public Property getInputData() {
		return (Property) this.propertySheet.getInput();
	}

	/**
	 * 入力値を保持したデータモデルを設定します。
	 * 
	 * @param property
	 */
	public void setInputData(Property property) {
		propertySheet.setInput(property);
		this.update();
	}

	/**
	 * @return Returns the uid.
	 */
	public String getUid() {
		return this.id;
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		if (this.mode == PropertyConstant.MODE_MODIFY) {
			return Messages.getString("modify");
		}
		else if (this.mode == PropertyConstant.MODE_ADD) {
			return Messages.getString("register");
		}
		else{
			return Messages.getString("ok");
		}
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * OKボタンを押されたときの処理
	 * 
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	@Override
	protected void okPressed() {
		ValidateResult result = this.validate();

		if (result == null || result.isValid()) {

			//作成時
			if(this.mode == PropertyConstant.MODE_ADD) {
				this.action();
			}
			//変更時、表示時
			else{
				super.okPressed();
			}

		} else {
			this.displayError(result);
		}
	}
}