/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.collectiverun.util;

import java.util.ArrayList;
import java.util.List;

import com.clustercontrol.ws.collectiverun.CommandInfo;
import com.clustercontrol.ws.collectiverun.CommandParameterInfo;
import com.clustercontrol.ws.collectiverun.EndStatusInfo;
import com.clustercontrol.ws.collectiverun.ItemInfo;
import com.clustercontrol.ws.collectiverun.ParameterInfo;
import com.clustercontrol.ws.collectiverun.ParameterSelectInfo;
import com.clustercontrol.ws.collectiverun.TreeItem;
import com.clustercontrol.ws.collectiverun.TypeInfo;

/**
 * 一括制御階層ツリーユーティリティクラス
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class TreeItemUtil {

	/**
	 * TreeItemクラスをコピーする
	 * 
	 * @param original
	 * @return コピーしたTreeItemクラス
	 * @since 2.1.0
	 */
	public static TreeItem copy(TreeItem original) {
		TreeItem clone = null;
		if(original != null){
			clone = clone(original);
		}

		return clone;
	}

	/**
	 * 一括制御階層ツリーからパラメータID一覧を取得
	 * 
	 * @param top
	 * @return
	 * @since 2.1.0
	 */
	public static List<String> getParameterIdList(TreeItem top) {
		ArrayList<String> list = new ArrayList<String>();

		List<TreeItem> children = top.getChildren();
		if (children != null) {
			List<TreeItem> parametaer = children.get(1).getChildren();
			if (parametaer != null) {
				for(int i = 0; i < parametaer.size(); i++){
					ParameterInfo info = parametaer.get(i).getParameterInfo();
					list.add(info.getId());
				}
			}
		}
		return list;
	}

	/**
	 * 一括制御階層ツリーからパラメータIDによりパラメータ名を取得
	 * 
	 * @param paramId
	 * @param top
	 * @return
	 * @since 2.1.0
	 */
	public static String getParameterName(String paramId, TreeItem top) {
		List<TreeItem> children = top.getChildren();
		if (children != null) {
			List<TreeItem> parametaer = children.get(1).getChildren();

			if (parametaer != null) {
				for(int i = 0; i < parametaer.size(); i++){
					ParameterInfo info = parametaer.get(i).getParameterInfo();
					if(info.getId().equals(paramId))
						return info.getName();
				}
			}
		}

		return null;
	}

	/**
	 * TreeItemクラスの最上位のインスタンスを取得する
	 * 
	 * @param item
	 * @return 最上位のインスタンス
	 * @since 2.1.0
	 */
	public static TreeItem getTopTreeItem(TreeItem item) {
		if(item == null)
			return null;

		while (item.getParent() != null) {
			item = item.getParent();
		}

		return item;
	}

	/**
	 * TreeItemクラスの同位のインスタンスから最大のorderNoを取得する
	 * 
	 * @param item
	 * @return
	 * @since 2.1.0
	 */
	public static int getMaxOrderNo(TreeItem parent) {
		if(parent == null)
			return -1;

		List<TreeItem> children = parent.getChildren();

		int max = -1;
		if (children != null) {
			for(TreeItem child : children){
				if(child.getCommandInfo() != null
						&& child.getCommandInfo().getOrderNo() > max){
					max = child.getCommandInfo().getOrderNo();

				} else if(child.getCommandParameterInfo() != null
						&& child.getCommandParameterInfo().getOrderNo() > max){
					max = child.getCommandParameterInfo().getOrderNo();

				} else if(child.getParameterInfo() != null
						&& child.getParameterInfo().getOrderNo() > max){
					max = child.getParameterInfo().getOrderNo();

				} else if(child.getParameterSelectInfo() != null
						&& child.getParameterSelectInfo().getOrderNo() > max){
					max = child.getParameterSelectInfo().getOrderNo();

				} else if(child.getTypeInfo() != null
						&& child.getTypeInfo().getOrderNo() > max){
					max = child.getTypeInfo().getOrderNo();

				}
			}
		}

		return max;
	}

	/**
	 * TreeItemクラスのインスタンスの上下移動の対象インスタンスを取得する
	 * 
	 * @param item
	 * @param up
	 * @return
	 * @since 2.1.0
	 */
	public static TreeItem getUpDownTargetItem(TreeItem item, Boolean up) {
		if(item == null)
			return null;

		TreeItem target = null;
		TreeItem parent = item.getParent();

		List<TreeItem> children = parent.getChildren();

		if (children != null) {
			CommandInfo commandInfo = item.getCommandInfo();
			CommandParameterInfo commandParameterInfo = item.getCommandParameterInfo();
			ParameterInfo parameterInfo = item.getParameterInfo();
			ParameterSelectInfo parameterSelectInfo = item.getParameterSelectInfo();
			TypeInfo typeInfo = item.getTypeInfo();

			int min = Integer.MAX_VALUE;
			int max = Integer.MIN_VALUE;
			for(TreeItem child : children) {
				CommandInfo childCommandInfo = child.getCommandInfo();
				CommandParameterInfo childCommandParameterInfo = child.getCommandParameterInfo();
				ParameterInfo childParameterInfo = child.getParameterInfo();
				ParameterSelectInfo childParameterSelectInfo = child.getParameterSelectInfo();
				TypeInfo childTypeInfo = child.getTypeInfo();
				int orderNo = 0;
				int childOrderNo = 0;
				if (childCommandInfo != null) {
					orderNo = commandInfo.getOrderNo();
					childOrderNo = childCommandInfo.getOrderNo();
				} else if (childCommandParameterInfo != null){
					orderNo = commandParameterInfo.getOrderNo();
					childOrderNo = childCommandParameterInfo.getOrderNo();
				} else if (childParameterInfo != null) {
					orderNo = parameterInfo.getOrderNo();
					childOrderNo = childParameterInfo.getOrderNo();
				} else if (childParameterSelectInfo != null) {
					orderNo = parameterSelectInfo.getOrderNo();
					childOrderNo = childParameterSelectInfo.getOrderNo();
				} else if (childTypeInfo != null) {
					orderNo = typeInfo.getOrderNo();
					childOrderNo = childTypeInfo.getOrderNo();
				}

				if (up) {
					if (childOrderNo > max && childOrderNo < orderNo) {
						max = childOrderNo;
						target = child;
					}
				} else {
					if (childOrderNo < min && childOrderNo > orderNo) {
						min = childOrderNo;
						target = child;
					}
				}
			}
		}

		return target;
	}

	/**
	 * 種別名を検索する
	 * 
	 * @param typeName
	 * @param item
	 * @return
	 * @since 2.1.0
	 */
	public static boolean findTypeName(String typeName, TreeItem item) {
		boolean find = false;
		String name;

		//種別名をチェック
		if (item.getCommandInfo() != null) {
			name = item.getCommandInfo().getName();
			if(name != null && name.length() > 0){
				if(name.compareTo(typeName) == 0){
					find = true;
					return find;
				}
			}
		} else if (item.getCommandParameterInfo() != null) {
			name = item.getCommandParameterInfo().getName();
			if(name != null && name.length() > 0){
				if(name.compareTo(typeName) == 0){
					find = true;
					return find;
				}
			}
		} else if (item.getParameterInfo() != null) {
			name = item.getParameterInfo().getName();
			if(name != null && name.length() > 0){
				if(name.compareTo(typeName) == 0){
					find = true;
					return find;
				}
			}
		} else if (item.getParameterSelectInfo() != null) {
			name = item.getParameterSelectInfo().getName();
			if(name != null && name.length() > 0){
				if(name.compareTo(typeName) == 0){
					find = true;
					return find;
				}
			}
		} else if (item.getTypeInfo() != null) {
			name = item.getTypeInfo().getName();
			if(name != null && name.length() > 0){
				if(name.compareTo(typeName) == 0){
					find = true;
					return find;
				}
			}
		}

		//子TreeItemを取得
		List<TreeItem> children = item.getChildren();
		if (children != null) {
			for(int i = 0; i < children.size(); i++){
				find = findTypeName(typeName, children.get(i));
				if(find){
					break;
				}
			}
		}

		return find;
	}

	/**
	 * TreeItemクラスの同位のインスタンスから一致するIDを検索する
	 * 
	 * @param id
	 * @param parent
	 * @return
	 */
	public static boolean findId(String id, TreeItem parent) {
		if(parent == null)
			return false;

		List<TreeItem> children = parent.getChildren();

		if (children != null) {
			for(int i = 0; i < children.size(); i++){
				CommandInfo commandInfo = children.get(i).getCommandInfo();
				CommandParameterInfo commandParameterInfo = children.get(i).getCommandParameterInfo();
				ParameterInfo parameterInfo = children.get(i).getParameterInfo();
				ParameterSelectInfo parameterSelectInfo = children.get(i).getParameterSelectInfo();
				TypeInfo typeInfo = children.get(i).getTypeInfo();

				if(commandInfo != null
						&& id.equals(commandInfo.getId())){
					return true;
				} else if(commandParameterInfo != null
						&& id.equals(commandParameterInfo.getId())){
					return true;
				} else if(parameterInfo != null
						&& id.equals(parameterInfo.getId())){
					return true;
				} else if(parameterSelectInfo != null
						&& id.equals(parameterSelectInfo.getId())){
					return true;
				} else if(typeInfo != null
						&& id.equals(typeInfo.getId())){
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * TreeItemクラスから該当する子を削除する
	 * 
	 * @param parent
	 * @param child
	 * @since 2.1.0
	 */
	public static void removeChildren(TreeItem parent, TreeItem child) {
		for (int i = 0; i < parent.getChildren().size(); i++) {
			if (child.equals(parent.getChildren().get(i))) {
				parent.getChildren().remove(i);
				break;
			}
		}
	}

	/**
	 * TreeItemクラスに該当する子を追加する
	 * 
	 * @param parent
	 * @param child
	 * @since 2.1.0
	 */
	public static void addChildren(TreeItem parent, TreeItem child) {
		child.setParent(parent);
		parent.getChildren().add(child);
	}

	/**
	 * CommandInfoクラスのcloneを作成する
	 * 
	 * @param CommandInfo
	 */
	public static CommandInfo clone(CommandInfo origInfo) {
		CommandInfo cloneInfo = new CommandInfo();
		copyItemInfo(origInfo, cloneInfo);

		// CommandInfo
		cloneInfo.setCommand(origInfo.getCommand());
		cloneInfo.setCommandType(origInfo.getCommandType());
		cloneInfo.setCommandTypeId(origInfo.getCommandTypeId());
		cloneInfo.setId(origInfo.getId());
		List<EndStatusInfo> list_orig = origInfo.getEndStatus();
		if (list_orig != null) {
			List<EndStatusInfo> list_clone = new ArrayList<EndStatusInfo>();
			for (EndStatusInfo endStatusInfo_orig : list_orig) {
				if(endStatusInfo_orig != null){
					EndStatusInfo endStatusInfo = new EndStatusInfo();
					endStatusInfo.setEndRangeValue(endStatusInfo_orig.getEndRangeValue());
					endStatusInfo.setEndStatus(endStatusInfo_orig.getEndStatus());
					endStatusInfo.setStartRangeValue(endStatusInfo_orig.getStartRangeValue());
					list_clone.add(endStatusInfo);
				}
			}
			cloneInfo.getEndStatus().addAll(list_clone);
		}
		return cloneInfo;
	}

	/**
	 * CommandParameterInfoクラスのcloneを作成する
	 * 
	 * @param CommandParameterInfo
	 */
	public static CommandParameterInfo clone(CommandParameterInfo origInfo) {
		CommandParameterInfo cloneInfo = new CommandParameterInfo();
		copyItemInfo(origInfo, cloneInfo);

		// CommandParameterInfo
		cloneInfo.setId(origInfo.getId());
		cloneInfo.setPrefix(origInfo.getPrefix());
		ParameterInfo paramMst_orig = origInfo.getParamMst();
		if (paramMst_orig != null) {
			ParameterInfo paramMst_clone = new ParameterInfo();
			paramMst_clone.setId(paramMst_orig.getId());
			paramMst_clone.setParamType(paramMst_orig.getParamType());

			List<ParameterSelectInfo> list_orig = paramMst_orig.getSelectMsts();
			if (list_orig != null) {
				List<ParameterSelectInfo> list_clone = new ArrayList<ParameterSelectInfo>();
				for (ParameterSelectInfo parameterSelectInfo_orig : list_orig) {
					if(parameterSelectInfo_orig != null){
						ParameterSelectInfo parameterSelectInfo = clone(parameterSelectInfo_orig);
						list_clone.add(parameterSelectInfo);
					}
				}
				paramMst_clone.getSelectMsts().addAll(list_clone);
			}
			cloneInfo.setParamMst(paramMst_clone);
		}

		return cloneInfo;
	}

	/**
	 * ParameterInfoクラスのcloneを作成する
	 * 
	 * @param ParameterInfo
	 */
	public static ParameterInfo clone(ParameterInfo origInfo) {
		ParameterInfo cloneInfo = new ParameterInfo();
		cloneInfo.setParamType(0);
		cloneInfo.setOrderNo(0);
		cloneInfo.setType(0);
		copyItemInfo(origInfo, cloneInfo);

		// ParameterInfo
		cloneInfo.setId(origInfo.getId());
		cloneInfo.setParamType(origInfo.getParamType());

		List<ParameterSelectInfo> list_orig = origInfo.getSelectMsts();
		if (list_orig != null) {
			List<ParameterSelectInfo> list_clone = new ArrayList<ParameterSelectInfo>();
			for (ParameterSelectInfo parameterSelectInfo_orig : list_orig) {
				if(parameterSelectInfo_orig != null){
					ParameterSelectInfo parameterSelectInfo = clone(parameterSelectInfo_orig);
					list_clone.add(parameterSelectInfo);
				}
			}
			cloneInfo.getSelectMsts().addAll(list_clone);
		}

		return cloneInfo;
	}

	/**
	 * ParameterSelectInfoクラスのcloneを作成する
	 * 
	 * @param ParameterSelectInfo
	 */
	public static ParameterSelectInfo clone(ParameterSelectInfo origInfo) {
		ParameterSelectInfo cloneInfo = new ParameterSelectInfo();
		copyItemInfo(origInfo, cloneInfo);

		// ParameterSelectInfo
		cloneInfo.setId(origInfo.getId());
		cloneInfo.setValue(origInfo.getValue());

		return cloneInfo;
	}

	/**
	 * TypeInfoクラスのcloneを作成する
	 * 
	 * @param TypeInfo
	 */
	public static TypeInfo clone(TypeInfo origInfo) {
		TypeInfo cloneInfo = new TypeInfo();
		copyItemInfo(origInfo, cloneInfo);

		// TypeInfo
		cloneInfo.setId(origInfo.getId());

		return cloneInfo;
	}

	/**
	 * TreeItemクラスのcloneを作成する
	 * 
	 * @param TreeItem
	 */
	public static TreeItem clone(TreeItem origInfo) {
		TreeItem cloneInfo = new TreeItem();

		cloneInfo.setCommandInfo(clone(origInfo.getCommandInfo()));
		cloneInfo.setCommandParameterInfo(clone(origInfo.getCommandParameterInfo()));
		cloneInfo.setParameterInfo(clone(origInfo.getParameterInfo()));
		cloneInfo.setParameterSelectInfo(clone(origInfo.getParameterSelectInfo()));
		cloneInfo.setTypeInfo(clone(origInfo.getTypeInfo()));

		cloneInfo.setParent(clone(origInfo.getParent()));

		List<TreeItem> list_orig = origInfo.getChildren();
		if (list_orig != null) {
			List<TreeItem> list_clone = new ArrayList<TreeItem>();
			for (TreeItem treeItem_orig : list_orig) {
				if(treeItem_orig != null){
					TreeItem treeItem = clone(treeItem_orig);
					list_clone.add(treeItem);
				}
			}
			cloneInfo.getChildren().addAll(list_clone);
		}
		return cloneInfo;
	}

	/**
	 * ItemInfoをコピーする
	 * 
	 * @param CommandInfo
	 */
	private static void copyItemInfo(ItemInfo origInfo, ItemInfo cloneInfo) {
		if (cloneInfo != null
				&& origInfo != null) {
			cloneInfo.setName(origInfo.getName());
			cloneInfo.setOrderNo(origInfo.getOrderNo());
			cloneInfo.setType(origInfo.getType());
		}

		return;
	}
}