/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.composite;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.EndStatusConstant;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.bean.SizeConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.composite.NumberVerifyListener;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.dialog.ScopeTreeDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.jobmanagement.bean.ProcessingMethodConstant;
import com.clustercontrol.jobmanagement.bean.SystemParameterConstant;
import com.clustercontrol.repository.FacilityPath;
import com.clustercontrol.repository.bean.FacilityConstant;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.jobmanagement.JobCommandInfo;
import com.clustercontrol.ws.repository.FacilityTreeItem;

/**
 * コマンドタブ用のコンポジットクラスです。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class CommandComposite extends Composite {
	/** スコープ用テキスト */
	protected Text m_scope = null;
	/** 起動コマンド用テキスト */
	protected Text m_startCommand = null;
	/** 停止コマンド用テキスト */
	protected Text m_stopCommand = null;
	/** 実効ユーザ用テキスト */
	protected Text m_user = null;
	/** ジョブ変数用ラジオボタン */
	protected Button m_scopeJobParam = null;
	/** 固定値用ラジオボタン */
	protected Button m_scopeFixedValue = null;
	/** スコープ参照用ボタン */
	protected Button m_scopeSelect = null;
	/** 全てのノードで実行用ラジオボタン */
	protected Button m_allNode = null;
	/** 正常終了するまでノードを順次リトライ用ラジオボタン */
	protected Button m_retry = null;
	/** コマンド実行失敗時に終了する用チェックボタン */
	protected Button m_errorCondition = null;
	/** 終了値用テキスト */
	protected Text m_errorEndValue = null;
	/** コマンド実行失敗時に終了する用グループ */
	protected Group m_errorEndConditionGroup = null;
	/** ファシリティID */
	protected String m_facilityId = null;
	/** スコープ */
	protected String m_facilityPath = null;
	/** ジョブコマンド情報 */
	protected JobCommandInfo m_execute = null;
	/** シェル */
	protected Shell m_shell = null;

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親コンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public CommandComposite(Composite parent, int style) {
		super(parent, style);
		initialize();
		m_shell = this.getShell();
	}

	/**
	 * コンポジットを配置します。
	 */
	private void initialize() {
		RowLayout layout = new RowLayout();
		layout.type = SWT.VERTICAL;
		layout.spacing = 10;
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.fill = true;
		this.setLayout(layout);

		Composite composite = null;
		RowLayout rowLayout = null;
		Label label = null;

		Group group = new Group(this, SWT.NONE);
		group.setText(Messages.getString("scope"));
		rowLayout = new RowLayout();
		rowLayout.type = SWT.VERTICAL;
		rowLayout.marginWidth = 10;
		rowLayout.marginHeight = 10;
		group.setLayout(rowLayout);

		composite = new Composite(group, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);
		m_scopeJobParam = new Button(composite, SWT.RADIO);
		m_scopeJobParam.setText(Messages.getString("job.parameter") + " : ");
		m_scopeJobParam.setLayoutData(
				new RowData(100, SizeConstant.SIZE_BUTTON_HEIGHT));
		m_scopeJobParam.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					m_scopeFixedValue.setSelection(false);
					m_scopeSelect.setEnabled(false);
				}
				update();
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});
		label = new Label(composite, SWT.LEFT);
		label.setText(
				SystemParameterConstant.getParamText(SystemParameterConstant.FACILITY_ID));
		label.setLayoutData(
				new RowData(100, SizeConstant.SIZE_LABEL_HEIGHT));

		composite = new Composite(group, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);

		m_scopeFixedValue = new Button(composite, SWT.RADIO);
		m_scopeFixedValue.setText(Messages.getString("fixed.value") + " : ");
		m_scopeFixedValue.setLayoutData(new RowData(100,
				SizeConstant.SIZE_BUTTON_HEIGHT));
		m_scopeFixedValue.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					m_scopeJobParam.setSelection(false);
					m_scopeSelect.setEnabled(true);
				}
				update();
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});
		this.m_scope = new Text(composite, SWT.BORDER | SWT.READ_ONLY);
		this.m_scope.setLayoutData(new RowData(200, SizeConstant.SIZE_TEXT_HEIGHT));
		this.m_scope.setText("");
		this.m_scope.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		m_scopeSelect = new Button(composite, SWT.NONE);
		m_scopeSelect.setText(Messages.getString("refer"));
		m_scopeSelect.setLayoutData(new RowData(80,
				SizeConstant.SIZE_BUTTON_HEIGHT));
		m_scopeSelect.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				ScopeTreeDialog dialog = new ScopeTreeDialog(m_shell);
				if (dialog.open() == IDialogConstants.OK_ID) {
					FacilityTreeItem selectItem = dialog.getSelectItem();
					if (selectItem.getData().getFacilityType() == FacilityConstant.TYPE_COMPOSITE) {
						m_facilityPath = "";
						m_facilityId = "";
					} else {
						FacilityPath path = new FacilityPath(
								ClusterControlPlugin.getDefault()
								.getSeparator());
						m_facilityPath = path.getPath(selectItem);
						m_facilityId = selectItem.getData().getFacilityId();
					}

					m_scope.setText(m_facilityPath);
					update();
				}
			}
		});

		group = new Group(this, SWT.NONE);
		group.setText(Messages.getString("scope.process"));
		rowLayout = new RowLayout();
		rowLayout.type = SWT.VERTICAL;
		rowLayout.marginWidth = 10;
		rowLayout.marginHeight = 10;
		group.setLayout(rowLayout);

		m_allNode = new Button(group, SWT.RADIO);
		m_allNode.setText(Messages.getString("scope.process.all.nodes"));
		m_allNode.setLayoutData(
				new RowData(350,SizeConstant.SIZE_BUTTON_HEIGHT));
		m_retry = new Button(group, SWT.RADIO);
		m_retry.setText(Messages.getString("scope.process.retry.nodes"));
		m_retry.setLayoutData(
				new RowData(350, SizeConstant.SIZE_BUTTON_HEIGHT));

		composite = new Composite(this, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);

		label = new Label(composite, SWT.NONE);
		label.setText(Messages.getString("start.command") + " : ");
		label.setLayoutData(new RowData(100,
				SizeConstant.SIZE_LABEL_HEIGHT));

		this.m_startCommand = new Text(composite, SWT.BORDER);
		this.m_startCommand.setLayoutData(new RowData(300,
				SizeConstant.SIZE_TEXT_HEIGHT));
		this.m_startCommand.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_1024));
		this.m_startCommand.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		composite = new Composite(this, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);

		label = new Label(composite, SWT.NONE);
		label.setText(Messages.getString("stop.command") + " : ");
		label.setLayoutData(new RowData(100,
				SizeConstant.SIZE_LABEL_HEIGHT));

		this.m_stopCommand = new Text(composite, SWT.BORDER);
		this.m_stopCommand.setLayoutData(new RowData(300,
				SizeConstant.SIZE_TEXT_HEIGHT));
		this.m_stopCommand.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_1024));
		this.m_stopCommand.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		composite = new Composite(this, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);

		label = new Label(composite, SWT.NONE);
		label.setText(Messages.getString("effective.user") + " : ");
		label.setLayoutData(new RowData(100, SizeConstant.SIZE_LABEL_HEIGHT));

		this.m_user = new Text(composite, SWT.BORDER);
		this.m_user.setLayoutData(new RowData(300, SizeConstant.SIZE_TEXT_HEIGHT));
		this.m_user.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_64));
		this.m_user.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		//エラー時終了
		composite = new Composite(this, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.VERTICAL;
		composite.setLayout(rowLayout);

		m_errorCondition = new Button(composite, SWT.CHECK);
		m_errorCondition.setText(Messages.getString("command.error.ended"));
		m_errorCondition.setLayoutData(new RowData(220,
				SizeConstant.SIZE_BUTTON_HEIGHT));
		m_errorCondition.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					m_errorEndConditionGroup.setEnabled(true);
					m_errorEndValue.setEnabled(true);
				} else {
					m_errorEndConditionGroup.setEnabled(false);
					m_errorEndValue.setEnabled(false);
				}
				update();
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});

		m_errorEndConditionGroup = new Group(this, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.VERTICAL;
		rowLayout.marginWidth = 10;
		rowLayout.marginHeight = 10;
		rowLayout.spacing = 10;
		m_errorEndConditionGroup.setLayout(rowLayout);

		composite = new Composite(m_errorEndConditionGroup, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);
		Label skipEndValueTitle = new Label(composite, SWT.LEFT);
		skipEndValueTitle.setText(Messages.getString("end.value") + " : ");
		skipEndValueTitle.setLayoutData(new RowData(80,
				SizeConstant.SIZE_LABEL_HEIGHT));
		m_errorEndValue = new Text(composite, SWT.BORDER);
		m_errorEndValue.setLayoutData(new RowData(100,
				SizeConstant.SIZE_TEXT_HEIGHT));
		m_errorEndValue.addVerifyListener(
				new NumberVerifyListener(DataRangeConstant.SMALLINT_LOW, DataRangeConstant.SMALLINT_HIGH));
		m_errorEndValue.addModifyListener(
				new ModifyListener(){
					@Override
					public void modifyText(ModifyEvent arg0) {
						update();
					}
				}
		);
	}

	/**
	 * 更新処理
	 * 
	 */
	@Override
	public void update(){
		// 必須項目を明示
		if(m_scopeFixedValue.getSelection() && "".equals(this.m_scope.getText())){
			this.m_scope.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_scope.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		if("".equals(this.m_startCommand.getText())){
			this.m_startCommand.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_startCommand.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		if("".equals(this.m_stopCommand.getText())){
			this.m_stopCommand.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_stopCommand.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		if("".equals(this.m_user.getText())){
			this.m_user.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_user.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		if(m_errorCondition.getSelection() && "".equals(m_errorEndValue.getText())) {
			this.m_errorEndValue.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_errorEndValue.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}

	/**
	 * ジョブコマンド情報をコンポジットに反映する。
	 * 
	 * @see com.clustercontrol.jobmanagement.bean.JobCommandInfo
	 */
	public void reflectCommandInfo() {

		// 初期値
		m_scopeJobParam.setSelection(false);
		m_scopeFixedValue.setSelection(true);
		m_scope.setText("");
		m_allNode.setSelection(true);
		m_startCommand.setText("");
		m_stopCommand.setText("");
		m_user.setText("");
		m_errorCondition.setSelection(true);
		m_errorEndValue.setText(String
				.valueOf(EndStatusConstant.INITIAL_VALUE_ABNORMAL));

		if (m_execute != null) {
			//スコープ設定
			m_facilityPath = m_execute.getScope();
			m_facilityId = m_execute.getFacilityID();
			if(SystemParameterConstant.isParam(
					m_facilityId,
					SystemParameterConstant.FACILITY_ID)){
				//ファシリティIDがジョブ変数の場合
				m_facilityId = "";
				m_facilityPath = "";
				m_scope.setText(m_facilityPath);
				m_scopeJobParam.setSelection(true);
				m_scopeFixedValue.setSelection(false);
			}
			else{
				if (m_facilityPath != null && m_facilityPath.length() > 0) {
					m_scope.setText(m_facilityPath);
				}
				m_scopeJobParam.setSelection(false);
				m_scopeFixedValue.setSelection(true);
			}
			//処理方法設定
			if (m_execute.getProcessingMethod() == ProcessingMethodConstant.TYPE_ALL_NODE) {
				m_allNode.setSelection(true);
				m_retry.setSelection(false);
			} else {
				m_allNode.setSelection(false);
				m_retry.setSelection(true);
			}
			//起動コマンド設定
			if (m_execute.getStartCommand() != null
					&& m_execute.getStartCommand().length() > 0) {
				m_startCommand.setText(m_execute.getStartCommand());
			}
			//停止コマンド設定
			if (m_execute.getStopCommand() != null
					&& m_execute.getStopCommand().length() > 0) {
				m_stopCommand.setText(m_execute.getStopCommand());
			}
			//ユーザー設定
			if (m_execute.getUser() != null && m_execute.getUser().length() > 0) {
				m_user.setText(m_execute.getUser());
			}

			//エラー時終了
			m_errorCondition.setSelection(YesNoConstant.typeToBoolean(m_execute
					.getErrorEndFlg()));

			//エラー時終了値
			m_errorEndValue.setText(String
					.valueOf(m_execute.getErrorEndValue()));
		}

		//スコープ
		if (m_scopeJobParam.getSelection()) {
			m_scopeSelect.setEnabled(false);
		} else {
			m_scopeSelect.setEnabled(true);
		}

		//スキップ
		if (m_errorCondition.getSelection()) {
			m_errorEndConditionGroup.setEnabled(true);
			m_errorEndValue.setEnabled(true);
		} else {
			m_errorEndConditionGroup.setEnabled(false);
			m_errorEndValue.setEnabled(false);
		}
	}

	/**
	 * ジョブコマンド情報を設定する。
	 * 
	 * @param execute ジョブコマンド情報
	 */
	public void setCommandInfo(JobCommandInfo execute) {
		m_execute = execute;
	}

	/**
	 * ジョブコマンド情報を返す。
	 * 
	 * @return ジョブコマンド情報
	 */
	public JobCommandInfo getCommandInfo() {
		return m_execute;
	}

	/**
	 * コンポジットの情報から、ジョブコマンド情報を作成する。
	 * 
	 * @return 入力値の検証結果
	 * 
	 * @see com.clustercontrol.jobmanagement.bean.JobCommandInfo
	 */
	public ValidateResult createCommandInfo() {
		ValidateResult result = null;

		//実行内容情報クラスのインスタンスを作成・取得
		m_execute = new JobCommandInfo();

		//スコープ取得
		if(m_scopeJobParam.getSelection()){
			//ジョブ変数の場合
			m_execute.setFacilityID(
					SystemParameterConstant.getParamText(SystemParameterConstant.FACILITY_ID));
			m_execute.setScope("");
		}
		else{
			//固定値の場合
			if (m_facilityId != null && m_facilityId.length() > 0){
				m_execute.setFacilityID(m_facilityId);
				m_execute.setScope(m_facilityPath);
			} else {
				result = new ValidateResult();
				result.setValid(false);
				result.setID(Messages.getString("message.hinemos.1"));
				result.setMessage(Messages.getString("message.hinemos.3"));
				return result;
			}
		}

		//処理方法取得
		if (m_allNode.getSelection()) {
			m_execute
			.setProcessingMethod(ProcessingMethodConstant.TYPE_ALL_NODE);
		} else {
			m_execute.setProcessingMethod(ProcessingMethodConstant.TYPE_RETRY);
		}

		//起動コマンド取得
		if (m_startCommand.getText().length() > 0) {
			m_execute.setStartCommand(m_startCommand.getText());
		} else {
			result = new ValidateResult();
			result.setValid(false);
			result.setID(Messages.getString("message.hinemos.1"));
			result.setMessage(Messages.getString("message.job.3"));
			return result;
		}

		//停止コマンド取得
		if (m_stopCommand.getText().length() > 0) {
			m_execute.setStopCommand(m_stopCommand.getText());
		} else {
			result = new ValidateResult();
			result.setValid(false);
			result.setID(Messages.getString("message.hinemos.1"));
			result.setMessage(Messages.getString("message.job.4"));
			return result;
		}

		//ユーザー取得
		if (m_user.getText().length() > 0) {
			m_execute.setUser(m_user.getText());
		} else {
			result = new ValidateResult();
			result.setValid(false);
			result.setID(Messages.getString("message.hinemos.1"));
			result.setMessage(Messages.getString("message.job.5"));
			return result;
		}

		//エラー時終了
		m_execute.setErrorEndFlg(YesNoConstant.booleanToType(m_errorCondition
				.getSelection()));

		//エラー時終了値取得
		try {
			if (m_errorEndValue.getText().length() > 0) {
				m_execute.setErrorEndValue(Integer.parseInt(m_errorEndValue
						.getText()));
			}
		} catch (NumberFormatException e) {
			if (m_execute.getErrorEndValue() == YesNoConstant.TYPE_YES) {
				result = new ValidateResult();
				result.setValid(false);
				result.setID(Messages.getString("message.hinemos.1"));
				result.setMessage(Messages.getString("message.job.21"));
				return result;
			}
		}

		return null;
	}
}