/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.run.dialog;

import java.util.Collection;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ProcessConstant;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.util.StringValueInfoManager;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.monitor.MonitorStringValueInfo;
import com.clustercontrol.ws.notify.NotifyRelationInfo;

/**
 * フィルタ[作成・変更]ダイアログクラス<BR>
 * 
 * @version 4.0.0
 * @since 2.1.0
 */
public class StringValueInfoCreateDialog extends CommonDialog {

	/** カラム数（タイトル）。 */
	public static final int WIDTH_TITLE = 4;

	/** カラム数（値）。 */
	public static final int WIDTH_VALUE = 2;

	/** 入力値を保持するオブジェクト。 */
	private MonitorStringValueInfo m_inputData = null;

	/** 入力値の正当性を保持するオブジェクト。 */
	private ValidateResult m_validateResult = null;

	/** 変更対象の識別キー。 */
	private String m_identifier = null;

	/** 変更対象の順序。 */
	private int m_order = Integer.MAX_VALUE;

	/** 説明。 */
	private Text m_textDescription = null;

	/** パターンマッチ表現。 */
	private Text m_textPattern = null;

	/** 条件に一致したら処理しない。 */
	private Button m_radioNotProcess = null;

	/** 条件に一致したら処理する。 */
	private Button m_radioProcess = null;

	/** 大文字・小文字を区別しない */
	private Button m_checkCaseSensitive = null;

	/** 重要度。 */
	private Combo m_comboPriority = null;

	/** メッセージ。 */
	private Text m_textMessage = null;

	/** この設定を有効にする。 */
	private Button m_buttonValid = null;

	/** メッセージにデフォルト値を入れるフラグ */
	private boolean logLineFlag = false;

	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
	public StringValueInfoCreateDialog(Shell parent, boolean logLineFlag) {
		super(parent);
		this.logLineFlag = logLineFlag;
	}

	/**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param identifier 変更する文字列監視の判定情報の識別キー
	 */
	public StringValueInfoCreateDialog(Shell parent, String identifier) {
		super(parent);

		this.m_identifier = identifier;
	}

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.composite.NotifyIdListComposite#NotifyIdListComposite(Composite, int, boolean)
	 * @see com.clustercontrol.monitor.run.composite.StringRunJobComposite#StringRunJobComposite(Composite, int)
	 * @see com.clustercontrol.monitor.run.util.StringValueInfoManager#get(String)
	 * @see com.clustercontrol.monitor.run.bean.MonitorStringValueInfo
	 * @see #setInputData(MonitorStringValueInfo)
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();

		// タイトル
		shell.setText(Messages.getString("dialog.monitor.run.create.modify.string"));

		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = 15;
		parent.setLayout(layout);

		/*
		 * 説明
		 */
		// ラベル
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("description") + " : ");
		// テキスト
		this.m_textDescription = new Text(parent, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = 10;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textDescription.setLayoutData(gridData);
		this.m_textDescription.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		/*
		 * 監視条件グループ
		 */
		// グループ
		Group monitorRuleGroup = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 15;
		monitorRuleGroup.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		monitorRuleGroup.setLayoutData(gridData);
		monitorRuleGroup.setText(Messages.getString("monitor.rule"));

		/*
		 * パターンマッチ表現
		 */
		// ラベル
		label = new Label(monitorRuleGroup, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("pattern.matching.expression") + " : ");
		// テキスト
		this.m_textPattern = new Text(monitorRuleGroup, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = 10;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textPattern.setLayoutData(gridData);
		this.m_textPattern.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		/*
		 * 出力の条件
		 */

		// 条件に一致したら処理する
		this.m_radioProcess = new Button(monitorRuleGroup, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalSpan = 7;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_radioProcess.setLayoutData(gridData);
		this.m_radioProcess.setText(Messages.getString("process.if.matched"));
		this.m_radioProcess.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				setEnabled(m_radioProcess.getSelection());
				update();
			}
		});

		// 空白
		label = new Label(monitorRuleGroup, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		// 大文字・小文字をチェックしない
		this.m_checkCaseSensitive = new Button(monitorRuleGroup, SWT.CHECK);
		gridData = new GridData();
		gridData.horizontalSpan = 7;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_checkCaseSensitive.setLayoutData(gridData);
		this.m_checkCaseSensitive.setText(Messages.getString("case.sensitive"));

		// 条件に一致したら処理しない
		this.m_radioNotProcess = new Button(monitorRuleGroup, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalSpan = 7;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_radioNotProcess.setLayoutData(gridData);
		this.m_radioNotProcess.setText(Messages.getString("don't.process.if.matched"));
		this.m_radioNotProcess.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				setEnabled(!m_radioNotProcess.getSelection());
				update();
			}
		});

		// 空白
		label = new Label(monitorRuleGroup, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 8;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * 処理グループ
		 */
		Group executeGroup = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 15;
		executeGroup.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		executeGroup.setLayoutData(gridData);
		executeGroup.setText(Messages.getString("process"));

		/*
		 * 重要度
		 */
		// ラベル
		label = new Label(executeGroup, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("priority") + " : ");
		// コンボボックス
		this.m_comboPriority = new Combo(executeGroup, SWT.DROP_DOWN | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_comboPriority.setLayoutData(gridData);
		this.m_comboPriority.add(PriorityConstant.STRING_CRITICAL);
		this.m_comboPriority.add(PriorityConstant.STRING_WARNING);
		this.m_comboPriority.add(PriorityConstant.STRING_INFO);
		this.m_comboPriority.add(PriorityConstant.STRING_UNKNOWN);
		this.m_comboPriority.setText(PriorityConstant.STRING_CRITICAL);
		// 空白
		label = new Label(executeGroup, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 8;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * メッセージ
		 */
		// ラベル
		label = new Label(executeGroup, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("message") + " : ");
		// テキスト
		this.m_textMessage = new Text(executeGroup, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalSpan = 10;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textMessage.setLayoutData(gridData);
		this.m_textMessage.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});
		// デフォルト #[LOG_LINE]
		if (logLineFlag) {
			this.m_textMessage.setText("#[LOG_LINE]");
		}

		/*
		 * 有効／無効
		 */
		this.m_buttonValid = new Button(parent, SWT.CHECK);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_buttonValid.setLayoutData(gridData);
		this.m_buttonValid.setText(Messages.getString("setting.valid.confirmed"));

		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 15;
		line.setLayoutData(gridData);

		// サイズを最適化
		// グリッドレイアウトを用いた場合、こうしないと横幅が画面いっぱいになります。
		shell.pack();
		shell.setSize(new Point(550, shell.getSize().y));

		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);
		Collection<NotifyRelationInfo> modifyNotify = null;

		// 識別子が指定されている場合、その情報を初期表示する。
		MonitorStringValueInfo info = StringValueInfoManager.getInstance().get(this.m_identifier);
		if (info == null) {
			info = new MonitorStringValueInfo();
			info.setOrderNo(Integer.MAX_VALUE);
			info.setProcessType(ProcessConstant.TYPE_YES);
			info.setValidFlg(ValidConstant.BOOLEAN_VALID);
		}

		this.setInputData(info, modifyNotify);
	}

	/**
	 * 更新処理
	 * 
	 */
	public void update(){
		// 必須項目を可視化
		if("".equals(this.m_textPattern.getText())){
			this.m_textPattern.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_textPattern.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		if(this.m_radioProcess.getSelection() && "".equals(this.m_textMessage.getText())){
			this.m_textMessage.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_textMessage.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}



	/**
	 * 入力値を保持した文字列監視の判定情報を返します。
	 * 
	 * @return 判定情報
	 */
	public MonitorStringValueInfo getInputData() {
		return this.m_inputData;
	}

	/**
	 * 引数で指定された判定情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる判定情報
	 */
	protected void setInputData(MonitorStringValueInfo info, Collection<NotifyRelationInfo> modifyNotify) {

		this.m_inputData = info;

		// 各項目に反映
		this.m_identifier = info.getIdentifier();
		this.m_order = info.getOrderNo();

		// 説明
		if (info.getDescription() != null) {
			this.m_textDescription.setText(info.getDescription());
		}

		// パターンマッチ表現
		if (info.getPattern() != null) {
			this.m_textPattern.setText(info.getPattern());
		}

		// 処理する／しない
		if (info.getProcessType() == ProcessConstant.TYPE_YES) {
			this.m_radioProcess.setSelection(true);
		} else {
			this.m_radioNotProcess.setSelection(true);
		}

		// 大文字・小文字を区別しない
		if (info.isCaseSensitivityFlg()){
			this.m_checkCaseSensitive.setSelection(true);
		}

		// 重要度
		this.m_comboPriority.setText(PriorityConstant.typeToString(info.getPriority()));

		// メッセージ
		if (info.getMessage() != null) {
			this.m_textMessage.setText(info.getMessage());
		}


		// 有効／無効
		if (info.isValidFlg() == ValidConstant.BOOLEAN_VALID) {
			this.m_buttonValid.setSelection(true);
		}

		// 入力制御
		this.setEnabled(this.m_radioProcess.getSelection());

		// 必須項目を可視化
		this.update();

	}

	/**
	 * 引数で指定された判定情報に、入力値を設定します。
	 * <p>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return 判定情報
	 * 
	 * @see #setValidateResult(String, String)
	 */
	protected MonitorStringValueInfo createInputData() {
		MonitorStringValueInfo info = new MonitorStringValueInfo();

		info.setIdentifier(this.m_identifier);
		info.setOrderNo(this.m_order);

		// 説明
		if (this.m_textDescription.getText() != null
				&& !"".equals((this.m_textDescription.getText()).trim())) {
			info.setDescription(this.m_textDescription.getText());
		}

		// パターンマッチ表現
		if (this.m_textPattern.getText() != null
				&& !"".equals((this.m_textPattern.getText()).trim())) {

			try{
				Pattern.compile(this.m_textPattern.getText());
				info.setPattern(this.m_textPattern.getText());
			}
			catch(PatternSyntaxException e){
				this.setValidateResult(Messages.getString("message.hinemos.1"),
						Messages.getString("message.monitor.27"));
				return null;
			}

		} else {
			this.setValidateResult(Messages.getString("message.hinemos.1"),
					Messages.getString("message.monitor.27"));
			return null;
		}

		// 処理する／しない
		if (this.m_radioProcess.getSelection()) {
			info.setProcessType(ProcessConstant.TYPE_YES);
		} else {
			info.setProcessType(ProcessConstant.TYPE_NO);
		}

		// 大文字・小文字を区別する/しない
		if (this.m_checkCaseSensitive.getSelection()){
			info.setCaseSensitivityFlg(true);
		}else{
			info.setCaseSensitivityFlg(false);
		}

		// 重要度
		String priorityText = this.m_comboPriority.getText();
		info.setPriority(PriorityConstant.stringToType(priorityText));

		// メッセージ
		if (this.m_textMessage.getText() != null
				&& !"".equals((this.m_textMessage.getText()).trim())) {
			info.setMessage(this.m_textMessage.getText());
		} else {
			if (this.m_radioProcess.getSelection()) {
				this.setValidateResult(Messages.getString("message.hinemos.1"),
						Messages.getString("message.monitor.29"));
				return null;
			}
		}

		// 有効
		if (this.m_buttonValid.getSelection()) {
			info.setValidFlg(ValidConstant.BOOLEAN_VALID);
		} else {
			info.setValidFlg(ValidConstant.BOOLEAN_INVALID);
		}

		return info;
	}

	/**
	 * 無効な入力値をチェックをします。
	 * 
	 * @return 検証結果
	 * 
	 * @see #createInputData()
	 */
	@Override
	protected ValidateResult validate() {
		// 入力値生成
		this.m_inputData = this.createInputData();

		if (this.m_inputData != null) {
			return super.validate();
		} else {
			return m_validateResult;
		}
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("ok");
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * コンポジットの選択可/不可を設定します。
	 * 
	 * @param enable 選択可の場合、<code> true </code>
	 */
	private void setEnabled(boolean enable) {
		this.m_comboPriority.setEnabled(enable);
		this.m_textMessage.setEnabled(enable);
	}

	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 */
	private void setValidateResult(String id, String message) {

		this.m_validateResult = new ValidateResult();
		this.m_validateResult.setValid(false);
		this.m_validateResult.setID(id);
		this.m_validateResult.setMessage(message);
	}

	/**
	 * 入力値の判定を行います。
	 * 
	 * @return true：正常、false：異常
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		boolean result = false;

		MonitorStringValueInfo info = this.m_inputData;
		if(info != null){
			result = true;
		}

		return result;
	}
}