/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.dialog;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.PriorityColorConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.composite.TextWithParameterComposite;
import com.clustercontrol.dialog.ScopeTreeDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.notify.action.AddNotify;
import com.clustercontrol.notify.action.GetNotify;
import com.clustercontrol.notify.action.ModifyNotify;
import com.clustercontrol.notify.bean.ExecFacilityConstant;
import com.clustercontrol.notify.bean.SyslogFacilityConstant;
import com.clustercontrol.notify.bean.SyslogSeverityConstant;
import com.clustercontrol.repository.FacilityPath;
import com.clustercontrol.repository.bean.FacilityConstant;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.notify.NotifyInfo;
import com.clustercontrol.ws.notify.NotifyLogEscalateInfo;
import com.clustercontrol.ws.repository.FacilityTreeItem;

/**
 * 通知（ログエスカレーション）作成・変更ダイアログクラス<BR>
 * 
 * @version 4.0.0
 * @since 3.0.0
 */
public class NotifyLogEscalateCreateDialog extends NotifyBasicCreateDialog {

	/** カラム数（重要度）。 */
	private static final int WIDTH_PRIORITY 		= 1;

	/** カラム数（チェックボックス）。 */
	private static final int WIDTH_CHECK 			= 1;

	/** カラム数（コンボボックス）。 */
	private static final int WIDTH_COMBO 			= 3;

	/** カラム数（メッセージ）。 */
	private static final int WIDTH_MESSAGE			= 5;

	/** カラム数（抑制）。 */
	private static final int WIDTH_INHIBITION_FLG	= 2;



	// ----- instance フィールド ----- //

	/** 通知タイプ
	 * @see com.clustercontrol.bean.NotifyTypeConstant
	 */
	private final int TYPE_LOG_ESCALATE=4;

	/** 入力値を保持するオブジェクト。 */
	private NotifyInfo inputData = null;

	/** 入力値の正当性を保持するオブジェクト。 */
	private ValidateResult validateResult = null;

	/** スコープ用テキスト */
	private Text m_textScope = null;

	/** ファシリティID */
	private String m_facilityId = null;

	/** スコープ */
	private String m_facilityPath = null;

	/** 出力ノード用ラジオボタン */
	private Button m_radioGenerationNodeValue = null;

	/** 固定値用ラジオボタン */
	private Button m_radioFixedValue = null;

	/** スコープ参照用ボタン */
	private Button m_scopeSelect = null;

	/** エスカレート先ポートテキストボックス。 */
	private Text m_textEscalatePort = null;

	/** 実行（重要度：通知） チェックボックス。 */
	private Button m_checkLogEscalateRunInfo = null;
	/** 実行（重要度：警告） チェックボックス。 */
	private Button m_checkLogEscalateRunWarning = null;
	/** 実行（重要度：危険） チェックボックス。 */
	private Button m_checkLogEscalateRunCritical = null;
	/** 実行（重要度：不明） チェックボックス。 */
	private Button m_checkLogEscalateRunUnknown = null;

	/** Facility（重要度：通知） コンボボックス。 */
	private Combo m_comboLogEscalateFacilityInfo = null;
	/** Facility（重要度：警告） コンボボックス。 */
	private Combo m_comboLogEscalateFacilityWarning = null;
	/** Facility（重要度：危険） コンボボックス。 */
	private Combo m_comboLogEscalateFacilityCritical = null;
	/** Facility（重要度：不明） コンボボックス。 */
	private Combo m_comboLogEscalateFacilityUnknown = null;

	/** Severity（重要度：通知） コンボボックス。 */
	private Combo m_comboLogEscalateSeverityInfo = null;
	/** Severity（重要度：警告） コンボボックス。 */
	private Combo m_comboLogEscalateSeverityWarning = null;
	/** Severity（重要度：危険） コンボボックス。 */
	private Combo m_comboLogEscalateSeverityCritical = null;
	/** Severity（重要度：不明） コンボボックス。 */
	private Combo m_comboLogEscalateSeverityUnknown = null;

	/** メッセージ（重要度：通知） テキスト。 */
	private TextWithParameterComposite m_textLogEscalateMessageInfo = null;
	/** メッセージ（重要度：警告） テキスト。 */
	private TextWithParameterComposite m_textLogEscalateMessageWarning = null;
	/** メッセージ（重要度：危険） テキスト。 */
	private TextWithParameterComposite m_textLogEscalateMessageCritical = null;
	/** メッセージ（重要度：不明） テキスト。 */
	private TextWithParameterComposite m_textLogEscalateMessageUnknown = null;

	// ----- コンストラクタ ----- //

	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
	public NotifyLogEscalateCreateDialog(Shell parent) {
		super(parent);
	}

	public NotifyLogEscalateCreateDialog(Shell parent, int notifyIdType) {
		super(parent, notifyIdType);
	}

	/**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param notifyId 変更する通知情報の通知ID
	 * @param referenceFlg 参照フラグ（参照のみで変更不可の場合、<code> true </code>）
	 */
	public NotifyLogEscalateCreateDialog(Shell parent, String notifyId, boolean referenceFlg) {
		super(parent, notifyId, referenceFlg);
	}

	// ----- instance メソッド ----- //

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.dialog.NotifyBasicCreateDialog#customizeDialog(Composite)
	 * @see com.clustercontrol.notify.action.GetNotify#getNotify(String)
	 * @see #setInputData(NotifyInfo)
	 */
	@Override
	protected void customizeDialog(Composite parent) {

		super.customizeDialog(parent);

		// 通知IDが指定されている場合、その情報を初期表示する。
		NotifyInfo info = null;
		if(this.notifyId != null){
			info = new GetNotify().getNotify(this.notifyId);
		}
		else{
			info = new NotifyInfo();
		}
		this.setInputData(info);

	}

	/**
	 * 親のクラスから呼ばれ、各通知用のダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.dialog.NotifyBasicCreateDialog#customizeDialog(Composite)
	 */
	@Override
	protected void customizeSettingDialog(Composite parent) {
		final Shell shell = this.getShell();

		// タイトル
		shell.setText(Messages.getString("dialog.notify.log.escalate.create.modify"));

		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = 15;
		parent.setLayout(layout);

		/*
		 * ログエスカレーション
		 */

		// ログエスカレーショングループ
		Group groupLogEscalate = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 15;
		groupLogEscalate.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupLogEscalate.setLayoutData(gridData);
		groupLogEscalate.setText(Messages.getString("notifies.log.escalate") + " : ");

		/*
		 * スコープグループ
		 */
		Group groupScope = new Group(groupLogEscalate, SWT.NONE);
		groupScope.setText(Messages.getString("notify.log.escalate.scope"));
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 15;
		groupScope.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupScope.setLayoutData(gridData);

		// イベント発生ノード ラジオボタン
		this.m_radioGenerationNodeValue = new Button(groupScope, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_radioGenerationNodeValue.setLayoutData(gridData);
		this.m_radioGenerationNodeValue.setText(Messages.getString("notify.node.generation") + " : ");
		this.m_radioGenerationNodeValue.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					m_radioFixedValue.setSelection(false);
					m_scopeSelect.setEnabled(false);
				}
				update();
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});

		// 固定値 ラジオボタン
		this.m_radioFixedValue = new Button(groupScope, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_radioFixedValue.setLayoutData(gridData);
		this.m_radioFixedValue.setText(Messages.getString("notify.node.fix") + " : ");
		this.m_radioFixedValue.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					m_radioGenerationNodeValue.setSelection(false);
					m_scopeSelect.setEnabled(true);
				}
				update();
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});

		this.m_textScope = new Text(groupScope, SWT.BORDER | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = 5;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textScope.setLayoutData(gridData);
		this.m_textScope.setText("");
		this.m_textScope.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		this.m_scopeSelect = new Button(groupScope, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_scopeSelect.setLayoutData(gridData);
		this.m_scopeSelect.setText(Messages.getString("refer"));
		this.m_scopeSelect.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				ScopeTreeDialog dialog = new ScopeTreeDialog(shell);
				if (dialog.open() == IDialogConstants.OK_ID) {
					FacilityTreeItem selectItem = dialog.getSelectItem();
					if (selectItem.getData().getFacilityType() == FacilityConstant.TYPE_COMPOSITE) {
						m_facilityPath = "";
						m_facilityId = "";
					} else {
						FacilityPath path = new FacilityPath(
								ClusterControlPlugin.getDefault()
								.getSeparator());
						m_facilityPath = path.getPath(selectItem);
						m_facilityId = selectItem.getData().getFacilityId();
					}

					m_textScope.setText(m_facilityPath);
					update();
				}
			}
		});
		// 空白
		label = new Label(groupScope, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 5;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		// エスカレート先ポート
		label = new Label(groupScope, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("port.number") + " : ");

		this.m_textEscalatePort = new Text(groupScope, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textEscalatePort.setLayoutData(gridData);
		this.m_textEscalatePort.setText("514");
		this.m_textEscalatePort.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		// 空白
		label = new Label(groupScope, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 10;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		// 空行
		label = new Label(groupLogEscalate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * 重要度 ごとの設定
		 */
		// ラベル（重要度）
		label = new Label(groupLogEscalate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_PRIORITY;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("priority"));

		// ラベル（実行する）
		label = new Label(groupLogEscalate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_CHECK;
		gridData.horizontalAlignment = GridData.CENTER;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("notify.attribute"));

		// ラベル（Facility）
		label = new Label(groupLogEscalate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_COMBO;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText("Syslog Facility");

		// ラベル（Severity）
		label = new Label(groupLogEscalate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_COMBO;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText("Syslog Severity");

		// ラベル（メッセージ）
		label = new Label(groupLogEscalate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_MESSAGE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("message"));

		// 空欄
		label = new Label(groupLogEscalate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_INHIBITION_FLG;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		//　ログエスカレーション　重要度：通知
		label = this.getLabelPriority(groupLogEscalate, Messages.getString("info"),PriorityColorConstant.COLOR_INFO);
		this.m_checkLogEscalateRunInfo = this.getCheckLogEscalateRun(groupLogEscalate);
		this.m_comboLogEscalateFacilityInfo = this.getComboLogEscalateFacility(groupLogEscalate);
		this.m_comboLogEscalateSeverityInfo = this.getComboLogEscalateSeverity(groupLogEscalate);
		this.m_textLogEscalateMessageInfo = this.getStyledTextMessage(groupLogEscalate);

		// 空欄
		label = new Label(groupLogEscalate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_INHIBITION_FLG;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		//　ログエスカレーション　重要度：警告
		label = this.getLabelPriority(groupLogEscalate, Messages.getString("warning"),PriorityColorConstant.COLOR_WARNING);
		this.m_checkLogEscalateRunWarning = this.getCheckLogEscalateRun(groupLogEscalate);
		this.m_comboLogEscalateFacilityWarning = this.getComboLogEscalateFacility(groupLogEscalate);
		this.m_comboLogEscalateSeverityWarning = this.getComboLogEscalateSeverity(groupLogEscalate);
		this.m_textLogEscalateMessageWarning = this.getStyledTextMessage(groupLogEscalate);

		// 空欄
		label = new Label(groupLogEscalate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_INHIBITION_FLG;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		//　ログエスカレーション　重要度：危険
		label = this.getLabelPriority(groupLogEscalate, Messages.getString("critical"),PriorityColorConstant.COLOR_CRITICAL);
		this.m_checkLogEscalateRunCritical = this.getCheckLogEscalateRun(groupLogEscalate);
		this.m_comboLogEscalateFacilityCritical = this.getComboLogEscalateFacility(groupLogEscalate);
		this.m_comboLogEscalateSeverityCritical = this.getComboLogEscalateSeverity(groupLogEscalate);
		this.m_textLogEscalateMessageCritical = this.getStyledTextMessage(groupLogEscalate);

		// 空欄
		label = new Label(groupLogEscalate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_INHIBITION_FLG;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		//　ログエスカレーション　重要度：不明
		label = this.getLabelPriority(groupLogEscalate, Messages.getString("unknown"),PriorityColorConstant.COLOR_UNKNOWN);
		this.m_checkLogEscalateRunUnknown = this.getCheckLogEscalateRun(groupLogEscalate);
		this.m_comboLogEscalateFacilityUnknown = this.getComboLogEscalateFacility(groupLogEscalate);
		this.m_comboLogEscalateSeverityUnknown = this.getComboLogEscalateSeverity(groupLogEscalate);
		this.m_textLogEscalateMessageUnknown = this.getStyledTextMessage(groupLogEscalate);

		// 空欄
		label = new Label(groupLogEscalate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_INHIBITION_FLG;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

	}

	/**
	 * 更新処理
	 * 
	 */
	public void update(){
		// 必須項目を明示

		// 固定スコープ
		if(this.m_radioFixedValue.getSelection() && "".equals(this.m_textScope.getText())){
			this.m_textScope.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_textScope.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}

		// ポート番号
		if("".equals(this.m_textEscalatePort.getText())){
			this.m_textEscalatePort.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_textEscalatePort.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}

	/**
	 * 入力値を保持した通知情報を返します。
	 * 
	 * @return 通知情報
	 */
	@Override
	public NotifyInfo getInputData() {
		return this.inputData;
	}

	/**
	 * 引数で指定された通知情報の値を、各項目に設定します。
	 * 
	 * @param notify 設定値として用いる通知情報
	 */
	@Override
	protected void setInputData(NotifyInfo notify) {

		this.inputData = notify;

		// 通知基本情報
		super.m_notifyBasic.setInputData(notify);

		// 初回通知情報
		super.m_notifyInitial.setInputData(notify);

		// 抑制情報
		super.m_notifyInhibition.setInputData(notify);

		// ログエスカレート情報
		List<NotifyLogEscalateInfo> list = notify.getNotifyLogEscalateInfo();
		if (list != null && list.size() > 0) {
			for (int index=0; index<list.size(); index++){
				NotifyLogEscalateInfo log = list.get(index);
				if(log != null){
					// 通知の場合
					if(PriorityConstant.TYPE_INFO == log.getPriority()){
						this.setInputDataForLogEscalate(log, m_checkLogEscalateRunInfo, m_comboLogEscalateFacilityInfo, m_comboLogEscalateSeverityInfo, m_textLogEscalateMessageInfo);
						// 通知のみログエスカレーション通知の共通部分の設定を行う
						if (log.getEscalateFacility() != null) {
							this.m_facilityId = log.getEscalateFacility();
							this.m_textScope.setText(log.getEscalateScope());
						}
						if (log.getEscalatePort() != null) {
							this.m_textEscalatePort.setText(log.getEscalatePort().toString());
						}
						if (log.getEscalateFacilityFlg() != null && log.getEscalateFacilityFlg() == ExecFacilityConstant.TYPE_GENERATION) {
							this.m_radioGenerationNodeValue.setSelection(true);
							this.m_scopeSelect.setEnabled(false);
						}
						else {
							this.m_radioFixedValue.setSelection(true);
							this.m_scopeSelect.setEnabled(true);
						}
					}
					// 警告の場合
					else if(PriorityConstant.TYPE_WARNING == log.getPriority()){
						this.setInputDataForLogEscalate(log, m_checkLogEscalateRunWarning, m_comboLogEscalateFacilityWarning, m_comboLogEscalateSeverityWarning, m_textLogEscalateMessageWarning);
					}
					// 危険の場合
					else if(PriorityConstant.TYPE_CRITICAL == log.getPriority()){
						this.setInputDataForLogEscalate(log, m_checkLogEscalateRunCritical, m_comboLogEscalateFacilityCritical, m_comboLogEscalateSeverityCritical, m_textLogEscalateMessageCritical);
					}
					// 不明の場合
					else if(PriorityConstant.TYPE_UNKNOWN == log.getPriority()){
						this.setInputDataForLogEscalate(log, m_checkLogEscalateRunUnknown, m_comboLogEscalateFacilityUnknown, m_comboLogEscalateSeverityUnknown, m_textLogEscalateMessageUnknown);
					}
				}
			}
		} else {
			// 新規追加の場合
			this.m_radioGenerationNodeValue.setSelection(true);
			this.m_scopeSelect.setEnabled(false);
		}

		// 有効／無効
		if (notify.getValidFlg() != null && notify.getValidFlg() == ValidConstant.TYPE_INVALID) {
			this.m_confirmValid.setSelection(false);
		}
		else {
			this.m_confirmValid.setSelection(true);
		}

		// 入力制御
		if(this.referenceFlg){
			this.setEnabled(false);
		}
		else{
			this.setEnabled(true);
		}

	}

	/**
	 * 引数で指定された通知ログエスカレーション情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる通知ログエスカレーション情報
	 * @param checkLogEscalateRun 通知チェックボックス
	 * @param comboLogEscalateFacility ファシリティコンボボックス
	 * @param comboLogEscalatePriority プライオリティコンボボックス
	 * @param textLogEscalateMessage メッセージテキストボックス
	 * @param checkLogEscalateInhibition 抑制チェックボックス
	 */
	private void setInputDataForLogEscalate(NotifyLogEscalateInfo info,
			Button checkLogEscalateRun,
			Combo comboLogEscalateFacility,
			Combo comboLogEscalatePriority,
			TextWithParameterComposite textLogEscalateMessage
	) {
		// ログエスカレーション実行
		if (info.getValidFlg() != null && info.getValidFlg().intValue() == ValidConstant.TYPE_VALID) {
			checkLogEscalateRun.setSelection(true);
		}
		// ファシリティ
		if (info.getSyslogFacility() != null) {
			comboLogEscalateFacility.setText(SyslogFacilityConstant.typeToString(info.getSyslogFacility()));
		}
		// プライオリティ
		if (info.getSyslogPriority() != null) {
			comboLogEscalatePriority.setText(SyslogSeverityConstant.typeToString(info.getSyslogPriority()));
		}
		// ジョブID
		if (info.getEscalateMessage() != null) {
			textLogEscalateMessage.setText(info.getEscalateMessage());
		}
	}

	/**
	 * 入力値を設定した通知情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return 通知情報
	 * 
	 * @see #createInputDataForLogEscalate(ArrayList, int, Button, Combo, Button, Combo, Button, Text)
	 */
	@Override
	protected NotifyInfo createInputData() {
		NotifyInfo info = new NotifyInfo();

		// 通知タイプの設定
		info.setNotifyType(TYPE_LOG_ESCALATE);

		// 通知基本情報
		validateResult = super.m_notifyBasic.createInputData(info, this.notifyId);
		if(validateResult != null){
			return null;
		}

		// 初回通知情報
		validateResult = super.m_notifyInitial.createInputData(info);
		if(validateResult != null){
			return null;
		}

		// 抑制情報
		validateResult = super.m_notifyInhibition.createInputData(info);
		if(validateResult != null){
			return null;
		}

		// ログエスカレーション情報

		ArrayList<NotifyLogEscalateInfo> list = new ArrayList<NotifyLogEscalateInfo>();
		list = this.createInputDataForLogEscalate(list, PriorityConstant.TYPE_INFO,		this.m_checkLogEscalateRunInfo,			this.m_comboLogEscalateFacilityInfo, 		this.m_comboLogEscalateSeverityInfo, 		this.m_textLogEscalateMessageInfo, null);
		list = this.createInputDataForLogEscalate(list, PriorityConstant.TYPE_WARNING,	this.m_checkLogEscalateRunWarning,		this.m_comboLogEscalateFacilityWarning, 	this.m_comboLogEscalateSeverityWarning, 	this.m_textLogEscalateMessageWarning, null);
		list = this.createInputDataForLogEscalate(list, PriorityConstant.TYPE_CRITICAL,	this.m_checkLogEscalateRunCritical,		this.m_comboLogEscalateFacilityCritical, 	this.m_comboLogEscalateSeverityCritical, 	this.m_textLogEscalateMessageCritical, null);
		list = this.createInputDataForLogEscalate(list, PriorityConstant.TYPE_UNKNOWN,	this.m_checkLogEscalateRunUnknown,		this.m_comboLogEscalateFacilityUnknown, 	this.m_comboLogEscalateSeverityUnknown, 	this.m_textLogEscalateMessageUnknown, null);


		if(list == null){
			return null;
		}

		List<NotifyLogEscalateInfo> notifyLogEscalateInfo = info.getNotifyLogEscalateInfo();
		notifyLogEscalateInfo.clear();
		notifyLogEscalateInfo.addAll(list);

		// 有効/無効
		if (this.m_confirmValid.getSelection()) {
			info.setValidFlg(ValidConstant.TYPE_VALID);
		} else {
			info.setValidFlg(ValidConstant.TYPE_INVALID);
		}

		return info;
	}

	/**
	 * 入力値を設定した通知ログエスカレーション情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return 通知ログエスカレーション情報
	 * 
	 */
	protected ArrayList<NotifyLogEscalateInfo> createInputDataForLogEscalate(ArrayList<NotifyLogEscalateInfo> list,
			int priority,
			Button checkLogEscalateRun,
			Combo comboLogEscalateFacility,
			Combo comboLogEscalatePriority,
			TextWithParameterComposite textLogEscalateMessage,
			Button checkLogEscalateInhibition
	) {

		if(list == null)
			return null;

		NotifyLogEscalateInfo log = new NotifyLogEscalateInfo();

		// 重要度
		log.setPriority(priority);

		// 実行
		if (checkLogEscalateRun.getSelection()) {
			log.setValidFlg(new Integer(ValidConstant.TYPE_VALID));
		} else {
			log.setValidFlg(new Integer(ValidConstant.TYPE_INVALID));
		}

		// ファシリティ
		if (comboLogEscalateFacility.getText() != null
				&& !"".equals(comboLogEscalateFacility.getText())){
			log.setSyslogFacility(SyslogFacilityConstant.stringToType(comboLogEscalateFacility.getText()));
		}

		// プライオリティ
		if (comboLogEscalatePriority.getText() != null
				&& !"".equals(comboLogEscalatePriority.getText())){
			log.setSyslogPriority(SyslogSeverityConstant.stringToType(comboLogEscalatePriority.getText()));
		}

		// メッセージ
		if (textLogEscalateMessage.getText() != null
				&& !"".equals(comboLogEscalatePriority.getText())){
			log.setEscalateMessage(textLogEscalateMessage.getText());
		}

		// 共通部分登録
		// 実行ファシリティID
		if (this.m_textScope.getText() != null
				&& !"".equals(this.m_textScope.getText())) {
			log.setEscalateFacility(this.m_facilityId);
			log.setEscalateScope(this.m_textScope.getText());
		}
		// 実行ファシリティ
		if (this.m_radioGenerationNodeValue.getSelection()) {
			log.setEscalateFacilityFlg(ExecFacilityConstant.TYPE_GENERATION);
		}
		else if (this.m_radioFixedValue.getSelection()){
			log.setEscalateFacilityFlg(ExecFacilityConstant.TYPE_FIX);
		}
		// ポート番号
		if (this.m_textEscalatePort.getText() != null
				&& !"".equals(this.m_textEscalatePort.getText())) {
			log.setEscalatePort(Integer.parseInt(this.m_textEscalatePort.getText()));
		}

		list.add(log);
		return list;
	}


	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	@Override
	protected ValidateResult validate() {
		// 入力値生成
		this.inputData = this.createInputData();

		if (this.inputData != null) {
			return super.validate();
		} else {
			return validateResult;
		}
	}

	/**
	 * 入力値をマネージャに登録します。
	 * 
	 * @return true：正常、false：異常
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		boolean result = false;

		NotifyInfo info = this.getInputData();
		if(info != null){
			if(this.notifyId == null){
				// 作成の場合
				result = new AddNotify().add(info);
			}
			else{
				// 変更の場合
				result = new ModifyNotify().modify(info);
			}
		}

		return result;
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("ok");
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 */
	@Override
	protected void setValidateResult(String id, String message) {

		this.validateResult = new ValidateResult();
		this.validateResult.setValid(false);
		this.validateResult.setID(id);
		this.validateResult.setMessage(message);
	}

	/**
	 * ボタンを生成します。<BR>
	 * 参照フラグが<code> true </code>の場合は閉じるボタンを生成し、<code> false </code>の場合は、デフォルトのボタンを生成します。
	 * 
	 * @param parent ボタンバーコンポジット
	 * 
	 * @see #createButtonsForButtonBar(Composite)
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {

		if(!this.referenceFlg){
			super.createButtonsForButtonBar(parent);
		}
		else{
			// 閉じるボタン
			this.createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CLOSE_LABEL, false);
		}
	}

	/**
	 * コンポジットの選択可/不可を設定します。
	 * 
	 * @param enable 選択可の場合、<code> true </code>
	 */
	@Override
	protected void setEnabled(boolean enable) {
		super.m_notifyBasic.setEnabled(enable);
		super.m_notifyInhibition.setEnabled(enable);
	}

	/**
	 * 重要度のラベルを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param text ラベルに表示するテキスト
	 * @param background ラベルの背景色
	 * @return 生成されたラベル
	 */
	protected Label getLabelPriority(Composite parent,
			String text,
			Color background
	) {

		// ラベル（重要度）
		Label label = new Label(parent, SWT.NONE);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_PRIORITY;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(text + " : ");
		label.setBackground(background);

		return label;
	}

	/**
	 * ログエスカレーションの実行チェックボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
	protected Button getCheckLogEscalateRun(Composite parent) {

		// チェックボックス（実行）
		Button button = new Button(parent, SWT.CHECK);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_CHECK;
		gridData.horizontalAlignment = GridData.CENTER;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);

		return button;
	}

	/**
	 * ログエスカレーションのFacilityコンボボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたコンボボックス
	 */
	protected Combo getComboLogEscalateFacility(Composite parent) {

		int blank = 0;

		// コンボボックス（通知状態）
		Combo combo = new Combo(parent, SWT.DROP_DOWN | SWT.READ_ONLY);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_COMBO - blank;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		combo.setLayoutData(gridData);

		// Facilityをコンボボックスに登録する
		combo.add(SyslogFacilityConstant.STRING_AUTH);
		combo.add(SyslogFacilityConstant.STRING_AUTHPRIV);
		combo.add(SyslogFacilityConstant.STRING_CRON);
		combo.add(SyslogFacilityConstant.STRING_DAEMON);
		combo.add(SyslogFacilityConstant.STRING_FTP);
		combo.add(SyslogFacilityConstant.STRING_KERN);
		combo.add(SyslogFacilityConstant.STRING_LPR);
		combo.add(SyslogFacilityConstant.STRING_MAIL);
		combo.add(SyslogFacilityConstant.STRING_NEWS);
		combo.add(SyslogFacilityConstant.STRING_SYSLOG);
		combo.add(SyslogFacilityConstant.STRING_USER);
		combo.add(SyslogFacilityConstant.STRING_UUCP);
		combo.add(SyslogFacilityConstant.STRING_LOCAL0);
		combo.add(SyslogFacilityConstant.STRING_LOCAL1);
		combo.add(SyslogFacilityConstant.STRING_LOCAL2);
		combo.add(SyslogFacilityConstant.STRING_LOCAL3);
		combo.add(SyslogFacilityConstant.STRING_LOCAL4);
		combo.add(SyslogFacilityConstant.STRING_LOCAL5);
		combo.add(SyslogFacilityConstant.STRING_LOCAL6);
		combo.add(SyslogFacilityConstant.STRING_LOCAL7);

		combo.setText(SyslogFacilityConstant.STRING_USER);

		// 空白
		if( blank > 0){
			Label label = new Label(parent, SWT.NONE);
			gridData = new GridData();
			gridData.horizontalSpan = blank;
			gridData.horizontalAlignment = GridData.FILL;
			gridData.grabExcessHorizontalSpace = true;
			label.setLayoutData(gridData);
		}

		return combo;
	}

	/**
	 * ログエスカレーションのPriorityコンボボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたコンボボックス
	 */
	protected Combo getComboLogEscalateSeverity(Composite parent) {

		int blank = 0;

		// コンボボックス（通知状態）
		Combo combo = new Combo(parent, SWT.DROP_DOWN | SWT.READ_ONLY);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_COMBO - blank;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		combo.setLayoutData(gridData);

		// Priorityをコンボボックスに登録する
		combo.add(SyslogSeverityConstant.STRING_EMERG);
		combo.add(SyslogSeverityConstant.STRING_ALERT);
		combo.add(SyslogSeverityConstant.STRING_CRIT);
		combo.add(SyslogSeverityConstant.STRING_ERR);
		combo.add(SyslogSeverityConstant.STRING_WARNING);
		combo.add(SyslogSeverityConstant.STRING_NOTICE);
		combo.add(SyslogSeverityConstant.STRING_INFO);
		combo.add(SyslogSeverityConstant.STRING_DEBUG);

		combo.setText(SyslogSeverityConstant.STRING_ERR);
		// 空白
		if( blank > 0){
			Label label = new Label(parent, SWT.NONE);
			gridData = new GridData();
			gridData.horizontalSpan = blank;
			gridData.horizontalAlignment = GridData.FILL;
			gridData.grabExcessHorizontalSpace = true;
			label.setLayoutData(gridData);
		}

		return combo;
	}

	/**
	 * ログエスカレーションのメッセージテキストボックスを返します。
	 * 
	 * @param parent 親のインスタンス
	 * @return 生成されたテキストボックス
	 */
	protected TextWithParameterComposite getStyledTextMessage(Composite parent/*, VerifyListener listener*/) {

		// テキスト（送信先（セミコロン区切り））
		TextWithParameterComposite text = new TextWithParameterComposite(parent, SWT.BORDER | SWT.LEFT);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_MESSAGE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;

		text.setToolTipText(Messages.getString("notify.parameter.tooltip"));
		text.setLayoutData(gridData);
		text.setColor(new Color(parent.getDisplay(), new RGB(0, 0, 255)));
		return text;
	}

	/**
	 * ログエスカレーションの抑制のチェックボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
	protected Button getCheckLogEscalateInhibition(Composite parent) {

		// チェックボックス（抑制する）
		Button button = new Button(parent, SWT.CHECK);
		GridData gridData = new GridData();
		gridData.horizontalSpan = WIDTH_INHIBITION_FLG;
		gridData.horizontalAlignment = GridData.CENTER;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);

		return button;
	}
}