/*

Copyright (C) since 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.accesscontrol.factory;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.util.Base64;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.UserNotFound;
import com.clustercontrol.accesscontrol.bean.UserFilterInfo;
import com.clustercontrol.accesscontrol.bean.UserInfo;
import com.clustercontrol.accesscontrol.ejb.entity.UserLocal;
import com.clustercontrol.accesscontrol.ejb.entity.UserRoleLocal;
import com.clustercontrol.accesscontrol.ejb.entity.UserUtil;
import com.clustercontrol.commons.util.ObjectValidator;

/**
 * ユーザ情報を検索するファクトリクラス<BR>
 *
 * @version 1.0.0
 * @since 3.2.0
 */
public class LoginUserSelector {

	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog(LoginUserSelector.class);


	/**
	 * 指定のユーザ情報を取得する。<BR>
	 * 
	 * @return ユーザ情報
	 */
	public static UserInfo getUserInfo(String userId) {
		UserInfo info = null;
		if(userId != null && userId.compareTo("") != 0){
			try {
				// ユーザを取得
				UserLocal cal = UserUtil.getLocalHome().findByPrimaryKey(userId);

				info = new UserInfo();
				info.setId(cal.getUserId());
				info.setName(cal.getUserName());
				info.setDescription(cal.getDescription());
				info.setCreateUserId(cal.getCreateUserId());
				if (cal.getCreateDatetime() != null) {
					info.setCreateDate(cal.getCreateDatetime().getTime());
				}
				info.setModifyUserId(cal.getModifyUserId());
				if (cal.getModifyDatetime() != null) {
					info.setModifyDate(cal.getModifyDatetime().getTime());
				}

				for(UserRoleLocal roleLocal : (Collection<UserRoleLocal>)cal.getUserRole()){
					info.addRole(roleLocal.getUserRole());
				}

			} catch (FinderException e) {
				m_log.info("found no user.", e);
			} catch (Exception e) {
				m_log.warn("failure to get all user.", e);
				throw new EJBException("failure to get all user.", e);
			}
		}
		return info;
	}

	/**
	 * ユーザ名とパスワードからUserInfoを取得する。
	 * @param userid
	 * @param password
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public static void getUserInfoByPassword(String userId, String password, ArrayList<String> roleList)
	throws InvalidUserPass, InvalidRole, HinemosUnknown {

		UserLocal user = null;
		boolean invalidUserFlag = true;
		try {
			user = UserUtil.getLocalHome().findByPrimaryKey(userId);
			invalidUserFlag = false;
		} catch (FinderException e) {
			invalidUserFlag = true;
		} catch (NamingException e) {
			String message = "getUserInfoByPassword : NamingException " + e.getMessage();
			m_log.warn(message);
			throw new HinemosUnknown(message,e);
		}
		if(user == null){
			invalidUserFlag = true;
		}
		else if (!user.getPassword().equals(hash(password))) {
			invalidUserFlag = true;
		}
		if (invalidUserFlag) {
			String message = "user(" + userId + ")/password is invalid combination";
			m_log.warn(message);
			throw new InvalidUserPass(message);
		}

		UserInfo userInfo = getUserInfo(userId);
		for (String role : roleList) {
			if (!userInfo.getRoleList().contains(role)) {
				String message = "user-role(" + list2String(userInfo.getRoleList()) +
				",need-role " + list2String(roleList);
				m_log.warn(message);
				throw new InvalidRole(message);
			}
		}
	}

	private static String hash(String password) {
		MessageDigest md = null;
		try {
			md = MessageDigest.getInstance("MD5");
		} catch (NoSuchAlgorithmException e) {
			m_log.error("NoSuchAlgorithmException " + e.getMessage());
			return null;
		}
		byte[] dat = password.getBytes();
		md.update(dat);
		return Base64.encodeBytes(md.digest());
	}

	/**
	 * 処理速度を測定するためのサンプルスクリプト
	 * @param args
	 */
	public static void main (String args[]) {
		int maxN = 1000000;
		String str = null;
		System.out.println("start (" + maxN + " loops) : " + new Date());
		for (int i = 0; i < maxN; i ++ ) {
			str = hash("HINEMOS_AGENT");
		}
		System.out.println("hash : " + str);
		System.out.println("end   (" + maxN + " loops) : " + new Date());
	}

	/**
	 * リストを文字列に変換する関数。
	 * debug用。
	 * @param list
	 * @return
	 */
	private static String list2String (Collection<String> list) {
		String ret = "";
		if (list == null) {
			return ret;
		}
		for (String s : list) {
			ret += s;
		}
		return ret;
	}

	/**
	 * ユーザ一覧情報を取得する。<BR>
	 * 
	 * @return ユーザ情報のリスト
	 * @throws UserNotFound
	 */
	public static ArrayList<UserInfo> getUserInfoList() throws UserNotFound{
		ArrayList<UserInfo> list = new ArrayList<UserInfo>();
		Collection<UserLocal> users = null;

		m_log.debug("getting all user...");

		try {
			// 全ユーザを取得
			users = UserUtil.getLocalHome().findAllLoginUser();
		} catch (FinderException e) {
			m_log.info("found no user.", e);
			throw new UserNotFound(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("failure to get all user.", e);
			throw new EJBException("failure to get all user.", e);
		}

		if(users != null){
			Iterator<UserLocal> itr = users.iterator();
			while(itr.hasNext()){
				UserLocal cal = itr.next();

				UserInfo info = new UserInfo();
				info.setId(cal.getUserId());
				info.setName(cal.getUserName());
				info.setDescription(cal.getDescription());
				info.setCreateUserId(cal.getCreateUserId());
				if (cal.getCreateDatetime() != null) {
					info.setCreateDate(cal.getCreateDatetime().getTime());
				}
				info.setModifyUserId(cal.getModifyUserId());
				if (cal.getModifyDatetime() != null) {
					info.setModifyDate(cal.getModifyDatetime().getTime());
				}

				list.add(info);
			}
		}

		m_log.debug("successful in getting all user.");
		return list;
	}

	/**
	 * ユーザIDに付与されたユーザ名を取得する。<BR>
	 * 
	 * @param userId ユーザID
	 * @return ユーザ名
	 * @throws NamingException
	 * @throws UserNotFound
	 */
	public static String getUserName(String userId) throws NamingException, UserNotFound {
		UserLocal user = null;

		if(userId != null && userId.compareTo("") != 0){
			// 該当するユーザを取得
			try {
				user = UserUtil.getLocalHome().findByPrimaryKey(userId);
			} catch (FinderException e) {
				m_log.warn("failure to get a user's name.", e);
				throw new UserNotFound(e.getMessage(), e);
			} catch (NamingException e) {
				throw e;
			}
		}
		return user.getUserName();
	}

	/**
	 * 条件に一致するユーザ一覧情報を取得する。<BR>
	 * 
	 * @param info ユーザ検索条件
	 * @return ユーザ情報のリスト
	 * @throws UserNotFound
	 */
	public static ArrayList<UserInfo> getUserInfoList(UserFilterInfo filter) throws UserNotFound {
		ArrayList<UserInfo> list = null;

		if(filter != null){

			try {
				list = new ArrayList<UserInfo>();

				// 検索対象のユーザIDの取得
				String userId = filter.getUserId();

				// 検索対象のユーザ名の取得
				String userName = filter.getUserName();

				// 検索対象の説明の取得
				String description = filter.getDescription();

				// 検索対象のユーザ権限の取得
				ArrayList<String> roleList = filter.getRoleList();

				// 検索条件に基づいた絞込み
				boolean userIdFlg = false;
				boolean userNameFlg = false;
				boolean descriptionFlg = false;

				if (! ObjectValidator.isEmptyString(userId)) {
					userIdFlg = true;
				}
				if (! ObjectValidator.isEmptyString(userName)) {
					userNameFlg = true;
				}
				if (! ObjectValidator.isEmptyString(description)) {
					descriptionFlg = true;
				}
				Collection<UserLocal> users = UserUtil.getLocalHome().findAllLoginUser();

				m_log.debug("getUserInfoList : " +
						"userIdFlg=" + userIdFlg + ",userId=" + userId +
						",userNameFlg=" + userNameFlg + ",userName=" + userName +
						",descriptionFlg=" + descriptionFlg + ",description" + description);

				for (UserLocal user : users) {
					// ユーザID・ユーザ名・説明が入力されている場合、部分一致しなかったらヒットしない
					if (userIdFlg && user.getUserId().indexOf(userId) == -1) {
						continue;
					}
					if (userNameFlg && user.getUserName().indexOf(userName) == -1) {
						continue;
					}
					if (descriptionFlg && user.getDescription().indexOf(description) == -1) {
						continue;
					}

					// 検索条件に権限が入力されていない場合、権限の保持に関するチェックは行わず、ヒットする
					if (roleList == null || roleList.size() == 0) {
						UserInfo info = new UserInfo();
						info.setId(user.getUserId());
						info.setName(user.getUserName());
						info.setDescription(user.getDescription());
						info.setCreateUserId(user.getCreateUserId());
						if (user.getCreateDatetime() != null) {
							info.setCreateDate(user.getCreateDatetime().getTime());
						}
						info.setModifyUserId(user.getModifyUserId());
						if (user.getModifyDatetime() != null) {
							info.setModifyDate(user.getModifyDatetime().getTime());
						}

						list.add(info);
					}
					// 検索条件に指定されたすべての権限がユーザの保持する権限の集合に含まれる場合、ヒットする
					else {
						int accessMatchCount = 0;

						for (UserRoleLocal role : (Collection<UserRoleLocal>)user.getUserRole()) {
							if ( roleList.contains(role.getUserRole()) ) {
								accessMatchCount++;
							}
						}
						if (accessMatchCount == roleList.size()) {
							UserInfo info = new UserInfo();
							info.setId(user.getUserId());
							info.setName(user.getUserName());
							info.setDescription(user.getDescription());
							info.setCreateUserId(user.getCreateUserId());
							if (user.getCreateDatetime() != null) {
								info.setCreateDate(user.getCreateDatetime().getTime());
							}
							info.setModifyUserId(user.getModifyUserId());
							if (user.getModifyDatetime() != null) {
								info.setModifyDate(user.getModifyDatetime().getTime());
							}

							list.add(info);
						}
					}
				}
			} catch (FinderException e) {
				m_log.info("found no user.", e);
				throw new UserNotFound(e.getMessage(), e);
			} catch (Exception e) {
				m_log.info("failure to get users by using filter.", e);
				throw new EJBException("failure to get users by using filter.", e);
			}
		}
		return list;
	}
}
