/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.calendar.factory;

import java.sql.Time;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CalendarInvalid;
import com.clustercontrol.fault.CalendarNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.bean.CalendarAddInfo;
import com.clustercontrol.calendar.bean.CalendarDTO;
import com.clustercontrol.calendar.bean.CalendarInfo;
import com.clustercontrol.calendar.bean.CalendarIrregularDTO;
import com.clustercontrol.calendar.bean.CalendarIrregularDTO.CalendarIrregularType;
import com.clustercontrol.calendar.bean.CalendarWeekdayDTO;
import com.clustercontrol.calendar.bean.RunConstant;
import com.clustercontrol.calendar.bean.ScheduleInfo;
import com.clustercontrol.calendar.bean.ScheduleTableConstant;
import com.clustercontrol.calendar.bean.WeekdayInfo;
import com.clustercontrol.calendar.ejb.entity.CalendarAddInfoLocal;
import com.clustercontrol.calendar.ejb.entity.CalendarAddInfoPK;
import com.clustercontrol.calendar.ejb.entity.CalendarAddInfoUtil;
import com.clustercontrol.calendar.ejb.entity.CalendarInfoLocal;
import com.clustercontrol.calendar.ejb.entity.CalendarInfoUtil;
import com.clustercontrol.calendar.ejb.entity.CalendarWeekdayInfoLocal;
import com.clustercontrol.calendar.ejb.entity.CalendarWeekdayInfoPK;
import com.clustercontrol.calendar.ejb.entity.CalendarWeekdayInfoUtil;


/**
 * カレンダを検索するファクトリークラス<BR>
 *
 * @version 2.0.0
 * @since 2.0.0
 */
public class SelectCalendar {

	private static Log m_log = LogFactory.getLog( SelectCalendar.class );

	/**
	 * カレンダ情報を取得します。
	 * 
	 * @param id
	 * @return カレンダ情報
	 * @throws CalendarNotFound
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	public CalendarInfo getCalendar(String id) throws CalendarNotFound, NamingException {
		CalendarInfo ret = null;

		if(id != null && id.compareTo("") != 0){
			//カレンダ取得
			CalendarInfoLocal info = null;

			try{
				info = CalendarInfoUtil.getLocalHome().findByPrimaryKey(id);
				//カレンダ情報のDTOを生成
				ret = new CalendarInfo();

				if(info != null){

					//id
					ret.setId(info.getCalendar_id());
					//名前
					ret.setName(info.getCalendar_name());
					//有効期間(From)
					if (info.getValid_time_from() != null) {
						ret.setValidTimeFrom(info.getValid_time_from().getTime());
					}
					//有効期間(To)
					if (info.getValid_time_to() != null) {
						ret.setValidTimeTo(info.getValid_time_to().getTime());
					}
					//説明
					ret.setDescription(info.getDescription());
					//登録者
					ret.setRegUser(info.getReg_user());
					//登録日時
					if (info.getReg_date() != null) {
						ret.setRegDate(info.getReg_date().getTime());
					}
					//更新者
					ret.setUpdateUser(info.getUpdate_user());
					//更新日時
					if (info.getUpdate_date() != null) {
						ret.setUpdateDate(info.getUpdate_date().getTime());
					}
				}
			}catch(FinderException e){
				m_log.debug("getCalendar():" + e.getMessage());
				throw new CalendarNotFound("getCalendar() find error : calendarId=" + id);
			}catch(NamingException e){
				throw e;
			}
		}
		return ret;
	}

	/**
	 * カレンダ情報一覧を取得します。
	 * 
	 * @return カレンダ情報のリスト
	 * @throws NamingException
	 * @throws CalendarNotFound
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<CalendarInfo> getCalendarList() throws NamingException, CalendarNotFound {
		ArrayList<CalendarInfo> list = new ArrayList<CalendarInfo>();

		try {
			//全カレンダを取得
			Collection<CalendarInfoLocal> ct = CalendarInfoUtil.getLocalHome().findAll();

			if(ct != null){
				Iterator<CalendarInfoLocal> itr = ct.iterator();
				while(itr.hasNext()){
					CalendarInfoLocal cal = itr.next();

					CalendarInfo info = new CalendarInfo();
					info.setId(cal.getCalendar_id());
					info.setName(cal.getCalendar_name());
					if (cal.getValid_time_from() != null) {
						info.setValidTimeFrom(cal.getValid_time_from().getTime());
					}
					if (cal.getValid_time_to() != null) {
						info.setValidTimeTo(cal.getValid_time_to().getTime());
					}
					info.setDescription(cal.getDescription());
					info.setRegUser(cal.getReg_user());
					if (cal.getReg_date() != null) {
						info.setRegDate(cal.getReg_date().getTime());
					}
					info.setUpdateUser(cal.getUpdate_user());
					if (cal.getUpdate_date() != null) {
						info.setUpdateDate(cal.getUpdate_date().getTime());
					}
					info.setCheck(Boolean.valueOf(true));
					list.add(info);
				}
			}
		} catch (FinderException e) {
			m_log.debug("getCalendarList():" + e.getMessage());
			throw new CalendarNotFound("SelectCalendar.getCalendarList() find error : " + e.getMessage());
		} catch (NamingException e) {
			throw e;
		}

		return list;
	}

	/**
	 * カレンダ曜日別情報一覧を取得します。
	 * 
	 * @param id
	 * @param weekday
	 * @param from
	 * @param to
	 * @return カレンダ曜日別情報のリスト
	 * @throws CalendarNotFound
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	public WeekdayInfo getWeekday(String id, Integer weekday, Long from, Long to) throws CalendarNotFound, NamingException {
		WeekdayInfo info = null;

		if(id != null && id.compareTo("") != 0){
			CalendarWeekdayInfoLocal infoLocal = null;

			try{
				infoLocal = CalendarWeekdayInfoUtil.getLocalHome().findByPrimaryKey(
						new CalendarWeekdayInfoPK(id, weekday, from==null?null:new Time(from), to==null?null:new Time(to)));

				info = new WeekdayInfo();
				if(info != null){
					//曜日
					info.setWeekday(infoLocal.getWeekday());
					//時間帯(From)
					if(infoLocal.getTime_from() != null) {
						info.setTimeFrom(infoLocal.getTime_from().getTime());
					}
					//時間帯(To)
					if(infoLocal.getTime_to() != null) {
						info.setTimeTo(infoLocal.getTime_to().getTime());
					}
					//説明
					info.setDescription(infoLocal.getDescription());
					//登録者
					info.setRegUser(infoLocal.getReg_user());
					//登録日時
					if (infoLocal.getReg_date() != null) {
						info.setRegDate(infoLocal.getReg_date().getTime());
					}
					//更新者
					info.setUpdateUser(infoLocal.getUpdate_user());
					//更新日時
					if (infoLocal.getUpdate_date() != null) {
						info.setUpdateDate(infoLocal.getUpdate_date().getTime());
					}
				}
			}catch(FinderException e){
				throw new CalendarNotFound("getWeekday() find error : calendarId=" + id);
			}catch(NamingException e){
				throw e;
			}
		}

		return info;
	}

	/**
	 * カレンダ曜日別情報一覧を取得します。
	 * 
	 * @param id
	 * @return カレンダ曜日別情報のリスト
	 * @throws NamingException
	 * @throws CalendarNotFound
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<WeekdayInfo> getWeekdayList(String id) throws NamingException, CalendarNotFound {
		ArrayList<WeekdayInfo> list = new ArrayList<WeekdayInfo>();

		try {
			//カレンダIDの曜日別情報を取得
			Collection<CalendarWeekdayInfoLocal> ct = CalendarWeekdayInfoUtil.getLocalHome().findByCalendarId(id);

			if(ct != null){
				Iterator<CalendarWeekdayInfoLocal> itr = ct.iterator();
				while(itr.hasNext()){
					CalendarWeekdayInfoLocal cal = itr.next();

					WeekdayInfo info = new WeekdayInfo();
					info.setWeekday(cal.getWeekday());
					if (cal.getTime_from() != null) {
						info.setTimeFrom(cal.getTime_from().getTime());
					}
					if (cal.getTime_to() != null) {
						info.setTimeTo(cal.getTime_to().getTime());
					}
					info.setDescription(cal.getDescription());
					info.setRegUser(cal.getReg_user());
					if (cal.getReg_date() != null) {
						info.setRegDate(cal.getReg_date().getTime());
					}
					info.setUpdateUser(cal.getUpdate_user());
					if (cal.getUpdate_date() != null) {
						info.setUpdateDate(cal.getUpdate_date().getTime());
					}
					list.add(info);
				}
			}
		} catch (FinderException e) {
			m_log.warn("getWeekdayList()" + e.getMessage());
			throw new CalendarNotFound("SelectCalendar.getWeekdayList() find error : calendarId=" + id);
		} catch (NamingException e) {
			throw e;
		}

		return list;
	}

	/**
	 * カレンダ追加情報(例外実行情報) を取得します。
	 * 
	 * @param id
	 * @param from
	 * @param to
	 * @return カレンダ追加情報(例外実行情報)のリスト
	 * @throws NamingException
	 * @throws CalendarNotFound
	 */
	@SuppressWarnings("unchecked")
	public CalendarAddInfo getCalendarAdd(String id, Long from, Long to) throws NamingException, CalendarNotFound {
		CalendarAddInfo info = null;

		if(id != null && id.compareTo("") != 0){
			CalendarAddInfoLocal infoLocal = null;

			try{
				//カレンダ曜日別情報を取得
				infoLocal = CalendarAddInfoUtil.getLocalHome().findByPrimaryKey(
						new CalendarAddInfoPK(id, from==null?null:new Date(from), to==null?null:new Date(to)));
				info = new CalendarAddInfo();

				if(info != null){
					//時間帯(From)
					if (infoLocal.getTime_from() != null) {
						info.setTimeFrom(infoLocal.getTime_from().getTime());
					}
					//時間帯(To)
					if (infoLocal.getTime_to() != null) {
						info.setTimeTo(infoLocal.getTime_to().getTime());
					}
					//実行
					info.setExecuteFlag(Boolean.valueOf(ValidConstant.typeToBoolean(infoLocal.getExecute_flg().intValue())));
					//説明
					info.setDescription(infoLocal.getDescription());
					//登録者
					info.setRegUser(infoLocal.getReg_user());
					//登録日時
					if (infoLocal.getReg_date() != null) {
						info.setRegDate(infoLocal.getReg_date().getTime());
					}
					//更新者
					info.setUpdateUser(infoLocal.getUpdate_user());
					//更新日時
					if (infoLocal.getUpdate_date() != null) {
						info.setUpdateDate(infoLocal.getUpdate_date().getTime());
					}
				}
			}catch(FinderException e){
				throw new CalendarNotFound("SelectCalendar.getCalendarAdd() find error : calendarId=" + id);
			}catch(NamingException e){
				throw e;
			}
		}

		return info;
	}


	/**
	 * カレンダ追加情報(例外実行情報) 一覧を取得します。
	 * 
	 * @param id
	 * @return カレンダ追加情報(例外実行情報)のリスト
	 * @throws CalendarNotFound
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<CalendarAddInfo> getAddList(String id) throws CalendarNotFound, NamingException {
		ArrayList<CalendarAddInfo> list = new ArrayList<CalendarAddInfo>();

		try {
			//カレンダIDの追加情報を取得
			Collection<CalendarAddInfoLocal> ct = CalendarAddInfoUtil.getLocalHome().findByCalendarId(id);

			if(ct != null){
				Iterator<CalendarAddInfoLocal> itr = ct.iterator();
				while(itr.hasNext()){
					CalendarAddInfoLocal cal = itr.next();

					CalendarAddInfo info = new CalendarAddInfo();
					info.setExecuteFlag(Boolean.valueOf(RunConstant.typeToBoolean(cal.getExecute_flg().intValue())));
					if (cal.getTime_from() != null) {
						info.setTimeFrom(cal.getTime_from().getTime());
					}
					if (cal.getTime_to() != null) {
						info.setTimeTo(cal.getTime_to().getTime());
					}
					info.setDescription(cal.getDescription());
					info.setRegUser(cal.getReg_user());
					if (cal.getReg_date() != null) {
						info.setRegDate(cal.getReg_date().getTime());
					}
					info.setUpdateUser(cal.getUpdate_user());
					if (cal.getUpdate_date() != null) {
						info.setUpdateDate(cal.getUpdate_date().getTime());
					}
					list.add(info);
				}
			}
		} catch (FinderException e) {
			m_log.warn("getAddList()" + e.getMessage());
			throw new CalendarNotFound("SelectCalendar.getAddList() find error : calendarId=" + id);
		} catch (NamingException e) {
			throw e;
		}

		return list;
	}

	/**
	 * カレンダ追加情報(例外実行情報)一覧を取得します。
	 * 
	 * @param base
	 * @return カレンダ追加情報(例外実行情報)のリスト
	 * @throws FinderException
	 * @throws NamingException
	 * @throws CalendarNotFound
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<ScheduleInfo> getSchedule(Long base) throws CalendarNotFound, NamingException {
		ArrayList<ScheduleInfo> list = new ArrayList<ScheduleInfo>();

		//全カレンダを取得
		Collection<CalendarInfoLocal> ct = null;
		try {
			ct = CalendarInfoUtil.getLocalHome().findAll();
			if(ct != null){
				Iterator<CalendarInfoLocal> itr = ct.iterator();
				while(itr.hasNext()){
					CalendarInfoLocal cal = itr.next();

					ScheduleInfo info = new ScheduleInfo();
					info.setId(cal.getCalendar_id());

					for(int i = 0; i < ScheduleTableConstant.DAYS; i++){
						Calendar baseCalendar = Calendar.getInstance();
						if (base != null) {
							baseCalendar.setTime(new Date(base));
						}
						baseCalendar.add(Calendar.DAY_OF_MONTH, i);
						Date work = baseCalendar.getTime();
						if(isSchedule(cal.getCalendar_id(), work)){
							info.addDate(work.getTime());
						}else{
							info.addDate(null);
						}
					}
					list.add(info);
				}
			}
		} catch (FinderException e) {
			m_log.warn("getSchedule()" + e.getMessage());
			CalendarNotFound ce = new CalendarNotFound(e.getMessage(), e);
			throw ce;
		} catch (NamingException e) {
			throw e;
		}
		return list;
	}

	/**
	 * 実行可能かをチェックします。<BR>
	 * 
	 * 指定カレンダにて、指定した日時が実行可能かチェックし、Bool値を返します。
	 * 
	 * @param id
	 * @param checkTimestamp
	 * @return 指定した日時が実行可能か
	 * @throws NamingException
	 * @throws CalendarNotFound
	 */
	public Boolean isRun(String id, Long checkTimestamp) throws NamingException, CalendarNotFound {
		Calendar weekdayCal = null;
		Calendar checkCal = null;

		try {
			CalendarDTO config = getCalendarDTO(id);
			Date date = new Date(checkTimestamp);

			// 有効期間外の場合はfalse
			if (checkTimestamp < config.getBeginDate() || config.getEndDate() < checkTimestamp) {
				return false;
			}

			// 実行例外の時間帯に含まれている場合は、その実行種別を返す
			for (CalendarIrregularDTO dto : config.getIrregulars()) {
				if (dto.getBeginDate() <= checkTimestamp && checkTimestamp <= dto.getEndDate()) {
					return dto.getType() == CalendarIrregularType.ON ? true : false;
				}
			}

			// 実行規則の曜日に含まれているかどうか
			for (CalendarWeekdayDTO dto : config.getWeekdays()) {
				weekdayCal = Calendar.getInstance();
				weekdayCal.setTime(date);
				int weekday = weekdayCal.get(Calendar.DAY_OF_WEEK);

				// 実行規則で定義されている曜日の場合
				if (dto.getWeekday() == weekday) {
					checkCal = Calendar.getInstance();
					checkCal.set(Calendar.YEAR, 1970);
					checkCal.set(Calendar.MONTH, 0);
					checkCal.set(Calendar.DAY_OF_MONTH, 1);
					checkCal.set(Calendar.HOUR_OF_DAY, weekdayCal.get(Calendar.HOUR_OF_DAY));
					checkCal.set(Calendar.MINUTE, weekdayCal.get(Calendar.MINUTE));
					checkCal.set(Calendar.SECOND, weekdayCal.get(Calendar.SECOND));
					Time checkTime = new Time(checkCal.getTime().getTime());

					// 実行規則で定義されている時間帯内の場合
					if (dto.getBeginTime() <= checkTime.getTime() && checkTime.getTime() <= dto.getEndTime()) {
						return true;
					}
				}
			}
		} catch (CalendarInvalid e) {
			throw new EJBException(e.getMessage(), e);
		} catch (HinemosUnknown e) {
			throw new EJBException(e.getMessage(), e);
		}

		// 有効期間内だが、実行例外・実行規則の時間帯に含まれない場合
		return false;
	}

	/**
	 * 実行チェック（追加情報）
	 * 
	 * 指定カレンダの追加情報にて、指定した日時が実行可能かチェックし、Bool値を返す。
	 * 
	 * @param id
	 * @param checkTimestamp
	 * @return 指定した日時が実行可能か
	 * @throws FinderException
	 * @throws NamingException
	 */
	private Boolean isRunByAddInfo(CalendarAddInfoLocal add, Date checkTimestamp) {
		Boolean run = null;

		if(add != null){
			if(checkTimestamp.after(add.getTime_from()) &&
					checkTimestamp.before(add.getTime_to())){
				run = Boolean.valueOf(RunConstant.typeToBoolean(add.getExecute_flg().intValue()));
			}
		}

		return run;
	}

	/**
	 * 実行チェック（曜日別情報）
	 * 
	 * 指定カレンダの曜日別情報にて、指定した日時が実行可能かチェックし、Bool値を返す。
	 * 
	 * @param id
	 * @param checkTimestamp
	 * @return 指定した日時が実行可能か
	 * @throws FinderException
	 * @throws NamingException
	 */
	private Boolean isRunByWeekdayInfo(CalendarWeekdayInfoLocal weekday, Date checkTimestamp) {
		Boolean run = Boolean.valueOf(false);

		Calendar work = Calendar.getInstance();
		work.setTime(checkTimestamp);
		int dayOfWeek = work.get(Calendar.DAY_OF_WEEK);

		Calendar checkCalendar = Calendar.getInstance();
		checkCalendar.set(Calendar.YEAR, 1970);
		checkCalendar.set(Calendar.MONTH, 0);
		checkCalendar.set(Calendar.DAY_OF_MONTH, 1);
		checkCalendar.set(Calendar.HOUR_OF_DAY, work.get(Calendar.HOUR_OF_DAY));
		checkCalendar.set(Calendar.MINUTE, work.get(Calendar.MINUTE));
		checkCalendar.set(Calendar.SECOND, work.get(Calendar.SECOND));
		Time checkTime = new Time(checkCalendar.getTime().getTime());

		if(weekday != null && dayOfWeek == weekday.getWeekday().intValue()){
			if(checkTime.after(weekday.getTime_from()) &&
					checkTime.before(weekday.getTime_to())){
				run = Boolean.valueOf(true);
			}
		}

		return run;
	}

	/**
	 * 予定チェック
	 * 
	 * 指定カレンダにて、指定した日付に実行可能な時間帯があるかチェックし、Bool値を返す。
	 * 
	 * @param id
	 * @param checkTimestamp
	 * @return 指定した日付に実行可能な時間帯があるか
	 * @throws FinderException
	 * @throws NamingException
	 * @throws CalendarNotFound
	 */
	public Boolean isSchedule(String id, Date checkTimestamp) throws NamingException, CalendarNotFound {
		Boolean run = Boolean.valueOf(false);

		try {
			//カレンダ情報を取得
			CalendarInfoLocal cal = CalendarInfoUtil.getLocalHome().findByPrimaryKey(id);
			//有効期間（開始）時刻から、その日の00:00:00を取得
			Calendar validTimeFromCalendar = Calendar.getInstance();
			validTimeFromCalendar.setTime(cal.getValid_time_from());
			validTimeFromCalendar.set(Calendar.HOUR_OF_DAY, 0);
			validTimeFromCalendar.set(Calendar.MINUTE, 0);
			validTimeFromCalendar.set(Calendar.SECOND, 0);
			Date validTimeFrom = validTimeFromCalendar.getTime();

			//有効期間（終了）時刻から、その日の23:59:59を取得
			Calendar validTimeToCalendar = Calendar.getInstance();
			validTimeToCalendar.setTime(cal.getValid_time_to());
			validTimeToCalendar.set(Calendar.HOUR_OF_DAY, 23);
			validTimeToCalendar.set(Calendar.MINUTE, 59);
			validTimeToCalendar.set(Calendar.SECOND, 59);
			Date validTimeToDate = validTimeToCalendar.getTime();

			if(cal != null){
				if(checkTimestamp.after(validTimeFrom) &&
						checkTimestamp.before(validTimeToDate)){
					//追加情報を取得
					if(isScheduleByAddInfo(id, checkTimestamp)){
						run = Boolean.valueOf(true);
						return run;
					}

					//曜日別情報を取得
					if(isScheduleByWeekdayInfo(id, checkTimestamp)){
						run = Boolean.valueOf(true);
						return run;
					}
				}
			}
		} catch (FinderException e) {
			CalendarNotFound ce = new CalendarNotFound(e.getMessage(), e);
			ce.setCalendarId(id);
			throw ce;
		} catch (NamingException e) {
			throw e;
		}

		return run;
	}

	/**
	 * 予定チェック（追加情報）
	 * 
	 * 指定カレンダの追加情報にて、指定した日付に実行可能な時間帯があるかチェックし、Bool値を返す。
	 * 
	 * @param id
	 * @param checkTimestamp
	 * @return 指定した日付に実行可能な時間帯があるか
	 * @throws FinderException
	 * @throws NamingException
	 */
	private Boolean isScheduleByAddInfo(String id, Date checkTimestamp) throws FinderException, NamingException {
		Boolean run = Boolean.valueOf(false);

		try {
			if(id != null && id.length() > 0){
				// 大元のカレンダ情報（CalendarInfo）の有効期間を取得する
				CalendarInfoLocal calInfo= CalendarInfoUtil.getLocalHome().findByPrimaryKey(id);
				Date mstTimeFrom = calInfo.getValid_time_from(); // 有効期間（開始）
				Date mstTimeTo = calInfo.getValid_time_to(); // 有効期間（終了）

				Collection ct = CalendarAddInfoUtil.getLocalHome().selectSchdule(id, mstTimeFrom, mstTimeTo, checkTimestamp);
				if(ct.size() > 0){
					run = Boolean.valueOf(true);
				}
			}
		} catch (FinderException e) {
			throw new FinderException("SelectCalendar.isScheduleByAddInfo() find error : calendarId=" + id + ", checkTimestamp=" + checkTimestamp);
		} catch (NamingException e) {
			throw e;
		}

		return run;
	}

	/**
	 * 予定チェック（曜日別情報）
	 * 
	 * 指定カレンダの曜日別情報にて、指定した日付に実行可能な時間帯があるかチェックし、Bool値を返す。
	 * 
	 * @param id
	 * @param checkTimestamp
	 * @return 指定した日付に実行可能な時間帯があるか
	 * @throws FinderException
	 * @throws NamingException
	 */
	private Boolean isScheduleByWeekdayInfo(String id, Date checkTimestamp) throws FinderException, NamingException {
		Boolean run = Boolean.valueOf(false);

		// 大元のカレンダ情報（CalendarInfo）の有効期間を取得する
		CalendarInfoLocal calInfo= CalendarInfoUtil.getLocalHome().findByPrimaryKey(id);
		Date mstTimeFrom = calInfo.getValid_time_from(); // 有効期間（開始）
		Date mstTimeTo = calInfo.getValid_time_to(); // 有効期間（終了）

		Calendar work = Calendar.getInstance();
		work.setTime(checkTimestamp);
		int dayOfWeek = work.get(Calendar.DAY_OF_WEEK);

		try {
			if(id != null && id.length() > 0){
				Collection ct = CalendarWeekdayInfoUtil.getLocalHome().findByWeekday(id, Integer.valueOf(dayOfWeek));
				if(ct.size() > 0){

					Iterator itr = ct.iterator();
					while(itr.hasNext()){
						// 曜日情報を取得
						CalendarWeekdayInfoLocal weekday = (CalendarWeekdayInfoLocal)itr.next();

						long timeFrom = weekday.getTime_from()==null?0:weekday.getTime_from().getTime();
						long timeTo = weekday.getTime_to()==null?0:weekday.getTime_to().getTime();

						// 指定カレンダの追加情報にて、指定した日付に実行不可能な時間帯があるかチェック
						Calendar baseCalendar = Calendar.getInstance();
						baseCalendar.setTimeInMillis(timeFrom);
						work.set(Calendar.HOUR_OF_DAY, baseCalendar.get(Calendar.HOUR_OF_DAY));
						work.set(Calendar.MINUTE, baseCalendar.get(Calendar.MINUTE));
						work.set(Calendar.SECOND,baseCalendar.get(Calendar.SECOND));
						work.set(Calendar.MILLISECOND,0);
						Date start = work.getTime();

						baseCalendar.setTimeInMillis(timeTo);
						work.set(Calendar.HOUR_OF_DAY, baseCalendar.get(Calendar.HOUR_OF_DAY));
						work.set(Calendar.MINUTE, baseCalendar.get(Calendar.MINUTE));
						work.set(Calendar.SECOND, baseCalendar.get(Calendar.SECOND));
						work.set(Calendar.MILLISECOND,0);
						Date end = work.getTime();

						// 該当曜日の開始時刻が、カレンダの有効期間（終了）よりも未来の場合、
						// および、該当曜日の終了時刻が、カレンダの有効期間（開始）よりも昔の場合は、
						// 該当曜日設定自体が無効なため、false を返す
						if((mstTimeTo != null && start.getTime() > mstTimeTo.getTime()) ||
								(mstTimeFrom != null && end.getTime() < mstTimeFrom.getTime())){
							return run;
						}

						// 該当曜日の開始時刻が、カレンダの有効期間（開始）よりも昔の場合は、
						// カレンダの有効期間（開始）時刻からを対象とする
						if(mstTimeFrom != null && start.getTime() < mstTimeFrom.getTime()){
							start = mstTimeFrom;
						}

						// 該当曜日の終了時刻が、カレンダの有効期間（終了）よりも未来の場合は、
						// カレンダの有効期間（開始）時刻までを対象とする
						if(mstTimeTo != null && end.getTime() > mstTimeTo.getTime()){
							end = mstTimeTo;
						}

						Collection ctNo = CalendarAddInfoUtil.getLocalHome().selectNoSchdule(id, start, end);
						if(ctNo.size() <= 0){
							run = Boolean.valueOf(true);
							return run;
						}
					}
				}
			}
		} catch (FinderException e) {
			throw new FinderException("SelectCalendar.isScheduleByWeekdayInfo() find error : calendarId=" + id + ", checkTimestamp=" + checkTimestamp);
		} catch (NamingException e) {
			throw e;
		}

		return run;
	}

	/**
	 * カレンダID一覧を取得します。<BR>
	 * 
	 * @return カレンダID一覧
	 * @throws NamingException
	 * @throws CalendarNotFound
	 */
	public ArrayList<String> getCalendarIdList() throws NamingException, CalendarNotFound {
		ArrayList<String> list = new ArrayList<String>();

		try {
			//全カレンダを取得
			Collection<CalendarInfoLocal> ct = CalendarInfoUtil.getLocalHome().findAll();
			if(ct != null){
				Iterator<CalendarInfoLocal> itr = ct.iterator();
				while(itr.hasNext()){
					CalendarInfoLocal cal = itr.next();
					list.add(cal.getCalendar_id());
				}
			}
		} catch (FinderException e) {
			CalendarNotFound ce = new CalendarNotFound(e.getMessage(), e);
			throw ce;
		} catch (NamingException e) {
			throw e;
		}

		return list;
	}

	/**
	 * 指定したカレンダIDに該当する設定情報をDTOインスタンスとして返す
	 * @param calendarId カレンダID
	 * @return カレンダDTOインスタンス
	 * @throws CalendarInvalid 指定されたカレンダIDの格納情報が不適切な場合(不具合の可能性あり)
	 * @throws HinemosUnknown 内部で予期せぬエラーが発生した場合
	 * @throws CalendarNotFound 指定されたカレンダIDに該当する情報が格納されていない場合
	 */
	@SuppressWarnings("unchecked")
	public CalendarDTO getCalendarDTO(String calendarId) throws CalendarInvalid, HinemosUnknown, CalendarNotFound {
		// Local Variables
		CalendarDTO dto = null;
		CalendarWeekdayDTO weekday = null;
		ArrayList<CalendarWeekdayDTO> weekdays = null;
		CalendarIrregularDTO irregular = null;
		ArrayList<CalendarIrregularDTO> irregulars = null;

		CalendarInfoLocal calendarInfo = null;

		// MAIN
		if (calendarId == null) {
			return null;
		}
		try {
			calendarInfo = CalendarInfoUtil.getLocalHome().findByPrimaryKey(calendarId);

			weekdays = new ArrayList<CalendarWeekdayDTO>();
			for (CalendarWeekdayInfoLocal weekdayInfo : (Collection<CalendarWeekdayInfoLocal>)calendarInfo.getCalendarWeekdayInfo()) {
				weekday = new CalendarWeekdayDTO(weekdayInfo.getWeekday().intValue(),
						weekdayInfo.getTime_from()==null?null:weekdayInfo.getTime_from().getTime(),
								weekdayInfo.getTime_to()==null?null:weekdayInfo.getTime_to().getTime(),
										weekdayInfo.getDescription(),
										weekdayInfo.getReg_date()==null?null:weekdayInfo.getReg_date().getTime(),
												weekdayInfo.getReg_user(),
												weekdayInfo.getUpdate_date()==null?null:weekdayInfo.getUpdate_date().getTime(),
														weekdayInfo.getUpdate_user());

				weekdays.add(weekday);
			}

			irregulars = new ArrayList<CalendarIrregularDTO>();
			for (CalendarAddInfoLocal irregularInfo : (Collection<CalendarAddInfoLocal>)calendarInfo.getCalendarAddInfo()) {
				irregular = new CalendarIrregularDTO(
						irregularInfo.getTime_from()==null?null:irregularInfo.getTime_from().getTime(),
								irregularInfo.getTime_to()==null?null:irregularInfo.getTime_to().getTime(),
										(ValidConstant.typeToBoolean(irregularInfo.getExecute_flg()) ? CalendarIrregularType.ON : CalendarIrregularType.OFF),
										irregularInfo.getDescription(),
										irregularInfo.getReg_date()==null?null:irregularInfo.getReg_date().getTime(),
												irregularInfo.getReg_user(),
												irregularInfo.getUpdate_date()==null?null:irregularInfo.getUpdate_date().getTime(),
														irregularInfo.getUpdate_user());
				irregulars.add(irregular);
			}

			dto = new CalendarDTO(calendarId, calendarInfo.getCalendar_name(), calendarInfo.getDescription(),
					calendarInfo.getValid_time_from()==null?null:calendarInfo.getValid_time_from().getTime(),
							calendarInfo.getValid_time_to()==null?null:calendarInfo.getValid_time_to().getTime(),
									calendarInfo.getReg_date()==null?null:calendarInfo.getReg_date().getTime(),
											calendarInfo.getReg_user(),
											calendarInfo.getUpdate_date()==null?null:calendarInfo.getUpdate_date().getTime(),
													calendarInfo.getUpdate_user(),
													weekdays, irregulars);

		} catch (NamingException e) {
			m_log.warn("access failure to entity. (calendarId = " + calendarId + ")");
			throw new HinemosUnknown("failed to access entity. (calendarId = " + calendarId + ")", e);
		} catch (FinderException e) {
			m_log.warn("entity not found. (calendarId = " + calendarId + ")");
			throw new CalendarNotFound("entity bean not found. (calendarId = " + calendarId + ")", e);
		}

		return dto;
	}

}
