/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CalendarNotFound;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.JobInfoNotFound;
import com.clustercontrol.fault.JobInvalid;
import com.clustercontrol.fault.JobMasterNotFound;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.fault.UsedFacility;
import com.clustercontrol.fault.UserNotFound;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.ejb.session.CalendarControllerLocal;
import com.clustercontrol.calendar.ejb.session.CalendarControllerUtil;
import com.clustercontrol.commons.ejb.session.CheckFacility;
import com.clustercontrol.commons.scheduler.TriggerSchedulerException;
import com.clustercontrol.commons.util.EntityCacheUtil;
import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.bean.JobForwardFile;
import com.clustercontrol.jobmanagement.bean.JobHistoryFilter;
import com.clustercontrol.jobmanagement.bean.JobHistoryList;
import com.clustercontrol.jobmanagement.bean.JobInfo;
import com.clustercontrol.jobmanagement.bean.JobNodeDetail;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.JobTriggerInfo;
import com.clustercontrol.jobmanagement.bean.JobTriggerTypeConstant;
import com.clustercontrol.jobmanagement.bean.NodeOperationInfo;
import com.clustercontrol.jobmanagement.bean.JobSchedule;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobScheduleLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSchedulePK;
import com.clustercontrol.jobmanagement.ejb.entity.JobScheduleUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionUtil;
import com.clustercontrol.jobmanagement.factory.FullJob;
import com.clustercontrol.jobmanagement.factory.JobOperationProperty;
import com.clustercontrol.jobmanagement.factory.ModifyJob;
import com.clustercontrol.jobmanagement.factory.ModifySchedule;
import com.clustercontrol.jobmanagement.factory.SelectJob;
import com.clustercontrol.jobmanagement.factory.SelectSchedule;
import com.clustercontrol.jobmanagement.util.JobUtil;
import com.clustercontrol.jobmanagement.util.JobValidator;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ジョブ管理機能の管理を行う Session Bean クラス<BR>
 * <p>クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。<BR>
 * 
 * @ejb.bean name="JobController"
 *           jndi-name="JobController"
 *           type="Stateless"
 *           transaction-type="Container"
 *           view-type="local"
 * 
 * @ejb.transaction type="Required"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobCommandInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobCommandMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobEndInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobEndMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobFileInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobFileMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobNoticeInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobNoticeMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobParamInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobParamMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobRelationInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobRelationMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobSessionLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobSessionJobLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobSessionNodeLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartJobInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartJobMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartTimeInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartTimeMasterLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 */
public abstract class JobControllerBean implements javax.ejb.SessionBean, CheckFacility {
	/** ログ出力のインスタンス<BR> */
	private static Log m_log = LogFactory.getLog( JobControllerBean.class );

	/** コンテキスト情報<BR> */
	@SuppressWarnings("unused")
	private SessionContext m_context;

	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean がインスタンスプールに格納される際に行う処理を実装します。<BR>
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * セパレータ文字列を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @return セパレータ文字列
	 */
	public String getSeparator() {
		return SelectJob.SEPARATOR;
	}

	/**
	 * ジョブツリー情報を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param treeOnly true=ジョブ情報を含まない, false=ジョブ情報含む
	 * @param locale ロケール情報
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * @throws JobMasterNotFound
	 * @throws UserNotFound
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getJobTree(boolean, Locale)
	 */
	public JobTreeItem getJobTree(boolean treeOnly, Locale locale) throws NotifyNotFound, HinemosUnknown, JobMasterNotFound, UserNotFound {
		m_log.debug("getJobTree() : treeOnly=" + treeOnly + ", locale=" + locale);

		String loginUser = m_context.getCallerPrincipal().getName();

		JobTreeItem item = null;
		try {
			//ジョブツリーを取得
			SelectJob select = new SelectJob();
			item = select.getJobTree(treeOnly, locale, loginUser);
		} catch (NamingException e) {
			m_log.error("SelectJob.getJobTree() : " + e.getMessage(), e );
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (JobMasterNotFound e) {
			throw e;
		} catch (UserNotFound e) {
			throw e;
		}
		removeParent(item);
		return item;
	}

	/**
	 * webサービスでは双方向の参照を保持することができないので、
	 * 親方向への参照を消す。
	 * クライアント側で参照を付与する。
	 * @param jobTreeItem
	 */
	private void removeParent(JobTreeItem jobTreeItem) {
		jobTreeItem.setParent(null);
		for (JobTreeItem child : jobTreeItem.getChildren()) {
			removeParent(child);
		}
	}

	/**
	 * removeParentの逆操作
	 */
	private void setParent(JobTreeItem jobTreeItem) {
		for (JobTreeItem child : jobTreeItem.getChildren()) {
			child.setParent(jobTreeItem);
			setParent(child);
		}
	}

	/**
	 * ジョブツリー情報を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param jobInfo ジョブ情報(ツリー情報のみ)
	 * @return ジョブ情報(Full)
	 * @throws NotifyNotFound
	 * @throws JobMasterNotFound
	 * @throws UserNotFound
	 * @throws HinemosUnknown
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getJobFull(JobInfo jobInfo)
	 */
	public JobInfo getJobFull(JobInfo jobInfo) throws NotifyNotFound,
	JobMasterNotFound, UserNotFound, HinemosUnknown {
		m_log.debug("getJobFull() : jobunitId=" + jobInfo.getJobunitId() +
				", jobId=" + jobInfo.getId());

		JobInfo ret = null;
		try {
			//ジョブツリーを取得
			FullJob fullJob = new FullJob();
			ret = fullJob.getJobFull(jobInfo);
		} catch (NamingException e) {
			m_log.error("SelectJob.getJobTree() : " + e.getMessage(), e );
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.error("SelectJob.getJobTree() : " + e.getMessage(), e );
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return ret;
	}

	/**
	 * ジョブツリー情報を登録する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param item ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws HinemosUnknown
	 * @throws JobMasterNotFound
	 * @throws JobInvalid
	 * @throws JobMasterNotFound
	 * @see com.clustercontrol.jobmanagement.factory.ModifyJob#registerJob(JobTreeItem, String)
	 */
	public void registerJob(JobTreeItem item) throws HinemosUnknown, JobMasterNotFound, JobInvalid, InvalidSetting {
		m_log.debug("registerJob() start");

		Principal pri = m_context.getCallerPrincipal();
		String loginUser = pri.getName();

		////////////////
		// Check
		////////////////
		// ジョブユニットID重複チェック
		try {
			JobUtil.findDuplicateJobunitId(item);
		} catch (JobInvalid e) {
			m_log.warn("registerJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		}
		// ジョブユニット内ジョブID重複チェック
		try {
			JobUtil.findDuplicateJobId(item, true);
		} catch (JobInvalid e) {
			m_log.warn("registerJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		}

		////////////////
		// Main
		// 0. 内部DBのジョブマスタ取得：クライアントのジョブツリーと比較用に取得する
		// 1. ジョブユニットの削除：内部DBに存在 & クライアントのジョブツリーに存在しない、ジョブユニットを削除する
		// 2. ジョブユニットの登録：削除対象以外のジョブユニットをリプレイス(削除 & 再登録)する
		////////////////

		// Exceptionが発生したかどうかのフラグ
		boolean error = false;
		// 失敗したジョブユニットIDのリスト
		HashMap<String, String> resultList = new HashMap<String, String>();

		// topを取得
		JobTreeItem top = item.getChildren(0);
		JobTreeItem[] jobunits = top.getChildrenArray();

		// ジョブツリー最上位の一つ下は必ずジョブユニットになる。
		// クライアントから操作した場合は、JobInvalidは発生しないはず。
		for (JobTreeItem jobunit : jobunits) {
			int type = jobunit.getData().getType();
			if (type != JobConstant.TYPE_JOBUNIT){
				String jobId = jobunit.getData().getId();
				String topId = "";
				if (top.getData() != null) {
					topId = top.getData().getId();
				}
				String message = "There is not jobunit but job(" + type + ", " + jobId + ") " +
				"top=" + topId;
				m_log.error("registerJob() : " + message);
				throw new JobInvalid(message);
			}
		}

		////////////////
		// 0. 内部DBのジョブマスタ取得：クライアントのジョブツリーと比較用に取得する
		////////////////

		//登録済みジョブユニット一覧を取得
		//一覧取得に失敗した場合は全件失敗とするため、リターン
		Collection<JobMasterLocal> registeredJobunits;
		try {
			registeredJobunits = JobMasterUtil.getLocalHome().findByJobType(JobConstant.TYPE_JOBUNIT);
		} catch (FinderException e) {
			m_log.warn("registerJob() : findByJobType " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			JobMasterNotFound je = new JobMasterNotFound(e.getMessage(), e);
			je.setJobType(JobConstant.TYPE_JOBUNIT);
			throw je;
		} catch (NamingException e) {
			m_log.error("registerJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		////////////////
		// 1. ジョブユニットの削除：内部DBに存在 & クライアントのジョブツリーに存在しない、ジョブユニットを削除する
		//
		// 注意事項(マルチクライアントアクセス時)
		// ・「ジョブユニットの削除」時に他クライアントから管理ユーザを変更され権限がない場合は、削除を行わない
		////////////////

		//itemに含まれないものは、クライアントサイドで削除されたものとする
		for (JobMasterLocal mst : registeredJobunits) {

			boolean deleteFlg = true;
			for (JobTreeItem jobunit : jobunits) {
				if (jobunit.getData().getType() == JobConstant.TYPE_JOBUNIT){
					m_log.debug("registerJob() : registeredJobunitId=" + mst.getJobunit_id() + ", jobunitId=" + jobunit.getData().getJobunitId());
					if (mst.getJobunit_id().equals(jobunit.getData().getJobunitId())) {
						deleteFlg = false;
						break;
					}
				}
			}

			m_log.debug("registerJob() : registeredJobunitId=" + mst.getJobunit_id() + ", deleteFlg=" + deleteFlg);

			if (deleteFlg) {

				// 削除に失敗した場合、ジョブユニットIDを記録する
				String jobunitId = mst.getJobunit_id();
				try {
					m_log.info("registerJob() : jobunit " + mst.getJobunit_id() + " is delete");

					JobControllerUtil.getLocalHome().create().deleteJobunit(jobunitId);

					resultList.put(jobunitId, Messages.getString("message.job.75"));
				} catch (Exception e) {
					m_log.warn("registerJob() : deleteJobunit=" + jobunitId + ", " + e.getClass().getSimpleName() +
							", " + e.getMessage(), e);

					error = true;
					resultList.put(jobunitId, Messages.getString("message.job.76") + "[" + e.getMessage() + "]");
				}
			}
		}

		////////////////
		// 2. ジョブユニットの登録：削除対象以外のジョブユニットをリプレイス(削除 & 再登録)する
		//
		// 注意事項(マルチクライアントアクセス時)
		// ・「ジョブユニットの登録」時に他クライアントから管理ユーザを変更され権限がない場合は、リプレイスを行わない
		// ・「ジョブユニットの登録」時に他クライアントから管理ユーザを変更され権限を付与されている場合は、リプレイスを行わない（クライアント側にジョブツリーを持たないため）
		////////////////

		//ジョブユニット単位でregisterJobunitメソッドをコールする
		for (JobTreeItem jobunit : jobunits) {
			// 登録処理に失敗した場合、ジョブユニットIDを記録する
			if(JobUtil.isReferableJobunit(jobunit, loginUser)){
				m_log.debug("registerJob() : registerJobunit : jobunit " + jobunit.getData().getJobunitId());

				String jobunitId = jobunit.getData().getJobunitId();
				String jobId = jobunit.getData().getId();
				boolean replace = false;
				try{
					// jobunitIdとjobIdが既に存在するか
					JobValidator.validateJobId(jobunitId, jobId);
					replace = true;
					m_log.info("registerJob() : jobunit " + jobunitId + " is exist");
				} catch (InvalidSetting e) {
					m_log.info("registerJob() : jobunit " + jobunitId + " is new jobunit");
				}

				// 登録 or 置換
				if(replace){

					try {

						JobControllerUtil.getLocalHome().create().replaceJobunit(jobunit);

						resultList.put(jobunitId, Messages.getString("message.job.77"));
					} catch (Exception e) {
						m_log.error("registerJob() : registerJobunit=" + jobunit.getData().getJobunitId() + ", " + e.getClass().getSimpleName() +
								", " + e.getMessage(), e);
						error = true;
						resultList.put(jobunitId, Messages.getString("message.job.78") + "[" + e.getMessage() + "]");
					}
				}
				else{

					try {

						JobControllerUtil.getLocalHome().create().registerJobunit(jobunit);

						resultList.put(jobunitId, Messages.getString("message.job.79"));
					} catch (Exception e) {
						m_log.error("registerJob() : registerJobunit=" + jobunit.getData().getJobunitId() + ", " + e.getClass().getSimpleName() +
								", " + e.getMessage(), e);
						error = true;
						resultList.put(jobunitId, Messages.getString("message.job.80") + "[" + e.getMessage() + "]");
					}
				}
			}
			else {
				m_log.warn("registerJob() : registerJobunit : not replace jobunit " + jobunit.getData().getJobunitId()
						+ ", because this jobunit is not referable by " + loginUser);
			}


		}

		// 登録処理に失敗していたらExceptionをスローする
		if (error) {
			StringBuffer detailMsg = new StringBuffer();

			ArrayList<String> entries = new ArrayList<String>(resultList.keySet());
			Collections.sort(entries, new Comparator<String>() {
				@Override
				public int compare(String o1, String o2) {
					return o1.compareTo(o2);
				}
			});

			for(String key : entries){
				detailMsg.append(key + ":" + resultList.get(key) + "\n");
			}

			String[] args = {detailMsg.toString()};
			throw new JobInvalid(Messages.getString("message.job.67", args));
		}
	}

	/**
	 * ジョブユニット単位でジョブツリー情報を置換する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param jobunit
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * @throws JobMasterNotFound
	 * @throws UserNotFound
	 * @throws JobInvalid
	 */
	public void replaceJobunit(JobTreeItem jobunit) throws NotifyNotFound, HinemosUnknown, JobMasterNotFound, UserNotFound, JobInvalid, InvalidSetting {
		m_log.debug("replaceJobunit() : jobunitId = " + jobunit.getData().getJobunitId());

		// 編集されていないと何もしない
		if(!jobunit.getData().isEdit()){
			m_log.info("replaceJobunit() : jobunit " + jobunit.getData().getJobunitId() + " is not edited");
			return;
		}

		try{
			// 変更の場合は、ジョブはプロパティ値が一部不完全なので、DBより参照して完全なものとする。
			FullJob fullJob = new FullJob();
			fullJob.setJobTreeFull(jobunit);

			// 削除(validateなし)
			deleteJobunit(jobunit.getData().getJobunitId(), false);

			// 登録
			registerJobunit(jobunit);

			// 登録後のvalidate
			JobValidator.validateJobMaster();
		} catch (EJBException e) {
			m_log.error("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (JobInvalid e) {
			m_log.warn("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NotifyNotFound e) {
			m_log.warn("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (HinemosUnknown e) {
			m_log.warn("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (JobMasterNotFound e) {
			m_log.warn("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (InvalidSetting e) {
			m_log.warn("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (CreateException e) {
			m_log.error("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * ジョブユニット単位でジョブツリー情報を登録する。登録のみで置換ではないため、既に登録済みの定義と重複がある場合は例外が発生する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param jobunit
	 * @param validCheck
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * @throws JobMasterNotFound
	 * @throws InvalidSetting
	 * @throws JobInvalid
	 */
	public void registerJobunit(JobTreeItem jobunit) throws HinemosUnknown, JobMasterNotFound, JobInvalid, InvalidSetting {
		m_log.debug("registerJobunit() : jobunitId = " + jobunit.getData().getJobunitId());

		Principal pri = m_context.getCallerPrincipal();
		String loginUser = pri.getName();

		try {
			// 待ち条件チェック
			setParent(jobunit);
			JobUtil.checkWaitRule(jobunit);

			// Validate
			JobValidator.validateJobUnit(jobunit);

			// 登録
			ModifyJob modify = new ModifyJob();
			modify.registerJobunit(jobunit, loginUser);

			// キャッシュをリフレッシュする。
			// ジョブの詳細情報はPostgreSQLのCASCADEで消されているため、
			// PostgreSQLとentity-beanに差異が発生する。
			String[] jobDetail =
			{
					"jboss.j2ee:jndiName=JobCommandMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobEndMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobFileMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobParamMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobRelationMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobStartJobMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobStartMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobStartTimeMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobUserRelationMasterLocal,service=EJB"
			};
			EntityCacheUtil.clear(jobDetail);

		} catch (EJBException e) {
			m_log.error("registerJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("registerJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.error("registerJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (JobInvalid e) {
			m_log.error("registerJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (HinemosUnknown e) {
			m_log.error("registerJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (JobMasterNotFound e) {
			m_log.error("registerJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (InvalidSetting e) {
			m_log.warn("registerJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		}
	}

	/**
	 * ジョブユニット単位でジョブツリー情報を削除する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param jobunitId
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * @throws JobMasterNotFound
	 * @throws InvalidSetting
	 */
	public void deleteJobunit(String jobunitId) throws NotifyNotFound, HinemosUnknown, JobMasterNotFound, InvalidSetting {
		m_log.debug("deleteJobunit() : jobunitId = " + jobunitId);
		deleteJobunit(jobunitId, true);
	}

	/**
	 * ジョブユニット単位でジョブツリー情報を削除する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param jobunitId
	 * @param validCheck
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * @throws JobMasterNotFound
	 */
	public void deleteJobunit(String jobunitId, boolean refCheck) throws NotifyNotFound, HinemosUnknown, JobMasterNotFound, InvalidSetting {
		m_log.debug("deleteJobunit() : jobunitId = " + jobunitId + ", refCheck = " + refCheck);

		Principal pri = m_context.getCallerPrincipal();
		String loginUser = pri.getName();

		try {
			// 削除
			ModifyJob modify = new ModifyJob();
			modify.deleteJobunit(jobunitId, loginUser);

			// キャッシュをリフレッシュする。
			// ジョブの詳細情報はPostgreSQLのCASCADEで消されているため、
			// PostgreSQLとentity-beanに差異が発生する。
			String[] jobDetail =
			{
					"jboss.j2ee:jndiName=JobCommandMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobEndMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobFileMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobParamMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobRelationMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobStartJobMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobStartMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobStartTimeMasterLocal,service=EJB",
					"jboss.j2ee:jndiName=JobUserRelationMasterLocal,service=EJB"
			};
			EntityCacheUtil.clear(jobDetail);

			// 登録後のvalidate
			if(refCheck){
				JobValidator.validateJobMaster();
			}
		} catch (EJBException e) {
			m_log.error("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (RemoveException e) {
			m_log.error("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.error("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (JobInvalid e) {
			m_log.warn("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NotifyNotFound e) {
			m_log.warn("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (HinemosUnknown e) {
			m_log.warn("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (JobMasterNotFound e) {
			m_log.warn("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (InvalidSetting e) {
			m_log.warn("deleteJobunit() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
	}

	/**
	 * ジョブ操作開始用プロパティを返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @param locale ロケール情報
	 * @return ジョブ操作開始用プロパティ
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.JobOperationProperty#getStartProperty(String, String, String, String, Locale)
	 */
	public ArrayList<String> getAvailableStartOperation(String sessionId, String jobunitId, String jobId, String facilityId, Locale locale) {
		JobOperationProperty jobProperty = new JobOperationProperty();
		return jobProperty.getAvailableStartOperation(sessionId, jobunitId, jobId, facilityId, locale);
	}

	/**
	 * ジョブ操作停止用プロパティを返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @param locale ロケール情報
	 * @return ジョブ操作停止用プロパティ
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.JobOperationProperty#getStopProperty(String, String, String, String, Locale)
	 */
	public ArrayList<String> getAvailableStopOperation(String sessionId, String jobunitId,  String jobId, String facilityId, Locale locale) {
		JobOperationProperty jobProperty = new JobOperationProperty();
		return jobProperty.getAvailableStopOperation(sessionId, jobunitId, jobId, facilityId, locale);
	}

	/**
	 * ジョブを実行します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param jobunitId
	 * @param jobId ジョブID
	 * @param triggerInfo 実行契機情報
	 * @throws FacilityNotFound
	 * @throws HinemosUnknown
	 * @throws JobInfoNotFound
	 * @throws JobMasterNotFound
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#createJobInfo(String, String, NotifyRequestMessage, JobTriggerInfo)
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#runJob(String, String)
	 */
	public String runJob(String jobunitId, String jobId, JobTriggerInfo triggerInfo) throws FacilityNotFound, HinemosUnknown, JobMasterNotFound, JobInfoNotFound {
		m_log.debug("runJob() : jobId=" + jobId);

		// triggerInfo の実行契機種別が「TRIGGER_MANUAL」の場合、ユーザ名を登録
		if (triggerInfo.getTrigger_type() == JobTriggerTypeConstant.TYPE_MANUAL) {
			Principal pri = m_context.getCallerPrincipal();
			triggerInfo.setTrigger_info(pri.getName());
		}

		String sessionId = null;
		try {
			//ジョブ情報作成 新たなトランザクションを生成して実行するため、EJBメソッドで実行
			JobControllerLocal jobController = JobControllerUtil.getLocalHome().create();
			sessionId = jobController.createJobInfo(jobunitId, jobId, null, triggerInfo);
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {jobId};
			apllog.put("SYS", "003", args);
			m_log.error("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {jobId};
			apllog.put("SYS", "003", args);
			m_log.error("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (FacilityNotFound e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {jobId};
			apllog.put("SYS", "003", args);
			m_log.warn("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (HinemosUnknown e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {jobId};
			apllog.put("SYS", "003", args);
			m_log.warn("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (JobMasterNotFound e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {jobId};
			apllog.put("SYS", "003", args);
			m_log.warn("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (JobInfoNotFound e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {jobId};
			apllog.put("SYS", "003", args);
			m_log.warn("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		try {
			//ジョブ実行
			JobRunManagementLocal jobRunManagement = JobRunManagementUtil.getLocalHome().create();
			jobRunManagement.runJob(sessionId, jobunitId, jobId);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {sessionId, jobId};
			apllog.put("SYS", "002", args);
			m_log.error("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.error("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (HinemosUnknown e) {
			m_log.error("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (JobInfoNotFound e) {
			m_log.error("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		}
		return sessionId;
	}

	/**
	 * ジョブを実行します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param info ログ出力情報
	 * @param triggerInfo 実行契機情報
	 * @throws HinemosUnknown
	 * @throws JobInfoNotFound
	 * @throws JobMasterNotFound
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#createJobInfo(String, String, NotifyRequestMessage, JobTriggerInfo}
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#runJob(String, String)
	 */
	public String runJob(String jobunitId, String jobId, OutputBasicInfo info, JobTriggerInfo triggerInfo)
	throws FacilityNotFound, HinemosUnknown, JobInfoNotFound, JobMasterNotFound {
		m_log.debug("runJob() : jobId=" + jobId);

		if (info == null) {
			// triggerInfo の実行契機種別が「TRIGGER_MANUAL」の場合、ユーザ名を登録
			if (triggerInfo.getTrigger_type() == JobTriggerTypeConstant.TYPE_MANUAL) {
				Principal pri = m_context.getCallerPrincipal();
				m_log.info("runJob() : pri.getName()=" + pri.getName());
				triggerInfo.setTrigger_info(pri.getName());
			}
		}

		String sessionId = null;
		try {
			//ジョブ情報作成 新たなトランザクションを生成して実行するため、EJBメソッドで実行
			JobControllerLocal jobController = JobControllerUtil.getLocalHome().create();
			sessionId = jobController.createJobInfo(jobunitId, jobId, info, triggerInfo);
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {jobId};
			apllog.put("SYS", "003", args);
			m_log.error("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {jobId};
			apllog.put("SYS", "003", args);
			m_log.error("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (FacilityNotFound e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {jobId};
			apllog.put("SYS", "003", args);
			m_log.warn("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (HinemosUnknown e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {jobId};
			apllog.put("SYS", "003", args);
			m_log.warn("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (JobMasterNotFound e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {jobId};
			apllog.put("SYS", "003", args);
			m_log.warn("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (JobInfoNotFound e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {jobId};
			apllog.put("SYS", "003", args);
			m_log.warn("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		try {
			//ジョブ実行
			JobRunManagementLocal jobRunManagement = JobRunManagementUtil.getLocalHome().create();
			jobRunManagement.runJob(sessionId, jobunitId, jobId);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {sessionId, jobId};
			apllog.put("SYS", "002", args);
			m_log.error("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.error("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (HinemosUnknown e) {
			m_log.error("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (JobInfoNotFound e) {
			m_log.error("runJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		}
		return sessionId;
	}

	/**
	 * ジョブをスケジュール実行します。<BR>
	 * Quartzからスケジュール実行時に呼び出されます。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * 
	 * トランザクション開始はユーザが制御する。
	 * また、追加実装により、トランザクションの入れ子が予期せず生じることを避けるため、Neverを採用する。
	 * @ejb.transaction type="Never"
	 * 
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param calendarId カレンダID
	 * @param triggerInfo 実行契機情報
	 * @throws FacilityNotFound
	 * @throws HinemosUnknown
	 * @throws JobInfoNotFound
	 * @throws JobMasterNotFound
	 * @throws CalendarNotFound
	 * @throws JobMasterNotFound
	 * @see com.clustercontrol.calendar.ejb.session.CalendarControllerBean#isRun(String, Date)}
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#runJob(String, String, JobTriggerInfo)
	 */
	public void scheduleRunJob(String jobunitId, String jobId, String calendarId, JobTriggerInfo triggerInfo)
	throws FacilityNotFound, HinemosUnknown, JobInfoNotFound, CalendarNotFound, JobMasterNotFound {
		m_log.debug("runJob() : jobId=" + jobId + ", calendarId=" + calendarId);

		//カレンダをチェック
		try {
			boolean check = false;
			if(calendarId != null && calendarId.length() > 0){
				CalendarControllerLocal calendar =
					CalendarControllerUtil.getLocalHome().create();

				//カレンダによる実行可/不可のチェック
				if(calendar.isRun(calendarId, new Date().getTime()).booleanValue()){
					check = true;
				}
			}
			else{
				check = true;
			}

			if(!check)
				return;

			//ジョブ実行
			runJob(jobunitId, jobId, triggerInfo);
		} catch (CreateException e) {
			m_log.error("scheduleRunJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("scheduleRunJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (FacilityNotFound e) {
			m_log.warn("scheduleRunJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (CalendarNotFound e) {
			m_log.warn("scheduleRunJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (JobMasterNotFound e) {
			AplLogger apllog = new AplLogger(HinemosModuleConstant.JOB, "job");
			String[] args = { jobId, triggerInfo.getTrigger_info() };
			apllog.put("SYS", "016", args);
			m_log.warn("scheduleRunJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (HinemosUnknown e) {
			m_log.warn("scheduleRunJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (JobInfoNotFound e) {
			m_log.warn("scheduleRunJob() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
	}

	/**
	 * ジョブの実行用情報を作成します。<BR>
	 * 
	 * @ejb.interface-method
	 *     view-type="local"
	 * 
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param info ログ出力情報
	 * @param triggerInfo 実行契機情報
	 * @return セッションID
	 * @throws FacilityNotFound
	 * @throws HinemosUnknown
	 * @throws JobInfoNotFound
	 * @throws JobMasterNotFound
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionLocalHome#makeSession(java.lang.String, com.clustercontrol.notify.message.NotifyRequestMessage)
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionBean#ejbHomeMakeSession(String, NotifyRequestMessage)
	 */
	public String createJobInfo(String jobunitId, String jobId, OutputBasicInfo info, JobTriggerInfo triggerInfo)
	throws FacilityNotFound, HinemosUnknown, JobMasterNotFound, JobInfoNotFound {
		m_log.debug("createJobInfo() : jobId=" + jobId);

		try {
			return JobSessionUtil.getLocalHome().makeSession(jobunitId, jobId, info, triggerInfo);
		} catch (FacilityNotFound e) {
			m_log.warn("createJobInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (CreateException e) {
			m_log.error("createJobInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("createJobInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (JobMasterNotFound e) {
			m_log.error("createJobInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (JobInfoNotFound e) {
			m_log.error("createJobInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		}
	}

	/**
	 * ジョブ操作を行います。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param property ジョブ操作用プロパティ
	 * @throws HinemosUnknown
	 * @throws JobInfoNotFound
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#operationJob(Property)
	 */
	public void operationJob(NodeOperationInfo property) throws HinemosUnknown, JobInfoNotFound {
		m_log.debug("operationJob()");

		try {
			JobRunManagementLocal JobRunManagement = JobRunManagementUtil.getLocalHome().create();
			JobRunManagement.operationJob(property);
		} catch (CreateException e) {
			m_log.error("createJobInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("createJobInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (HinemosUnknown e) {
			m_log.error("createJobInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (JobInfoNotFound e) {
			m_log.error("createJobInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		}
	}

	/**
	 * ジョブ履歴一覧情報を返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param property 履歴フィルタ用プロパティ
	 * @param histories 表示履歴数
	 * @return ジョブ履歴一覧情報（Objectの2次元配列）
	 * @throws JobInfoNotFound
	 * @throws HinemosUnknown
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getHistoryList(Property, int)
	 */
	public JobHistoryList getJobHistoryList(JobHistoryFilter property, int histories) throws JobInfoNotFound, HinemosUnknown {
		m_log.debug("getHistoryList()");

		String userId = m_context.getCallerPrincipal().getName();

		JobHistoryList list = null;
		try {
			SelectJob select = new SelectJob();
			list = select.getHistoryList(userId, property, histories);
		} catch (NamingException e) {
			m_log.error("getHistoryList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (SQLException e) {
			m_log.error("getHistoryList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (JobInfoNotFound e) {
			throw e;
		}

		return list;
	}

	/**
	 * ジョブ詳細一覧情報を返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @return ジョブ詳細一覧情報（Objectの2次元配列）
	 * @throws JobInfoNotFound
	 * @throws HinemosUnknown
	 * @throws JobMasterNotFound
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getDetailList(String)
	 */
	public JobTreeItem getJobDetailList(String sessionId) throws JobInfoNotFound, HinemosUnknown, JobMasterNotFound {
		m_log.debug("getDetailList() : sessionId=" + sessionId);

		JobTreeItem item = null;
		try {
			SelectJob select = new SelectJob();
			item = select.getDetailList(sessionId);
		} catch (NamingException e) {
			m_log.error("getDetailList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		removeParent(item);
		return item;
	}

	/**
	 * ノード詳細一覧情報を返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param locale ロケール情報
	 * @return ノード詳細一覧情報（Objectの2次元配列）
	 * @throws JobInfoNotFound
	 * @throws HinemosUnknown
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getNodeDetailList(String, String, String, Locale)
	 */
	public ArrayList<JobNodeDetail> getNodeDetailList(String sessionId, String jobunitId, String jobId, Locale locale) throws JobInfoNotFound, HinemosUnknown {
		m_log.debug("getNodeDetailList() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId);

		ArrayList<JobNodeDetail> list;
		try {
			SelectJob select = new SelectJob();
			list = select.getNodeDetailList(sessionId, jobunitId, jobId, locale);
		} catch (NamingException e) {
			m_log.error("getNodeDetailList() : " + e.getMessage(), e );
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.error("getNodeDetailList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (JobInfoNotFound e) {
			throw e;
		}

		return list;
	}

	/**
	 * ファイル転送一覧情報を返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @return ファイル転送一覧情報（Objectの2次元配列）
	 * @throws JobInfoNotFound
	 * @throws HinemosUnknown
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getForwardFileList(String, String)
	 */
	public ArrayList<JobForwardFile> getForwardFileList(String sessionId, String jobunitId, String jobId) throws JobInfoNotFound, HinemosUnknown {
		m_log.debug("getForwardFileList() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId);

		ArrayList<JobForwardFile> list = null;
		try {
			SelectJob select = new SelectJob();
			list = select.getForwardFileList(sessionId, jobunitId, jobId);
		} catch (NamingException e) {
			m_log.error("getForwardFileList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.error("getForwardFileList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (JobInfoNotFound e) {
			throw e;
		}

		return list;
	}

	/**
	 * スケジュール情報を登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param info スケジュール情報
	 * @throws HinemosUnknown
	 * @throws InvalidSetting
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#addSchedule(JobSchedule, String)
	 */
	public void addSchedule(JobSchedule info) throws HinemosUnknown, InvalidSetting {
		m_log.debug("addSchedule() : scheduleId=" + info.getId() + ",  Schedule = " + info.getSchedule()
				+ ", ScheduleType = " + info.getSchedule().getType() + ",  ScheduleDate = " + info.getSchedule());

		m_log.debug("addSchedule() : CreateTime=" + info.getCreateTime() + ",  UpdateTime = " + info.getUpdateTime());

		// 新規登録ユーザ、最終変更ユーザを設定
		Principal pri = m_context.getCallerPrincipal();
		info.setUpdateUser(pri.getName());
		info.setCreateUser(pri.getName());

		// DBにスケジュール情報を保存
		Date now = new Date();  // 現在時刻を、CreateTimeとUpdateTimeに設定する。
		try {
			// 入力チェック
			JobValidator.validateJobSchedule(info);

			// IDの重複チェック
			String id = info.getId();
			JobSchedulePK pk = new JobSchedulePK(id);
			try {
				JobScheduleUtil.getLocalHome().findByPrimaryKey(pk);
				String[] args = {id};
				throw new InvalidSetting(Messages.getString("message.job.83", args));
			} catch (FinderException e) {
			} catch (NamingException e) {
				m_log.warn("validateJobSchedule " + e.getMessage());
			}

			JobScheduleUtil.getLocalHome().create(
					info.getId(),
					info.getName(),
					info.getJobunitId(),
					info.getJobId(),
					info.getCalendarId(),
					info.getSchedule().getType(),
					info.getSchedule().getMonth(),
					info.getSchedule().getDay(),
					info.getSchedule().getWeek(),
					info.getSchedule().getHour(),
					info.getSchedule().getMinute(),
					info.getValid(),
					now,
					now,
					info.getCreateUser(),
					info.getUpdateUser());
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {info.getId(), info.getJobId()};
			apllog.put("SYS", "004", args);
			m_log.warn("JobSchedule.create() : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {info.getId(), info.getJobId()};
			apllog.put("SYS", "004", args);
			m_log.warn("JobSchedule.create() : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (InvalidSetting e) {
			m_context.setRollbackOnly();
			throw e;
		}


		ModifySchedule modify = new ModifySchedule();
		try {
			modify.addSchedule(info, pri.getName());
		} catch (TriggerSchedulerException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {info.getId(), info.getJobId()};
			apllog.put("SYS", "004", args);
			m_log.warn("addSchedule() : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * スケジュール情報を変更します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param info スケジュール情報
	 * @throws HinemosUnknown
	 * @throws InvalidSetting
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#modifySchedule(JobSchedule, String)
	 */
	public void modifySchedule(JobSchedule info) throws HinemosUnknown, InvalidSetting {
		m_log.debug("modifySchedule() : scheduleId=" + info.getId());

		// 最終変更ユーザを設定
		Principal pri = m_context.getCallerPrincipal();
		info.setUpdateUser(pri.getName());

		// DBにスケジュール情報を保存
		try {
			// 入力チェック
			JobValidator.validateJobSchedule(info);

			JobScheduleLocal bean  = JobScheduleUtil.getLocalHome().findByPrimaryKey(new JobSchedulePK(info.getId()));

			bean.setSchedule_id(info.getId());
			bean.setSchedule_name(info.getName());
			bean.setJobunit_id(info.getJobunitId());
			bean.setJob_id(info.getJobId());
			bean.setCalendar_id(info.getCalendarId());
			bean.setSchedule_type(info.getSchedule().getType());
			bean.setMonth(info.getSchedule().getMonth());
			bean.setDay(info.getSchedule().getDay());
			bean.setWeek(info.getSchedule().getWeek());
			bean.setHour(info.getSchedule().getHour());
			bean.setMinute(info.getSchedule().getMinute());
			bean.setValid_flg(info.getValid());
			if (info.getCreateTime() != null) {
				bean.setReg_date(new Date(info.getCreateTime()));
			}
			bean.setReg_user(info.getCreateUser());
			if (info.getUpdateTime() != null) {
				bean.setUpdate_date(new Date(info.getUpdateTime()));
			}
			bean.setUpdate_user(info.getUpdateUser());
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {info.getId(), info.getJobId()};
			apllog.put("SYS", "004", args);
			m_log.warn("JobSchedule.findByPrimaryKey() : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {info.getId(), info.getJobId()};
			apllog.put("SYS", "004", args);
			m_log.warn("JobSchedule.findByPrimaryKey() : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (InvalidSetting e) {
			m_context.setRollbackOnly();
			throw e;
		}

		ModifySchedule modify = new ModifySchedule();
		try {
			modify.addSchedule(info, pri.getName());
		} catch (TriggerSchedulerException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {info.getId(), info.getJobId()};
			apllog.put("SYS", "004", args);
			m_log.warn("modifySchedule() : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * スケジュール情報を削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param scheduleId スケジュールID
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#deleteSchedule(String)
	 */
	public void deleteSchedule(String scheduleId) throws HinemosUnknown {
		m_log.debug("deleteSchedule() : scheduleId=" + scheduleId);

		// DBのスケジュール情報を削除
		try {
			JobScheduleUtil.getLocalHome().findByPrimaryKey(new JobSchedulePK(scheduleId)).remove();
		} catch (EJBException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {scheduleId};
			apllog.put("SYS", "005", args);
			m_log.warn("deleteSchedule() : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (RemoveException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {scheduleId};
			apllog.put("SYS", "005", args);
			m_log.warn("deleteSchedule() : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {scheduleId};
			apllog.put("SYS", "005", args);
			m_log.warn("deleteSchedule() : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {scheduleId};
			apllog.put("SYS", "005", args);
			m_log.warn("deleteSchedule() : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}

		ModifySchedule modify = new ModifySchedule();
		try {
			modify.deleteSchedule(scheduleId);
		} catch (TriggerSchedulerException e) {
			AplLogger apllog = new AplLogger("JOB", "job");
			String[] args = {scheduleId};
			apllog.put("SYS", "005", args);
			m_log.warn("deleteSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * スケジュール一覧情報を返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return スケジュール一覧情報
	 * @throws JobMasterNotFound
	 * @throws HinemosUnknown
	 * @see com.clustercontrol.jobmanagement.factory.SelectSchedule#getScheduleList()
	 */
	public ArrayList<JobSchedule> getScheduleList() throws JobMasterNotFound, HinemosUnknown  {
		m_log.debug("getScheduleList()");

		ArrayList<JobSchedule> list;
		try {
			SelectSchedule select = new SelectSchedule();
			list = select.getScheduleList(m_context.getCallerPrincipal().getName());
		} catch (NamingException e) {
			m_log.error("getScheduleList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		return list;
	}

	/**
	 * 指定されたジョブスケジュールIDの有効/無効を変更します
	 * 
	 * @ejb.interface-method
	 * 
	 * @param scheduleId
	 * @param validFlag
	 * @throws HinemosUnknown
	 * @throws InvalidSetting
	 */
	public void setScheduleStatus(String scheduleId, boolean validFlag) throws HinemosUnknown, InvalidSetting{
		m_log.debug("setScheduleStatus() scheduleId = " + scheduleId + ", validFlag = " + validFlag);
		// null check
		if(scheduleId == null || "".equals(scheduleId)){
			throw new HinemosUnknown("target scheduleId is null or empty.");
		}

		try{
			SelectSchedule select = new SelectSchedule();
			JobSchedule info = select.getJobSchedule(scheduleId);
			if(validFlag){
				info.setValid(ValidConstant.TYPE_VALID);
			}
			else{
				info.setValid(ValidConstant.TYPE_INVALID);
			}
			modifySchedule(info);
		} catch (HinemosUnknown e) {
			m_log.warn("setScheduleStatus() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (JobMasterNotFound e) {
			m_log.warn("setScheduleStatus() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.warn("setScheduleStatus() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (FinderException e) {
			m_log.warn("setScheduleStatus() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (InvalidSetting e) {
			m_context.setRollbackOnly();
			throw e;
		}
	}

	/**
	 * カレンダIDリストを返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return カレンダIDのリスト
	 * @throws HinemosUnknown
	 * @throws CalendarNotFound
	 * @throws
	 * 
	 * @see com.clustercontrol.calendar.ejb.session.CalendarControllerBean#getCalendarIdList()
	 */
	public ArrayList<String> getCalendarIdList() throws HinemosUnknown, CalendarNotFound {
		m_log.debug("getCalendarIdList()");

		ArrayList<String> list = null;
		try {
			CalendarControllerLocal calendar =
				CalendarControllerUtil.getLocalHome().create();

			list = calendar.getCalendarIdList();
		} catch (CreateException e) {
			m_log.error("getCalendarIdList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getCalendarIdList() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		return list;
	}

	/**
	 * セッションジョブ情報を返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws JobInfoNotFound
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getSessionJobInfo(String, String, String)
	 */
	public JobTreeItem getSessionJobInfo(String sessionId, String jobunitId, String jobId) throws HinemosUnknown, NotifyNotFound, JobInfoNotFound {
		m_log.debug("getSessionJobInfo() : sessionId=" + sessionId + ", jobId=" + jobId);

		JobTreeItem item = null;
		try {
			SelectJob select = new SelectJob();
			item = select.getSessionJobInfo(sessionId, jobunitId, jobId);
			item.getData().setPropertyFull(true);
		} catch (NamingException e) {
			m_log.error("getSessionJobInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.error("getSessionJobInfo() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		return item;
	}

	/**
	 * ファシリティIDが使用されているかチェックします。<BR>
	 * <P>
	 * 使用されていたら、UsedFacility がスローされる。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId ファシリティ
	 * @throws UsedFacility
	 * 
	 * @see com.clustercontrol.commons.ejb.session.CheckFacility#isUseFacilityId(java.lang.String)
	 * @see com.clustercontrol.bean.PluginConstant;
	 */
	@Override
	public void isUseFacilityId(String facilityId) throws UsedFacility {
		try {
			Collection<JobCommandInfoLocal> ct = null;

			// ファシリティIDが使用されているジョブコマンドマスタを取得する。
			ct = JobCommandMasterUtil.getLocalHome().findByFacilityId(facilityId);
			if(ct != null && ct.size() > 0) {
				throw new UsedFacility(PluginConstant.TYPE_JOBMANAGEMENT);
			}

		} catch (FinderException e) {
			m_log.warn("isUseFacilityId() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
		} catch (NamingException e) {
			m_log.warn("isUseFacilityId() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
		}
	}
}
