/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.run.factory;

import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.custom.bean.CustomCheckInfo;
import com.clustercontrol.http.bean.HttpCheckInfo;
import com.clustercontrol.logfile.bean.LogfileCheckInfo;
import com.clustercontrol.monitor.bean.MonitorFilterInfo;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorNumericValueInfo;
import com.clustercontrol.monitor.run.bean.MonitorStringValueInfo;
import com.clustercontrol.monitor.run.bean.MonitorTrapValueInfo;
import com.clustercontrol.monitor.run.bean.MonitorTruthValueInfo;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.monitor.run.ejb.entity.MonitorNumericValueInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorStringValueInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorTrapValueInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorTruthValueInfoLocal;
import com.clustercontrol.notify.ejb.session.NotifyControllerLocal;
import com.clustercontrol.notify.ejb.session.NotifyControllerUtil;
import com.clustercontrol.performance.monitor.bean.PerfCheckInfo;
import com.clustercontrol.ping.bean.PingCheckInfo;
import com.clustercontrol.port.bean.PortCheckInfo;
import com.clustercontrol.process.bean.ProcessCheckInfo;
import com.clustercontrol.repository.bean.FacilityConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.repository.factory.FacilitySelector;
import com.clustercontrol.snmp.bean.SnmpCheckInfo;
import com.clustercontrol.snmptrap.bean.TrapCheckInfo;
import com.clustercontrol.sql.bean.SqlCheckInfo;
import com.clustercontrol.util.apllog.AplLogger;
import com.clustercontrol.winservice.bean.WinServiceCheckInfo;

/**
 * 監視情報を検索する抽象クラス<BR>
 * <p>
 * 監視種別（真偽値，数値，文字列）の各クラスで継承してください。
 *
 * @version 4.0.0
 * @since 2.0.0
 */
public class SelectMonitor {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( SelectMonitor.class );

	/** 監視情報のローカルコンポーネント。 */
	protected MonitorInfoLocal m_monitor;

	/** 監視対象ID。 */
	protected String m_monitorTypeId;

	/** 監視項目ID。 */
	protected String m_monitorId;

	/**
	 * 引数で指定された監視情報を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定された監視情報を取得します。</li>
	 * <li>Quartzより、有効/無効を取得します。</li>
	 * <li>監視情報より判定情報を取得します。各監視種別（真偽値，数値，文字列）のサブクラスで実装します（{@link #getJudgementInfo()}）。</li>
	 * <li>監視情報よりチェック条件を取得します。各監視管理のサブクラスで実装します（{@link #getCheckInfo()}）。</li>
	 * </ol>
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @param monitorId 監視項目ID
	 * @return 監視情報
	 * @throws CreateException
	 * @throws NamingException
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws MonitorNotFound
	 * @see com.clustercontrol.monitor.run.ejb.entity.MonitorInfoBean
	 * @see com.clustercontrol.monitor.run.factory.SelectSchedule#getValid(String, String)
	 * @see #getJudgementInfo()
	 * @see #getCheckInfo()
	 */

	public MonitorInfo getMonitor(String monitorTypeId, String monitorId)
	throws CreateException, NamingException, NotifyNotFound, HinemosUnknown, MonitorNotFound {

		m_monitorTypeId = monitorTypeId;
		m_monitorId = monitorId;

		MonitorInfo bean = null;
		try
		{
			// 監視情報を取得
			m_monitor = MonitorInfoUtil.getLocalHome().findByPrimaryKey(m_monitorId);

			// スコープの取得
			RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
			String facilityPath = repository.getFacilityPath(m_monitor.getFacilityId(), null);

			NotifyControllerLocal nc = NotifyControllerUtil.getLocalHome().create();
			Collection notifyId = nc.getNotifyRelation(m_monitor.getNotifyGroupId());

			bean = new MonitorInfo(
					m_monitor.getApplication(),
					m_monitor.getCalendarId(),
					m_monitor.getDescription(),
					facilityPath,
					m_monitor.getFacilityId(),
					m_monitor.getFailurePriority(),
					m_monitorId,
					m_monitorTypeId,
					m_monitor.getMonitorType(),
					m_monitor.getRegDate()==null?null:m_monitor.getRegDate().getTime(),
							m_monitor.getRegUser(),
							m_monitor.getRunInterval(),
							m_monitor.getNotifyGroupId(),
							notifyId,
							m_monitor.getUpdateDate()==null?null:m_monitor.getUpdateDate().getTime(),
									m_monitor.getUpdateUser(),
									m_monitor.getMonitorFlg(),
									m_monitor.getCollectorFlg(),
									m_monitor.getItemName(),
									m_monitor.getMeasure(),
									getNumericValueInfo(),
									getStringValueInfo(),
									getTruthValueInfo(),
									getTrapValueInfo(),
									getHttpCheckInfo(),
									getPerfCheckInfo(),
									getPingCheckInfo(),
									getPortCheckInfo(),
									getProcessCheckInfo(),
									getSnmpCheckInfo(),
									getSqlCheckInfo(),
									getTrapCheckInfo(),
									getCommandCheckInfo(),
									getLogfileCheckInfo(),
									getWinServiceCheckInfo()
			);
		} catch (CreateException e) {
			outputLog(e, "getMonitor()", "010");
			throw e;
		} catch (FinderException e) {
			outputLog(e, "getMonitor()", "010");
			throw new MonitorNotFound(e.getMessage(), e);
		} catch (NamingException e) {
			outputLog(e, "getMonitor()", "010");
			throw e;
		}

		return bean;
	}

	/**
	 * 判定情報を返します。
	 * <p>
	 * @return 判定情報（{@link com.clustercontrol.monitor.run.bean.MonitorJudgementInfo}のリスト）
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected ArrayList<MonitorNumericValueInfo> getNumericValueInfo() throws FinderException, NamingException {
		// 数値監視判定情報を取得
		Collection<MonitorNumericValueInfoLocal> ct = m_monitor.getMonitorNumericValueInfo();
		ArrayList<MonitorNumericValueInfo> valueList = new ArrayList<MonitorNumericValueInfo>();

		for (MonitorNumericValueInfoLocal local : ct) {
			MonitorNumericValueInfo value = new MonitorNumericValueInfo(
					local.getMessage(),
					local.getMessageId(),
					local.getMonitorId(),
					local.getPriority(),
					local.getThresholdLowerLimit(),
					local.getThresholdUpperLimit());
			valueList.add(value);
		}
		return valueList;
	}

	/**
	 * 判定情報を返します。
	 * <p>
	 * @return 判定情報（{@link com.clustercontrol.monitor.run.bean.MonitorJudgementInfo}のリスト）
	 * @throws FinderException
	 * @throws NamingException
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 */
	protected ArrayList<MonitorStringValueInfo> getStringValueInfo() throws FinderException, NamingException, HinemosUnknown, NotifyNotFound {

		// 文字列監視判定情報を取得
		Collection<MonitorStringValueInfoLocal> ct = m_monitor.getMonitorStringValueInfo();
		ArrayList<MonitorStringValueInfo> valueList = new ArrayList<MonitorStringValueInfo>();

		for (MonitorStringValueInfoLocal local : ct) {
			MonitorStringValueInfo value = new MonitorStringValueInfo(
					local.getMonitorId(),
					local.getOrderNo().intValue(),
					local.getDescription(),
					local.getProcessType().intValue(),
					local.getPattern(),
					local.getPriority().intValue(),
					local.getMessage(),
					ValidConstant.typeToBoolean(local.getCaseSensitivityFlg()),
					ValidConstant.typeToBoolean(local.getValidFlg().intValue())
			);
			valueList.add(value);
		}
		return valueList;
	}

	/**
	 * 判定情報を返します。
	 * <p>
	 * @return 判定情報（{@link com.clustercontrol.monitor.run.bean.MonitorJudgementInfo}のリスト）
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected ArrayList<MonitorTruthValueInfo> getTruthValueInfo() throws FinderException, NamingException {
		// 真偽値監視判定情報を取得
		Collection<MonitorTruthValueInfoLocal> ct = m_monitor.getMonitorTruthValueInfo();

		ArrayList<MonitorTruthValueInfo> valueList = new ArrayList<MonitorTruthValueInfo>();

		for (MonitorTruthValueInfoLocal local : ct) {
			MonitorTruthValueInfo value = new MonitorTruthValueInfo(
					local.getMessage(),
					local.getMessageId(),
					local.getMonitorId(),
					local.getPriority(),
					local.getTruthValue());
			valueList.add(value);
		}
		return valueList;
	}

	/**
	 * 判定情報を返します。
	 * <p>
	 * @return 判定情報（{@link com.clustercontrol.monitor.run.bean.MonitorJudgementInfo}のリスト）
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected ArrayList<MonitorTrapValueInfo> getTrapValueInfo() throws FinderException, NamingException {
		m_log.debug("getTrapValueInfo() : start");

		// 真偽値監視判定情報を取得
		Collection<MonitorTrapValueInfoLocal> ct = m_monitor.getMonitorTrapValueInfo();

		ArrayList<MonitorTrapValueInfo> valueList = new ArrayList<MonitorTrapValueInfo>();

		for (MonitorTrapValueInfoLocal local : ct) {
			m_log.debug("getTrapValueInfo() : add Uei = " + local.getUei());

			MonitorTrapValueInfo value = new MonitorTrapValueInfo(
					local.getMonitorId(),
					local.getMib(),
					local.getTrapOid(),
					local.getGenericId(),
					local.getSpecificId(),
					local.getUei(),
					YesNoConstant.typeToBoolean(local.getValidFlg()),
					local.getPriority(),
					local.getLogmsg(),
					local.getDescr());
			valueList.add(value);
		}

		m_log.debug("getTrapValueInfo() : end");
		return valueList;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected HttpCheckInfo getHttpCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected PerfCheckInfo getPerfCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected PingCheckInfo getPingCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected PortCheckInfo getPortCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected ProcessCheckInfo getProcessCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected SnmpCheckInfo getSnmpCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected SqlCheckInfo getSqlCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected TrapCheckInfo getTrapCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報(コマンド監視)を返す。
	 * @return チェック条件情報(コマンド監視)
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected CustomCheckInfo getCommandCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報(ログファイル監視)を返す。
	 * @return チェック条件情報(ログファイル監視)
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected LogfileCheckInfo getLogfileCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * チェック条件情報(Windowsサービス監視)を返す。
	 * @return チェック条件情報(Windowsサービス監視)
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected WinServiceCheckInfo getWinServiceCheckInfo() throws FinderException, NamingException {
		return null;
	}

	/**
	 * 監視情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定された監視対象の監視情報を取得します。</li>
	 * <li>１監視情報をテーブルのカラム順（{@link com.clustercontrol.monitor.run.bean.MonitorTabelDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１監視情報を保持するリストを、監視情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>監視情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 監視情報1 {カラム1の値, カラム2の値, … }, 監視情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @return 監視情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws NamingException
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws NotifyNotFound
	 * @see com.clustercontrol.monitor.run.bean.MonitorTabelDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList<MonitorInfo> getMonitorList(String monitorTypeId) throws CreateException, NamingException, HinemosUnknown, MonitorNotFound, NotifyNotFound {

		ArrayList<MonitorInfo> list = new ArrayList<MonitorInfo>();
		try
		{
			// 監視情報一覧を取得
			Collection<MonitorInfoLocal> ct =
				MonitorInfoUtil.getLocalHome().findByMonitorTypeId(monitorTypeId);

			for (MonitorInfoLocal info : ct) {
				list.add(getMonitor(info.getMonitorTypeId(), info.getMonitorId()));
			}
		} catch (CreateException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (FinderException e) {
			outputLog(e, "getMonitorList()", "011");
			throw new MonitorNotFound(e.getMessage(), e);
		} catch (NamingException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (HinemosUnknown e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (NotifyNotFound e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		}
		return list;
	}

	/**
	 * 監視情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>全ての監視対象の監視情報を取得します。</li>
	 * <li>１監視情報をテーブルのカラム順（{@link com.clustercontrol.monitor.run.bean.MonitorTabelDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１監視情報を保持するリストを、監視情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>監視情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 監視情報1 {カラム1の値, カラム2の値, … }, 監視情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 監視情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws NamingException
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws NotifyNotFound
	 * @see com.clustercontrol.monitor.run.bean.MonitorTabelDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList<MonitorInfo> getMonitorList() throws CreateException, NamingException, HinemosUnknown, MonitorNotFound, NotifyNotFound {
		m_log.debug("getMonitorList() : start");

		ArrayList<MonitorInfo> list = new ArrayList<MonitorInfo>();
		try
		{
			// 監視情報一覧を取得
			Collection<MonitorInfoLocal> ct =
				MonitorInfoUtil.getLocalHome().findAll();

			for (MonitorInfoLocal info : ct) {
				list.add(getMonitor(info.getMonitorTypeId(), info.getMonitorId()));

				// for debug
				if(m_log.isDebugEnabled()){
					m_log.debug("getMonitorList() : " +
							"monitorId = " + info.getMonitorId() +
							", monitorTypeId = " + info.getMonitorTypeId() +
							", monitorType = " + info.getMonitorType() +
							", description = " + info.getDescription() +
							", facilityId = " + info.getFacilityId() +
							", runInterval = " + info.getRunInterval() +
							", calendarId = " + info.getCalendarId() +
							", failurePriority = " + info.getFailurePriority() +
							", notifyGroupId = " + info.getNotifyGroupId() +
							", application = " + info.getApplication() +
							", monitorFlg = " + info.getMonitorFlg() +
							", collectorFlg = " + info.getCollectorFlg() +
							", regDate = " + info.getRegDate() +
							", updateDate = " + info.getUpdateDate() +
							", regUser = " + info.getRegUser() +
							", updateUser = " + info.getUpdateUser());
				}
			}
		} catch (CreateException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (FinderException e) {
			outputLog(e, "getMonitorList()", "011");
			throw new MonitorNotFound(e.getMessage(), e);
		} catch (NamingException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (HinemosUnknown e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (NotifyNotFound e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		}

		m_log.debug("getMonitorList() : end");
		return list;
	}

	/**
	 * 指定したフィルタにマッチする監視情報一覧を返します。
	 * @param condition
	 * @return
	 * @throws CreateException
	 * @throws NamingException
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws NotifyNotFound
	 */
	public ArrayList<MonitorInfo> getMonitorList(MonitorFilterInfo condition) throws CreateException, NamingException, HinemosUnknown, MonitorNotFound, NotifyNotFound {
		m_log.debug("getMonitorList() condition ");
		if(m_log.isDebugEnabled()){
			if(condition != null){
				m_log.debug("getMonitorList() " +
						"monitorId = " + condition.getMonitorId() +
						", monitorTypeId = " + condition.getMonitorTypeId() +
						", description = " + condition.getDescription() +
						", facilityId = " + condition.getFacilityId() +
						", calendarId = " + condition.getCalendarId() +
						", regUser = " + condition.getRegUser() +
						", regFromDate = " + condition.getRegFromDate() +
						", regToDate = " + condition.getRegToDate() +
						", updateUser = " + condition.getUpdateUser() +
						", updateFromDate = " + condition.getUpdateFromDate() +
						", updateToDate = " + condition.getUpdateToDate() +
						", monitorFlg = " + condition.getMonitorFlg() +
						", collectorFlg = " + condition.getCollectorFlg());
			}
		}

		ArrayList<MonitorInfo> filterList = new ArrayList<MonitorInfo>();
		// 条件未設定の場合は空のリストを返却する
		if(condition == null){
			m_log.debug("getMonitorList() condition is null");
			return filterList;
		}

		// DBではなくJavaAPにてフィルタリングを行う
		for(MonitorInfo info : getMonitorList()){
			m_log.debug("getMonitorList() condition : target = " + info.getMonitorId() + " start!");
			// facilityId 以外でマッチングを行いふるいにかける

			// monitorId
			if(condition.getMonitorId() != null && !"".equals(condition.getMonitorId())){
				if(info.getMonitorId() == null || (info.getMonitorId() != null && !info.getMonitorId().matches(".*" + condition.getMonitorId() + ".*"))){
					m_log.debug("getMonitorList() continue : monitorId target = " + info.getMonitorId() + ", filter = " + condition.getMonitorId());
					continue;
				}
			}

			// monitorTypeId
			if(condition.getMonitorTypeId() != null && !"".equals(condition.getMonitorTypeId())){
				if(info.getMonitorTypeId() == null || (info.getMonitorTypeId() != null && !info.getMonitorTypeId().matches(".*" + condition.getMonitorTypeId() + ".*"))){
					m_log.debug("getMonitorList() continue : monitorTypeId target = " + info.getMonitorTypeId() + ", filter = " + condition.getMonitorTypeId());
					continue;
				}
			}

			// description
			if(condition.getDescription() != null && !"".equals(condition.getDescription())){
				if(info.getDescription() == null || (info.getDescription() != null && !info.getDescription().matches(".*" + condition.getDescription() + ".*"))){
					m_log.debug("getMonitorList() continue : description target = " + info.getDescription() + ", filter = " + condition.getDescription());
					continue;
				}
			}

			// calendarId
			if(condition.getCalendarId() != null && !"".equals(condition.getCalendarId())){
				if(info.getCalendarId() == null || (info.getCalendarId() != null && !info.getCalendarId().matches(".*" + condition.getCalendarId() + ".*"))){
					m_log.debug("getMonitorList() continue : calendarId target = " + info.getCalendarId() + ", filter = " + condition.getCalendarId());
					continue;
				}
			}

			// regUser
			if(condition.getRegUser() != null && !"".equals(condition.getRegUser())){
				if(info.getRegUser() == null || (info.getRegUser() != null && !info.getRegUser().matches(".*" + condition.getRegUser() + ".*"))){
					m_log.debug("getMonitorList() continue : regUser target = " + info.getRegUser() + ", filter = " + condition.getRegUser());
					continue;
				}
			}

			// regFromDate
			if(condition.getRegFromDate() > 0){
				if(!(info.getRegDate() >= condition.getRegFromDate())){
					m_log.debug("getMonitorList() continue : regFromDate target = " + info.getRegDate() + ", filter = " + condition.getRegFromDate());
					continue;
				}
			}
			// regToDate
			if(condition.getRegToDate() > 0){
				if(!(info.getRegDate() <= condition.getRegToDate())){
					m_log.debug("getMonitorList() continue : regToDate target = " + info.getRegDate() + ", filter = " + condition.getRegToDate());
					continue;
				}
			}

			// updateUser
			if(condition.getUpdateUser() != null && !"".equals(condition.getUpdateUser())){
				if(info.getUpdateUser() == null || (info.getUpdateUser() != null && !info.getUpdateUser().matches(".*" + condition.getUpdateUser() + ".*"))){
					m_log.debug("getMonitorList() continue : updateUser target = " + info.getUpdateUser() + ", filter = " + condition.getUpdateUser());
					continue;
				}
			}

			// updateFromDate
			if(condition.getUpdateFromDate() > 0){
				if(!(info.getUpdateDate() >= condition.getUpdateFromDate())){
					m_log.debug("getMonitorList() continue : updateFromDate target = " + info.getUpdateDate() + ", filter = " + condition.getUpdateFromDate());
					continue;
				}
			}

			// updateToDate
			if(condition.getUpdateToDate() > 0){
				if(!(info.getUpdateDate() <= condition.getUpdateToDate())){
					m_log.debug("getMonitorList() continue : updateToDate target = " + info.getUpdateDate() + ", filter = " + condition.getUpdateToDate());
					continue;
				}
			}

			// monitorFlg
			if(condition.getMonitorFlg() >= 0){
				if(info.getMonitorFlg() != condition.getMonitorFlg()){
					m_log.debug("getMonitorList() continue : monitorFlg target = " + info.getMonitorFlg() + ", filter = " + condition.getMonitorFlg());
					continue;
				}
			}

			// collectorFlg
			if(condition.getCollectorFlg() >= 0){
				if(info.getCollectorFlg() != condition.getCollectorFlg()){
					m_log.debug("getMonitorList() continue : collectorFlg target = " + info.getCollectorFlg() + ", filter = " + condition.getCollectorFlg());
					continue;
				}
			}

			// facilityId
			if(condition.getFacilityId() != null && !"".equals(condition.getFacilityId()) && info.getFacilityId() != null){
				// FacilitySelector.getNodeFacilityIdListの第一引数が登録ノード全ての場合は、空リストを返す。そのため、下記のifを追加。
				if (!FacilityConstant.STRING_COMPOSITE.equals(info.getFacilityId())) {
					ArrayList<String> searchIdList = FacilitySelector.getNodeFacilityIdList(info.getFacilityId(), RepositoryControllerBean.ALL, false, true);

					if(!searchIdList.contains(condition.getFacilityId())){
						m_log.debug("getMonitorList() continue : collectorFlg target = " + info.getFacilityId() + ", filter = " + condition.getFacilityId());
						continue;
					}
				}
			}

			m_log.debug("getMonitorList() add display list : target = " + info.getMonitorId());
			filterList.add(info);
		}



		return filterList;
	}
	/**
	 * アプリケーションログにログを出力します。
	 * 
	 * @param e 例外
	 * @param method メソッド名
	 * @param index アプリケーションログのインデックス
	 */
	private void outputLog(Exception e, String method, String index) {
		AplLogger apllog = new AplLogger("MON", "mon");
		String[] args = {m_monitorTypeId, m_monitorId };
		apllog.put("SYS", index, args);
		m_log.debug(method + ":" + e.getMessage());
	}
}
