/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.factory;

import java.util.Collection;

import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.notify.ejb.entity.NotifyHistoryLocal;
import com.clustercontrol.notify.ejb.entity.NotifyHistoryUtil;
import com.clustercontrol.notify.ejb.entity.NotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 通知情報を削除するクラスです。
 *
 * @version 3.0.0
 * @since 1.0.0
 */
public class DeleteNotify {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( DeleteNotify.class );

	/**
	 * 通知情報を削除します。
	 * <p>
	 * <ol>
	 *  <li>通知IDより、通知情報を取得します。</li>
	 *  <li>通知情報に設定されている通知イベント情報を削除します。</li>
	 *  <li>通知情報を削除します。</li>
	 *  <li>キャッシュ更新用の通知情報を生成し、ログ出力キューへ送信します。
	 *      監視管理機能で、監視管理機能で保持している通知情報キャッシュから削除されます。</li>
	 * </ol>
	 * 
	 * @param notifyId 削除対象の通知ID
	 * @return 削除に成功した場合、<code> true </code>
	 * @throws NotifyNotFound
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws JMSException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 * @see com.clustercontrol.notify.ejb.entity.NotifyEventInfoBean
	 * @see #deleteEvents(Collection)
	 * @see com.clustercontrol.notify.util.SendQueue
	 */
	public boolean delete(String notifyId) throws NotifyNotFound, RemoveException, NamingException, FinderException {

		NotifyInfoLocal notify = null;
		try
		{
			// 通知設定を取得
			notify = NotifyInfoUtil.getLocalHome().findByPrimaryKey(notifyId);

		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { notifyId };
			apllog.put("SYS", "003", args);
			m_log.debug("delete():" + e.getMessage());
			throw new NotifyNotFound(e.getMessage(), e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { notifyId };
			apllog.put("SYS", "003", args);
			m_log.debug("delete():" + e.getMessage());
			throw e;
		}

		try
		{
			// 通知関連情報を削除
			DeleteNotifyRelation deleteRelation = new DeleteNotifyRelation();
			deleteRelation.deleteBaseOnNotifyId(notifyId);

			// この通知設定の結果として通知された通知履歴を削除する
			Collection<NotifyHistoryLocal> historyList =
				NotifyHistoryUtil.getLocalHome().findByNotifyId(notify.getNotifyId());
			for(NotifyHistoryLocal history : historyList){
				history.remove();
			}

			// 通知情報を削除
			notify.remove();
		} catch (RemoveException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { notifyId };
			apllog.put("SYS", "003", args);
			m_log.debug("getNotify():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { notifyId };
			apllog.put("SYS", "003", args);
			m_log.debug("getNotify():" + e.getMessage());
			throw e;
		}
		return true;
	}
}
