/*

Copyright (C) 2008 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.mail.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.MailTemplateNotFound;
import com.clustercontrol.notify.mail.bean.MailTemplateInfo;
import com.clustercontrol.notify.mail.ejb.entity.MailTemplateInfoLocal;
import com.clustercontrol.notify.mail.ejb.entity.MailTemplateInfoUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * メールテンプレート情報を検索するクラス<BR>
 *
 * @version 2.4.0
 * @since 2.4.0
 */
public class SelectMailTemplate {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( SelectMailTemplate.class );

	/**
	 * メールテンプレート情報を返します。
	 * 
	 * @param mailTemplateId 取得対象のメールテンプレートID
	 * @return メールテンプレート情報
	 * @throws MailTemplateNotFound
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.MailTemplateInfoBean
	 */
	public MailTemplateInfo getMailTemplateInfo(String mailTemplateId) throws MailTemplateNotFound, HinemosUnknown {

		MailTemplateInfo data = null;

		try {
			// メールテンプレート情報を取得
			MailTemplateInfoLocal infoLocal = MailTemplateInfoUtil.getLocalHome().findByPrimaryKey(mailTemplateId);

			data = new MailTemplateInfo();
			data.setMailTemplateId(infoLocal.getMailTemplateId());
			data.setDescription(infoLocal.getDescription());
			data.setSubject(infoLocal.getSubject());
			data.setBody(infoLocal.getBody());
			if(infoLocal.getRegDate() == null){
				data.setRegDate(null);
			}
			else{
				data.setRegDate(infoLocal.getRegDate().getTime());
			}
			if(infoLocal.getUpdateDate() == null){
				data.setUpdateDate(null);
			}
			else{
				data.setUpdateDate(infoLocal.getUpdateDate().getTime());
			}
			data.setRegUser(infoLocal.getRegUser());
			data.setUpdateUser(infoLocal.getUpdateUser());

		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("MAILTEMP", "mailTemplate");
			String[] args = { mailTemplateId };
			apllog.put("SYS", "004", args);
			m_log.warn("getMailTemplate():" + e.getMessage());
			throw new MailTemplateNotFound(e.getMessage(), e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("MAILTEMP", "mailTemplate");
			String[] args = { mailTemplateId };
			apllog.put("SYS", "004", args);
			m_log.warn("getMailTemplate():" + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return data;
	}

	/**
	 * テンプレートID一覧を返します。
	 * <p>
	 * テンプレートIDの昇順に並んだメールテンプレートID一覧を返します。
	 * 
	 * @return テンプレートID一覧
	 * @throws HinemosUnknown
	 * @throws MailTemplateNotFound
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.MailTemplateInfoBean
	 */
	public ArrayList<String> getMailTemplateIdList() throws HinemosUnknown, MailTemplateNotFound {

		ArrayList<String> list = null;
		try
		{
			// メールテンプレートID一覧を取得
			Collection ct = MailTemplateInfoUtil.getLocalHome().findAllOrderByMailTemplateId();

			Iterator itr = ct.iterator();
			while(itr.hasNext())
			{
				if(list == null){
					list = new ArrayList<String>();
				}

				MailTemplateInfoLocal mailTemplate = (MailTemplateInfoLocal)itr.next();
				list.add(mailTemplate.getMailTemplateId());
			}

		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("MAILTEMP", "mailTemplate");
			apllog.put("SYS", "005");
			m_log.warn("getMailTemplateIdList():" + e.getMessage());
			throw new MailTemplateNotFound(e.getMessage(), e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("MAILTEMP", "mailTemplate");
			apllog.put("SYS", "005");
			m_log.warn("getMailTemplateIdList():" + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return list;
	}

	/**
	 * メールテンプレート情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>メールテンプレートIDの昇順に並んだ全てのメールテンプレート情報を取得します。</li>
	 * <li>１メールテンプレート情報をテーブルのカラム順（{@link com.clustercontrol.notify.bean.MailTemplateTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１メールテンプレート情報を保持するリストを、メールテンプレート情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>メールテンプレート情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ メールテンプレート情報1 {カラム1の値, カラム2の値, … }, メールテンプレート情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return メールテンプレート情報一覧（Objectの2次元配列）
	 * @throws HinemosUnknown
	 * @throws MailTemplateNotFound
	 * 
	 * @see com.clustercontrol.notify.bean.MailTemplateTableDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList<MailTemplateInfo> getMailTemplateList() throws HinemosUnknown, MailTemplateNotFound {
		m_log.debug("getMailTemplateList() : start");
		ArrayList<MailTemplateInfo> list = new ArrayList<MailTemplateInfo>();
		try
		{
			// メールテンプレート情報一覧を取得
			Collection<MailTemplateInfoLocal> ct = MailTemplateInfoUtil.getLocalHome().findAllOrderByMailTemplateId();

			for(MailTemplateInfoLocal info : ct){
				list.add(getMailTemplateInfo(info.getMailTemplateId()));
				// for debug
				if(m_log.isDebugEnabled()){
					m_log.debug("getMailTemplateList() : " +
							"mailTemplateId = " + info.getMailTemplateId() +
							", description = " + info.getDescription() +
							", subject = " + info.getSubject() +
							", body = " + info.getBody() +
							", regUser = " + info.getRegUser() +
							", updateUser = " + info.getUpdateUser() +
							", regDate = " + info.getRegDate() +
							", updateDate = " + info.getUpdateDate());
				}
			}

		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("MAILTEMP", "mailTemplate");
			apllog.put("SYS", "006");
			m_log.warn("getMailTemplateList():" + e.getMessage());
			throw new MailTemplateNotFound(e.getMessage(), e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("MAILTEMP", "mailTemplate");
			apllog.put("SYS", "006");
			m_log.warn("getMailTemplateList():" + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return list;
	}

	/**
	 * メールテンプレート情報マップを返します。
	 * <p>
	 * <ol>
	 * <li>全てのメールテンプレート情報を取得します。</li>
	 * <li>メールテンプレート情報マップに、メールテンプレート情報をセットします。</li>
	 *  <dl>
	 *   <dt>メールテンプレート情報マップ</dt>
	 *   <dd>キー：メールテンプレートID</dd>
	 *   <dd>値　：メールテンプレート情報</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return メールテンプレート情報マップ
	 * @throws MailTemplateNotFound
	 * @throws HinemosUnknown
	 */
	public HashMap<String, MailTemplateInfo> getMailTemplateMap() throws MailTemplateNotFound, HinemosUnknown {

		HashMap<String, MailTemplateInfo> map = new HashMap<String, MailTemplateInfo>();
		try
		{
			// メールテンプレート情報一覧を取得
			Collection ct = MailTemplateInfoUtil.getLocalHome().findAll();
			Iterator itr = ct.iterator();
			MailTemplateInfoLocal mailTemplate = null;

			while(itr.hasNext()){

				mailTemplate = (MailTemplateInfoLocal)itr.next();

				MailTemplateInfo data = new MailTemplateInfo();
				data.setMailTemplateId(mailTemplate.getMailTemplateId());
				data.setDescription(mailTemplate.getDescription());
				data.setSubject(mailTemplate.getSubject());
				data.setBody(mailTemplate.getBody());
				if(data.getRegDate() == null){
					data.setRegDate(null);
				}
				else{
					data.setRegDate(mailTemplate.getRegDate().getTime());
				}
				if(data.getUpdateDate() == null){
					data.setUpdateDate(null);
				} else {
					data.setUpdateDate(mailTemplate.getUpdateDate().getTime());
				}
				data.setRegUser(mailTemplate.getRegUser());
				data.setUpdateUser(mailTemplate.getUpdateUser());

				map.put(mailTemplate.getMailTemplateId(), data);
			}

		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyMap():" + e.getMessage());
			throw new MailTemplateNotFound(e.getMessage(), e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyMap():" + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return map;
	}
}
