/*

 Copyright (C) 2009 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.sql.Timestamp;
import java.util.Collection;
import java.util.concurrent.ConcurrentHashMap;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.postgresql.util.PSQLException;

import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.ejb.entity.MonitorStatusLocal;
import com.clustercontrol.notify.ejb.entity.MonitorStatusPK;
import com.clustercontrol.notify.ejb.entity.MonitorStatusUtil;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;

public class MonitorResultStatusUpdater {
	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog( MonitorResultStatusUpdater.class );

	/**
	 * 同一重要度カウンタの最大値（この値を超えた場合は、以降は更新されない）
	 */
	public static int MAX_INITIAL_NOTIFY_COUNT = 10;

	// 排他制御用のキーセット
	private static ConcurrentHashMap<MonitorStatusPK, MonitorStatusPK> m_keyMap = new ConcurrentHashMap<MonitorStatusPK, MonitorStatusPK>();

	private static ConcurrentHashMap<MonitorStatusPK, Long> m_monitorStatusCounterCache =
		new ConcurrentHashMap<MonitorStatusPK, Long>();
	private static ConcurrentHashMap<MonitorStatusPK, Integer> m_monitorStatusPrioCache =
		new ConcurrentHashMap<MonitorStatusPK, Integer>();

	static{
		// プロパティファイルの読み込み
		try {
			MAX_INITIAL_NOTIFY_COUNT = Integer.parseInt(
					HinemosProperties.getProperty("notify.initial.count.max", "100"));
		} catch (Exception e) {
			m_log.warn(e.getMessage(), e);
		}

		// 最初にキャッシュ化しておく。
		m_log.info("init cache (monitorStatus)");
		try {
			Collection<MonitorStatusLocal> c = MonitorStatusUtil.getLocalHome().findAll();
			for (MonitorStatusLocal local : c) {
				MonitorStatusPK pk = (MonitorStatusPK)local.getPrimaryKey();
				m_monitorStatusCounterCache.put(pk, local.getCounter());
				m_monitorStatusPrioCache.put(pk, local.getPriority());
			}
		} catch (Exception e) {
			m_log.warn("failed creating cache : " + e.getMessage());
		}
	}

	public static MonitorStatusPK getLock(MonitorStatusPK pk) {
		MonitorStatusPK lockObject = null;
		synchronized (m_keyMap) {
			// タプルの存在確認とinsertまでをクリティカルセクションとするためにロックオブジェクトを管理する。
			lockObject = m_keyMap.get(pk);
			if(lockObject == null){
				lockObject = pk;
				// このキャッシュを削除する機構は実装していない。
				// 問題にならない容量であり、削除する機構を実装すると複雑なので。
				m_keyMap.put(pk, lockObject);
			}
		}
		return lockObject;
	}

	// このメソッドを利用する場合は、getLockを使用すること。
	public static Long getCounter (MonitorStatusPK pk) {
		Long count = null;
		count = m_monitorStatusCounterCache.get(pk);
		if (count == null) {
			m_log.warn("counter is null. pk=" + pk);
		}
		return count;
	}

	// このメソッドを利用する場合は、getLockを使用すること。
	public static Integer getPriority (MonitorStatusPK pk) {
		Integer priority = null;
		priority = m_monitorStatusPrioCache.get(pk);
		if (priority == null) {
			m_log.warn("priority is null. pk=" + pk);
		}
		return priority;
	}

	// このメソッドを利用する場合は、getLockを使用すること。
	public static MonitorStatusLocal refreshCache (MonitorStatusPK pk) throws FinderException{
		m_log.debug("refreshCache size=" + m_monitorStatusCounterCache.size() + "," +
				m_monitorStatusPrioCache.size());
		if (m_monitorStatusCounterCache.get(pk) != null) {
			return null;
		}

		// キャッシュにヒットしない場合。
		MonitorStatusLocal monitorStatus = null;
		try {
			// cc_monitor_statusが非常に多い場合は、findByPrimaryKeyに失敗する。
			// 前回と今回の監視結果の集計がバッティングしてしまうため。
			monitorStatus = MonitorStatusUtil.getLocalHome().findByPrimaryKey(pk);

			long counter = monitorStatus.getCounter();
			int priority = monitorStatus.getPriority();
			m_monitorStatusCounterCache.put(pk, counter);
			m_monitorStatusPrioCache.put(pk, priority);
		} catch (NamingException e) {
			m_log.warn("getCounter : " + e.getMessage());
		}
		return monitorStatus;
	}

	// このメソッドは利用する場合は、getLockを使用すること。
	// MonitorStatusを変更、削除する際に利用する必要あり。
	// MonitorStatusを追加する際は利用する必要なし。
	public static void removeCache (MonitorStatusPK pk){
		m_monitorStatusCounterCache.remove(pk);
		m_monitorStatusPrioCache.remove(pk);
	}
	
	public static void clear() {
		m_log.info("clear Cache");
		
		// FIXME
		m_monitorStatusCounterCache.clear();
		m_monitorStatusPrioCache.clear();
	}

	/**
	 * 直前の監視結果と現在の監視結果の重要度を比較し、変更がある場合は、DBで保持している同一重要度カウンタをリセット。
	 * 戻り値として、trueを返す。
	 * 重要度変化がない場合は、同一重要度カウンタをカウントアップし、DB情報を更新。
	 * 戻り値として、falseを返す。
	 * 
	 * @param facilityId ファシリティID
	 * @param pluginId プラグインID
	 * @param monitorId 監視項目ID
	 * @param generateDate 監視結果生成時刻
	 * @param currentPriority 現在の監視結果の重要度
	 * @return 重要度変化の有無（有:true / 無:false）
	 */
	private static boolean update(String facilityId, String pluginId, String monitorId,
			String subkey, Long generateDate, int currentPriority){
		m_log.debug("update() facilityId = " + facilityId +
				", pluginId = " + pluginId +
				", monitorId = " + monitorId +
				", subkey = " + subkey +
				", generateDate = " + generateDate +
				", currentPriority = " + currentPriority);

		MonitorStatusLocal monitorStatus = null;

		MonitorStatusPK pk = new MonitorStatusPK(facilityId, pluginId, monitorId, subkey);

		MonitorStatusPK lockObject = getLock(pk);
		synchronized (lockObject) {
			try {
				try {
					// キャッシュをリフレッシュする。
					// キャッシュが存在せず、DBにも存在しない場合は、FinderException。
					monitorStatus = refreshCache(pk);

					// 重要度が変化しているか確認
					if(currentPriority != getPriority(pk)){
						// 重要度が変化している場合
						if (monitorStatus == null) {
							monitorStatus = MonitorStatusUtil.getLocalHome().findByPrimaryKey(pk);
						}
						if(m_log.isDebugEnabled()){
							m_log.debug("prioityChangeFlag = true. " + pk + " ," +
									monitorStatus.getPriority() + " to " +
									currentPriority);
						}

						// 重要度を更新
						monitorStatus.setPriority(currentPriority);

						// 同一重要度カウンタを1にリセット
						monitorStatus.setCounter(1l);

						// キャッシュをクリア
						removeCache(pk);
						return true;
					} else {
						// 重要度は変化していない
						// 50行ぐらい下の箇所で、カウンタをアップする。
					}
				} catch (FinderException e) {
					// 検索の結果該当タプルが存在しない場合
					m_log.debug("create new entity. " + pk);

					try {
						FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
					} catch (FinderException e1) {
						// 出力先のファシリティIDが存在しない場合
						// 未登録のファシリティIDで通知される場合もある
						// その場合ここを通るため、エラーログは出さない。
						m_log.debug(e1.getMessage() + " FacilityId = " + facilityId);
						return true;
					}

					// 新規タプルを生成
					try {
						// 同一重要度カウンタは1で生成
						monitorStatus = MonitorStatusUtil.getLocalHome().create(
								facilityId,
								pluginId,
								monitorId,
								subkey,
								currentPriority,
								new Timestamp(generateDate),
								1l);
						// キャッシュをクリアする。
						removeCache(pk);
						return true;
					} catch (CreateException e1) {
						if (e1.getCause() instanceof PSQLException) {
							// 同じPKが同時にきたときに、ここに到達することがある。
							m_log.warn("CreateException : " + e.getMessage());
						} else {
							m_log.error(e.getMessage(), e);
						}
						return true;
					}
				}
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
				return true;
			}

			// 同一重要度カウンタをアップ
			long oldCount = 0;
			try {
				MonitorStatusLocal tmp = refreshCache(pk);
				if (tmp != null) {
					monitorStatus = tmp;
				}
			} catch (FinderException e) {
				// ここは通らないはず。
				m_log.error("update :: " + e.getMessage());
			}
			oldCount = getCounter(pk);

			// 最大カウント数を超えると、それ以上は増やさない（DBへのupdateを減らすための方策）
			if(oldCount <= MAX_INITIAL_NOTIFY_COUNT){
				if (monitorStatus == null) {
					try {
						monitorStatus = MonitorStatusUtil.getLocalHome().findByPrimaryKey(pk);
					} catch (Exception e) {
						// ここは通らないはず。
						m_log.warn("update : " + e.getClass().getSimpleName() +
								", " + e.getMessage() + pk);
						removeCache(pk);
					}
				}
				if (monitorStatus == null) {
					// ここは通らないはず。
					m_log.warn("monitorStatus == null " +
							getCounter(pk) + ", " + getPriority(pk));
					removeCache(pk);
				} else {
					monitorStatus.setCounter(oldCount+1);
					monitorStatus.setLastUpdate(new Timestamp(generateDate));
					removeCache(pk);
				}
			}
		}
		return false;
	}

	public static boolean update(OutputBasicInfo output){
		if(output.getSubKey() == null){
			m_log.warn("SubKey is null. PluginId = " + output.getPluginId() +
					", MonitorId = " + output.getMonitorId() +
					", FacilityId = " + output.getFacilityId());
			output.setSubKey("");
		}

		return update(output.getFacilityId(), output.getPluginId(), output.getMonitorId(),
				output.getSubKey(), output.getGenerationDate(), output.getPriority());
	}
}
