/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.sql.Timestamp;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.monitor.bean.ConfirmConstant;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoPK;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoUtil;
import com.clustercontrol.notify.monitor.ejb.entity.EventLogUtil;
import com.clustercontrol.notify.util.DependDbNotifier;


/**
 * イベント情報を更新するクラス<BR>
 *
 * @version 4.0.0
 * @since 3.0.0
 */
public class OutputEvent implements DependDbNotifier {

	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog( OutputEvent.class );

	/**
	 * 最終出力日時
	 * cc_event_logテーブルは主キーに出力日時を含む。
	 * 同じプラグインID,監視項目ID, ファシリティIDの同一イベントを出力するためには、出力日時を変更する必要がある
	 * **/
	private volatile static long _lastOutputDateTime = 0;

	/**
	 * イベントの出力を行います。
	 * 
	 * @param message 出力・通知情報
	 */
	@Override
	public void notify(NotifyRequestMessage message){
		outputEvent(message.getOutputInfo(), message.getNotifyId());
	}

	/**
	 * イベントの出力を行います。
	 * 
	 * @param outputInfo 出力・通知情
	 * @param notifyId 通知ID
	 */
	private void outputEvent(OutputBasicInfo outputInfo, String notifyId){
		// FIXME
		if(notifyId == null || "SYS".equals(outputInfo.getPluginId())) {
			// インターナルイベントの場合
			this.insertEventLog(outputInfo, ConfirmConstant.TYPE_UNCONFIRMED);
		} else {
			try {
				NotifyEventInfoLocal notifyEventInfo = NotifyEventInfoUtil.getLocalHome().findByPrimaryKey(
						new NotifyEventInfoPK(notifyId, outputInfo.getPriority()));
				this.insertEventLog(outputInfo, notifyEventInfo.getEventNormalState());
			} catch (FinderException e) {
				m_log.error(e.getMessage(), e);
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
			}
		}
	}

	/**
	 * イベント情報を作成します。
	 * @param output 出力情報
	 * @param confirmState 確認フラグ ConfirmConstant.TYPE_UNCONFIRMED / ConfirmConstant.TYPE_CONFIRMED
	 * @throws NamingException
	 * @throws CreateException
	 */
	public void insertEventLog(OutputBasicInfo output, int confirmState) {
		if (m_log.isDebugEnabled())
			m_log.debug("start event creation. (monitorId=" + output.getMonitorId() + ", pluginId=" + output.getPluginId()
					+ ", facilityId=" + output.getFacilityId() + ", generationDate=" + output.getGenerationDate()
					+ ", priority=" + output.getPriority() + ", message=" + output.getMessage() + ")");

		try {
			// 出力日時を生成
			Timestamp outputDate = createOutputDate();

			EventLogUtil.getLocalHome().create(
					output.getMonitorId(),
					output.getSubKey(),
					output.getPluginId(),
					output.getFacilityId(),
					output.getScopeText(),
					output.getApplication(),
					output.getMessageId(),
					output.getMessage(),
					output.getMessageOrg(),
					new Integer(output.getPriority()),
					new Integer(confirmState),
					null,
					"",
					new Integer(0),
					new Timestamp(output.getGenerationDate()),
					outputDate,
					new Integer(YesNoConstant.TYPE_NO),
					"",
					null,
					""
			);
		} catch(NamingException e) {
			m_log.error(e.getMessage(), e);
		} catch(CreateException e) {
			// 同じキーを持つオブジェクトがすでに存在する場合はエラー終了。再送処理は行わない。
			m_log.error(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("failed to create new Event Entity.", e);
		}

		if (m_log.isDebugEnabled())
			m_log.debug("created event successfully. (monitorId=" + output.getMonitorId() + ", pluginId=" + output.getPluginId()
					+ ", facilityId=" + output.getFacilityId() + ", generationDate=" + output.getGenerationDate()
					+ ", priority=" + output.getPriority() + ", message=" + output.getMessage() + ")");
	}

	/**
	 * イベントの出力日時を払い出します。
	 * 前回払い出した日時とは重複しないように払い出します。
	 * 
	 * @return 出力日時
	 */
	private static synchronized Timestamp createOutputDate(){
		// 現在時刻を取得
		long now = System.currentTimeMillis();
		long outputDateTime = 0;

		if((_lastOutputDateTime - 1000) < now && now <= _lastOutputDateTime){
			// 現在時刻と最終出力日時の誤差が1秒以内であり（時刻同期により大幅にずれた場合を想定）、
			// 現在時刻が最後に払い出した出力日時より昔の場合は、最終出力日時より1msだけ進める
			outputDateTime = _lastOutputDateTime + 1;
			m_log.warn("create OutputDate=" + outputDateTime);
		} else {
			outputDateTime = now;
		}

		_lastOutputDateTime = outputDateTime;

		return new Timestamp(outputDateTime);
	}

	/**
	 * 通知失敗時の内部エラー通知を定義します
	 */
	@Override
	public void internalErrorNotify(String notifyId, String msgID, String detailMsg) throws Exception {
		//FIXME
		// 何もしない
	}
}
