/*

 Copyright (C) 2008 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 */

package com.clustercontrol.poller.impl;

import java.math.BigInteger;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.CountDownLatch;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.opennms.protocols.snmp.SnmpCounter64;
import org.opennms.protocols.snmp.SnmpEndOfMibView;
import org.opennms.protocols.snmp.SnmpHandler;
import org.opennms.protocols.snmp.SnmpInt32;
import org.opennms.protocols.snmp.SnmpObjectId;
import org.opennms.protocols.snmp.SnmpOctetString;
import org.opennms.protocols.snmp.SnmpParameters;
import org.opennms.protocols.snmp.SnmpPduBulk;
import org.opennms.protocols.snmp.SnmpPduPacket;
import org.opennms.protocols.snmp.SnmpPduRequest;
import org.opennms.protocols.snmp.SnmpPeer;
import org.opennms.protocols.snmp.SnmpSession;
import org.opennms.protocols.snmp.SnmpSyntax;
import org.opennms.protocols.snmp.SnmpUInt32;
import org.opennms.protocols.snmp.SnmpVarBind;

import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.poller.bean.PollerProtocolConstant;
import com.clustercontrol.repository.util.SearchNodeProperties;
import com.clustercontrol.sharedtable.DataTable;
import com.clustercontrol.sharedtable.TableEntry;

/**
 * SNMPのポーリングを実行するクラス
 * 
 * @version 3.0.0
 * @since 3.0.0
 */
public class SnmpBulkPollerImpl implements SnmpHandler {
	private static Log m_log = LogFactory.getLog(SnmpBulkPollerImpl.class);

	// デフォルトのポート番号
	private static final int DEFAULT_PORT = 161;

	// デフォルトのリトライ回数
	private static final int DEFAULT_RETRIES = 3;

	// デフォルトのタイムアウト値(ms)
	private static final int DEFAULT_TIMEOUT = 3000;

	// IPアドレス（ポーリングの度に更新）
	private String m_ipAddress;

	// 収集のターゲットとなるOID（全てのポーリングが終了するまで値を保持）
	private String[] m_oidText;

	// 収集を開始するOID（ポーリングの度に更新）
	private ArrayList<SnmpObjectId> m_startOids;

	// 収集を終了するOID（ポーリングの度に更新）
	private ArrayList<SnmpObjectId> m_stopOids;

	// 取得したデータを格納するテーブル（ポーリングの度に更新）
	private DataTable m_dataTable;

	// 収集したOIDの最後のインデックスをチェックするか否かのフラグ（ポーリングの度に更新）
	private boolean m_indexCheckFlg = false;

	// m_indexCheckFlg = true の場合に、取得したOIDのインデックスが異なることで
	// リトライした回数を保持（ポーリングの度に更新）
	private int m_indexCheckDiffIndexRetryCount = 0;

	// m_indexCheckFlg = true の場合に、取得できたVarBindの数が異なることで
	// リトライした回数を保持（ポーリングの度に更新）
	private int m_indexCheckVarBindSizeRetryCount = 0;

	// m_indexCheckFlg = true の場合にリトライした回数の合計を保持（ポーリングの度に更新）
	private int m_indexCheckRetryCountTotal = 0;

	// m_indexCheckFlg = true の場合にリトライする回数の最大値
	private final int MAX_RETRY_COUNT = 10;

	// ポーリング対象のスコープを抜ける際のOIDが本当にそのスコープの最後のOIDなのかをチェックする
	// （Net-SNMPのHOST-RESOURCES-MIBでプロセス情報を取得した際に、
	//  プロセスのリストを最後まで取得できない場合があることへの対処）
	// そのために最後のOIDでリトライ処理をするその回数を保持
	// （ポーリングの度に更新）
	private int m_indexCheckLastIndexRetryCount = 0;

	// 最期にポーリングを行なった際のOIDのリスト m_indexCheckFlg = true の場合に使用する
	private ArrayList<String> m_lastPollingOids;

	/**
	 * アプリケーションスレッドがポーリング完了を待つためのラッチ。
	 * アプリケーションからの呼び出し箇所では、joeSNMPに実行を依頼した後、
	 * このラッチを待つ。このラッチは、「setPollerStatus」関数によって
	 * 解放される。
	 */
	private final CountDownLatch latchPollingEnd = new CountDownLatch(1);

	enum PollerStatus {
		/** まだこのポーラーが使用されていない状態<br>
		 * POLLING/INTERNALにのみ状態遷移する。 */
		NOUSING,
		/** ポーリング最中でjoeSNMPのスレッドによりコールバックされる状態<br>
		 * TIMEOUT/INTERNAL/COMPLETEのいずれかに遷移する */
		POLLING,
		/** joeSNMPへの依頼に失敗したか、ポーリング中に何らかのエラーが発生した後の状態<br>
		 * これ以降結果は格納しない。また、これ以降他のステータスに遷移しない。 */
		INTERNAL,
		/** タイムアウトエラーが発生した後の状態<br>
		 * これ以降結果は格納しない。また、これ以降他のステータスに遷移しない。 */
		TIMEOUT,
		/** ポーリングでリトライ回数を使い果たした状態<br>
		 * これ以降結果は格納しない。また、これ以降他のステータスに遷移しない。 */
		RETRYOVER,
		/** エラーが発生せず正常に終了した状態<br>
		 * これ以降テーブルに操作しない。またこれ以降他のステータスに遷移しない。 */
		COMPLETE
	}
	/**
	 * ポーラーのステータスを表す。（getter/setterを必ず使ってアクセスすること）
	 * この変数を読み書きする際には必ずlockModifyStatusのでガードが必要となる
	 */
	private PollerStatus pollerStatus = PollerStatus.NOUSING;

	/**
	 * GETBULKの変数
	 */
	private static int nonRepeaters = 0;
	/**
	 * GETBULKの変数
	 * maxRepetitionsを大きくすると、1回のやりとりで取得できる量が増える。
	 * しかし、不要なデータを取得する割合が増える。
	 * maxRepetitionsを小さくすると、やりとりの回数が増え、SNMPGETと近い状況になる。
	 */
	private static int maxRepetitions = 10;
	static {
		String nkey = "poller.snmp.bulk.nonrepeaters";
		try {
			String str = HinemosProperties.getProperty(nkey);
			if (str != null && !"".equals(str)) {
				maxRepetitions = Integer.parseInt(str);
			}
		} catch (Exception e) {
			m_log.warn(nkey + " is not available. " + e.getMessage());
		}
		String mkey = "poller.snmp.bulk.maxrepetitions";
		try {
			String str = HinemosProperties.getProperty(mkey);
			if (str != null && !"".equals(str)) {
				maxRepetitions = Integer.parseInt(str);
			}
		} catch (Exception e) {
			m_log.warn(mkey + " is not available. " + e.getMessage());
		}
	}


	/**
	 * ポーラーのステータスを取得する。この関数はスレッドセーフではない。
	 * 呼び出し側は、「lockModifyStatus」オブジェクトによって同期化処理が必要となる。
	 * この値を基にして、m_dataTable に触るか否かを決定する場合、必ず
	 * この関数の呼び出しからm_dataTableに対する処理までを１つのsynchronized
	 * ブロックにくくらなくてはならない。
	 */
	private PollerStatus getPollerStatus() {
		return pollerStatus;
	}
	/**
	 * このポーラー全体のステータスを変更するメソッド。このメソッドは「lockModifyStatus」
	 * によってスレッドセーフ性が保障されている。また、このメソッドには副作用がある。
	 * ステートが変更され、かつ、新しいステートが
	 * 「INTERNAL/TIMEOUT/RETRYOVER/COMPLETE」
	 * のいずれかである場合、「latchPollingEnd」ラッチが解放される。
	 * （ポーラーの終了条件を満たした段階でポーリングの終了を通知するラッチを開放する）
	 */
	private void setPollerStatus(PollerStatus newStatus) {
		synchronized (lockModifyStatus) {
			// 既にINTERNAL/TIMEOUT/RETRYOVER/COMPLETEに
			// なっている場合には、状態遷移は行わない
			switch(pollerStatus) {
			case INTERNAL:
			case TIMEOUT:
			case RETRYOVER:
			case COMPLETE:
				return;
			}

			pollerStatus = newStatus;

			// 新規ステータスが INTERNAL/TIMEOUT/RETRYOVER/COMPLETEの
			// いずれかになるになる場合には、 ポーリング終了待ちのラッチを開放する
			switch(pollerStatus) {
			case INTERNAL:
			case TIMEOUT:
			case RETRYOVER:
			case COMPLETE:
				latchPollingEnd.countDown();
			}
		}

	}

	/**
	 * ポーラーのステータスを参照・更新変更する際に取得するロック。statusの
	 * 読み書き・読んだ結果を基にアクションする際には必ずこのロックを取得する。
	 */
	private final Object lockModifyStatus = new Object();

	// クラス変数の初期化
	private void init(){
		m_indexCheckDiffIndexRetryCount = 0;
		m_indexCheckVarBindSizeRetryCount = 0;
		m_indexCheckRetryCountTotal = 0;
		m_indexCheckLastIndexRetryCount = 0;
	}

	/**
	 * メインルーチン
	 * IPアドレスと　DataTableを受け取り、
	 * ポーリングした結果をDataTableに代入する
	 * 
	 * @param ipAddress IPアドレス
	 * @param port ポート番号
	 * @param version バージョン（0:SNMP V1 protocol, 1:SNMP V2 protocol）
	 * @param community コミュニティ
	 * @param retries １回のポーリングでのリトライ回数
	 * @param timeout ポーリングのタイムアウト
	 * @param oidList 対象OIDのリスト
	 * @param indexCheckFlg ポーリング結果のインデックスが揃っているかのチェック
	 */
	public DataTable polling(
			String ipAddress,
			int port,
			int version,
			String community,
			int retries,
			int timeout,
			List<String> oidList,
			boolean indexCheckFlg) {
		// クラス変数を初期化
		init();

		// ポーリングの結果を返すインスタンス
		m_dataTable = new DataTable();

		m_ipAddress = ipAddress;
		m_indexCheckFlg = indexCheckFlg;

		// SnmpPeer の定義
		SnmpPeer peer;
		try {
			peer = new SnmpPeer(InetAddress.getByName(ipAddress));
		} catch (UnknownHostException e) {
			m_log.error("polling() error  :" + ipAddress.toString(), e);
			return m_dataTable;
		}
		if (port < 0){
			m_log.trace("set Port. " + port + " to " + DEFAULT_PORT);
			port = DEFAULT_PORT;
		}

		if (timeout < 0){
			m_log.trace("set Timeout. " + timeout + " to " + DEFAULT_TIMEOUT);
			timeout = DEFAULT_TIMEOUT;
		}

		if (retries < 0){
			m_log.trace("set Retries. " + retries + " to " + DEFAULT_RETRIES);
			retries = DEFAULT_RETRIES;
		}

		// パラメータ設定
		peer.setPort(port);
		peer.setTimeout(timeout);
		peer.setRetries(retries);

		// peer を初期化
		SnmpParameters parms = peer.getParameters();
		parms.setVersion(version);
		if (community != null)
			parms.setReadCommunity(community);

		// デバッグ出力
		if (m_log.isTraceEnabled()) {
			m_log.trace("polling() start :" + ipAddress.toString());
			m_log.trace("Port            : " + peer.getPort());
			m_log.trace("Version         : " + peer.getParameters().getVersion());
			m_log.trace("Community       : " + peer.getParameters().getReadCommunity());
			m_log.trace("Retries         : " + peer.getRetries());
			m_log.trace("Timeout         : " + peer.getTimeout());
			m_log.trace("IndexCheckFlg   : " + indexCheckFlg);
		}

		// sessionを生成
		SnmpSession session = null;
		try {
			session = new SnmpSession(peer);
		} catch (SocketException e) {
			m_log.error("polling() warning  :" + ipAddress.toString()
					+ " SocketException creating the SNMP session");
			setPollerStatus(PollerStatus.INTERNAL);
			return m_dataTable;
		}
		session.setDefaultHandler(this);

		try {
			m_oidText = new String[oidList.size()];
			m_startOids = new ArrayList<SnmpObjectId>(oidList.size());
			m_stopOids = new ArrayList<SnmpObjectId>(oidList.size());
			// 最期にポーリングを行なった際のOIDのリスト(m_indexCheckFlg = true の場合に使用)
			m_lastPollingOids = new ArrayList<String>(oidList.size());
			int i = 0;

			SnmpPduBulk pdu = new SnmpPduBulk(nonRepeaters, maxRepetitions, null);
			pdu.setRequestId(SnmpPduPacket.nextSequence());

			Iterator<String> itr = oidList.iterator();
			while (itr.hasNext()) {
				m_oidText[i] = itr.next();

				// OIDの最後が0で終わる場合は、.0を削除する
				// GETNEXTで取得するため
				if(m_oidText[i].endsWith(".0")){
					m_oidText[i] = m_oidText[i].substring(0, m_oidText[i].lastIndexOf(".0"));
				}

				// 開始OIDを設定
				SnmpObjectId startOid = new SnmpObjectId(m_oidText[i]);

				// 終了するOIDを設定
				SnmpObjectId stopOid = new SnmpObjectId(m_oidText[i]);
				int[] ids = stopOid.getIdentifiers();
				++ids[ids.length - 1];
				stopOid.setIdentifiers(ids);

				SnmpObjectId oId = new SnmpObjectId(m_oidText[i]);
				pdu.addVarBind(new SnmpVarBind(oId));
				m_startOids.add(startOid);
				m_stopOids.add(stopOid);
				m_lastPollingOids.add(m_oidText[i]);

				i++;
			}

			// ステータスをPOLLING中に変更し、実際にjoeSNMPに処理委譲する
			setPollerStatus(PollerStatus.POLLING);
			session.send(pdu);
			if(m_log.isDebugEnabled()){
				String str = "";
				for (String s : m_oidText) {
					str += s + ",";
				}
				m_log.debug("polling() : " + m_ipAddress + " send SnmpPduRequest, oid=" + str);
			}
			// SNMPの処理が完了（orエラーで失敗）するまで待つ。
			latchPollingEnd.await();

		} catch (InterruptedException e) {
			m_log.error("polling() warning :" + ipAddress.toString()
					+ " polling failed at InterruptedException");
			setPollerStatus(PollerStatus.INTERNAL);
		} finally {
			try {

				// ポーリングステータスが成功して完了した以外の状態になっている場合、
				// テーブルに含まれているデータを全て削除する。
				synchronized (lockModifyStatus) {
					if (getPollerStatus() != PollerStatus.COMPLETE) {
						m_dataTable.clear();
					}
				}

				session.close();

			} catch (Exception e) {
				/**
				 * FIXME
				 * Joe-SNMPの不具合の回避コード
				 * Joe-SNMPで不具合が修正されたら削除すること
				 */
				m_log.warn("polling():" + m_ipAddress
						+ " Session close failed");
			}
		}

		// リトライ回数が多い場合は出力する
		if(m_indexCheckRetryCountTotal >= 100){
			m_log.warn("polling():" + m_ipAddress +
					" too many retries. count : " + m_indexCheckRetryCountTotal);
		}

		// デバッグ出力
		if (m_log.isTraceEnabled()) {
			m_log.trace("polling() " + m_ipAddress + " retry total count : " + m_indexCheckRetryCountTotal);
		}
		if(m_log.isDebugEnabled()){
			m_log.debug("polling() end :" + ipAddress.toString());
		}
		return m_dataTable;
	}

	/**
	 * SNMPポーリングの結果を受け取った際にコールバックされるメソッド
	 */
	@Override
	public void snmpReceivedPdu(SnmpSession session, int cmd, SnmpPduPacket pdu) {
		//		m_log.debug("snmpReceivedPdu() start");

		try {
			long time = System.currentTimeMillis(); // 取得時刻

			SnmpPduRequest req = null;
			if (pdu instanceof SnmpPduRequest) {
				req = (SnmpPduRequest) pdu;
			} else {
				m_log.error("polling() error :" + session.toString()
						+ " Received non-request pdu");
				setPollerStatus(PollerStatus.INTERNAL);
				return;
			}

			if (pdu.getCommand() != SnmpPduPacket.RESPONSE) {
				m_log.error("polling() error :" + session.toString()
						+ "  Received non-response command "
						+ pdu.getCommand());
				setPollerStatus(PollerStatus.INTERNAL);
				return;
			}

			if (req.getErrorStatus() != 0) {
				String message = "polling() error :" + session.toString() +
				", ErrorStatus=" + req.getErrorStatus();
				if (req.getLength() != 0) {
					message += ", oid=" + req.getVarBindAt(0).getName();
				}
				m_log.error(message);
				setPollerStatus(PollerStatus.INTERNAL);
				return;
			}

			// 次の pduを作成
			SnmpPduBulk nxt = new SnmpPduBulk(nonRepeaters, maxRepetitions, null);
			nxt.setRequestId(SnmpPduPacket.nextSequence());

			// インデックスチェックフラグが設定されている場合にOIDの最後のインデックスが
			// 一致しているか否かをチェックするのに使用
			String lastIndexStr = null;

			// インデックスの最大値を保持する(m_indexCheckFlgがtrueの場合のみ利用)
			// 取得できたOIDのインデックスに差異がない場合は-lのまま
			int maxIndex = -1;

			// 次回のポーリングでpduにセットするOIDの開始と終了のOIDを保持する
			ArrayList<SnmpObjectId> nextStartOids = new ArrayList<SnmpObjectId>(m_startOids.size());
			ArrayList<SnmpObjectId> nextStopOids = new ArrayList<SnmpObjectId>(m_stopOids.size());
			// 最期にポーリングを行なった際に取得したOIDのリスト(m_indexCheckFlg = true の場合に使用)
			ArrayList<String> lastPollingOids = new ArrayList<String>(m_startOids.size());
			// インデックスチェックが必要な場合は、この時点ではテーブルに反映することはできないため、
			// 一時的にArrayListに保持する(m_indexCheckFlg = true の場合に使用)
			ArrayList<TableEntry> entryBuffer = new ArrayList<TableEntry>(m_startOids.size());

			// pduの戻りとして帰ってきた SnmpVarBindを順次処理する

			/*
			 * pduの戻りは下記のとおり。
			 * 
			 * [oidSize*maxRepetitionsが大きい場合]:
			 * 0, 1, 2, ..... , n-1
			 * n, n+1, n+2, ..... , 2n-1
			 * ...
			 * m*n, m*n+1, ..., p
			 * 
			 * n=oidSize (GETBULKにセットしたoidの数)
			 * p=PDUパケットサイズの限界値
			 * getVarBindAt(a)の次のOIDがgetVarBindAt(a+n)となる。
			 * 
			 * 
			 * [oidSize*maxRepetitionsが小さい場合]:
			 * 0, 1, 2, ..... , n-1
			 * n, n+1, n+2, ..... , 2n-1
			 * ...
			 * m*n, m*n+1, ..., (m+1)*n-1
			 * 
			 * n=oidSize (GETBULKにセットしたoidの数)
			 * m=maxRepetitions+1
			 * getVarBindAt(a)の次のOIDがgetVarBindAt(a+n)となる。
			 */
			int length = pdu.getLength();
			int oidSize = m_startOids.size();
			for (int i = 0; i < length; i++) {
				SnmpVarBind var = pdu.getVarBindAt(i);

				// ポーリングの終了判定
				// 条件：
				// 戻りの型が EndOfMibView の場合
				// 停止条件のOIDにマッチした場合は終了
				if (var.getValue().typeId() == SnmpEndOfMibView.ASNTYPE
						|| ((m_stopOids.get(i % oidSize) != null &&
								m_stopOids.get(i % oidSize).compare(var.getName()) < 0))) {

					// 該当OIDの収集を終了します
					if (m_log.isTraceEnabled()) {
						m_log.trace(m_ipAddress + " stop polling. " + i + ", "
								+ m_startOids.get(i % oidSize) + ", "
								+ m_stopOids.get(i % oidSize)
								+ " - " + var.getName());
					}
					continue;
				}

				// OIDを設定
				String oidString = var.getName().toString();

				// デバッグ出力
				if (m_log.isTraceEnabled()) {
					m_log.trace("get " + m_ipAddress + " " + oidString + " "
							+ time + " " + var.getValue().getClass().getName()
							+ " " + var.getValue().toString());
				}

				if (!m_startOids.get(i % oidSize).isRootOf(var.getName())) {
					// 帰ってきたMIB値のOIDが開始OIDのツリーの配下でない場合は、
					// 該当OIDの収集を終了します
					if (m_log.isTraceEnabled()) {
						m_log.trace(m_ipAddress + " stop polling.. " + i + ", "
								+ m_startOids.get(i % oidSize) + ", "
								+ m_stopOids.get(i % oidSize)
								+ " - " + var.getName());
					}
					continue;
				} else {
					// インデックスチェックフラグが設定されている場合は、OIDの最後のインデックスが
					// 一致しているか否かをチェックする
					if (m_indexCheckFlg == true) {
						if (lastIndexStr == null) { // ループの1回目（i=0の場合）
							lastIndexStr = oidString.substring(oidString
									.lastIndexOf(".")+1);
						} else { // ループの2回目以降は値が設定されている
							String indexStr = oidString.substring(oidString
									.lastIndexOf(".")+1);

							// OIDの最後のインデックスが一致しない場合
							if (!lastIndexStr.equals(indexStr)) {
								m_log.trace("Index not match. " + m_ipAddress
										+ ", " + lastIndexStr + ", "
										+ oidString);

								// 最大のインデックスを設定
								int lastIndex = Integer.parseInt(lastIndexStr);
								if(lastIndex > maxIndex){
									maxIndex = lastIndex;
								}
							}
							lastIndexStr = indexStr;
						}
					}

					// デバッグ
					if (m_log.isTraceEnabled()) {
						m_log.trace("set " + m_ipAddress + ","
								+ oidString + "," + time + ","
								+ var.getValue().toString() + ", "
								+ var.getValue());
					}

					if (var.getValue() instanceof SnmpUInt32) {
						long longValue = ((SnmpUInt32)var.getValue()).getValue();
						if(m_indexCheckFlg == true){
							entryBuffer.add(new TableEntry(getEntryKey(oidString), time, longValue));

							m_log.trace("snmpReceivedPdu() dataTable put : " +
									"entryKey : " + getEntryKey(oidString) +
									", time : " + time +
									", longValue : " + longValue);
						} else {
							synchronized (lockModifyStatus) {
								if (getPollerStatus() != PollerStatus.POLLING) {
									return;
								}
								m_dataTable.putValue(getEntryKey(oidString), time, longValue);
							}

							m_log.trace("snmpReceivedPdu() dataTable put : " +
									"entryKey : " + getEntryKey(oidString) +
									", time : " + time +
									", longValue : " + longValue);
						}
					} else if (var.getValue() instanceof SnmpInt32) {
						long longValue = ((SnmpInt32)var.getValue()).getValue();
						if(m_indexCheckFlg == true){
							entryBuffer.add(new TableEntry(getEntryKey(oidString), time, longValue));

							m_log.trace("snmpReceivedPdu() dataTable put : " +
									"entryKey : " + getEntryKey(oidString) +
									", time : " + time +
									", longValue : " + longValue);
						} else {
							synchronized (lockModifyStatus) {
								if (getPollerStatus() != PollerStatus.POLLING) {
									return;
								}
								m_dataTable.putValue(getEntryKey(oidString), time, longValue);
							}

							m_log.trace("snmpReceivedPdu() dataTable put : " +
									"entryKey : " + getEntryKey(oidString) +
									", time : " + time +
									", longValue : " + longValue);
						}
					} else if (var.getValue() instanceof SnmpCounter64) {
						BigInteger bigInt = ((SnmpCounter64)var.getValue()).getValue();
						// BigInteger が大き過ぎて long に適さない場合は、下位の 64 ビットだけが返される。
						long longValue = bigInt.longValue();
						if(m_indexCheckFlg == true){
							entryBuffer.add(new TableEntry(getEntryKey(oidString), time, longValue));

							m_log.trace("snmpReceivedPdu() dataTable put : " +
									"entryKey : " + getEntryKey(oidString) +
									", time : " + time +
									", longValue : " + longValue);
						} else {
							synchronized (lockModifyStatus) {
								if (getPollerStatus() != PollerStatus.POLLING) {
									return;
								}
								m_dataTable.putValue(getEntryKey(oidString), time, longValue);
							}

							m_log.trace("snmpReceivedPdu() dataTable put : " +
									"entryKey : " + getEntryKey(oidString) +
									", time : " + time +
									", longValue : " + longValue);
						}
					} else if (var.getValue() instanceof SnmpOctetString) {
						byte[] stringBytes = ((SnmpOctetString)var.getValue()).getString();
						String value = null;

						if (oidString.startsWith(SearchNodeProperties.DEFAULT_OID_NIC_MAC_ADDRESS) && stringBytes.length == 6) {
							// 6 byteのbinaryのOctetString
							// 00:0A:1F:5F:30 という値として扱う
							String hex = "";
							value = "";
							for (int j = 0; j < stringBytes.length; j++) {
								hex = String.format("%02x", stringBytes[j]).toUpperCase();
								value += "".equals(value) ? hex : ":" + hex;
							}
						} else {
							// WindowsのNIC名には0x00が含まれることがあるので除外する
							int stringLen = stringBytes.length;
							for (int j = 0; j < stringBytes.length; j++) {
								if (stringBytes[j] == 0x00) {
									stringLen = j;
									break;
								}
							}
							value = new String(stringBytes, 0 , stringLen);
						}

						if(m_log.isTraceEnabled()){
							m_log.trace("SnmpOctetString getValue string = " + value);
						}

						if(m_indexCheckFlg == true){
							entryBuffer.add(new TableEntry(getEntryKey(oidString), time, value)
							);

							m_log.trace("snmpReceivedPdu() dataTable put : " +
									"entryKey : " + getEntryKey(oidString) +
									", time : " + time +
									", longValue : " + value);
						} else {
							synchronized (lockModifyStatus) {
								if (getPollerStatus() != PollerStatus.POLLING) {
									return;
								}
								m_dataTable.putValue(getEntryKey(oidString), time, value);
							}

							m_log.trace("snmpReceivedPdu() dataTable put : " +
									"entryKey : " + getEntryKey(oidString) +
									", time : " + time +
									", longValue : " + value);
						}
					} else {
						if(m_indexCheckFlg == true){
							entryBuffer.add(new TableEntry(getEntryKey(oidString), time, var.getValue()));

							m_log.trace("snmpReceivedPdu() dataTable put : " +
									"entryKey : " + getEntryKey(oidString) +
									", time : " + time +
									", longValue : " + var.getValue().toString());
						} else {
							synchronized (lockModifyStatus) {
								if (getPollerStatus() != PollerStatus.POLLING) {
									return;
								}
								m_dataTable.putValue(getEntryKey(oidString), time, var.getValue());
							}

							m_log.trace("snmpReceivedPdu() dataTable put : " +
									"entryKey : " + getEntryKey(oidString) +
									", time : " + time +
									", longValue : " + var.getValue().toString());
						}
					}

					if (length - oidSize <= i) {
						// 次のポーリングで収集するものだけを追加
						nextStartOids.add(m_startOids.get(i % oidSize));
						nextStopOids.add(m_stopOids.get(i % oidSize));
						nxt.addVarBind(new SnmpVarBind(var.getName()));
						// 最期に実行したポーリングで取得したOIDのリストを保持する
						// m_indexCheckFlg が true の場合のリトライ処理のため
						lastPollingOids.add(var.getName().toString());
					}
				}
			}

			/*
			 * GETBULKではインデックスチェックをしない。
			 */
			m_indexCheckFlg = false;

			// インデックスチェックを行なっている場合
			if(m_indexCheckFlg == true){
				// 正常に値取得できている場合は、値をテーブルに設定する
				if (maxIndex == -1
						&& ((lastPollingOids.size() == m_lastPollingOids.size())
								|| (lastPollingOids.size() == 0))) {
					// 一時的にArrayListで保持されていた取得値をセットする
					Iterator<TableEntry> itr = entryBuffer.iterator();
					// ステータスを確認し、ポーリング中以外の状態になっている場合には以降の処理をキャンセルする
					synchronized (lockModifyStatus) {
						if (getPollerStatus() != PollerStatus.POLLING) {
							return;
						}
						while(itr.hasNext()){
							TableEntry entry = itr.next();

							// デバッグ
							if (m_log.isTraceEnabled()) {
								m_log.trace("set (buffer to table) " + m_ipAddress + ","
										+ entry.getKey() + "," + time + ","
										+ entry.getValue().toString() + ", "
										+ entry.getValue());
							}

							m_dataTable.putValue(entry);
						}
					}

					// リトライ回数をリセット
					m_indexCheckDiffIndexRetryCount = 0;
					m_indexCheckVarBindSizeRetryCount = 0;
				} else {
					m_log.trace("detect error. " + m_ipAddress);
				}

				// 対応しているエラーパターン（VarBindの数をn個とする）
				// 1) n個全て取得できているが、対象のMIBツリーの最後ではない
				// 2) n個全て取得できているが、indexが揃っていない（少なくともどれかのindexが異なる）
				// 3) n個全ては取得できていない。かつ、indexが揃っていない（少なくともどれかのindexが異なる）
				// 4) n個全ては取得できていないが、indexは全て同じ
				if (lastPollingOids.size() == 0) {  // エラーパターン1への対応
					// 次に実行するOIDが１つもない場合
					// ポーリング対象のスコープを抜ける際のOIDが本当にそのスコープの最後のOIDなのかをチェックする
					// （Net-SNMPのHOST-RESOURCES-MIBでプロセス情報を取得した際に、
					//  プロセスのリストを最後まで取得できない場合があることへの対処）

					// リトライ回数をリセット
					m_indexCheckDiffIndexRetryCount = 0;
					m_indexCheckVarBindSizeRetryCount = 0;

					if (m_indexCheckLastIndexRetryCount >= 1) {
						// 2回連続で同じOID（同じVarBindのセット）でポーリングの結果、全てのポーリングの戻りの
						// PDUに入っている「次の問い合わせ対象OID」が取得対象のMIBツリーからはずれているため
						// 正常に取得できたと判断
					} else {
						// ポーリング対処のOIDのリストの全てが同時に終了した場合、
						// 対象スコープを抜ける際のOIDが本当にそのスコープの最後のOIDなのかをチェックする。
						// （Net-SNMPのHOST-RESOURCES-MIBでプロセス情報を取得した際に、
						// プロセスのリストを最後まで取得できない場合があることへの対処）
						if (m_log.isTraceEnabled()) {
							m_log.trace("check retry. count="
									+ m_indexCheckLastIndexRetryCount + " "
									+ m_ipAddress);
						}

						nextStartOids.clear();
						nextStopOids.clear();
						lastPollingOids.clear();
						nxt = new SnmpPduBulk(nonRepeaters, maxRepetitions, null);
						nxt.setRequestId(SnmpPduPacket.nextSequence());

						for (int i = 0; i < m_lastPollingOids.size(); i++) {
							if (m_log.isTraceEnabled()) {
								m_log.trace("retry last polling set VarBind "
										+ m_ipAddress + ","
										+ m_lastPollingOids.get(i));
							}
							nextStartOids.add(m_startOids.get(i));
							nextStopOids.add(m_stopOids.get(i));
							nxt.addVarBind(new SnmpVarBind(m_lastPollingOids
									.get(i)));
							lastPollingOids = m_lastPollingOids;
						}

						// 最後のOIDでリトライ処理を実行した回数をカウントアップ
						m_indexCheckLastIndexRetryCount++;
						m_indexCheckRetryCountTotal++;
					}
				} else {  // エラーパターン1以外の対応（正常動作の場合も含む）
					// 次に実行するOIDが１つ以上ある場合

					// ログ出力
					if (m_log.isTraceEnabled()) {
						if (m_indexCheckLastIndexRetryCount != 0) {
							m_log.trace("detect error." + m_ipAddress);
						}
					}

					// 最後のOIDでリトライ処理を実行した回数をリセット
					m_indexCheckLastIndexRetryCount = 0;

					// インデックスの異なるものが戻ってきていた場合
					// 再度ポーリングを実行する
					if (maxIndex >= 0){  // エラーパターン2or3の対応
						// ログ出力
						if (m_log.isTraceEnabled()){
							m_log.trace("recovery(index not match) " + m_ipAddress
									+ "  max index:" + maxIndex
									+ "  next VarBind size:" + lastPollingOids.size());
						}

						nextStartOids.clear();
						nextStopOids.clear();
						lastPollingOids.clear();
						nxt = new SnmpPduBulk(nonRepeaters, maxRepetitions, null);
						nxt.setRequestId(SnmpPduPacket.nextSequence());

						if(m_indexCheckDiffIndexRetryCount < MAX_RETRY_COUNT){
							for (int i = 0; i < m_lastPollingOids.size(); i++) {
								String lastOid = m_lastPollingOids.get(i);
								// 最大のインデックスのものがリトライによって収集されるように
								// maxIndex-lをインデックスに設定する
								String nextOid = lastOid.substring(0, lastOid.lastIndexOf(".") + 1)
								+ (maxIndex - 1);

								nextStartOids.add(m_startOids.get(i));
								nextStopOids.add(m_stopOids.get(i));
								nxt.addVarBind(new SnmpVarBind(nextOid));
								lastPollingOids.add(nextOid);

								// ログ出力
								if (m_log.isTraceEnabled()) {
									m_log.trace("set VarBind " + m_ipAddress + "," + nextOid);
								}
							}
						} else { // リトライ回数の上限を超えた場合
							m_log.error("too many retries(index not match). " + m_ipAddress
									+ " count:" + m_indexCheckDiffIndexRetryCount);

							// ステータスをリトライ回数越えに変更し、ポーリング結果を待つスレッドに
							// 終了を通知する。また、以降の処理は全てキャンセルする
							setPollerStatus(PollerStatus.RETRYOVER);
							return;
						}

						// リトライ回数をカウントアップ
						m_indexCheckDiffIndexRetryCount++;
						m_indexCheckVarBindSizeRetryCount = 0;
						m_indexCheckRetryCountTotal++;
					} else if(lastPollingOids.size() != m_lastPollingOids.size()) {  // エラーパターン4の対応
						// ログ出力
						if (m_log.isTraceEnabled()){
							m_log.trace("recovery(next count:" + lastPollingOids.size() + ") "
									+ m_ipAddress);
						}

						nextStartOids.clear();
						nextStopOids.clear();
						lastPollingOids.clear();
						nxt = new SnmpPduBulk(nonRepeaters, maxRepetitions, null);
						nxt.setRequestId(SnmpPduPacket.nextSequence());

						// 最大リトライ回数（デフォルト3回）までは、同じOIDでリトライ処理
						// 永遠に同じOIDでリトライ処理が実行されないように、最大リトライ回数を超えたものは、
						// IF文分岐の下のロジックを通る
						if(m_indexCheckVarBindSizeRetryCount < MAX_RETRY_COUNT){
							for (int i = 0; i < m_lastPollingOids.size(); i++) {
								String nextOid = m_lastPollingOids.get(i);

								nextStartOids.add(m_startOids.get(i));
								nextStopOids.add(m_stopOids.get(i));
								nxt.addVarBind(new SnmpVarBind(nextOid));
								lastPollingOids.add(nextOid);

								// ログ出力
								if (m_log.isTraceEnabled()) {
									m_log.trace("set VarBind " + m_ipAddress + "," + nextOid);
								}
							}
						} else { // リトライ回数の上限を超えた場合
							m_log.error("too many retries(VarBind size not match). " + m_ipAddress
									+ " count:" + m_indexCheckVarBindSizeRetryCount);

							// ステータスをリトライ回数越えに変更し、ポーリング結果を待つスレッドに
							// 終了を通知する。また、以降の処理は全てキャンセルする
							setPollerStatus(PollerStatus.RETRYOVER);
							return;
						}

						// リトライ回数をカウントアップ
						m_indexCheckDiffIndexRetryCount = 0;
						m_indexCheckVarBindSizeRetryCount++;
						m_indexCheckRetryCountTotal++;
					}
				}
			}

			if (nxt.getLength() != 0) {
				m_startOids = nextStartOids;
				m_stopOids = nextStopOids;
				m_lastPollingOids = lastPollingOids;
				session.send(nxt, this);
			} else {
				// ステータスを正常完了として、ポーリング完了待ちスレッドに通知する
				setPollerStatus(PollerStatus.COMPLETE);
				return;
			}
		} catch (Exception e) { // 何か例外が生じた場合は収集を停止する
			m_log.error("InternalError ", e);

			// ステータスをインターナルエラーとして、ポーリング完了待ちスレッドに通知する
			setPollerStatus(PollerStatus.INTERNAL);
		}
	}

	/**
	 * 内部エラー発生時にコールバックされるメソッド
	 * 
	 * @see org.opennms.protocols.snmp.SnmpHandler#snmpInternalError(org.opennms.protocols.snmp.SnmpSession, int, org.opennms.protocols.snmp.SnmpSyntax)
	 */
	@Override
	public void snmpInternalError(SnmpSession session, int err, SnmpSyntax pdu) {
		String message = "InternalError. The error code is " + err + ". IP:"
		+ m_ipAddress + " OID:" + m_oidText[0];
		m_log.error("snmpInternalError():" + session.toString() + " " + message);

		// ステータスをインターナルエラーとし、ポーリング完了待ちスレッドに通知する
		setPollerStatus(PollerStatus.INTERNAL);
	}

	/**
	 * ポーリングでタイムアウト発生時にコールバックされるメソッド
	 * @see org.opennms.protocols.snmp.SnmpHandler#snmpTimeoutError(org.opennms.protocols.snmp.SnmpSession, org.opennms.protocols.snmp.SnmpSyntax)
	 */
	@Override
	public void snmpTimeoutError(SnmpSession session, SnmpSyntax pdu) {
		//		m_log.warn("snmpTimeoutError():"
		//		+ session.getPeer().getPeer().toString() + " "
		//		+ ((SnmpPduRequest) pdu).toVarBindArray()[0].getName()
		//		+ " polling failed at TimeoutError");

		SnmpVarBind[] bind = ((SnmpPduRequest) pdu).toVarBindArray();
		String ipAddress = session.getPeer().getPeer().toString();
		String message = "Polling failed at TimeoutError." + " IP:" + ipAddress
		+ " OID:" + bind[0].getName();

		// より詳細なメッセージを作成
		String addMessage = "";
		if(m_log.isDebugEnabled()){
			for (int i = 1; i < bind.length; i++) {
				addMessage = addMessage + "\n\t" + ipAddress + " " + bind[i].getName();
			}
			message = message + addMessage;
		}

		m_log.warn(message);

		// ステータスをTIMEOUTとし、ポーリング完了待ちスレッドに通知する
		setPollerStatus(PollerStatus.TIMEOUT);
	}

	/**
	 * DataTableに格納するためのEntryKeyを返すメソッド
	 * 
	 * @param oidString OID
	 */
	private String getEntryKey(String oidString){

		return PollerProtocolConstant.PROTOCOL_SNMP + "." + oidString;
	}


	/**
	 * 単体試験用
	 * 
	 * @param args
	 */
	public static void main(String[] args) {
		// パラメータを設定
		String ipAddress = "172.26.98.158";

		int port = 161;
		int version = 1;
		String community = "public";
		int retries = 3;
		int timeout = 3000;
		// boolean indexCheckFlg = false;
		ArrayList<String> oids = new ArrayList<String>();

		for(int i=7; i<args.length; i++){
			oids.add(args[i]);
		}
		oids.clear();
		oids.add(".1.3.6.1.2.1.2.2.1.10");
		print(ipAddress, port, version, community, retries, timeout,
				oids, false);

		oids.clear();
		oids.add(".1.3.6.1.2.1.2.2.1.10");
		oids.add(".1.3.6.1.2.1.2.2.1.11");
		oids.add(".1.3.6.1.2.1.2.2.1.14");
		oids.add(".1.3.6.1.2.1.2.2.1.16");
		oids.add(".1.3.6.1.2.1.2.2.1.17");
		oids.add(".1.3.6.1.2.1.2.2.1.2");
		oids.add(".1.3.6.1.2.1.2.2.1.20");
		oids.add(".1.3.6.1.2.1.25.1.1");
		oids.add(".1.3.6.1.2.1.25.2.3.1.3");
		oids.add(".1.3.6.1.2.1.25.2.3.1.5");
		oids.add(".1.3.6.1.2.1.25.2.3.1.6");
		oids.add(".1.3.6.1.2.1.25.3.2.1.1");
		oids.add(".1.3.6.1.2.1.25.3.3.1.2");
		oids.add(".1.3.6.1.4.1.2021.10.1.5");
		oids.add(".1.3.6.1.4.1.2021.11.3");
		oids.add(".1.3.6.1.4.1.2021.11.4");
		oids.add(".1.3.6.1.4.1.2021.11.50");
		oids.add(".1.3.6.1.4.1.2021.11.51");
		oids.add(".1.3.6.1.4.1.2021.11.52");
		oids.add(".1.3.6.1.4.1.2021.11.53");
		oids.add(".1.3.6.1.4.1.2021.11.54");
		oids.add(".1.3.6.1.4.1.2021.11.59");
		oids.add(".1.3.6.1.4.1.2021.11.60");
		oids.add(".1.3.6.1.4.1.2021.11.62");
		oids.add(".1.3.6.1.4.1.2021.11.63");
		oids.add(".1.3.6.1.4.1.2021.13.15.1.1.2");
		oids.add(".1.3.6.1.4.1.2021.13.15.1.1.3");
		oids.add(".1.3.6.1.4.1.2021.13.15.1.1.4");
		oids.add(".1.3.6.1.4.1.2021.13.15.1.1.5");
		oids.add(".1.3.6.1.4.1.2021.13.15.1.1.6");
		oids.add(".1.3.6.1.4.1.2021.4.11");
		oids.add(".1.3.6.1.4.1.2021.4.14");
		oids.add(".1.3.6.1.4.1.2021.4.15");
		oids.add(".1.3.6.1.4.1.2021.4.3");
		oids.add(".1.3.6.1.4.1.2021.4.4");
		oids.add(".1.3.6.1.4.1.2021.4.5");
		oids.add(".1.3.6.1.4.1.2021.4.6");
		print(ipAddress, port, version, community, retries, timeout,
				oids, false);

		oids.clear();
		oids.add(".1.3.6.1.2.1.25.4.2.1.2");
		oids.add(".1.3.6.1.2.1.25.4.2.1.4");
		oids.add(".1.3.6.1.2.1.25.4.2.1.5");
		print(ipAddress, port, version, community, retries, timeout,
				oids, true);
	}
	private static void print(String ipAddress, int port, int version, String community,
			int retries, int timeout, ArrayList<String> oids, boolean indexCheckFlg) {
		// ポーリングを行い値を収集
		try {
			long start = System.currentTimeMillis();
			SnmpBulkPollerImpl poller = new SnmpBulkPollerImpl();
			DataTable dataTable = poller.polling(
					ipAddress,
					port,
					version,
					community,
					retries,
					timeout,
					oids,
					indexCheckFlg);
			long end = System.currentTimeMillis();

			Set<String> keyset = dataTable.keySet();
			// OID順に並び替える
			TreeSet<String> keys = new TreeSet<String>(keyset);
			Iterator<String> keyItr = keys.iterator();

			// テーブルの中の値を取得時刻順に並び替える
			TreeSet<TableEntry> ts = new TreeSet<TableEntry>();
			while(keyItr.hasNext()){
				ts.add(dataTable.getValue(keyItr.next()));
			}

			// 取得値の一覧を表示
			Iterator<TableEntry> itr = ts.iterator();
			int i = 0;
			while(itr.hasNext()){
				TableEntry entry = itr.next();
				m_log.info(entry);
				// System.out.println("tableEntry " + i + "," +  entry);
				i++;
			}
			System.out.println("ts.size()=" + ts.size());
			System.out.println("time = " + (end - start) + "ms " + poller.getClass().getSimpleName());
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
}
