/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.repository.ejb.session;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;

import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.SessionContext;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.collectiverun.ejb.session.CollectiveRunControllerLocal;
import com.clustercontrol.collectiverun.ejb.session.CollectiveRunControllerUtil;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.fault.FacilityDuplicate;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.UsedFacility;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.hinemosagent.bean.TopicInfo;
import com.clustercontrol.hinemosagent.util.AgentConnectUtil;
import com.clustercontrol.jobmanagement.ejb.session.JobControllerLocal;
import com.clustercontrol.jobmanagement.ejb.session.JobControllerUtil;
import com.clustercontrol.monitor.ejb.session.MonitorControllerLocal;
import com.clustercontrol.monitor.ejb.session.MonitorControllerUtil;
import com.clustercontrol.notify.ejb.session.NotifyControllerLocal;
import com.clustercontrol.notify.ejb.session.NotifyControllerUtil;
import com.clustercontrol.repository.bean.AgentStatusInfo;
import com.clustercontrol.repository.bean.FacilityInfo;
import com.clustercontrol.repository.bean.FacilitySortOrderConstant;
import com.clustercontrol.repository.bean.FacilityTreeItem;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.RepositoryTableInfo;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.bean.TopicConstant;
import com.clustercontrol.repository.ejb.entity.CollectorPlatformMstLocal;
import com.clustercontrol.repository.ejb.entity.CollectorPlatformMstUtil;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.repository.factory.AgentLibDownloader;
import com.clustercontrol.repository.factory.FacilityModifier;
import com.clustercontrol.repository.factory.NodeProperty;
import com.clustercontrol.repository.factory.ScopeProperty;
import com.clustercontrol.repository.factory.SearchNodeBySNMP;
import com.clustercontrol.repository.factory.FacilitySelector;
import com.clustercontrol.repository.util.RepositoryManagerUtil;
import com.clustercontrol.repository.util.RepositoryUpdateTime;
import com.clustercontrol.repository.util.RepositoryValidator;
import com.clustercontrol.repository.util.SendTopic;
import com.clustercontrol.vm.ejb.entity.VmProtocolMstUtil;
import com.clustercontrol.vm.ejb.entity.VmSolutionMstLocal;
import com.clustercontrol.vm.ejb.entity.VmSolutionMstUtil;

/**
 *
 * <!-- begin-user-doc --> リポジトリ情報（ノード、スコープ）の生成、変更、削除、
 * 参照を行うSessionBean<BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 * <!-- end-user-doc --> *
 *
 * <!-- begin-xdoclet-definition -->
 * @ejb.bean name="RepositoryController"
 *           jndi-name="RepositoryController"
 *           type="Stateless"
 *           transaction-type="Container"
 *           view-type="local"
 * 
 * @ejb.transaction
 *   type="Required"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=FacilityLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=FacilityRelationLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NodeLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NodeHostnameLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NodeNoteLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NodeDeviceLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NodeFilesystemLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 * <!-- end-xdoclet-definition -->
 * @generated
 */
public abstract class RepositoryControllerBean implements javax.ejb.SessionBean {

	private static Log m_log = LogFactory.getLog( RepositoryControllerBean.class );

	private SessionContext m_context;

	public static final int ALL = 0;
	public static final int ONE_LEVEL = 1;

	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}


	/**********************
	 * ファシリティツリーのメソッド群
	 **********************/

	/**
	 * ファシリティツリー（スコープツリー）取得を取得します。
	 * <BR>
	 * 取得したファシリティツリーには割り当てられたノードを含みます。<BR>
	 * このメソッドはクライアントの画面情報を作成するために
	 * 呼び出されます。クライアントのロケールを引数をして必要とします。<BR>
	 * （最上位のスコープという表記をscopeをいう表記を切り替えるため。）
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param locale クライアントのロケール
	 * @return FacilityTreeItemの階層オブジェクト
	 */
	public FacilityTreeItem getFacilityTree(Locale locale) {
		/** メイン処理 */
		return FacilitySelector.getFacilityTree(locale, false, null);
	}

	/**
	 * ファシリティツリー（スコープツリー）取得を取得します。(有効なノードのみ)
	 * <BR>
	 * 取得したファシリティツリーには割り当てられたノードを含みます。<BR>
	 * このメソッドはクライアントの画面情報を作成するために
	 * 呼び出されます。クライアントのロケールを引数をして必要とします。<BR>
	 * （最上位のスコープという表記をscopeをいう表記を切り替えるため。）
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param locale クライアントのロケール
	 * @return FacilityTreeItemの階層オブジェクト
	 */
	public FacilityTreeItem getExecTargetFacilityTree(Locale locale) {
		/** メイン処理 */
		return FacilitySelector.getFacilityTree(locale, false, Boolean.TRUE);
	}

	/**
	 * ファシリティツリー（スコープツリー）取得を取得します。(有効なノードのみ)
	 * <BR>
	 * 取得したファシリティツリーには割り当てられたノードを含みます。<BR>
	 * このメソッドはクライアントの画面情報を作成するために
	 * 呼び出されます。クライアントのロケールを引数をして必要とします。<BR>
	 * （最上位のスコープという表記をscopeをいう表記を切り替えるため。）
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 4.0.0
	 * @since 4.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @param locale クライアントのロケール
	 * @return FacilityTreeItemの階層オブジェクト
	 */
	public FacilityTreeItem getExecTargetFacilityTree(String facilityId, Locale locale) {
		/** メイン処理 */
		return FacilitySelector.getFacilityTree(facilityId, locale, false, Boolean.TRUE);
	}

	/**********************
	 * ノードのメソッド群(getter)
	 **********************/
	/**
	 * ノード一覧を取得します。<BR>
	 * リポジトリに登録されているすべてのノードを取得します。<BR>
	 * 戻り値はNodeInfoのArrayListで、NodeInfoには
	 * ノードの下記情報のみ格納されています。
	 * ・ファシリティID
	 * ・ファシリティ名
	 * ・IPアドレスバージョン、IPv4, Ipv6
	 * ・説明
	 * getNodeFacilityIdListを利用すること。（getNodeと組み合わせて利用する。）
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @return NodeInfoの配列
	 */
	@Deprecated
	public ArrayList<NodeInfo> getNodeList() {
		/** メイン処理 */
		return FacilitySelector.getNodeList();
	}

	/**
	 * 詳細版ノード一覧を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 4.0.0
	 * @since 4.0.0
	 * 
	 * @return NodeInfoの配列
	 * @throws HinemosUnknown
	 */
	public List<NodeInfo> getNodeDetailList() throws HinemosUnknown {
		List<NodeInfo> list = new ArrayList<NodeInfo>();
		try {
			for (String facilityId : FacilitySelector.getFacilityIdList("REGISTERED", 0, false)) {
				NodeInfo nodeInfo = NodeProperty.getProperty(facilityId);
				list.add(nodeInfo);
			}
		} catch (Exception e) {
			m_log.error("getNodeList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return list;
	}

	/**
	 * ノード一覧を取得します。<BR>
	 * 
	 * クライアントなどで検索した場合に呼ばれ、該当するノード一覧を取得します。<BR>
	 * 引数はNodeInfoであり、"ファシリティID"、"ファシリティ名"、"説明"、
	 * "IPアドレス"、"OS名"、"OSリリース"、"管理者"、"連絡先"が１つ以上含まれており、
	 * その条件を元に該当するノードを戻り値とします。<BR>
	 * 戻り値はNodeInfoのArrayListで、NodeInfoには
	 * ノードの"ファシリティID"、"ファシリティ名"、"説明"のみ格納されています。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param property　検索条件のプロパティ
	 * @return NodeInfoの配列
	 */
	public ArrayList<NodeInfo> getFilterNodeList(NodeInfo property) {
		/** メイン処理 */
		return FacilitySelector.getFilterNodeList(property);
	}

	/**
	 * 
	 * 監視・ジョブ等の処理を実行する対象となる、ファシリティIDのリストを取得します。
	 * 引数で指定されたファシリティIDが、ノードかスコープによって、以下のようなリストを取得します。
	 * 
	 * ノードの場合
	 *   引数で指定されたfacilityIdが格納されたArrayList
	 *   ただし、管理対象（有効/無効フラグが真）の場合のみ
	 * 
	 * スコープの場合
	 *   配下に含まれるノードのファシリティIDが格納されたArrayList
	 *   ただし、管理対象（有効/無効フラグが真）のみ
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 3.0.0
	 * @since 3.0.0
	 * 
	 * 
	 * @param facilityId 処理を実行する対象となるファシリティID
	 * @return 有効なノードのリスト（有効なノードがひとつも含まれない場合は空のリスト）
	 */
	public ArrayList<String> getExecTargetFacilityIdList(String facilityId) {
		/** メイン処理 */
		return FacilitySelector.getNodeFacilityIdList(facilityId, RepositoryControllerBean.ALL, false, true);
	}

	/**
	 * ノードのファシリティIDリストを取得します。<BR>
	 * 
	 * リポジトリにあるすべてのノードのリストを取得します。<BR>
	 * 戻り値は ファシリティID(String)のArrayList<BR>
	 * 
	 * getNodeList() との違いはこちらの戻り値はArrayListの２次元ではなく、
	 * 単純にファシリティID（String）のみのArrayList
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @return ファシリティIDの配列
	 */
	public ArrayList<String> getNodeFacilityIdList() {
		return getNodeFacilityIdList(false);
	}

	/**
	 * ノードのファシリティIDリストを取得します。<BR>
	 * 
	 * リポジトリにあるすべてのノードのリストを取得します。<BR>
	 * 戻り値は ファシリティID(String)のArrayList
	 * 引数のsortにtrueをセットした場合には、listがCollator.compare()にしたがってソートされる。<BR>
	 * 
	 * getNodeList() との違いはこちらの戻り値はNodeInfoのArrayListではなく、
	 * 単純にファシリティID（String）のみのArrayList
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 2.1.0
	 * @since 2.1.0
	 * 
	 * @param sort sort ソートするか？(する:true しない:false)
	 * @return ファシリティIDの配列
	 */
	public ArrayList<String> getNodeFacilityIdList(boolean sort) {
		/** メイン処理 */
		return FacilitySelector.getNodeFacilityIdList(sort);
	}

	/**
	 * ノードの詳細プロパティを取得します。<BR>
	 * 
	 * faciliyIDで指定されるノードの詳細プロパティを取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @param locale クライアントのロケール
	 * @return ノード情報プロパティ
	 * @throws FacilityNotFound
	 */
	public NodeInfo getNode(String facilityId) throws FacilityNotFound {
		/** メイン処理 */
		return NodeProperty.getProperty(facilityId);
	}


	/**
	 * ファシリティパスを取得します。<BR>
	 * 
	 * 第一引数がノードの場合は、パスではなく、ファシリティ名。<BR>
	 * 第一引数がスコープの場合は、第二引数との相対的なファシリティパスを取得します。<BR>
	 * (例　○○スコープ>××システム>DBサーバ)<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @param parentFacilityId 上位のファシリティID
	 * @return String ファシリティパス
	 */
	public String getFacilityPath(String facilityId, String parentFacilityId) {
		/** メイン処理 */
		return FacilitySelector.getNodeScopePath(parentFacilityId, facilityId);
	}


	/**
	 * SNMPを利用してノードの情報を取得します。<BR>
	 * 
	 * 
	 * クライアントからSNMPで検出を行った際に呼び出されるメソッドです。<BR>
	 * SNMPポーリングにより、ノード詳細プロパティをセットし、クライアントに返す。
	 * 戻り値はNodeInfo
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 2.1.2
	 * @since 2.1.2
	 * 
	 * @param ポーリング対象のIPアドレス、コミュニティ名、バージョン、ポート
	 * @param locale クライアントのロケール
	 * @return ノード情報
	 * @throws HinemosUnknown
	 */
	public NodeInfo getNodePropertyBySNMP(String ipAddress, int port, String community, String version)
	throws HinemosUnknown {
		/** メイン処理 */
		NodeInfo property = null;
		try {
			property = SearchNodeBySNMP.searchNode(ipAddress, port, community, version);
		} catch (UnknownHostException e) {
			m_log.error("getNodePropertyBySNMP " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getNodePropertyBySNMP " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return property;
	}

	/**
	 * 条件のHashMapに該当するノードのファシリティIDのリストを返却する。<BR>
	 * このメソッドは性能が低いため、要注意。
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 3.1.0
	 * @since 3.1.0
	 * 
	 * @return ArrayList<String>
	 */
	@Deprecated
	public ArrayList<String> findByCondition(HashMap<String, String> condition){
		/** メイン処理 */
		return FacilitySelector.getNodeFacilityIdListByCondition(condition);
	}

	/**
	 * IPアドレスから該当するノードのファシリティID一覧を取得する。
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 4.0.0
	 * @since 4.0.0
	 * 
	 * @param ipaddr IPアドレス(Inet4Address or Inet6Address)
	 * @return ファシリティIDのリスト
	 * @throws HinemosUnknown 予期せぬ内部エラーが発生した場合
	 */
	public List<String> getFacilityIdByIpAddress(InetAddress ipaddr) throws HinemosUnknown {
		return FacilitySelector.getFacilityIdByIpAddress(ipaddr);
	}

	/**
	 * ファシリティIDリストを取得します。<BR>
	 * 
	 *  引数のホスト名（ノード名）またはIPアドレスに対応するノードのファシリティIDのリストを
	 *  取得します。<BR>
	 *  戻り値はファシリティID(String)のArrayList
	 * getNodeList(NodeInfo)を利用すること。
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param hostName ホスト名（ノード名）
	 * @param ipAddress　IPアドレス(v4)
	 * @return ファシリティIDの配列
	 */
	@Deprecated
	public ArrayList<String> getFacilityIdList(String hostName, String ipAddress) {
		/** メイン処理 */
		return FacilitySelector.getFacilityIdList(hostName, ipAddress);
	}

	/**********************
	 * ノードのメソッド群(getter以外)
	 **********************/

	/**
	 * ノードを新規に追加します。<BR>
	 * またこのメソッドで組み込みスコープである"登録済みノード"スコープにも
	 * 割り当てが行われます。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param nodeinfo 追加するノード情報のプロパティ
	 * @throws FacilityDuplicate
	 */
	public void addNode(NodeInfo nodeInfo) throws FacilityDuplicate,InvalidSetting {
		addNode(nodeInfo, true);
	}

	/**
	 * ノードを新規に追加します。（リポジトリ更新TOPIC未送信選択可能）<BR>
	 * またこのメソッドで組み込みスコープである"登録済みノード"スコープにも
	 * 割り当てが行われます。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param info 追加するノード情報のプロパティ
	 * @throws FacilityDuplicate
	 */
	public void addNode(NodeInfo info, boolean topicSendFlg) throws FacilityDuplicate,InvalidSetting {
		TransactionManager tm = null;
		boolean commit = false;

		/** メイン処理 */
		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			// 入力チェック
			RepositoryValidator.validateNodeInfo(info);

			FacilityModifier.addNode(info, m_context.getCallerPrincipal().getName(),
					FacilitySortOrderConstant.DEFAULT_SORT_ORDER_NODE, topicSendFlg);
			commit = true;
		} catch (DuplicateKeyException e) {
			m_log.warn("addNode() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new FacilityDuplicate(e.getMessage(), e);
		} catch (InvalidSetting e) {
			m_log.warn("addNode() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("addNode() : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new EJBException("addNode() : " + e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();
						NodeProperty.removeNode(info.getFacilityId());

						// ファシリティの木情報のキャッシュを初期化する
						FacilitySelector.initCacheFacilityTree();

						// read-committedのため、commit後に外部コンポーネントに通知する
						RepositoryManagerUtil.broadcastConfigured();
						

					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new EJBException(e.getMessage(), e);
				}
			}
		}

		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}

	/**
	 * ノードを変更します。<BR>
	 * 引数のpropertyには変更する属性のみを設定してください。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @version 2.0.0
	 * @since 1.0.0
	 * 
	 * @param info　変更するノード情報のプロパティ
	 */
	public void modifyNode(NodeInfo info) throws InvalidSetting, HinemosUnknown {
		TransactionManager tm = null;
		boolean commit = false;

		try{
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			// 入力チェック
			RepositoryValidator.validateNodeInfo(info);

			/** メイン処理 */
			FacilityModifier.modifyNode(info, m_context.getCallerPrincipal().getName(), true);
			commit = true;
		} catch (InvalidSetting e) {
			m_log.warn("modifyNode() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("modifyNode() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();
						NodeProperty.removeNode(info.getFacilityId());

						// ファシリティの木情報のキャッシュを初期化する
						FacilitySelector.initCacheFacilityTree();

						// read-committedのため、commit後に外部コンポーネントに通知する
						RepositoryManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new EJBException(e.getMessage(), e);
				}
			}
		}
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}

	/**
	 * ノード詳細情報を一括変更します。<BR>
	 * クライアントにあるスコープを指定して変更を行うためのメソッド
	 * 引数のpropertyには変更する属性のみを設定してください。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @version 2.0.0
	 * @since 1.0.0
	 * 
	 * @param info　変更するノード情報のプロパティ
	 * @param parentFacilityId　スコープのファシリティID
	 */
	public void modifyNode(NodeInfo info, String parentFacilityId) throws InvalidSetting, HinemosUnknown {
		TransactionManager tm = null;
		boolean commit = false;
		ArrayList<String> facilityIdList = new ArrayList<String>();

		try{
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			// 入力チェック
			RepositoryValidator.validateNodeInfo(info);

			/** メイン処理 */
			facilityIdList = FacilityModifier.modifyAllNode(info, parentFacilityId,
					m_context.getCallerPrincipal().getName(), true);
			commit = true;
		} catch (InvalidSetting e) {
			m_log.warn("modifyNode() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("modifyNode() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						for (String facilityId : facilityIdList) {
							NodeProperty.removeNode(facilityId);
						}

						// ファシリティの木情報のキャッシュを初期化する
						FacilitySelector.initCacheFacilityTree();

						// read-committedのため、commit後に外部コンポーネントに通知する
						RepositoryManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new EJBException(e.getMessage(), e);
				}
			}
		}
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}

	/**
	 * ノード情報を削除します。<BR>
	 * 
	 * faciityIDで指定されたノードをリポジトリから削除します。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @throws UsedFacility
	 * @throws HinemosUnknown
	 */
	public void deleteNode(String facilityId) throws UsedFacility, HinemosUnknown {
		TransactionManager tm = null;
		boolean commit = false;

		/** メイン処理 */
		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			checkIsUseFacility(facilityId);
			FacilityModifier.deleteNode(facilityId, m_context.getCallerPrincipal().getName(), true);
			commit = true;
		} catch (UsedFacility e) {
			m_log.warn("deleteNode() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("deleteNode() : " + e.getClass().getSimpleName() +", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(),e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();
						NodeProperty.removeNode(facilityId);

						// ファシリティの木情報のキャッシュを初期化する
						FacilitySelector.initCacheFacilityTree();

						// read-committedのため、commit後に外部コンポーネントに通知する
						RepositoryManagerUtil.broadcastConfigured();

					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new EJBException(e.getMessage(), e);
				}
			}
		}

		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}




	/**********************
	 * スコープのメソッド群
	 **********************/
	/**
	 * ファシリティID一覧を取得します。<BR>
	 * あるスコープを指定してその直下にあるスコープとノードを取得します。<BR>
	 * このメソッドは引数としてそのスコープのファシリティIDを要求します。<BR>
	 * 戻り値はArrayListで中のFacilityInfoには子の
	 * "ファシリティID"、"ファシリティ名"、"説明"のみ格納されています。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId ファシリティID
	 * @return ScopeInfoの配列
	 */
	public ArrayList<FacilityInfo> getFacilityList(String parentFacilityId) {
		/** メイン処理 */
		return FacilitySelector.getFacilityListAssignedScope(parentFacilityId);
	}

	/**
	 * スコープ用プロパティ情報を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @param locale クライアントのロケール
	 * @return スコープのプロパティ情報（ファシリティID、ファシリティ名、説明）
	 * @throws FacilityNotFound
	 * @throws HinemosUnknown
	 */
	public ScopeInfo getScope(String facilityId) throws FacilityNotFound, HinemosUnknown {
		/** メイン処理 */
		ScopeInfo property = null;
		try {
			property = ScopeProperty.getProperty(facilityId);
		} catch (NamingException e) {
			m_log.error("NamingException " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return property;
	}

	/**
	 * スコープを新規に追加します。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープの下にpropertyで指定されるスコープを
	 * 追加します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId
	 * @param property
	 * @throws FacilityDuplicate
	 */
	public void addScope(String parentFacilityId, ScopeInfo property)
	throws FacilityDuplicate,InvalidSetting,HinemosUnknown {
		addScope(parentFacilityId, property, FacilitySortOrderConstant.DEFAULT_SORT_ORDER_SCOPE);
	}

	/**
	 * スコープを新規に追加します(表示順指定)。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープの下にpropertyで指定されるスコープを
	 * 追加します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId
	 * @param property
	 * @param sortOrder
	 * @throws FacilityDuplicate
	 */
	public void addScope(String parentFacilityId, ScopeInfo property, int displaySortOrder)
	throws FacilityDuplicate,InvalidSetting,HinemosUnknown {
		addScope(parentFacilityId, property, displaySortOrder, true);
	}

	/**
	 * スコープを新規に追加します(表示順指定、リポジトリ更新TOPIC未送信選択可能)。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープの下にpropertyで指定されるスコープを
	 * 追加します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId
	 * @param info
	 * @param sortOrder
	 * @throws FacilityDuplicate
	 */
	public void addScope(String parentFacilityId, ScopeInfo info, int displaySortOrder, boolean topicSendFlg)
	throws FacilityDuplicate,InvalidSetting,HinemosUnknown {
		TransactionManager tm = null;
		boolean commit = false;

		/** メイン処理 */
		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			//入力チェック
			RepositoryValidator.validateScopeInfo(parentFacilityId, info, true);

			FacilityModifier.addScope(parentFacilityId, info, m_context.getCallerPrincipal().getName(), displaySortOrder, topicSendFlg);
			commit = true;
		} catch (DuplicateKeyException e) {
			m_log.warn("addScope() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new FacilityDuplicate(e.getMessage(), e);
		} catch (InvalidSetting e) {
			m_log.warn("addScope() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("addScope() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(),e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						// ファシリティの木情報のキャッシュを初期化する
						FacilitySelector.initCacheFacilityTree();

						// read-committedのため、commit後に外部コンポーネントに通知する
						RepositoryManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new EJBException(e.getMessage(), e);
				}
			}
		}

		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}

	/**
	 * スコープの情報を変更します。<BR>
	 * 
	 * 引数propertyで指定した内容でスコープ情報を更新します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * propertyに含まれるファシリティIDに対応するスコープの情報が変更されます。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param info
	 */
	public void modifyScope(ScopeInfo info) throws InvalidSetting,HinemosUnknown {
		TransactionManager tm = null;
		boolean commit = false;

		try{
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			//入力チェック
			RepositoryValidator.validateScopeInfo(null, info, false);

			/** メイン処理 */
			FacilityModifier.modifyScope(info, m_context.getCallerPrincipal().getName(), true);
			commit = true;
		} catch (InvalidSetting e) {
			m_log.warn("modifyScope() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("modifyScope() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(),e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						// ファシリティの木情報のキャッシュを初期化する
						FacilitySelector.initCacheFacilityTree();

						// read-committedのため、commit後に外部コンポーネントに通知する
						RepositoryManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new EJBException(e.getMessage(), e);
				}
			}
		}

		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}

	/**
	 * スコープ情報を削除します。<BR>
	 * 
	 * faciityIDで指定されたスコープをリポジトリから削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @throws UsedFacility
	 * @throws HinemosUnknown
	 */
	public void deleteScope(String facilityId) throws UsedFacility, HinemosUnknown {
		TransactionManager tm = null;
		boolean commit = false;

		/** メイン処理 */
		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			checkIsUseFacility(facilityId);
			FacilityModifier.deleteScope(facilityId, m_context.getCallerPrincipal().getName(), true);
			commit = true;
		} catch (UsedFacility e) {
			m_log.warn("deleteScope() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("deleteScope() : " + e.getClass().getSimpleName() +", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(),e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						// ファシリティの木情報のキャッシュを初期化する
						FacilitySelector.initCacheFacilityTree();

						// read-committedのため、commit後に外部コンポーネントに通知する
						RepositoryManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new EJBException(e.getMessage(), e);
				}
			}
		}

		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}



	/**********************
	 * ノード割り当てのメソッド群
	 **********************/
	/**
	 * 割当ノード一覧を取得します。<BR>
	 * 
	 * あるファシリティIDの配下または直下のノード一覧を取得します。<BR>
	 * このメソッドでは、引数levelで直下または配下を制御します。<BR>
	 * 戻り値はNodeInfoのArrayListで、NodeInfoには
	 * ノードの"ファシリティID"、"ファシリティ名"、"説明"のみ格納されています。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId
	 * @param level　取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return NodeInfoの配列
	 */
	public ArrayList<NodeInfo> getNodeList(String parentFacilityId, int level) {
		/** メイン処理 */
		return FacilitySelector.getNodeList(parentFacilityId, level);
	}


	/**
	 * 割当スコープ一覧を取得します。<BR>
	 * 割り当てスコープ一覧とは、あるノードが属しているスコープすべてを
	 * 一覧表示したものです。
	 * クライアントの割り当てスコープビューの表示データとなります。
	 * 戻り値はArrayListのArrayListで中のArrayListには"スコープ"が最上位からの
	 * スコープパス表記で（Stringで）格納されています。
	 * 外のArrayListには、そのレコードが順に格納されています。
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ノードのファシリティID
	 * @return Stringの配列
	 */
	public ArrayList<String> getNodeScopeList(String facilityId) {
		/** メイン処理 */
		return FacilitySelector.getNodeScopeList(facilityId);
	}

	/**
	 * ファシリティIDリストを取得します。<BR>
	 * 
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（スコープ、ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 2.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId ファシリティID
	 * @param level　取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return ファシリティIDの配列
	 */
	public ArrayList<String> getFacilityIdList(String parentFacilityId, int level) {
		/** メイン処理 */
		return getFacilityIdList(parentFacilityId, level, false);
	}

	/**
	 * ファシリティIDリストを取得します。<BR>
	 * 
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（スコープ、ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 引数のsortにtrueをセットした場合には、listがCollator.compare()にしたがってソートされる。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 2.1.0
	 * @since 2.1.0
	 * 
	 * @param parentFacilityId ファシリティID
	 * @param level 取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @param sort ソートするか？(する:true しない:false)
	 * @return ファシリティIDの配列
	 */
	public ArrayList<String> getFacilityIdList(String parentFacilityId, int level, boolean sort) {
		/** メイン処理 */
		return FacilitySelector.getFacilityIdList(parentFacilityId, level, sort);
	}

	/**
	 * ノードのファシリティIDリストを取得します。<BR>
	 * 
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 2.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId ファシリティID
	 * @param level  取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return ファシリティIDの配列
	 */
	public ArrayList<String> getNodeFacilityIdList(String parentFacilityId, int level) {
		/** メイン処理 */
		return getNodeFacilityIdList(parentFacilityId, level, false, true);
	}

	/**
	 * ノードのファシリティIDリスト取得<BR>
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 引数のsortにtrueをセットした場合には、listがCollator.compare()にしたがってソートされる。
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 2.1.0
	 * @since 2.1.0
	 * 
	 * @param parentFacilityId
	 * @param level   取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @param sort sort ソートするか？(する:true しない:false)
	 * @return ファシリティIDの配列
	 */
	public ArrayList<String> getNodeFacilityIdList(String parentFacilityId, int level, boolean sort, Boolean valid) {
		/** メイン処理 */
		return FacilitySelector.getNodeFacilityIdList(parentFacilityId, level, sort, valid);
	}

	/**
	 * スコープへのノードの割り当てを行います。（リポジトリ更新TOPIC未送信選択可能）<BR>
	 * 
	 * parentFacilityIdで指定されるスコープにfacilityIdsで指定されるノード群を
	 * 割り当てます。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId　ノードを割り当てるスコープ
	 * @param facilityIds 割り当てさせるノード(群)
	 */
	public void assignNodeScope(String parentFacilityId, String[] facilityIds, boolean topicSendFlg)
	throws InvalidSetting, HinemosUnknown {
		TransactionManager tm = null;
		boolean commit = false;

		try{
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			//入力チェック
			RepositoryValidator.validateaAssignNodeScope(parentFacilityId, facilityIds);

			/** メイン処理 */
			FacilityModifier.assignFacilityToScope(parentFacilityId, facilityIds, topicSendFlg);
			commit = true;
		} catch (InvalidSetting e) {
			m_log.warn("assignNodeScope() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("assignNodeScope() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(),e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						// ファシリティの木情報のキャッシュを初期化する
						FacilitySelector.initCacheFacilityTree();

						// read-committedのため、commit後に外部コンポーネントに通知する
						RepositoryManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new EJBException(e.getMessage(), e);
				}
			}
		}

		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}

	/**
	 * スコープへのノードの割り当てを行います。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープにfacilityIdsで指定されるノード群を
	 * 割り当てます。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId　ノードを割り当てるスコープ
	 * @param facilityIds 割り当てさせるノード(群)
	 */
	public void assignNodeScope(String parentFacilityId, String[] facilityIds)
	throws InvalidSetting, HinemosUnknown {
		assignNodeScope(parentFacilityId, facilityIds, true);
	}

	/**
	 * ノードをスコープから削除します。（割り当てを解除します。）<BR>
	 * parentFacilityIdで指定されるスコープからfacilityIdsで指定されるノード群を
	 * 削除（割り当て解除）します。
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId ノードを取り除くスコープ
	 * @param facilityIds 取り除かれるノード（群）
	 */
	public void releaseNodeScope(String parentFacilityId, String[] facilityIds)
	throws InvalidSetting, HinemosUnknown{
		releaseNodeScope(parentFacilityId, facilityIds, true);
	}


	/**
	 * ノードをスコープから削除します。（割り当てを解除します。リポジトリ更新TOPIC未送信選択可能）<BR>
	 * parentFacilityIdで指定されるスコープからfacilityIdsで指定されるノード群を
	 * 削除（割り当て解除）します。
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId ノードを取り除くスコープ
	 * @param facilityIds 取り除かれるノード（群）
	 */
	public void releaseNodeScope(String parentFacilityId, String[] facilityIds, boolean topicSendFlg)
	throws InvalidSetting, HinemosUnknown{
		TransactionManager tm = null;
		boolean commit = false;

		try{
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			//入力チェック
			RepositoryValidator.validateaAssignNodeScope(parentFacilityId, facilityIds);

			/** メイン処理 */
			FacilityModifier.releaseNodeFromScope(parentFacilityId, facilityIds, m_context.getCallerPrincipal().getName(), topicSendFlg);
			commit = true;
		} catch (InvalidSetting e) {
			m_log.warn("assignNodeScope() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.warn("assignNodeScope() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();

						// ファシリティの木情報のキャッシュを初期化する
						FacilitySelector.initCacheFacilityTree();

						// read-committedのため、commit後に外部コンポーネントに通知する
						RepositoryManagerUtil.broadcastConfigured();
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new EJBException(e.getMessage(), e);
				}
			}
		}
		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refleshUpdateTime();
	}


	/**********************
	 * その他のメソッド群
	 **********************/
	/**
	 * ファシリティがノードかどうかをチェックします。<BR>
	 * 
	 * ファシリティIDに対応するものがノードかチェックし、結果をbooleanで返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId　ファシリティID
	 * @return true：ノード　false:ノードではない（スコープ）
	 * @throws FacilityNotFound 指定されたIDに該当するファシリティが存在しない場合
	 */
	public boolean isNode(String facilityId) throws FacilityNotFound {
		/** メイン処理 */
		try {
			return FacilitySelector.isNode(facilityId);
		} catch (FacilityNotFound e) {
			m_log.warn("isNode() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
	}

	/**
	 * セパレータ文字列を取得します。<BR>
	 * 
	 * セパレータ文字列はスコープパス表示の際のスコープを区切る文字列
	 * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @return セパレータ文字列
	 */
	public String getSeparator() {
		/** メイン処理 */
		return FacilitySelector.SEPARATOR;
	}

	/**
	 * リポジトリ更新のTOPICを送信する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 3.1.0
	 * @since 3.1.0
	 * 
	 */
	public void sendRepositoryUpdateTopic(){
		/** メイン処理 */
		SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
	}


	/**
	 * ノード作成変更時に、利用可能プラットフォームを表示するためのメソッド。
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 3.2.0
	 * @since 3.2.0
	 * @return ArrayList<RepositoryTableInfo>
	 * @throws HinemosUnknown
	 */
	public ArrayList<RepositoryTableInfo> getPlatformList() throws HinemosUnknown{
		ArrayList<RepositoryTableInfo> list = new ArrayList<RepositoryTableInfo>();
		try {
			for (CollectorPlatformMstLocal bean : (Collection<CollectorPlatformMstLocal>)CollectorPlatformMstUtil.getLocalHome().findAll()) {
				list.add(new RepositoryTableInfo(bean.getPlatformId(), bean.getPlatformName()));
			}
		} catch (Exception e) {
			m_log.error("getPlatformList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return list;
	}

	/**
	 * ノード作成変更時に、仮想化種別hostのファシリティIDを表示するためのメソッド。
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 3.2.0
	 * @since 3.2.0
	 * @return ArrayList
	 * @throws HinemosUnknown
	 */
	public ArrayList<RepositoryTableInfo> getHostList() throws HinemosUnknown{
		ArrayList<RepositoryTableInfo> list = new ArrayList<RepositoryTableInfo>();
		try {
			for (FacilityLocal bean : (Collection<FacilityLocal>)FacilityUtil.getLocalHome().findAll()) {
				if (bean.isNode()) {
					if ("host".equals(bean.getNode().getVirtualizationNodeType()) ||
							"controller".equals(bean.getNode().getVirtualizationNodeType())) {
						list.add(new RepositoryTableInfo(bean.getFacilityId(), bean.getFacilityName()));
					}
				}
			}
		} catch (Exception e) {
			m_log.error("getHostList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return list;
	}

	/**
	 * OpenFlowスイッチのノードを取得する。
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 4.0.0
	 * @since 4.0.0
	 * @return List
	 * @throws HinemosUnknown
	 */
	public List<NodeInfo> getOpenFlowSwitchList() throws HinemosUnknown{
		List<NodeInfo> list = new ArrayList<NodeInfo>();
		for(NodeInfo node : this.getNodeDetailList()){
			if("NW_EQUIPMENT".equals(node.getPlatformFamily()) && node.getOpenFlowDataPathId() != null && ! "".equals(node.getOpenFlowDataPathId())){
				list.add(node);
			}
		}
		return list;
	}

	/**
	 * ノード作成変更時に、利用可能な仮想化ソリューションを表示するためのメソッド。
	 * 
	 * @ejb.interface-method
	 * 
	 * @version 3.2.0
	 * @since 3.2.0
	 * @return ArrayList
	 * @throws HinemosUnknown
	 */
	public ArrayList<RepositoryTableInfo> getVmSolutionMstList() throws HinemosUnknown{
		ArrayList<RepositoryTableInfo> list = new ArrayList<RepositoryTableInfo>();
		try {
			for (VmSolutionMstLocal bean : (Collection<VmSolutionMstLocal>)VmSolutionMstUtil.getLocalHome().findAll()) {
				list.add(new RepositoryTableInfo(bean.getVmSolutionId(), bean.getVmSolutionName()));
			}
		} catch (Exception e) {
			m_log.error("getVmSolutionMstList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return list;
	}

	/**
	 * ノード作成変更時に、利用可能な仮想化プロトコルを表示するためのメソッド。
	 * @ejb.interface-method
	 * 
	 * @version 3.2.0
	 * @since 3.2.0
	 * @return ArrayList
	 * @throws HinemosUnknown
	 */
	public ArrayList<String> getVmProtocolMstList() throws HinemosUnknown{
		ArrayList<String> list = new ArrayList<String>();
		try {
			for (String protocol : (Collection<String>)VmProtocolMstUtil.getLocalHome().findDistinctProtocol()) {
				list.add(protocol);
			}
		} catch (Exception e) {
			m_log.error("getPlatformList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return list;
	}
	/**
	 * リポジトリの最終更新時刻を取得
	 * 
	 * @ejb.interface-method
	 * 
	 * @return
	 */
	public Date getLastUpdate(){
		Date updateTime = RepositoryUpdateTime.getUpdateTime();
		m_log.debug("getLastUpdate() lastUpdate = " + updateTime.toString());
		return updateTime;
	}

	/**
	 * エージェントの状態を返します。<BR>
	 * 
	 * @ejb.interface-method
	 */
	public ArrayList<AgentStatusInfo> getAgentStatusList() {
		m_log.debug("getAgentStatusList() ");

		return AgentLibDownloader.getAgentStatusList();
	}

	/**
	 * エージェントを再起動、アップデートします。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param facilityId　ファシリティID
	 * @param agentCommand エージェントに実行するコマンド。
	 * @see com.clustercontrol.repository.bean.AgentCommandConstant
	 */
	public void restartAgent(ArrayList<String> facilityIdList, int agentCommand) {
		// Local Variables
		TopicInfo topicInfo = null;

		// MAIN
		topicInfo = new TopicInfo();

		/*
		 * com.clustercontrol.repository.bean.AgentCommandConstant
		 * public static int RESTART = 1;
		 * public static int UPDATE = 2;
		 */
		topicInfo.setAgentCommand(agentCommand);

		// 同時にアップデートされると困るので、ずらす。
		int restartSleep = 500;
		try {
			String str = HinemosProperties.getProperty("common.repository.restart.sleep");
			if (str != null &&  str.length() > 0) {
				restartSleep = Integer.parseInt(str);
				m_log.info("restart sleep = " + restartSleep);
			}
		} catch (Exception e) {
			m_log.warn("restartAgent, " + e.getMessage());
		}
		for (String facilityId : facilityIdList) {
			if (AgentConnectUtil.isValidAgent(facilityId)) {
				m_log.info("restart() : setTopic(" + facilityId + ")");
				AgentConnectUtil.setTopic(facilityId, topicInfo);
				try {
					Thread.sleep(restartSleep);
				} catch (InterruptedException e) {
					m_log.warn("restartAgent : " + e.getMessage());
				}
			} else {
				m_log.warn("restart() : agent does not connect. " +
						"(facilityId=" + facilityId + ")");
			}
		}
	}

	/**
	 * @ejb.interface-method
	 * 
	 * @param facilityId　ファシリティID
	 * @param agentCommand エージェントに実行するコマンド。
	 * @throws HinemosUnknown
	 * @see com.clustercontrol.repository.bean.AgentCommandConstant
	 */
	public HashMap<String, String> getAgentLibMap () throws HinemosUnknown {
		HashMap<String, String> map = null;
		try {
			map = AgentLibDownloader.getAgentLibMap();
		} catch (Exception e) {
			m_log.warn("getAgentLibMap() : " + e.getClass().getSimpleName() + ", " +
					e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return map;
	}

	private void checkIsUseFacility (String facilityId) throws HinemosUnknown, UsedFacility {
		JobControllerLocal job = null;
		CollectiveRunControllerLocal collectiveRun = null;
		MonitorControllerLocal monitor = null;
		NotifyControllerLocal notify = null;
		try {
			job = JobControllerUtil.getLocalHome().create();
			collectiveRun = CollectiveRunControllerUtil.getLocalHome().create();
			monitor = MonitorControllerUtil.getLocalHome().create();
			notify = NotifyControllerUtil.getLocalHome().create();
		} catch (Exception e) {
			throw new HinemosUnknown(e.getMessage(),e);
		}
		job.isUseFacilityId(facilityId);
		collectiveRun.isUseFacilityId(facilityId);
		monitor.isUseFacilityId(facilityId);
		notify.isUseFacilityId(facilityId);
	}
}