/*

Copyright (C) since 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.repository.factory;


import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.HashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.hinemosagent.bean.AgentInfo;
import com.clustercontrol.hinemosagent.util.AgentConnectUtil;
import com.clustercontrol.repository.bean.AgentStatusInfo;

/**
 * マネージャが保持しているエージェントライブラリを管理するクラス<BR>
 */
public class AgentLibDownloader {

	private static Log m_log = LogFactory.getLog(AgentLibDownloader.class);

	/**
	 * マネージャが保持しているエージェントライブラリのリストを返す。
	 * @return
	 */
	public static HashMap<String, String> getAgentLibMap() {
		HashMap<String, String> fileMap = new HashMap<String, String>();

		// TODO HINEMOS_HOMEを利用すること。
		String agentLibDir = "/opt/hinemos/lib/agent";
		File dir = new File(agentLibDir);
		File[] files = dir.listFiles();
		if (files != null) {
			for (File file : files) {
				if (file.isFile()) {
					fileMap.put(file.getName(), getMD5(file.getAbsolutePath()));
				}
			}
		} else {
			m_log.warn("agentLibDir=" + agentLibDir);
		}
		return fileMap;
	}

	/**
	 * エージェントの状態を返します。<BR>
	 * 
	 */
	public static ArrayList<AgentStatusInfo> getAgentStatusList() {
		m_log.debug("getAgentStatusList() ");

		HashMap<String, String> managerMap = getAgentLibMap();
		ArrayList<AgentStatusInfo> ret = new ArrayList<AgentStatusInfo>();
		for (AgentInfo agentInfo : AgentConnectUtil.getAgentList()) {
			// ファシリティID
			String facilityId = agentInfo.getFacilityId();
			if (facilityId == null) {
				continue;
			}
			AgentStatusInfo agent = null;
			agent = new AgentStatusInfo();
			agent.setFacilityId(facilityId);

			// ファシリティ名
			String facilityName = "";
			try {
				facilityName = NodeProperty.getProperty(facilityId).getFacilityName();
			} catch (FacilityNotFound e) {
				// ここは通らないはず。
				m_log.warn("getAgentStatusList() : " + e.getClass().getSimpleName(), e);
			}
			agent.setFacilityName(facilityName);

			// startup time
			agent.setStartupTime(agentInfo.getStartupTime());

			// last login
			agent.setLastLogin(agentInfo.getLastLogin());

			// 最新チェック
			boolean flag = true;
			HashMap<String, String> agentMap = AgentConnectUtil.getAgentLibMd5(facilityId);
			if (agentMap == null) {
				/*
				 *  該当ファシリティIDがエージェントリストに存在するけど、
				 *  ライブラリマップが存在しない状態だとここに到達する。
				 *  エージェント接続直後は、タイミングによってはこの状態になる。
				 */
				m_log.info("agentMap is null " + facilityId);
				continue;
			} else {
				for (String filename : managerMap.keySet()) {
					String agentMd5 = agentMap.get(filename);
					String managerMd5 = managerMap.get(filename);
					// マネージャが保持しているのに、エージェントが保持していない場合は、
					// エージェントが最新でない。
					if (agentMd5 == null) {
						m_log.info("agentMd5 is null + [" + facilityId +
								"] filename=" + filename);
						flag = false;
						break;
					}
					if (!agentMd5.equals(managerMd5)) {
						m_log.info("agentMd5 differs from managerMd5 [" + facilityId +
								"] filename=" + filename +
								", managerMd5=" + managerMd5 + ", agentMd5=" + agentMd5);
						flag = false;
						break;
					}
				}
			}
			agent.setNewFlag(flag);

			ret.add(agent);
		}
		return ret;
	}


	/**
	 * MD5を取得する。
	 * @param filepath
	 * @return
	 */
	private static String getMD5(String filepath) {
		MessageDigest md = null;
		DigestInputStream inStream = null;
		byte[] digest = null;
		try {
			md = MessageDigest.getInstance("MD5");
			inStream = new DigestInputStream(
					new BufferedInputStream(new FileInputStream(filepath)), md);
			while (inStream.read() != -1) {}
			digest = md.digest();
		} catch (Exception e) {
			m_log.error("getMD5() : filepath=" + filepath + ", " + e.getClass(), e);
		} finally {
			if (inStream != null) {
				try {
					inStream.close();
				} catch (Exception e) {
					m_log.error("getMD5() : close " + e.getClass());
				}
			}
		}
		return hashByte2MD5(digest);
	}

	private static String hashByte2MD5(byte []input) {
		String ret = "";
		for (byte b : input) {
			if ((0xff & b) < 0x10) {
				ret += "0" + Integer.toHexString((0xFF & b));
			} else {
				ret += Integer.toHexString(0xFF & b);
			}
		}
		return ret;
	}
}
