/*

Copyright (C) 2006 - 2008 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.snmptrap.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import javax.ejb.EJBException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.InitialContext;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.bean.SettingUpdateInfo;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapMibMasterData;
import com.clustercontrol.snmptrap.factory.AddMibMaster;
import com.clustercontrol.snmptrap.factory.AddOidMaster;
import com.clustercontrol.snmptrap.factory.DeleteMibMaster;
import com.clustercontrol.snmptrap.factory.DeleteOidMaster;
import com.clustercontrol.snmptrap.factory.ModifyMibMaster;
import com.clustercontrol.snmptrap.factory.ModifyOidMaster;
import com.clustercontrol.snmptrap.factory.SelectMibMaster;
import com.clustercontrol.snmptrap.util.SnmpTrapMasterCache;
import com.clustercontrol.snmptrap.util.SnmpTrapMibMasterCache;

/**
 * SNMPトラップ監視を制御するSession Bean <BR>
 * 
 * @ejb.bean name="MonitorSnmpTrapController"
 *           jndi-name="MonitorSnmpTrapController"
 *           type="Stateless"
 *           transaction-type="Container"
 *           view-type="local"
 * 
 * @ejb.transaction type="Required"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorTrapValueInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=SnmpTrapMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=SnmpTrapMibMasterLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 * @version 4.0.0
 * @since 2.1.0
 * 
 */
public abstract class MonitorSnmpTrapControllerBean implements SessionBean {

	private static Log m_log = LogFactory.getLog( MonitorSnmpTrapControllerBean.class );

	@SuppressWarnings("unused")
	private SessionContext m_context;

	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * MIB一覧を取得します。<BR>
	 * 
	 * 戻り値はMIBの名前(String)のArrayListです。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return MIB一覧
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<String> getMibList() throws MonitorNotFound, HinemosUnknown{
		m_log.debug("getMibList() : start");

		// MIB一覧を取得
		SelectMibMaster master = new SelectMibMaster();
		ArrayList<String> list = null;
		try {
			list = master.getMibList();
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getMibList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		m_log.debug("getMibList() : end");
		return list;
	}

	/**
	 * 監視マスタ情報一覧を取得します。<BR>
	 * 
	 * 
	 * 戻り値はSnmpTrapMasterInfoのArrayListです。
	 * 
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return MIB一覧
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 * @see com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo
	 */
	public ArrayList<SnmpTrapMasterInfo> getMasterList(String mib) throws MonitorNotFound, HinemosUnknown{
		m_log.debug("getMasterList() : start mib = " + mib);

		// 監視マスタ情報一覧を取得
		SelectMibMaster master = new SelectMibMaster();
		ArrayList<SnmpTrapMasterInfo> list = null;
		try {
			list = master.getMasterList(mib);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getMasterList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		m_log.debug("getMasterList() : end mib = " + mib);
		return list;
	}


	/**
	 * MIBの内容を取得します。<BR>
	 * 
	 * 戻り値はMIBのDTO(SnmpTrapMibMasterData)です。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return MIB一覧
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public SnmpTrapMibMasterData getMib(String Mib) throws MonitorNotFound, HinemosUnknown{
		m_log.debug("getMib() : start mib = " + Mib);

		SelectMibMaster  selector = new SelectMibMaster();

		SnmpTrapMibMasterData data = null;
		try {
			data = selector.getMibMasterData(Mib);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("SnmpTrapMibMasterData " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		m_log.debug("getMib() : end mib = " + Mib);
		return data;
	}

	/**
	 * MIB情報の登録
	 *
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param MibData
	 * @return
	 * @throws HinemosUnknown
	 */
	public boolean addMibMaster(SnmpTrapMibMasterData MibData) throws HinemosUnknown {
		m_log.debug("addMibMaster() : start mib = " + MibData.getMib());

		Principal pri = m_context.getCallerPrincipal();

		AddMibMaster adder = new AddMibMaster();

		TransactionManager tm = null;
		boolean commit = false;

		boolean flag;
		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			flag = adder.add(MibData , pri.getName());

			commit = flag;
		} catch (Exception e) {
			m_log.error("addMibMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();
						SnmpTrapMibMasterCache.clear();

						// read-committedのため、commit後に外部コンポーネントに通知する
						SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
		m_log.debug("addMibMaster() : end mib = " + MibData.getMib());
		return flag;
	}

	/**
	 * MIB情報の変更
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param MibData
	 * @return
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public boolean modifyMibMaster(SnmpTrapMibMasterData MibData) throws MonitorNotFound, HinemosUnknown{
		m_log.debug("modifyMibMaster() : start mib = " + MibData.getMib());

		Principal pri = m_context.getCallerPrincipal();

		ModifyMibMaster modifier = new ModifyMibMaster();

		TransactionManager tm = null;
		boolean commit = false;

		boolean flag;
		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			flag = modifier.modify(MibData , pri.getName());

			commit = flag;
		} catch (MonitorNotFound e) {
			m_log.warn("modifyMibMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("modifyMibMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();
						SnmpTrapMibMasterCache.clear();

						// read-committedのため、commit後に外部コンポーネントに通知する
						SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
		m_log.debug("modifyMibMaster() : end mib = " + MibData.getMib());
		return flag;
	}

	/**
	 * MIB情報の削除
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param mib
	 * @return
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public boolean deleteMibMaster(String Mib) throws MonitorNotFound, HinemosUnknown{
		m_log.debug("deleteMibMaster() : start mib = " + Mib);

		Principal pri = m_context.getCallerPrincipal();

		DeleteMibMaster eraser= new DeleteMibMaster();

		TransactionManager tm = null;
		boolean commit = false;

		boolean flag;
		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			flag = eraser.delete(Mib);

			commit = flag;
		} catch (MonitorNotFound e) {
			m_log.warn("deleteMibMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("deleteMibMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();
						SnmpTrapMibMasterCache.clear();

						// read-committedのため、commit後に外部コンポーネントに通知する
						SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
		m_log.debug("deleteMibMaster() : start mib = " + Mib);
		return flag;
	}


	/**
	 * OID情報の登録
	 *
	 * @ejb.interface-method
	 *
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 *
	 * @param MibData
	 * @return
	 * @throws HinemosUnknown
	 */
	public boolean addOidMaster(SnmpTrapMasterInfo Oid) throws HinemosUnknown {
		m_log.debug("addOidMaster() : start Uei = " + Oid.getUei());

		Principal pri = m_context.getCallerPrincipal();

		AddOidMaster adder = new AddOidMaster();

		TransactionManager tm = null;
		boolean commit = false;

		boolean flag;
		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			flag = adder.add(Oid , pri.getName());

			commit = flag;
		} catch (Exception e) {
			m_log.error("addOidMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();
						SnmpTrapMasterCache.clear();

						// read-committedのため、commit後に外部コンポーネントに通知する
						SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
		m_log.debug("addOidMaster() : end Uei = " + Oid.getUei());
		return flag;
	}

	/**
	 * OID情報の変更
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param MibData
	 * @return
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public boolean modifyOidMaster(SnmpTrapMasterInfo Oid) throws MonitorNotFound, HinemosUnknown{
		m_log.debug("modifyOidMaster() : start Uei = " + Oid.getUei());

		Principal pri = m_context.getCallerPrincipal();

		ModifyOidMaster modifier = new ModifyOidMaster();

		TransactionManager tm = null;
		boolean commit = false;

		boolean flag;
		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			flag = modifier.modify(Oid , pri.getName());

			commit = flag;
		} catch (MonitorNotFound e) {
			m_log.warn("modifyOidMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("modifyOidMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();
						SnmpTrapMasterCache.clear();

						// read-committedのため、commit後に外部コンポーネントに通知する
						SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
		m_log.debug("modifyOidMaster() : end Uei = " + Oid.getUei());
		return flag;
	}

	/**
	 * OID情報の削除
	 * 
	 * @ejb.interface-method
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param mib
	 * @return
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public boolean deleteOidMaster(SnmpTrapMasterInfo oid) throws MonitorNotFound, HinemosUnknown{
		m_log.debug("deleteOidMaster() : start Uei = " + oid.getUei());

		Principal pri = m_context.getCallerPrincipal();

		DeleteOidMaster eraser= new DeleteOidMaster();

		TransactionManager tm = null;
		boolean commit = false;

		boolean flag;
		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();

			flag = eraser.delete(oid);

			commit = flag;
		} catch (MonitorNotFound e) {
			m_log.warn("delteOidMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("delteOidMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (commit) {
						tm.commit();
						SnmpTrapMasterCache.clear();

						// read-committedのため、commit後に外部コンポーネントに通知する
						SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
		m_log.debug("deleteOidMaster() : end Uei = " + oid.getUei());
		return flag;
	}
}
