/*
Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.ws.endpoint;

import java.util.ArrayList;

import javax.annotation.Resource;
import javax.ejb.CreateException;
import javax.naming.NamingException;
import javax.xml.ws.WebServiceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.MaintenanceDuplicate;
import com.clustercontrol.fault.MaintenanceNotFound;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.maintenance.bean.MaintenanceInfo;
import com.clustercontrol.maintenance.bean.MaintenanceTypeMst;
import com.clustercontrol.maintenance.ejb.session.MaintenanceControllerLocal;
import com.clustercontrol.maintenance.ejb.session.MaintenanceControllerLocalHome;
import com.clustercontrol.notify.util.NotifyRelationCache;

/**
 * メンテナンス用のWebAPIエンドポイント
 */
@javax.jws.WebService(targetNamespace = "http://maintenance.ws.clustercontrol.com")
public class MaintenanceEndpoint {
	@Resource
	WebServiceContext wsctx;

	private static Log m_log = LogFactory.getLog( MaintenanceEndpoint.class );
	private static Log m_opelog = LogFactory.getLog("HinemosOperation");

	/**
	 * echo(WebサービスAPI疎通用)
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @param str
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String echo(String str) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<String> roleList = new ArrayList<String>();
		HttpAuthenticator.authCheck(wsctx, roleList);

		return str + ", " + str;
	}

	/**
	 * メンテナンス(履歴情報削除)用のSessionBeanの取得
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 */
	private MaintenanceControllerLocal getMaintenanceControllerLocal() throws HinemosUnknown, InvalidUserPass {
		MaintenanceControllerLocal local = null;
		try {
			String account = HttpAuthenticator.getAccount(wsctx);
			NamingContext namingContext = HttpAuthenticator.getContext(account);
			MaintenanceControllerLocalHome localHome =
				(MaintenanceControllerLocalHome)
				namingContext.lookup(MaintenanceControllerLocalHome.JNDI_NAME);
			local = localHome.create();
		} catch (CreateException e) {
			m_log.error("getMaintenanceControllerLocal CreateException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getMaintenanceControllerLocal NamingException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return local;
	}

	/**
	 * メンテナンス情報を追加します。
	 * 
	 * MaintenanceWrite権限が必要
	 * 
	 * @throws MaintenanceDuplicate
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * 
	 */
	public void addMaintenance(MaintenanceInfo info) throws HinemosUnknown, MaintenanceDuplicate, InvalidUserPass, InvalidRole,InvalidSetting
	{
		m_log.debug("addMaintenance");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MAINTENANCE_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", MaintenanceID=");
			msg.append(info.getMaintenanceId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MAINTENANCE + " Add, Method=addMaintenance, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		try{
			getMaintenanceControllerLocal().addMaintenance(info);
		} finally {
			// FIXME:本来はSessionBeanに実装すべき
			// トランザクションのコミット後にキャッシュをクリアする。
			NotifyRelationCache.refresh();
		}
	}

	/**
	 * メンテナンス情報を変更します。
	 * 
	 * MaintenanceWrite権限が必要
	 * 
	 * @throws MaintenanceNotFound
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 */
	public void modifyMaintenance(MaintenanceInfo info) throws HinemosUnknown, NotifyNotFound, MaintenanceNotFound, InvalidUserPass, InvalidRole,InvalidSetting
	{
		m_log.debug("modifyMaintenance");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MAINTENANCE_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", MaintenanceID=");
			msg.append(info.getMaintenanceId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MAINTENANCE + " Change, Method=modifyMaintenance, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		try{
			getMaintenanceControllerLocal().modifyMaintenance(info);
		} finally {
			// FIXME:本来はSessionBeanに実装すべき
			// トランザクションのコミット後にキャッシュをクリアする。
			NotifyRelationCache.refresh();
		}
	}

	/**
	 * メンテナンス情報を削除します。
	 * 
	 * MaintenanceWrite権限が必要
	 * 
	 * @throws NotifyNotFound
	 * @throws MaintenanceNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 */
	public void deleteMaintenance(String maintenanceId) throws HinemosUnknown, MaintenanceNotFound, NotifyNotFound, InvalidUserPass, InvalidRole {
		m_log.debug("deleteMaintenance");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MAINTENANCE_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", MaintenanceID=");
		msg.append(maintenanceId);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MAINTENANCE + " Delete, Method=deleteMaintenance, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		try {
			getMaintenanceControllerLocal().deleteMaintenance(maintenanceId);
		} finally {
			// FIXME:本来はSessionBeanに実装すべき
			// トランザクションのコミット後にキャッシュをクリアする。
			NotifyRelationCache.refresh();
		}
	}

	/**
	 * メンテナンス情報を取得します。
	 *
	 * MaintenanceRead権限が必要
	 *
	 * @return
	 * @throws HinemosUnknown
	 * @throws MaintenanceNotFound
	 * @throws NotifyNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 */
	public MaintenanceInfo getMaintenanceInfo(String maintenanceId) throws NotifyNotFound, MaintenanceNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getMaintenanceInfo");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MAINTENANCE_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", MaintenanceID=");
		msg.append(maintenanceId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MAINTENANCE + " Get, Method=getMaintenanceInfo, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getMaintenanceControllerLocal().getMaintenanceInfo(maintenanceId);
	}

	/**
	 * メンテナンス情報の一覧を取得します。<BR>
	 * 
	 * MaintenanceRead権限が必要
	 * 
	 * @return メンテナンス情報の一覧を保持するArrayList
	 * @throws MaintenanceNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<MaintenanceInfo> getMaintenanceList() throws HinemosUnknown, MaintenanceNotFound, InvalidUserPass, InvalidRole {
		m_log.debug("getMaintenanceList");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MAINTENANCE_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MAINTENANCE + " Get, Method=getMaintenanceList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return getMaintenanceControllerLocal().getMaintenanceList();
	}

	/**
	 * メンテナンス種別の一覧を取得します。<BR>
	 * 
	 * MaintenanceRead権限が必要
	 * 
	 * @return メンテナンス種別の一覧を保持するArrayList
	 * @throws HinemosUnknown
	 * @throws MaintenanceNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<MaintenanceTypeMst> getMaintenanceTypeList() throws MaintenanceNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getMaintenanceTypeList");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MAINTENANCE_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MAINTENANCE + " Get, Method=getMaintenanceTypeList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return getMaintenanceControllerLocal().getMaintenanceTypeList();
	}

	/**
	 * メンテナンスの有効、無効を変更するメソッドです。
	 * 
	 * MaintenanceWrite権限が必要
	 * 
	 * @throws HinemosUnknown
	 * @throws MaintenanceNotFound
	 * @throws NotifyNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 */
	public void setMaintenanceStatus(String maintenanceId, boolean validFlag) throws NotifyNotFound, MaintenanceNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("setMaintenanceStatus");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MAINTENANCE_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", MaintenanceID=");
		msg.append(maintenanceId);
		msg.append(", ValidFlag=");
		msg.append(validFlag);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MAINTENANCE + " Change Valid, Method=setMaintenanceStatus, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		getMaintenanceControllerLocal().setMaintenanceStatus(maintenanceId, validFlag);
	}
}