/*
Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.ws.endpoint;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;
import javax.ejb.CreateException;
import javax.naming.NamingException;
import javax.xml.ws.WebServiceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.performance.bean.CollectMasterInfo;
import com.clustercontrol.performance.ejb.session.PerformanceCollectMasterControllerLocal;
import com.clustercontrol.performance.ejb.session.PerformanceCollectMasterControllerLocalHome;
import com.clustercontrol.repository.ejb.entity.CollectorPlatformMstData;

/**
 * 収集項目マスタ用のWebAPIエンドポイント
 */
@javax.jws.WebService(targetNamespace = "http://collectmaster.ws.clustercontrol.com")
public class PerformanceCollectMasterEndpoint {
	@Resource
	WebServiceContext wsctx;

	private static Log m_log = LogFactory.getLog( PerformanceCollectMasterEndpoint.class );
	private static Log m_opelog = LogFactory.getLog("HinemosOperation");

	/**
	 * echo(WebサービスAPI疎通用)
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @param str
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String echo(String str) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<String> roleList = new ArrayList<String>();
		HttpAuthenticator.authCheck(wsctx, roleList);

		return str + ", " + str;
	}

	/**
	 * 収集項目マスタのSessionBeanの取得
	 * 
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 */
	private PerformanceCollectMasterControllerLocal getPerformanceCollectMasterControllerLocal() throws HinemosUnknown, InvalidUserPass {
		PerformanceCollectMasterControllerLocal local = null;
		try {
			String account = HttpAuthenticator.getAccount(wsctx);
			NamingContext namingContext = HttpAuthenticator.getContext(account);
			PerformanceCollectMasterControllerLocalHome localHome =
				(PerformanceCollectMasterControllerLocalHome)
				namingContext.lookup(PerformanceCollectMasterControllerLocalHome.JNDI_NAME);
			local = localHome.create();
		} catch (CreateException e) {
			m_log.error("getPerformanceCollectorMasterControllerLocal CreateException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getPerformanceCollectorMasterControllerLocal NamingException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return local;
	}

	/**
	 * 収集項目マスタデータを一括で登録します。
	 * @param collectMasterInfo
	 * @return 登録に成功した場合、true
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	public boolean addCollectMaster(CollectMasterInfo collectMasterInfo) throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("addCollectMaster");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.PERFORMANCE_EXECUTE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_PERFORMANCE + " Add Master, Method=addCollectMaster, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return getPerformanceCollectMasterControllerLocal().addCollectMaster(collectMasterInfo);
	}

	/**
	 * 収集項目のマスタ情報を全て削除します。
	 * 
	 * @return 削除に成功した場合、true
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	public boolean deleteCollectMasterAll() throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("deleteCollectMasterAll");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.PERFORMANCE_EXECUTE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_PERFORMANCE + " Delete Master, Method=deleteCollectMasterAll, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return getPerformanceCollectMasterControllerLocal().deleteCollectMasterAll();
	}

	/**
	 * 収集マスタ情報を一括で取得します。
	 * 
	 * @return 収集マスタ情報
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	public CollectMasterInfo getCollectMasterInfo() throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getCollectMasterInfo");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.PERFORMANCE_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_PERFORMANCE + " Get Master, Method=getCollectMasterInfo, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return getPerformanceCollectMasterControllerLocal().getCollectMasterInfo();
	}



	/**
	 * プラットフォーム定義を登録します。
	 * 
	 * @param data プラットフォーム定義情報
	 * @return 登録に成功した場合、true
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public boolean addCollectPlatformMaster(CollectorPlatformMstData data) throws  HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("addCollectPlatformMaster");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.PERFORMANCE_EXECUTE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", PlatformId=");
		msg.append(data.getPlatformId());
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_PERFORMANCE + " Add Platform, Method=addCollectPlatformMaster, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getPerformanceCollectMasterControllerLocal().addCollectPlatformMaster(data);
	}

	/**
	 * プラットフォーム定義を削除します。
	 * 
	 * @param platformId プラットフォームID
	 * @return 削除に成功した場合、true
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public boolean deleteCollectPlatformMaster(String platformId) throws  HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("deleteCollectPlatformMaster");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.PERFORMANCE_EXECUTE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", PlatformId=");
		msg.append(platformId);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_PERFORMANCE + " Delete Platform, Method=deleteCollectPlatformMaster, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getPerformanceCollectMasterControllerLocal().deleteCollectPlatformMaster(platformId);
	}

	/**
	 * プラットフォーム定義を取得します。
	 * 
	 * @return プラットフォーム定義のリスト
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public List<CollectorPlatformMstData> getCollectPlatformMaster() throws  HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getCollectPlatformMaster");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.PERFORMANCE_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_PERFORMANCE + " Get Platform, Method=getCollectPlatformMaster, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return getPerformanceCollectMasterControllerLocal().getCollectPlatformMaster();
	}
}
