/*
Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.ws.endpoint;

import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;

import javax.annotation.Resource;
import javax.ejb.CreateException;
import javax.naming.NamingException;
import javax.xml.ws.WebServiceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.fault.FacilityDuplicate;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.UsedFacility;
import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.repository.bean.AgentStatusInfo;
import com.clustercontrol.repository.bean.FacilityInfo;
import com.clustercontrol.repository.bean.FacilityTreeItem;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.RepositoryTableInfo;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocalHome;

/**
 * リポジトリ操作用のWebAPIエンドポイント
 */
@javax.jws.WebService(targetNamespace = "http://repository.ws.clustercontrol.com")
public class RepositoryEndpoint {
	@Resource
	WebServiceContext wsctx;

	private static Log m_log = LogFactory.getLog( RepositoryEndpoint.class );
	private static Log m_opelog = LogFactory.getLog("HinemosOperation");

	/**
	 * echo(WebサービスAPI疎通用)
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @param str
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String echo(String str) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<String> roleList = new ArrayList<String>();
		HttpAuthenticator.authCheck(wsctx, roleList);

		return str + ", " + str;
	}

	/**
	 * リポジトリ管理用のSessionBeanの取得
	 * 
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 */
	private RepositoryControllerLocal getRepositoryControllerLocal() throws HinemosUnknown, InvalidUserPass {
		RepositoryControllerLocal local = null;
		try {
			String account = HttpAuthenticator.getAccount(wsctx);
			NamingContext namingContext = HttpAuthenticator.getContext(account);
			RepositoryControllerLocalHome localHome =
				(RepositoryControllerLocalHome)
				namingContext.lookup(RepositoryControllerLocalHome.JNDI_NAME);
			local = localHome.create();
		} catch (CreateException e) {
			m_log.error("getRepositoryControllerLocal CreateException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getRepositoryControllerLocal NamingException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return local;
	}

	/**********************
	 * ファシリティツリーのメソッド群
	 **********************/
	/**
	 * ファシリティツリー（スコープツリー）取得を取得します。
	 * <BR>
	 * 取得したファシリティツリーには割り当てられたノードを含みます。<BR>
	 * このメソッドはクライアントの画面情報を作成するために
	 * 呼び出されます。クライアントのロケールを引数をして必要とします。<BR>
	 * （最上位のスコープという表記をscopeをいう表記を切り替えるため。）
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param locale クライアントのロケール
	 * @return FacilityTreeItemの階層オブジェクト
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public FacilityTreeItem getFacilityTree() throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getFacilityTree :");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getFacilityTree, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getFacilityTree(Locale.getDefault());
	}

	/**
	 * ファシリティツリー（スコープツリー）取得を取得します。
	 * <BR>
	 * 取得したファシリティツリーには割り当てられたノードを含みます。<BR>
	 * このメソッドはクライアントの画面情報を作成するために
	 * 呼び出されます。クライアントのロケールを引数をして必要とします。<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @param locale クライアントのロケール
	 * @return FacilityTreeItemの階層オブジェクト
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public FacilityTreeItem getExecTargetFacilityTreeByFacilityId(String facilityId) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getExecTargetFacilityTreeByFacilityId :");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getExecTargetFacilityTreeByFacilityId, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getExecTargetFacilityTree(facilityId, Locale.getDefault());
	}

	/**********************
	 * ノードのメソッド群(getter)
	 **********************/

	/**
	 * ノード一覧を取得します。<BR>
	 * リポジトリに登録されているすべてのノードを取得します。<BR>
	 * 戻り値はNodeInfoのArrayListで、NodeInfoには
	 * ノードの下記情報のみ格納されています。
	 * ・ファシリティID
	 * ・ファシリティ名
	 * ・IPアドレスバージョン、IPv4, Ipv6
	 * ・説明
	 * getNodeFacilityIdListを利用すること。（getNodeと組み合わせて利用する。）
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @return NodeInfoの配列
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<NodeInfo> getNodeListAll() throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getNodeListAll : ");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getNodeListAll, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getNodeList();
	}
	/**
	 * ノード一覧を取得します。<BR>
	 * 
	 * クライアントなどで検索した場合に呼ばれ、該当するノード一覧を取得します。<BR>
	 * 引数はNodeInfoであり、"ファシリティID"、"ファシリティ名"、"説明"、
	 * "IPアドレス"、"OS名"、"OSリリース"、"管理者"、"連絡先"が１つ以上含まれており、
	 * その条件を元に該当するノードを戻り値とします。<BR>
	 * 戻り値はNodeInfoのArrayListで、NodeInfoには
	 * ノードの"ファシリティID"、"ファシリティ名"、"説明"のみ格納されています。<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @param property　検索条件のプロパティ
	 * @return NodeInfoの配列
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<NodeInfo> getFilterNodeList(NodeInfo property) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getFilterNodeList : nodeInfo=" + property);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if (property != null) {
			StringBuffer msg = new StringBuffer();
			msg.append(", FacilityID=");
			msg.append(property.getFacilityId());
			m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getFilterNodeList, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getFilterNodeList(property);
	}

	/**
	 * 
	 * 監視・ジョブ等の処理を実行する対象となる、ファシリティIDのリストを取得します。
	 * 引数で指定されたファシリティIDが、ノードかスコープによって、以下のようなリストを取得します。
	 * 
	 * ノードの場合
	 *   引数で指定されたfacilityIdが格納されたArrauList
	 *   ただし、管理対象（有効/無効フラグが真）の場合のみ
	 * 
	 * スコープの場合
	 *   配下に含まれるノードのファシリティIDが格納されたArrayList
	 *   ただし、管理対象（有効/無効フラグが真）のみ
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @param facilityId 処理を実行する対象となるファシリティID
	 * @return 有効なノードのリスト（有効なノードがひとつも含まれない場合は空のリスト）
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<String> getExecTargetFacilityIdList(String facilityId) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getExecTargetFacilityIdList : facilityId=" + facilityId);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getExecTargetFacilityIdList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getExecTargetFacilityIdList(facilityId);
	}


	/**
	 * ノードの詳細プロパティを取得します。<BR>
	 * 
	 * faciliyIDで指定されるノードの詳細プロパティを取得します。<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @param facilityId ファシリティID
	 * @param locale クライアントのロケール
	 * @return ノード情報プロパティ
	 * @throws FacilityNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public NodeInfo getNode(String facilityId) throws FacilityNotFound, InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getNode : facilityId=" + facilityId);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getNode, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getNode(facilityId);
	}


	/**
	 * ファシリティパスを取得します。<BR>
	 * 
	 * 第一引数がノードの場合は、パスではなく、ファシリティ名。<BR>
	 * 第一引数がスコープの場合は、第二引数との相対的なファシリティパスを取得します。<BR>
	 * (例　○○スコープ>××システム>DBサーバ)<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @param facilityId ファシリティID
	 * @param parentFacilityId 上位のファシリティID
	 * @return String ファシリティパス
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public String getFacilityPath(String facilityId, String parentFacilityId) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getFacilityPath : facilityId=" + facilityId +
				", parentFacilityId=" + parentFacilityId);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		msg.append(", ParentFacilityID=");
		msg.append(parentFacilityId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getFacilityPath, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getFacilityPath(facilityId, parentFacilityId);
	}


	/**
	 * SNMPを利用してノードの情報を取得します。<BR>
	 * 
	 * 
	 * クライアントからSNMPで検出を行った際に呼び出されるメソッドです。<BR>
	 * SNMPポーリングにより、ノード詳細プロパティをセットし、クライアントに返す。
	 * 戻り値はNodeInfo
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @param pollingData ポーリング対象のIPアドレス、コミュニティ名、バージョン、ポート
	 * @param locale クライアントのロケール
	 * @return ノード情報
	 * @throws HinemosUnknown
	 * @throws FacilityNotFound
	 * @throws UnknownHostException
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public NodeInfo getNodePropertyBySNMP(String ipAddress, int port, String community, String version) throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getNodePropertyBySNMP : ipAddress=" + ipAddress + ", port=" + port +
				", community=" + community + ", version=" + version);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", IPAddress=");
		msg.append(ipAddress);
		msg.append(", Port=");
		msg.append(port);
		msg.append(", Community=");
		msg.append(community);
		msg.append(", Version=");
		msg.append(version);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getNodePropertyBySNMP, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getNodePropertyBySNMP(ipAddress, port, community, version);
	}


	/**********************
	 * ノードのメソッド群(getter以外)
	 **********************/
	/**
	 * ノードを新規に追加します。<BR>
	 * またこのメソッドで組み込みスコープである"登録済みノード"スコープにも
	 * 割り当てが行われます。
	 * 
	 * RepositoryWrite権限が必要
	 * 
	 * @param nodeinfo 追加するノード情報のプロパティ
	 * @throws FacilityDuplicate
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void addNode(NodeInfo nodeInfo) throws FacilityDuplicate, InvalidUserPass, InvalidRole, HinemosUnknown,InvalidSetting {
		m_log.debug("addNode : nodeInfo=" + nodeInfo);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(nodeInfo != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", FacilityID=");
			msg.append(nodeInfo.getFacilityId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Add Node, Method=addNode, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.addNode(nodeInfo);
	}


	/**
	 * ノードを変更します。<BR>
	 * 引数のpropertyには変更する属性のみを設定してください。<BR>
	 * 
	 * RepositoryWrite権限が必要
	 * 
	 * @version 2.0.0
	 * @since 1.0.0
	 * 
	 * @param property　変更するノード情報のプロパティ
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void modifyNode(NodeInfo property) throws InvalidUserPass, InvalidRole, HinemosUnknown,InvalidSetting {
		m_log.debug("modifyNode : nodeInfo=" + property);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(property != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", FacilityID=");
			msg.append(property.getFacilityId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Change Node, Method=modifyNode, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.modifyNode(property);
	}

	/**
	 * ノード詳細情報を一括変更します。<BR>
	 * クライアントにあるスコープを指定して変更を行うためのメソッド
	 * 引数のpropertyには変更する属性のみを設定してください。<BR>
	 * 
	 * RepositoryWrite権限が必要
	 * 
	 * @version 2.0.0
	 * @since 1.0.0
	 * 
	 * @param property　変更するノード情報のプロパティ
	 * @param parentFacilityId　スコープのファシリティID
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void modifyAllNode(NodeInfo property, String parentFacilityId) throws InvalidUserPass, InvalidRole, HinemosUnknown,InvalidSetting {
		m_log.debug("modifyAllNode : nodeInfo=" + property);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(property != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", ParentFacilityID=");
			msg.append(parentFacilityId);
			msg.append(", FacilityID=");
			msg.append(property.getFacilityId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Change Node, Method=modifyAllNode, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.modifyNode(property, parentFacilityId);
	}

	/**
	 * ノード情報を削除します。<BR>
	 * 
	 * faciityIDで指定されたノードをリポジトリから削除します。
	 * 
	 * RepositoryWrite権限が必要
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @throws UsedFacility
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void deleteNode(String facilityId) throws UsedFacility, InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("deleteNode : facilityId=" + facilityId);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Delete Node, Method=deleteNode, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.deleteNode(facilityId);
	}

	/**********************
	 * スコープのメソッド群
	 **********************/
	/**
	 * ファシリティ一覧を取得します。<BR>
	 * あるスコープを指定してその直下にあるファシリティを取得します。<BR>
	 * このメソッドは引数としてそのスコープのファシリティIDを要求します。<BR>
	 * 戻り値はArrayListで中のScopeInfoには子の
	 * "ファシリティID"、"ファシリティ名"、"説明"のみ格納されています。<BR>
	 * 
	 * RepositoryWrite権限が必要
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId　スコープのファシリティID
	 * @return ScopeInfoの配列
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<FacilityInfo> getFacilityList(String parentFacilityId) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", ParentFacilityID=");
		msg.append(parentFacilityId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getScopeList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return repository.getFacilityList(parentFacilityId);
	}

	/**
	 * スコープ用プロパティ情報を取得します。<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @param facilityId ファシリティID
	 * @param locale クライアントのロケール
	 * @return スコープのプロパティ情報（ファシリティID、ファシリティ名、説明）
	 * @throws HinemosUnknown
	 * @throws FacilityNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public ScopeInfo getScope(String facilityId) throws FacilityNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getScope : facilityId=" + facilityId);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getScope, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getScope(facilityId);
	}

	/**
	 * スコープを新規に追加します。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープの下にpropertyで指定されるスコープを
	 * 追加します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * 
	 * RepositoryWrite権限が必要
	 * 
	 * @param parentFacilityId
	 * @param property
	 * @throws FacilityDuplicate
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void addScope(String parentFacilityId, ScopeInfo property) throws FacilityDuplicate, InvalidUserPass, InvalidRole, HinemosUnknown,InvalidSetting {
		m_log.debug("addScope : parentFacilityId=" + parentFacilityId +
				", scopeInfo=" + property);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(property != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", ParentFacilityID=");
			msg.append(parentFacilityId);
			msg.append(", FacilityID=");
			msg.append(property.getFacilityId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Add Scope, Method=addScope, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.addScope(parentFacilityId, property);
	}

	/**
	 * スコープの情報を変更します。<BR>
	 * 
	 * 引数propertyで指定した内容でスコープ情報を更新します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * propertyに含まれるファシリティIDに対応するスコープの情報が変更されます。<BR>
	 * 
	 * RepositoryWrite権限が必要
	 * 
	 * @param property
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void modifyScope(ScopeInfo property) throws InvalidUserPass, InvalidRole, HinemosUnknown,InvalidSetting {
		m_log.debug("modifyScope : scopeInfo=" + property);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(property != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", FacilityID=");
			msg.append(property.getFacilityId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Change Scope, Method=modifyScope, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.modifyScope(property);
	}

	/**
	 * スコープ情報を削除します。<BR>
	 * 
	 * faciityIDで指定されたスコープをリポジトリから削除します。
	 * 
	 * RepositoryWrite権限が必要
	 * 
	 * @param facilityId ファシリティID
	 * @throws UsedFacility
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void deleteScope(String facilityId) throws UsedFacility, InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("deleteScope : facilityId=" + facilityId);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Delete Scope, Method=deleteScope, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.deleteScope(facilityId);
	}



	/**********************
	 * ノード割り当てのメソッド群
	 **********************/
	/**
	 * 割当ノード一覧を取得します。<BR>
	 * 
	 * あるファシリティIDの配下または直下のノード一覧を取得します。<BR>
	 * 引数がノードの場合は、そのノードのファシリティIDを返す。<BR>
	 * このメソッドでは、引数levelで直下または配下を制御します。<BR>
	 * 戻り値はNodeInfoのArrayListで、NodeInfoには
	 * ノードの"ファシリティID"、"ファシリティ名"、"説明"のみ格納されています。<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @param parentFacilityId
	 * @param level　取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return NodeInfoの配列
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<NodeInfo> getNodeList(String parentFacilityId, int level) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getNodeList : parentFacilityId=" + parentFacilityId + ", level=" + level);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", ParentFacilityID=");
		msg.append(parentFacilityId);
		msg.append(", Level=");
		msg.append(level);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getNodeList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getNodeList(parentFacilityId, level);
	}


	/**
	 * 割当スコープ一覧を取得します。<BR>
	 * 割り当てスコープ一覧とは、あるノードが属しているスコープすべてを
	 * 一覧表示したものです。
	 * クライアントの割り当てスコープビューの表示データとなります。
	 * 戻り値はArrayListのArrayListで中のArrayListには"スコープ"が最上位からの
	 * スコープパス表記で（Stringで）格納されています。
	 * 外のArrayListには、そのレコードが順に格納されています。
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ノードのファシリティID
	 * @return Stringの配列
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<String> getNodeScopeList(String facilityId) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getNodeScopeList : facilityId=" + facilityId);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getNodeScopeList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getNodeScopeList(facilityId);
	}

	/**
	 * ファシリティIDリストを取得します。<BR>
	 * 
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（スコープ、ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @param parentFacilityId ファシリティID
	 * @param level　取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return ファシリティIDの配列
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<String> getFacilityIdList(String parentFacilityId, int level) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getFacilityIdList : parentFacilityId=" + parentFacilityId + ", level=" + level);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", ParentFacilityID=");
		msg.append(parentFacilityId);
		msg.append(", Level=");
		msg.append(level);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getFacilityIdList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getFacilityIdList(parentFacilityId, level);
	}


	/**
	 * ノードのファシリティIDリストを取得します。<BR>
	 * 
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @param parentFacilityId ファシリティID
	 * @param level  取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return ファシリティIDの配列
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<String> getNodeFacilityIdList(String parentFacilityId, int level) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getNodeFacilityIdList : parentFacilityId=" + parentFacilityId + ", level=" + level);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", ParentFacilityID=");
		msg.append(parentFacilityId);
		msg.append(", Level=");
		msg.append(level);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getNodeFacilityIdList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getNodeFacilityIdList(parentFacilityId, level);
	}

	/**
	 * スコープへのノードの割り当てを行います。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープにfacilityIdsで指定されるノード群を
	 * 割り当てます。
	 * 
	 * RepositoryWrite権限が必要
	 * 
	 * @param parentFacilityId　ノードを割り当てるスコープ
	 * @param facilityIds 割り当てさせるノード(群)
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void assignNodeScope(String parentFacilityId, String[] facilityIds) throws InvalidUserPass, InvalidRole, HinemosUnknown,InvalidSetting {
		m_log.debug("assignNodeScope : parentFacilityId=" + parentFacilityId + ", facilityIds=" + facilityIds);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if (facilityIds != null) {
			StringBuffer msg = new StringBuffer();
			msg.append(", ParentFacilityID=");
			msg.append(parentFacilityId);
			msg.append(", FacilityID=");
			msg.append(Arrays.toString(facilityIds));
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Assign Node, Method=assignNodeScope, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.assignNodeScope(parentFacilityId, facilityIds);
	}

	/**
	 * ノードをスコープから削除します。（割り当てを解除します。）<BR>
	 * parentFacilityIdで指定されるスコープからfacilityIdsで指定されるノード群を
	 * 削除（割り当て解除）します。
	 * 
	 * RepositoryWrite権限が必要
	 * 
	 * @param parentFacilityId ノードを取り除くスコープ
	 * @param facilityIds 取り除かれるノード（群）
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public void releaseNodeScope(String parentFacilityId, String[] facilityIds) throws InvalidUserPass, InvalidRole, HinemosUnknown,InvalidSetting {
		m_log.debug("releaseNodeScope : parentFacilityId=" + parentFacilityId + ", facilityIds=" + facilityIds);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if (facilityIds != null) {
			StringBuffer msg = new StringBuffer();
			msg.append(", ParentFacilityID=");
			msg.append(parentFacilityId);
			msg.append(", FacilityID=");
			msg.append(Arrays.toString(facilityIds));
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Release Node, Method=releaseNodeScope, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.releaseNodeScope(parentFacilityId, facilityIds);
	}


	/**********************
	 * その他のメソッド群
	 **********************/
	/**
	 * ファシリティがノードかどうかをチェックします。<BR>
	 * 
	 * ファシリティIDに対応するものがノードかチェックし、結果をbooleanで返します。
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @param facilityId　ファシリティID
	 * @return true：ノード　false:ノードではない（スコープ）
	 * @throws FacilityNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public boolean isNode(String facilityId) throws FacilityNotFound, InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("isNode : facilityId=" + facilityId);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Check, Method=isNode, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.isNode(facilityId);
	}

	/**
	 * ノード作成変更時に、利用可能プラットフォームを表示するためのメソッド。
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @return ArrayList
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<RepositoryTableInfo> getPlatformList() throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getPlatformList : ");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getPlatformList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getPlatformList();
	}

	/**
	 * ノード作成変更時に、仮想化種別hostのファシリティIDを表示するためのメソッド。
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @return ArrayList
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<RepositoryTableInfo> getHostList() throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getHostList : ");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getHostList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getHostList();
	}

	/**
	 * OpenFlowスイッチであるノードを取得する。
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @return List
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public List<NodeInfo> getOpenFlowSwitchList() throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getOpenFlowSwitchList : ");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getOpenFlowSwitchList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getOpenFlowSwitchList();
	}

	/**
	 * ノード作成変更時に、利用可能な仮想化ソリューションを表示するためのメソッド。
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @version 3.2.0
	 * @since 3.2.0
	 * @return ArrayList
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<RepositoryTableInfo> getVmSolutionMstList() throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getVmSolutionMstList : ");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getVmSolutionMstList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getVmSolutionMstList();
	}

	/**
	 * ノードの「管理対象」フラグを変更します。
	 * 
	 * RepositoryWrite権限が必要
	 * 
	 * @return ArrayList
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws InvalidSetting
	 * @throws FacilityNotFound
	 */
	public void setValid(String facilityId, boolean validFlag) throws InvalidUserPass, InvalidRole, HinemosUnknown, FacilityNotFound, InvalidSetting {
		m_log.debug("setValid : ");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		msg.append(", ValidFlag=");
		msg.append(validFlag);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Change Valid, Method=setValid, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		NodeInfo info = repository.getNode(facilityId);
		info.setValid(validFlag);
		repository.modifyNode(info);
	}

	/**
	 * ノード作成変更時に、利用可能な仮想化プロトコルを表示するためのメソッド。
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @return ArrayList
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<String> getVmProtocolMstList() throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getVmProtocolMstList : ");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getVmProtocolMstList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getVmProtocolMstList();
	}
	/**
	 * リポジトリの最終更新時刻を取得
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @return
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws HinemosUnknown
	 * @throws InvalidRoleException
	 * @throws InvalidUserPassException
	 */
	public Long getLastUpdate() throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getLastUpdate : ");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getLastUpdate, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getLastUpdate()==null?null:repository.getLastUpdate().getTime();
	}

	/**
	 * エージェントの状態を返します。<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<AgentStatusInfo> getAgentStatusList() throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getAgentStatusList : ");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Get, Method=getAgentStatusList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getAgentStatusList();
	}

	/**
	 * エージェントを再起動、アップデートします。<BR>
	 * 
	 * RepositoryExecute権限が必要
	 * 
	 * @param facilityId　ファシリティID
	 * @param agentCommand エージェントに実行するコマンド。
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @see com.clustercontrol.repository.bean.AgentCommandConstant
	 */
	public void restartAgent(ArrayList<String> facilityIdList, int agentCommand) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getAgentStatusList : ");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_EXECUTE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		for (String facilityId : facilityIdList) {
			msg.append(facilityId);
			msg.append(",");
		}
		msg.append(" AgentCommand=");
		msg.append(agentCommand);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_REPOSITORY + " Restart Agent, Method=restartAgent, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.restartAgent(facilityIdList, agentCommand);
	}
}
