/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.agent;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.PropertyConfigurator;
import org.apache.log4j.helpers.LogLog;

import com.clustercontrol.agent.log.LogfileMonitorManager;
import com.clustercontrol.agent.util.AgentProperties;
import com.clustercontrol.repository.bean.AgentCommandConstant;
import com.clustercontrol.ws.agent.AgentInfo;
import com.clustercontrol.ws.agent.HinemosUnknown_Exception;
import com.clustercontrol.ws.agent.InvalidRole_Exception;
import com.clustercontrol.ws.agent.InvalidUserPass_Exception;

/**
 * エージェントメインクラス<BR>
 * 
 * 管理対象ノードで起動する際に呼び出されるクラスです。
 */
public class Agent {

	//ロガー
	private static Log m_log = LogFactory.getLog(Agent.class);

	private static ReceiveTopic m_receiveTopic;

	private static SendQueue m_sendQueue;

	/** Agentクラス用スケジューラ * */
	private ScheduledExecutorService m_log4jScheduler = Executors.newScheduledThreadPool(1);

	/** log4j設定 */
	public String m_log4jFileName = null;
	/** log4j設定ファイル再読み込み間隔 */
	private long m_reconfigLog4jInterval = 600;

	private static AgentInfo agentInfo = new AgentInfo();

	public static final Integer DEFAULT_CONNECT_TIMEOUT = 10;
	public static final Integer DEFAULT_REQUEST_TIMEOUT = 60;

	/*
	 * AgentHome
	 * /opt/hinemos_agentやC:\Program Files(x86)\Hinemos\Agent4.0.0
	 * など
	 */
	private static String agentHome = null;

	// Runtime.execのシリアル化実行用Lock Object(他にRuntime.exec用のLock Objectを設けず、必ずJVM内で共有すること)
	public static final Object runtimeExecLock = new Object();

	/**
	 * メイン処理
	 * 
	 * @param args プロパティファイル名
	 */
	public static void main(String[] args) throws Exception{

		// 引数チェック
		if(args.length != 2){
			System.out.println("Usage : java Agent [Agent.properties File Path] [log4j.properties File Path]");
			System.exit(1);
		}

		// Systemプロパティ
		m_log.info("starting Hinemos Agent...");
		m_log.info("java.vm.version = " + System.getProperty("java.vm.version"));
		m_log.info("java.vm.vendor = " + System.getProperty("java.vm.vendor"));
		m_log.info("java.home = " + System.getProperty("java.home"));
		m_log.info("os.name = " + System.getProperty("os.name"));
		m_log.info("os.arch = " + System.getProperty("os.arch"));
		m_log.info("os.version = " + System.getProperty("os.version"));
		m_log.info("user.name = " + System.getProperty("user.name"));
		m_log.info("user.dir = " + System.getProperty("user.dir"));
		m_log.info("user.country = " + System.getProperty("user.country"));
		m_log.info("user.language = " + System.getProperty("user.language"));
		m_log.info("file.encoding = " + System.getProperty("file.encoding"));

		// TODO プロパティファイルのフォルダの親がagentHome。
		// あまり良くない実装なので、修正予定。
		File file = new File(args[0]);
		agentHome = file.getParentFile().getParent() + "/";
		m_log.info("agentHome=" + agentHome);

		// 起動時刻
		long startDate = System.currentTimeMillis();
		m_log.info("start date = " + new Date(startDate) + "(" + startDate + ")");
		agentInfo.setStartupTime(startDate);

		// Agent設定の初期化
		m_log.info("Agent.properties = " + args[0]);
		// ログ再読み込みの機構初期化
		m_log.info("log4j.properties = " + args[1]);

		// Agentインスタンス作成
		Agent agent = new Agent(args[0], args[1]);

		//エージェント処理開始
		agent.exec();

		m_log.info("Hinemos Agent started");


		//終了待ち
		agent.waitAwakeAgent();
	}

	/**
	 * マネージャのawakeAgentを待つメソッド。
	 * UDP 24005にパケットが送られてきたら、ラッチを開放する(releaseLatch)。
	 * ラッチが開放されたReceiveTopicは、マネージャからTopicを取りに行く。
	 */
	public void waitAwakeAgent () {
		final int BUFSIZE = 1;
		final int port = 24005;
		byte[] buf = new byte[BUFSIZE];
		InetAddress cAddr;		// マネージャのIPアドレス
		int cPort;				// マネージャのポート
		DatagramSocket sock = null;
		try {
			sock = new DatagramSocket(port);
		} catch (Exception e) {
			m_log.warn("waitAwakeAgent " + e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			m_log.warn("waitAwakeAgent : sock is null");
		}
		while (true) {
			try {
				Thread.sleep(500);
				if (sock == null) {
					continue;
				}
				DatagramPacket recvPacket = new DatagramPacket(buf, BUFSIZE);
				sock.receive(recvPacket);
				cAddr = recvPacket.getAddress();
				cPort = recvPacket.getPort();
				m_log.info("waitAwakeAgent (" + cAddr.getHostAddress() +
						" onPort=" + cPort + ") buf.length=" + buf.toString().length());
				m_receiveTopic.releaseLatch();
			} catch (Exception e) {
				String msg = "waitAwakeAgent " + e.getClass().getSimpleName() + ", " + e.getMessage();
				m_log.warn(msg);
				m_log.debug(msg, e);
				try {
					Thread.sleep(60*1000);
				} catch (InterruptedException e1) {
					m_log.warn(e1,e1);
				}
			}
		}
	}

	/**
	 * コンストラクタ
	 */
	public Agent(String propFileName, String log4jFileName) throws Exception{

		//------------
		//-- 初期処理
		//------------

		// エージェントのIPアドレス、ホスト名をログに出力。
		getAgentInfo();
		m_log.info(getAgentStr());

		//プロパティファイル読み込み初期化
		AgentProperties.init(propFileName);

		// log4j設定ファイル再読み込み設定
		if(log4jFileName == null){
			m_log.error("log4j.properties does not specify!");
		}
		m_log4jFileName = log4jFileName;

		m_sendQueue = new SendQueue();

		int connectTimeout = DEFAULT_CONNECT_TIMEOUT;
		int requestTimeout = DEFAULT_REQUEST_TIMEOUT;
		try {
			String strConnect = AgentProperties.getProperty("connect.timeout");
			if (strConnect != null) {
				requestTimeout = Integer.parseInt(strConnect);
			}
			String strRequest = AgentProperties.getProperty("request.timeout");
			if (strRequest != null) {
				requestTimeout = Integer.parseInt(strRequest);
			}
		} catch (Exception e) {
			m_log.warn("timeout : " + e.getMessage());
		}


		try {
			EndpointManager.init(AgentProperties.getProperty("user"),
					AgentProperties.getProperty("password"),
					AgentProperties.getProperty("managerAddress"),
					connectTimeout, requestTimeout);
		} catch (Exception e) {
			m_log.error("EndPointWrapper.init error : " + e.getMessage(), e);
			m_log.error("current-dir=" + (new File(".")).getAbsoluteFile().getParent());
			throw e;
		}

		Runtime.getRuntime().addShutdownHook(new Thread() {
			@Override
			public void run() {
				terminate();
				m_log.info("Hinemos agent stopped");
			}
		});
	}

	/**
	 * エージェント処理実行実行します。<BR>
	 * 
	 * メインメソッドから呼び出されるメソッドで
	 * トピックへの接続を行います。
	 */
	public void exec() {
		//-----------------
		//-- トピック接続
		//-----------------
		m_log.debug("exec() : create topic ");

		m_receiveTopic = new ReceiveTopic(m_sendQueue);
		m_receiveTopic.setName("ReceiveTopicThread");
		m_log.info("receiveTopic start 1");
		m_receiveTopic.start();
		m_log.info("receiveTopic start 2");

		// log4j再読み込みタスク
		m_log4jScheduler.scheduleWithFixedDelay(new ReloadLog4jTask(),
				m_reconfigLog4jInterval,m_reconfigLog4jInterval,TimeUnit.SECONDS);

		// ログファイル読み込みスレッド開始
		LogfileMonitorManager manager = new LogfileMonitorManager(null);
		manager.start();

	}

	/**
	 * 終了処理を行います。<BR>
	 */
	public void terminate() {
		m_log.info("terminate() start");
		m_receiveTopic.logHistory();

		try {
			AgentEndPointWrapper.deleteAgent();
		} catch (InvalidRole_Exception e) {
			m_log.info("InvalidRoleException " + e.getMessage());
		} catch (InvalidUserPass_Exception e) {
			m_log.info("InvalidUserPassException " + e.getMessage());
		} catch (HinemosUnknown_Exception e) {
			m_log.info("HinemosUnknown " + e.getMessage());
		}

		m_log.info("terminate() end");
	}

	/**
	 * エージェントを再起動します。
	 */
	public static void restart(int agentCommand) {
		String osName = System.getProperty("os.name");
		m_log.info("Hinemos agent restart : os=" + osName);

		/** OSがWindowsか否(UNIX)かを自動判別する */
		if(osName != null && osName.startsWith("Windows")){
			restartWin(agentCommand);
		} else {
			restartUnix(agentCommand);
		}
	}

	private static void restartWin(int agentCommand) {
		String[] command = null;

		if (agentCommand == AgentCommandConstant.UPDATE) {
			command = new String[4];
			command[0] = "CMD";
			command[1] = "/C";
			command[2] = getAgentHome() + "/bin/RestartAgent.vbs";
			command[3] = "copy";
			m_log.info("command = " + command[0] + " " + command[1] + " " + command[2] +
					" " + command[3]);
		} else {
			command = new String[3];
			command[0] = "CMD";
			command[1] = "/C";
			command[2] = getAgentHome() + "/bin/RestartAgent.vbs";
			m_log.info("command = " + command[0] + " " + command[1] + " " + command[2]);
		}
		try {
			Runtime.getRuntime().exec(command);
		} catch (Exception e) {
			m_log.error("restart " + e.getMessage(), e);
		}
	}

	private static void restartUnix(int agentCommand) {
		String[] command = null;
		String script = "agent_restart.sh";
		command = new String[3];
		command[0] = "sh";
		command[1] = "-c";
		command[2] = getAgentHome() + "bin/" + script;
		if (agentCommand == AgentCommandConstant.UPDATE) {
			command[2] += " copy";
		}
		m_log.info("command = " + command[0] + " " + command[1] + " " + command[2]);
		try {
			Runtime.getRuntime().exec(command);
		} catch (Exception e) {
			m_log.error("restart " + e.getMessage(), e);
		}
	}

	/**
	 * ローカルディスクに存在するファイルのURLオブジェクトを作成する
	 * 
	 * @param localFilePath
	 * @return
	 */
	private URL toUrl(String localFilePath){
		URL url = null;

		// ファイルの存在チェック
		FileInputStream in = null;
		try{
			in = new FileInputStream(localFilePath);
		}catch (Exception e) {
			m_log.error(localFilePath + " does not exists!", e);
		}finally{
			if(in != null){
				try {
					in.close();
				} catch (Exception e) {
				}
			}
		}

		// URLクラスの作成
		try{
			url = new URL("file", "localhost", localFilePath);
		}catch (Exception e) {
			m_log.error(localFilePath + " : unknown exception", e);
		}

		return url;
	}

	/**
	 * ログ設定を再読み込みするタスク
	 * @author Hinemos
	 *
	 */
	protected class ReloadLog4jTask implements Runnable {

		/**
		 * デフォルトコンストラクタ
		 */
		public ReloadLog4jTask() {
		}

		// 最終更新時刻
		private long lastConfigured = -1;
		// log4j設定ファイルURL
		URL configURL = null;

		/**
		 * ログ設定ファイルリロードタスクの開始
		 */
		@Override
		public void run() {
			m_log.debug("ReloadLog4jTask.run() Checking if configuration changed");

			// File Load
			configURL = toUrl(m_log4jFileName);

			// 対象ファイルが存在する場合に限り実行
			if(configURL != null){

				// File Check
				try{
					URLConnection conn = configURL.openConnection();
					long lastModified = conn.getLastModified();

					if (lastConfigured < lastModified)
					{
						m_log.debug("ReloadLog4jTask.run() do re-configuration");
						reconfigure(conn);
					}

				} catch (IOException e) {
					m_log.warn("ReloadLog4jTask.run() Failed to check URL: " + configURL, e);
				}
			}
			m_log.debug("ReloadLog4jTask.run() finish");
		}

		/**
		 * ログ設定のリロード
		 * @param conn
		 */
		private void reconfigure(final URLConnection conn) {
			m_log.info("Configuring from URL: " + configURL);

			// reconfigure
			try{
				PropertyConfigurator.configure(configURL);

				/* Set the LogLog.QuietMode. As of log4j1.2.8 this needs to be set to
					avoid deadlock on exception at the appender level. See bug#696819.
				 */
				LogLog.setQuietMode(true);

				lastConfigured = System.currentTimeMillis();

				m_log.debug("ReloadLog4jTask.reconfigure() lastConfigured = " + lastConfigured);

			} catch (Exception e) {
				m_log.error("Failed to configure from URL: " + configURL, e);
			}
		}
	}

	public static AgentInfo getAgentInfo() {

		agentInfo.setFacilityId(AgentProperties.getProperty("facilityId"));

		// OS情報(識別したホスト名、IPアドレス)
		try {
			// ホスト名取得
			String hostname = System.getProperty("hostname");
			m_log.debug("hostname=[" + hostname + "]");
			agentInfo.setHostname(hostname);

			// IPアドレス取得
			Enumeration<NetworkInterface> networkInterfaces = NetworkInterface.getNetworkInterfaces();
			String newIpAddressStr = "";
			ArrayList<String> newIpAddressList = new ArrayList<String>();
			if (null != networkInterfaces) {
				while (networkInterfaces.hasMoreElements()) {
					NetworkInterface ni = networkInterfaces.nextElement();
					Enumeration<InetAddress> inetAddresses = ni.getInetAddresses();
					while (inetAddresses.hasMoreElements()) {
						InetAddress in = inetAddresses.nextElement();
						String hostAddress = in.getHostAddress();
						if (hostAddress != null && !hostAddress.equals("127.0.0.1") &&
								!hostAddress.startsWith("0:0:0:0:0:0:0:1") &&
								!hostAddress.equals("::1")) {
							newIpAddressList.add(hostAddress);
							newIpAddressStr += "," + hostAddress;
						}
					}
				}
			}
			if (agentInfo.getIpAddress().size() != newIpAddressList.size()) {
				m_log.info("ipAddress change : " + agentInfo.getIpAddress().size() +
						"," + newIpAddressList.size());
				agentInfo.getIpAddress().clear();
				agentInfo.getIpAddress().addAll(newIpAddressList);
			} else {
				if (!agentInfo.getIpAddress().containsAll(newIpAddressList)) {
					m_log.info("ipAddress change");
					agentInfo.getIpAddress().clear();
					agentInfo.getIpAddress().addAll(newIpAddressList);
				}
			}
		} catch (SocketException e) {
			m_log.error(e,e);
		}
		if (m_receiveTopic != null) {
			agentInfo.setInterval(m_receiveTopic.getTopicInterval());
		}
		m_log.debug(getAgentStr());

		return agentInfo;
	}

	public static String getAgentStr() {
		String str = "agentInfo=";
		if (agentInfo.getFacilityId() != null) {
			str += "[facilityID=" + agentInfo.getFacilityId() + "]";
		}
		str += "[hostname="+ agentInfo.getHostname() + "]";
		for (String ipAddress : agentInfo.getIpAddress()) {
			str += ", " +ipAddress;
		}
		return str;
	}

	public static String getAgentHome() {
		return agentHome;
	}
}
