/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.agent;

import java.util.ArrayList;
import java.util.ConcurrentModificationException;
import java.util.Date;
import java.util.Hashtable;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.bean.TopicFlagConstant;
import com.clustercontrol.agent.custom.CollectorManager;
import com.clustercontrol.agent.custom.CommandCollector;
import com.clustercontrol.agent.job.CheckSumThread;
import com.clustercontrol.agent.job.CommandThread;
import com.clustercontrol.agent.job.FileListThread;
import com.clustercontrol.agent.job.PublicKeyThread;
import com.clustercontrol.agent.job.RunHistoryUtil;
import com.clustercontrol.agent.log.LogfileMonitorManager;
import com.clustercontrol.agent.update.UpdateModuleUtil;
import com.clustercontrol.agent.util.AgentProperties;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.jobmanagement.bean.CommandConstant;
import com.clustercontrol.jobmanagement.bean.CommandTypeConstant;
import com.clustercontrol.jobmanagement.bean.RunStatusConstant;
import com.clustercontrol.repository.bean.AgentCommandConstant;
import com.clustercontrol.util.CommandMonitoringWSUtil;
import com.clustercontrol.ws.agent.HinemosUnknown_Exception;
import com.clustercontrol.ws.agent.InvalidRole_Exception;
import com.clustercontrol.ws.agent.InvalidUserPass_Exception;
import com.clustercontrol.ws.agent.CustomInvalid_Exception;
import com.clustercontrol.ws.agent.MonitorNotFound_Exception;
import com.clustercontrol.ws.agent.SettingUpdateInfo;
import com.clustercontrol.ws.agent.TopicInfo;
import com.clustercontrol.ws.jobmanagement.RunInstructionInfo;
import com.clustercontrol.ws.jobmanagement.RunResultInfo;
import com.clustercontrol.ws.monitor.CommandExecuteDTO;
import com.clustercontrol.ws.monitor.MonitorInfo;

/**
 * Topicを受信するクラス<BR>
 * Topicへの接続と、メッセージの受信を行います。
 * 
 * Topicでマネージャからのジョブ実行指示を受け取ります。
 */
public class ReceiveTopic extends Thread {

	//ロガー
	private static Log m_log = LogFactory.getLog(ReceiveTopic.class);

	//protected ArrayList<String> m_facilityIdList;
	private SendQueue m_sendQueue;

	private static Hashtable<String, Date> m_runHistory;

	private int m_topicInterval = 30000;
	private int m_topicFastInterval = 1000;
	private int m_topicFastCount = 10;
	private long m_runhistoryClearDelay = -1;

	// マネージャと接続できている場合、フラグはtrueになる。
	private long disconnectCounter = 1;
	private static boolean m_clearFlg = false;

	// 最後に受信した設定情報更新日時
	private static SettingUpdateInfo commandMonitorLastUpdateInfo = null;

	/**
	 * コンストラクタ
	 * @param agent ジョブエージェント
	 * @param facilityIdList ファシリティIDのリスト
	 * @param sendQueue　メッセージ送信用クラス
	 * @param props　プロパティファイル情報
	 */
	public ReceiveTopic(SendQueue sendQueue) {
		super();
		m_sendQueue = sendQueue;
		m_runHistory = new Hashtable<String, Date>();

		m_log.info("create ReceiveTopic ");

		// 再接続処理実行間隔取得
		String interval, str;
		str = "topic.interval";
		interval = AgentProperties.getProperty(str);
		if (interval != null) {
			try {
				// プロパティファイルにはmsecで記述
				m_topicInterval = Integer.parseInt(interval);
				m_log.info(str + " = " + m_topicInterval + " msec");
			} catch (NumberFormatException e) {
				m_log.error(str,e);
			}
		}

		str = "topic.fast.interval";
		interval = AgentProperties.getProperty(str);
		if (interval != null) {
			try {
				// プロパティファイルにはmsecで記述
				m_topicFastInterval = Integer.parseInt(interval);
				m_log.info(str + " = " + m_topicFastInterval + " msec");
			} catch (NumberFormatException e) {
				m_log.error(str,e);
			}
		}

		str = "topic.fast.count";
		interval = AgentProperties.getProperty(str);
		if (interval != null) {
			try {
				m_topicFastCount = Integer.parseInt(interval);
				m_log.info(str + " = " + m_topicFastCount);
			} catch (NumberFormatException e) {
				m_log.error(str,e);
			}
		}

		// ジョブ実行履歴の削除実行時間を取得
		String delay = AgentProperties.getProperty("runhistory.clear.delay");
		if (delay != null) {
			try {
				// プロパティファイルには秒で記述
				m_runhistoryClearDelay = Integer.parseInt(delay);
				m_log.info("runhistory.clear.delay = " + m_runhistoryClearDelay + " sec");
			} catch (NumberFormatException e) {
				m_log.error("runhistory.clear.delay",e);
			}
		}

		// ログファイル監視のマネージャを生成
		m_logManager = new LogfileMonitorManager(m_sendQueue);
	}


	/**
	 * マネージャからのTopic(即時実行など)が発行された際に、ラッチを開放する。
	 */
	private CountDownLatch countDownLatch = null;

	public void releaseLatch() {
		if (countDownLatch == null) {
			m_log.info("latch is null");
		}
		countDownLatch.countDown();
	}

	/**
	 * トピック受信処理
	 */
	@Override
	public void run() {

		m_log.info("run start");

		int fastCount = 1;
		while (true) {
			/*
			 * トピックの有無をマネージャにチェックし終わったら、sleepする。
			 * 
			 * 一度、Topicにジョブチェックが含まれている場合は、fastモードになり、sleep時間が短くなる。
			 * それ以外の場合は、slowモードになり、sleep時間が長くなる。
			 * (ジョブチェックのTopicの受信直後には、再度Topicが来る。)
			 */
			try {
				int interval;
				/*
				 * fastフラグが立っている場合、その後、
				 * m_topicFastCount回、intervalがm_topicFastIntervalとなる。
				 */
				if (fastCount > 0) {
					interval = m_topicFastInterval;
					fastCount --;
				} else {
					interval = m_topicInterval;
				}
				countDownLatch = new CountDownLatch(1);
				countDownLatch.await(interval, TimeUnit.MILLISECONDS);
			} catch (InterruptedException e) {
				m_log.warn("Interrupt " + e.getMessage());
			} catch (Exception e) {
				m_log.error("run() : " + e.getMessage(), e);
			}

			try {
				List<RunInstructionInfo> runInstructionList = new ArrayList<RunInstructionInfo>();
				m_log.info("getTopic " + Agent.getAgentStr() + " start");
				List<TopicInfo> topicInfoList = null;
				SettingUpdateInfo updateInfo = null;
				try {
					topicInfoList = AgentEndPointWrapper.getTopic();
					updateInfo = AgentEndPointWrapper.getSettingUpdateInfo();
				} catch (Exception e) {
					/*
					 * マネージャが停止している、もしくはマネージャと通信ができないと、
					 * ここに到達する。
					 */
					if (disconnectCounter < Long.MAX_VALUE) {
						disconnectCounter++;
					}
					// 一定時間、マネージャが応答していないと、ジョブ履歴を削除する。
					if (disconnectCounter * m_topicInterval / 1000 > m_runhistoryClearDelay) {
						clearJobHistory();
					}

					/*
					 * マネージャが停止している時などは、下記のログが出続ける。
					 * StackTraceを出すと、ログファイルが大きくなりすぎるので、
					 * StackTraceはつけない。
					 * どうしても見たかったらdebugにする。
					 */
					String message = "run() getTopic : " + e.getClass().getSimpleName() +
					", " + e.getMessage();
					m_log.warn(message);
					m_log.debug(message, e);
					continue; // whileまで戻る。
				}
				/*
				 * マネージャと接続直後の場合、ここで転送ログファイル、コマンド監視情報のリストを受け取る。
				 */
				if (disconnectCounter != 0) {
					reloadLogfileMonitor();
					reloadCommandMonitoring(updateInfo);
				}
				disconnectCounter = 0;
				setClearFlg(false);

				if (topicInfoList.size() == 0 && isCommandMonitoringUpdated(updateInfo)) {
					// コマンド監視のリロードを行う
					reloadCommandMonitoring(updateInfo);
				}

				for (TopicInfo topicInfo : topicInfoList) {
					m_log.info("getTopic facilityId=" + topicInfo.getFacilityId());
					RunInstructionInfo runInstructionInfo = topicInfo.getRunInstructionInfo();
					if (runInstructionInfo != null) {
						runInstructionList.add(runInstructionInfo);
						/*
						 * コマンドタイプがCHECKの場合のみfastFlagをtrueにする。
						 * その直後にメッセージが来るので。
						 */
						if (runInstructionInfo.getCommandType() == CommandTypeConstant.CHECK) {
							m_log.debug("run : command=check");
							fastCount = m_topicFastCount;
						} else {
							fastCount = 0;
						}
					}
					long topicFlag = topicInfo.getFlag();
					if ((topicFlag & TopicFlagConstant.REPOSITORY_CHANGED) != 0 ||
							(topicFlag & TopicFlagConstant.NEW_FACILITY) != 0 ||
							(topicFlag & TopicFlagConstant.CALENDAR_CHANGED) != 0 ||
							(topicFlag & TopicFlagConstant.LOGFILE_CHANGED) != 0) {
						reloadLogfileMonitor();
					}
					if ((topicFlag & TopicFlagConstant.REPOSITORY_CHANGED) != 0 ||
							(topicFlag & TopicFlagConstant.NEW_FACILITY) != 0 ||
							(topicFlag & TopicFlagConstant.CALENDAR_CHANGED) != 0 ||
							(topicFlag & TopicFlagConstant.CUSTOM_CHANGED) != 0) {
						// コマンド監視のリロードを行う
						reloadCommandMonitoring(updateInfo);
					}
					if (topicInfo.getAgentCommand() != 0) {
						int agentCommand = topicInfo.getAgentCommand();
						m_log.debug("agentCommand : " + agentCommand);
						if (agentCommand == AgentCommandConstant.UPDATE) {
							// 1つもファイルをダウンロードしていない場合は、再起動しない。
							if(!UpdateModuleUtil.update()) {
								agentCommand = 0;
							}
						}
						Agent.restart(agentCommand);
					}
					if ((topicFlag & TopicFlagConstant.NEW_FACILITY) != 0) {
						UpdateModuleUtil.setAgentLibMd5();
					}
				}
				m_log.debug("getTopic " + Agent.getAgentStr() + " end");
				if (runInstructionList.size() == 0) {
					m_log.debug("infoList" + ".size() 0");
					continue;
				}
				m_log.info("infoList.size() = " + runInstructionList.size());
				for (RunInstructionInfo info : runInstructionList){
					runJob(info);
				}
			} catch (Exception e) {
				m_log.error("run() : " + e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			}
		}
	}

	public LogfileMonitorManager m_logManager = null;

	private boolean isCommandMonitoringUpdated(SettingUpdateInfo updateInfo) {

		if (updateInfo == null) {
			return false;
		} else if (commandMonitorLastUpdateInfo == null) {
			return true;
		} else {
			if (commandMonitorLastUpdateInfo.getCustomMonitorUpdateTime() == updateInfo.getCustomMonitorUpdateTime()
					&& commandMonitorLastUpdateInfo.getCalendarUpdateTime() == updateInfo.getCalendarUpdateTime()
					&& commandMonitorLastUpdateInfo.getRepositoryUpdateTime() == updateInfo.getRepositoryUpdateTime()) {
				return false;
			} else {
				return true;
			}
		}

	}

	private void reloadCommandMonitoring(SettingUpdateInfo updateInfo) {
		// Local Variables
		ArrayList<CommandExecuteDTO> dtos = null;

		// MAIN
		m_log.info("reloading configuration of custom monitoring...");
		try {
			dtos = AgentEndPointWrapper.getCommandExecuteDTOs();

			// reset Command Collector
			CollectorManager.unregisterCollectorTask(PluginConstant.TYPE_CUSTOM_MONITOR);
			for (CommandExecuteDTO dto : dtos) {
				m_log.info("reloaded configuration : " + CommandMonitoringWSUtil.toStringCommandExecuteDTO(dto));
				CollectorManager.registerCollectorTask(new CommandCollector(dto));
			}

			// update last update time
			commandMonitorLastUpdateInfo = updateInfo;
			if (updateInfo != null) {
				m_log.info("command monitor updated. (custom = " + updateInfo.getCalendarUpdateTime()
						+ ", calendar = " + updateInfo.getCalendarUpdateTime()
						+ ", repository = " + updateInfo.getRepositoryUpdateTime() + ")");
			}

		} catch (HinemosUnknown_Exception e) {
			m_log.warn("un-expected internal failure occurs...", e);
		} catch (CustomInvalid_Exception e) {
			m_log.warn("monitor configuration is not valid...", e);
		} catch (InvalidRole_Exception e) {
			m_log.warn("reloadCommandMonitoring: " + e.getMessage());
		} catch (InvalidUserPass_Exception e) {
			m_log.warn("reloadCommandMonitoring: " + e.getMessage());
		}

	}

	private void reloadLogfileMonitor () {
		m_log.info("reloading configuration of logfile monitoring...");
		try {
			ArrayList<MonitorInfo> list = AgentEndPointWrapper.getMonitorLogfile();

			for (MonitorInfo info : list) {
				m_log.info("logfile: filepath=" + info.getLogfileCheckInfo().getLogfile() +
						", monitorId=" + info.getMonitorId() +
						", monitorFlg=" + info.getMonitorFlg());
			}
			m_logManager.setLogfileMonitor(list);
		} catch (HinemosUnknown_Exception e) {
			m_log.error(e,e);
		} catch (InvalidRole_Exception e) {
			m_log.warn("realoadLogfileMonitor: " + e.getMessage());
		} catch (InvalidUserPass_Exception e) {
			m_log.warn("realoadLogfileMonitor: " + e.getMessage());
		} catch (MonitorNotFound_Exception e) {
			m_log.warn("realoadLogfileMonitor: " + e.getMessage());
		}
	}

	private void runJob (RunInstructionInfo info) {
		m_log.debug("onMessage SessionID=" + info.getSessionId()
				+ ", JobID=" + info.getJobId()
				+ ", CommandType=" + info.getCommandType());

		if(info.getCommandType() == CommandTypeConstant.CHECK){
			//受信メッセージ応答
			m_log.debug("onMessage CommandType = CHECK");

			RunResultInfo resultInfo = new RunResultInfo();
			resultInfo.setSessionId(info.getSessionId());
			resultInfo.setJobunitId(info.getJobunitId());
			resultInfo.setJobId(info.getJobId());
			resultInfo.setFacilityId(info.getFacilityId());
			resultInfo.setCommandType(info.getCommandType());

			m_sendQueue.put(resultInfo);
		}
		else if(info.getCommandType() == CommandTypeConstant.DELETE_NORMAL_HISTORY ||
				info.getCommandType() == CommandTypeConstant.DELETE_STOP_HISTORY){
			//受信メッセージ応答
			m_log.debug("onMessage CommandType = DELETE_NORMAL_HISTORY or DELETE_STOP_HISTORY");

			//実行履歴削除
			RunHistoryUtil.delRunHistory(info, m_runHistory);
		}
		else {
			m_log.debug("onMessage CommandType != CHECK");

			//実行履歴チェック
			Date startDate = RunHistoryUtil.findRunHistory(info, m_runHistory);
			if(startDate != null){
				//実行履歴がある場合、開始メッセージを送信する

				RunResultInfo resultInfo = new RunResultInfo();
				resultInfo.setSessionId(info.getSessionId());
				resultInfo.setJobunitId(info.getJobunitId());
				resultInfo.setJobId(info.getJobId());
				resultInfo.setFacilityId(info.getFacilityId());
				resultInfo.setCommand(info.getCommand());
				resultInfo.setCommandType(info.getCommandType());
				resultInfo.setStatus(RunStatusConstant.START);
				resultInfo.setTime(startDate.getTime());

				m_log.info("run SessionID=" + info.getSessionId() + ", JobID=" + info.getJobId());

				//送信
				m_sendQueue.put(resultInfo);
			}
			else{
				if(info.getCommand().equals(CommandConstant.GET_PUBLIC_KEY) ||
						info.getCommand().equals(CommandConstant.ADD_PUBLIC_KEY) ||
						info.getCommand().equals(CommandConstant.DELETE_PUBLIC_KEY)){
					//公開鍵用スレッド実行
					m_log.debug("onMessage CommandType = GET_PUBLIC_KEY or ADD_PUBLIC_KEY or DELETE_PUBLIC_KEY");

					PublicKeyThread thread = new PublicKeyThread(info, m_sendQueue, m_runHistory);
					thread.start();
				}
				else if(info.getCommand().equals(CommandConstant.GET_FILE_LIST)){
					//ファイルリスト用スレッド実行
					m_log.debug("onMessage CommandType = GET_FILE_LIST");

					FileListThread thread = new FileListThread(info, m_sendQueue, m_runHistory);
					thread.start();
				}
				else if(info.getCommand().equals(CommandConstant.GET_CHECKSUM) ||
						info.getCommand().equals(CommandConstant.CHECK_CHECKSUM)){
					//チェックサム用スレッド実行
					m_log.debug("onMessage CommandType = GET_CHECKSUM or CHECK_CHECKSUM");

					CheckSumThread thread = new CheckSumThread(info, m_sendQueue, m_runHistory);
					thread.start();
				}
				else{
					//コマンド実行
					CommandThread thread = new CommandThread(info, m_sendQueue, m_runHistory);
					thread.start();
				}
			}
		}
	}

	/**
	 * 実行履歴をログに出力
	 * @return
	 */
	protected void logHistory() {

		synchronized (m_runHistory) {

			try {
				for(String key : m_runHistory.keySet()) {
					m_log.info("A running job is out of control due to stopped agent : " + key);
				}
			} catch (ConcurrentModificationException e) {
				m_log.warn("Log output process is stopped due to job execution history updated at the same time.");
			}
		}
	}


	/**
	 * clearFlgの設定
	 * @param clearFlg
	 */
	synchronized private static void setClearFlg(boolean clearFlg) {
		m_clearFlg = clearFlg;
	}

	/**
	 * clearFlgの取得
	 * @return
	 */
	synchronized static public boolean isHistoryClear(){
		return m_clearFlg;
	}

	/**
	 * ジョブ実行履歴削除
	 * 通信エラーとなった場合に、一定時間後、ジョブ履歴情報を削除する
	 */
	public void clearJobHistory() {
		m_log.debug("clearJobHistory start");
		try{
			if (RunHistoryUtil.clearRunHistory(m_runHistory)) {
				m_log.info("job history was deleted.");
				setClearFlg(true);
			}
		} catch (Exception e) {
			m_log.error("clearJobHistory : ", e);
		}
	}

	public int getTopicInterval() {
		return m_topicInterval;
	}
}
