/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.presenter;

import java.util.List;

import com.clustercontrol.cloud.commons.util.ControlUtil;
import com.clustercontrol.cloud.presenter.AccountResource.Region;
import com.clustercontrol.cloud.resources.Tag;
import com.clustercontrol.ws.cloud.CloudEndpoint;
import com.clustercontrol.ws.cloud.CloudManagerFault_Exception;
import com.clustercontrol.ws.cloud.InvalidRole_Exception;
import com.clustercontrol.ws.cloud.InvalidUserPass_Exception;
import com.clustercontrol.ws.cloud.NodeDetail;

public class CloudInstanceManager extends InternalContainer<CloudInstance, com.clustercontrol.ws.cloud.CloudInstance> implements ICloudInstanceManager {
	private Region region;
	private CloudComputeService cloudComputeService;
	
	public CloudInstanceManager(CloudComputeService cloudComputeService) {
		this.cloudComputeService = cloudComputeService;
		this.region = cloudComputeService.getRegion();
	}
	
	@Override
	public Region getRegion() {
		return region;
	}

	private CloudInstance getCloudInstance(String cloudInstanceId){
		for(CloudInstance cloudInstance: getComponents()){
			if(cloudInstance.getInstanceId().equals(cloudInstanceId)){
				return cloudInstance;
			}
		}
		return null;
	}
	
	@Override
	public CloudInstance[] getCloudInstances() {
		return getComponents().toArray(new CloudInstance[0]);
	}

	@Override
	public CloudInstance createCloudInstance(CreateCloudInstanceRequest createCloudInstanceRequest) {
		com.clustercontrol.ws.cloud.CreateInstanceRequest wsCreateCloudInstanceRequest = new com.clustercontrol.ws.cloud.CreateInstanceRequest();
		ControlUtil.exportObject(createCloudInstanceRequest, wsCreateCloudInstanceRequest);
		wsCreateCloudInstanceRequest.setNodeDetail(new NodeDetail());
		ControlUtil.exportObject(createCloudInstanceRequest, wsCreateCloudInstanceRequest.getNodeDetail());
		

		if(createCloudInstanceRequest.getTags() != null){
			for (Tag tag: createCloudInstanceRequest.getTags()) {
				com.clustercontrol.ws.cloud.Tag wsTag = new com.clustercontrol.ws.cloud.Tag();
				wsTag.setKey(tag.name);
				wsTag.setValue(tag.value);
				wsCreateCloudInstanceRequest.getTags().add(wsTag);
			}
		}
		try {
			com.clustercontrol.ws.cloud.CloudInstance reWsCloudInstance = getEndpoint().addInstance(getRoleId(), region.getRegion(), wsCreateCloudInstanceRequest);
			CloudInstance cloudInstance = CloudInstance.createCloudInstance(this, reWsCloudInstance);
			getComponents().add(cloudInstance);
			fireComponentAdded(cloudInstance);
			return cloudInstance;
		} catch (CloudManagerFault_Exception | InvalidRole_Exception | InvalidUserPass_Exception e) {
			throw new CloudModelException(e);
		}
	}

	@Override
	public CloudInstance createCloudInstanceByTemplate(CreateCloudInstanceByTemplateRequest createCloudInstanceByTemplateRequest) {
		com.clustercontrol.ws.cloud.CreateInstanceByTemplateRequest wsCreateCloudInstanceByTemplateRequest = new com.clustercontrol.ws.cloud.CreateInstanceByTemplateRequest();
		CreateCloudInstanceByTemplateRequest request = new CreateCloudInstanceByTemplateRequest();
		ControlUtil.importObject(request, createCloudInstanceByTemplateRequest);
		ControlUtil.exportObject(request, wsCreateCloudInstanceByTemplateRequest);
		wsCreateCloudInstanceByTemplateRequest.setNodeDetail(new NodeDetail());
		ControlUtil.exportObject(request, wsCreateCloudInstanceByTemplateRequest.getNodeDetail());
		
		if(createCloudInstanceByTemplateRequest.getTags() != null){
			for (Tag tag: createCloudInstanceByTemplateRequest.getTags()) {
				com.clustercontrol.ws.cloud.Tag wsTag = new com.clustercontrol.ws.cloud.Tag();
				wsTag.setKey(tag.name);
				wsTag.setValue(tag.value);
				wsCreateCloudInstanceByTemplateRequest.getTags().add(wsTag);
			}
		}
		if(createCloudInstanceByTemplateRequest.getArguments() != null && createCloudInstanceByTemplateRequest.getArguments().size() > 0){
			List<com.clustercontrol.ws.cloud.TemplateJobsArgument> wsArgs = wsCreateCloudInstanceByTemplateRequest.getArguments(); 
			for(TemplateJobsArgument arg: createCloudInstanceByTemplateRequest.getArguments()){
				com.clustercontrol.ws.cloud.TemplateJobsArgument wsArg = new com.clustercontrol.ws.cloud.TemplateJobsArgument();
				ControlUtil.exportObject(arg, wsArg);
				wsArgs.add(wsArg);
			}
		}
		try {
			com.clustercontrol.ws.cloud.CloudInstance reWsCloudInstance = getEndpoint().addInstanceByTemplate(getRoleId(), region.getRegion(), wsCreateCloudInstanceByTemplateRequest);
			CloudInstance cloudInstance = CloudInstance.createCloudInstance(this, reWsCloudInstance);
			getComponents().add(cloudInstance);
			fireComponentAdded(cloudInstance);
			return cloudInstance;
		} catch (CloudManagerFault_Exception | InvalidRole_Exception | InvalidUserPass_Exception e) {
			throw new CloudModelException(e);
		}
	}

	@Override
	public void removeCloudInstance(String roleId, String region, String instanceId) {
		try {
			getEndpoint().removeInstance(roleId, region, instanceId);
		} catch (CloudManagerFault_Exception | InvalidRole_Exception | InvalidUserPass_Exception e) {
			throw new CloudModelException(e);
		}
		CloudInstance cloudInstance = getCloudInstance(instanceId);
		getComponents().remove(cloudInstance);
		fireComponentRemoved(cloudInstance);
	}

	@Override
	public CloudInstance restoreCloudInstance(RestoreCloudInstanceRequest restoreCloudInstanceRequest) {
		com.clustercontrol.ws.cloud.RestoreInstanceRequest wsRequest = new com.clustercontrol.ws.cloud.RestoreInstanceRequest();
		ControlUtil.exportObject(restoreCloudInstanceRequest, wsRequest);
		wsRequest.setNodeDetail(new NodeDetail());
		ControlUtil.exportObject(restoreCloudInstanceRequest, wsRequest.getNodeDetail());
		
		try {
			com.clustercontrol.ws.cloud.CloudInstance restoredInstance = getEndpoint().restoreInstance(getRoleId(), getRegion().getRegion(), wsRequest);
			CloudInstance cloudInstance = CloudInstance.createCloudInstance(this, restoredInstance);
			getComponents().add(cloudInstance);
			fireComponentAdded(cloudInstance);
			// v202 追記
			return cloudInstance;
		} catch (CloudManagerFault_Exception | InvalidRole_Exception | InvalidUserPass_Exception e) {
			throw new CloudModelException(e);
		}
		// v202 コメントアウト
//		return null;
	}

	@Override
	protected List<com.clustercontrol.ws.cloud.CloudInstance> getSources() throws CloudModelException {
		try {
			return getEndpoint().getInstancesByRegion(getRoleId(), region.getRegion());
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
	}

	@Override
	protected CloudInstance createComponent(com.clustercontrol.ws.cloud.CloudInstance source) throws CloudModelException {
		CloudInstance cloudInstance = new CloudInstance(this);
		cloudInstance.overwrite(source);
		return cloudInstance;
	}

	@Override
	protected void fireComponentAdded(CloudInstance addedValue) {
		fireElementAdded(p2.cloudInstances, addedValue);
	}

	@Override
	protected void fireComponentRemoved(CloudInstance removedValue) {
		fireElementRemoved(p2.cloudInstances, removedValue);
	}
	
	private CloudEndpoint getEndpoint(){
		return getRegion().getAccountResource().getAccountResourceManager().getCloudResourceManager().getEndpoint(CloudEndpoint.class);
	}
	
	private String getRoleId(){
		return getRegion().getAccountResource().getActiveUser().getRoleId();
	}

	@Override
	public CloudComputeService getCloudComputeService() {
		return cloudComputeService;
	}

	@Override
	public void unregistNode(String instanceId) {
		try {
			getEndpoint().unregistNode(getRoleId(), getRegion().getRegion(), instanceId);
			CloudInstance tmpInstance = getCloudInstance(instanceId);
			getComponents().remove(tmpInstance);
			fireComponentRemoved(tmpInstance);
		} catch (CloudManagerFault_Exception | InvalidRole_Exception | InvalidUserPass_Exception e) {
			throw new CloudModelException(e);
		}
	}
}
