/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.factory.monitors;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;

import com.clustercontrol.cloud.IResourceManagement;
import com.clustercontrol.cloud.IResourceManagement.Instance;
import com.clustercontrol.cloud.IResourceManagement.Instance.BlockDeviceMapping;
import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.bean.CloudRegion;
import com.clustercontrol.cloud.bean.CloudService;
import com.clustercontrol.cloud.bean.CloudUser;
import com.clustercontrol.cloud.bean.InstanceStateKind;
import com.clustercontrol.cloud.commons.CloudPropertyConstants;
import com.clustercontrol.cloud.factory.IStorageOperator;
import com.clustercontrol.cloud.registry.ObjectRegistryService;
import com.clustercontrol.cloud.util.CloudCredential;
import com.clustercontrol.cloud.util.RepositoryControllerBeanWrapper;
import com.clustercontrol.cloud.util.ResourceRegion;
import com.clustercontrol.cloud.util.ResourceUtil;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.session.RepositoryControllerBean;


public class InstanceMonitorService {
	public interface InstanceTraceListner {
		void onNotify(String accountResourceId, Instance instance);
		void onError(String accountResourceId, String instanceId, Exception e);
	}

	private static class InstanceInfo {
		public String facilityId;
		public SessionService.ContextData data;
		public CloudService service;
		public CloudRegion region;
		public CloudUser user;
		public InstanceStateKind[] stoppedStatus;
		public int pingCount;
		public boolean updateStorage;
	}

	private Map<String, InstanceInfo> instanceInfoMap = new HashMap<String, InstanceInfo>();
	private final int interval;
	private final int maxPingCount;
	private final ScheduledExecutorService executor;

	private List<InstanceTraceListner> listeners = Collections.synchronizedList(new ArrayList<InstanceTraceListner>());

	/**
	 * instanceMap に合わせて同期される。
	 */
	private ScheduledFuture<?> sf;
	
	private InstanceMonitorService(int interval, int maxPingCount) {
		this.interval = interval;
		this.maxPingCount = maxPingCount;
		this.executor = new ScheduledThreadPoolExecutor(1, new ThreadFactory() {
				@Override
				public Thread newThread(final Runnable r) {
					return new Thread(r, "InstanceMonitorService-thread-1");
				}
			}) {

			@Override
			protected void afterExecute(Runnable r, Throwable t) {
				super.afterExecute(r, t);
				SessionService.current().close();
			}
		};
	}
	
	/**
	 * 追跡開始。
	 * 
	 * @param indtanecId
	 * @param listener
	 */
	public void startMonitor(String indtanecId, String facilityId, SessionService.ContextData data, CloudService service, CloudRegion region, CloudUser user, boolean updateStorage, InstanceStateKind... stoppedStatus) {
		if (CloudPropertyConstants.autoupdate_node.match(CloudPropertyConstants.off)) {
			return;
		}
		
		synchronized (instanceInfoMap) {
			InstanceInfo info = new InstanceInfo();
			info.facilityId = facilityId;
			info.data = data;
			info.service = service;
			info.region = region;
			info.user = user;
			info.stoppedStatus = stoppedStatus;
			info.pingCount = 0;
			info.updateStorage = updateStorage;
			
			// TODO インスタンス ID で登録をしているが、実際にはインスタンス ID だけでは一意にならないので、
			// 問題が発生する可能性がある。
			instanceInfoMap.put(indtanecId, info);
			
			if (sf == null) {
				sf = executor.scheduleWithFixedDelay(
					new Runnable() {
						@Override
						public void run() {
							synchronized(instanceInfoMap) {
								RepositoryControllerBean bean = RepositoryControllerBeanWrapper.bean();
								Logger logger = Logger.getLogger(InstanceMonitorService.class);
								
								// 既存のインスタンスの情報が取得できている場合、更新。
								for (final Map.Entry<String, InstanceInfo> entry: instanceInfoMap.entrySet()) {
									logger.debug("instanceid : " + entry.getKey() + ", condition : " + entry.getValue().stoppedStatus.toString());
									
									++entry.getValue().pingCount;
									
									try {
										SessionService.changeContext(entry.getValue().data);
										
										IResourceManagement rm = ObjectRegistryService.registry().get(IResourceManagement.class, entry.getValue().service.getCloudTypeId());
										if (rm == null) {
											throw new InternalManagerError();
										}
										rm.setAccessDestination(new CloudCredential(entry.getValue().user), new ResourceRegion(entry.getValue().region));
										Instance instance = rm.getInstance(entry.getKey());
										
										NodeInfo node = bean.getNode(entry.getValue().facilityId);
										
										if (node.getIpAddressV4() == null || (instance.getIpAddress() == null && !"123.123.123.123".equals(node.getIpAddressV4()) || (instance.getIpAddress() != null && !node.getIpAddressV4().equals(instance.getIpAddress())))) {
											String oldIpAddress = node.getIpAddressV4();
											if (instance.getIpAddress() != null) {
												node.setIpAddressV4(instance.getIpAddress());
											}
											else {
												node.setIpAddressV4("123.123.123.123");
											}
											logger.info("Change Node, Method=autoRegist, CloudUser=" + entry.getValue().user.getCloudUserId() + ", FacilityID=" + node.getFacilityId() + ", InstanceId=" + instance.getInstanceId() + ", OldIpAddress=" + oldIpAddress + ", NewIpAddress=" + node.getIpAddressV4());
											bean.modifyNode(node);
										}
										
										if (entry.getValue().updateStorage) { 
											// ストレージ情報を更新。
											IStorageOperator operator = ResourceUtil.getResourceOperator(IStorageOperator.class);
											for (BlockDeviceMapping ibd: instance.getBlockDeviceMappings()) {
												operator.updateStorage(ibd.getStorageId());
											}
										}
										
										for (InstanceStateKind state: entry.getValue().stoppedStatus) {
											if (state.equals(instance.getState())) {
												instanceInfoMap.remove(entry.getKey());
												break;
											}
										}

										if (maxPingCount == entry.getValue().pingCount) {
											instanceInfoMap.remove(entry.getKey());
										}

										for (InstanceTraceListner listener: listeners) {
											listener.onNotify(entry.getValue().user.getAccountResourceId(), instance);
										}
									}
									catch (Exception e) {
										logger.error(e.getMessage(), e);
										instanceInfoMap.remove(entry.getKey());

										for (InstanceTraceListner listener: listeners) {
											listener.onError(entry.getValue().user.getAccountResourceId(), entry.getKey(), e);
										}
									}
								}
								
								mustStop();
							}
						}
					},
					interval,
					interval,
					TimeUnit.MILLISECONDS); 
			}
		}
	}
	
	/**
	 * 追跡停止。
	 * 
	 * @param indtanecId
	 */
	public void stopMonitor(String indtanecId) {
		synchronized (instanceInfoMap) {
			instanceInfoMap.remove(indtanecId);
			mustStop();
		}
	}
	
	private void mustStop() {
		if (sf != null && instanceInfoMap.isEmpty()) {
			sf.cancel(true);
			sf = null;
		}
	}
	
	public void shutdown() {
		executor.shutdown();
		instanceInfoMap.clear();
	}
	
	public void addListener(InstanceTraceListner listner) {
		listeners.add(listner);
	}
	public void removeListener(InstanceTraceListner listner) {
		listeners.remove(listner);
	}

	private static InstanceMonitorService singleton;
	
	public static InstanceMonitorService getSingleton() {
		if (singleton == null) {
			synchronized (InstanceMonitorService.class) {
				if (singleton == null) {
		            singleton = new InstanceMonitorService(
		            		Integer.valueOf(CloudPropertyConstants.registcheck_interval.value()),
		            		Integer.valueOf(CloudPropertyConstants.registcheck_count.value()));
				}
			}
		}
		return singleton;
	}
}
