/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.composite;


import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableTreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.accesscontrol.util.LocalSession;
import com.clustercontrol.jobmanagement.action.GetJobDetailTableDefine;
import com.clustercontrol.jobmanagement.composite.action.JobDetailSelectionChangedListener;
import com.clustercontrol.jobmanagement.composite.action.SessionJobDoubleClickListener;
import com.clustercontrol.jobmanagement.util.JobEndpointWrapper;
import com.clustercontrol.jobmanagement.viewer.JobTableTreeViewer;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.jobmanagement.InvalidRole_Exception;
import com.clustercontrol.ws.jobmanagement.JobInfo;
import com.clustercontrol.ws.jobmanagement.JobTreeItem;

/**
 * ジョブ[ジョブ詳細]ビュー用のコンポジットクラスです。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class DetailComposite extends Composite {

	// ログ
	private static Log m_log = LogFactory.getLog( DetailComposite.class );

	/** テーブルビューアー */
	private JobTableTreeViewer m_viewer = null;
	/** セッションID */
	private String m_sessionId = null;
	/** 所属ジョブユニットのジョブID */
	private String m_jobunitId = null;
	/** ジョブID */
	private String m_jobId = null;
	/** セッションID用ラベル */
	private Label m_sessionIdLabel = null;

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public DetailComposite(Composite parent, int style) {
		super(parent, style);
		initialize();
	}

	/**
	 * コンポジットを配置します。
	 */
	private void initialize() {
		GridLayout layout = new GridLayout(1, true);
		this.setLayout(layout);
		layout.marginHeight = 0;
		layout.marginWidth = 0;

		//セッションIDラベル作成
		m_sessionIdLabel = new Label(this, SWT.LEFT);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		m_sessionIdLabel.setLayoutData(gridData);

		//ジョブ詳細テーブル作成
		TableTree tree = new TableTree(this, SWT.H_SCROLL | SWT.V_SCROLL
				| SWT.FULL_SELECTION | SWT.SINGLE);

		Table table = tree.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 1;
		tree.setLayoutData(gridData);

		m_viewer = new JobTableTreeViewer(tree);
		m_viewer.createTableColumn(GetJobDetailTableDefine.get(),
				GetJobDetailTableDefine.SORT_COLUMN_INDEX,
				GetJobDetailTableDefine.SORT_ORDER);

		m_viewer.addSelectionChangedListener(
				new JobDetailSelectionChangedListener(this));

		m_viewer.addDoubleClickListener(
				new SessionJobDoubleClickListener(this));

		update(null, null);
	}

	/**
	 * テーブルビューアーを更新します。<BR>
	 * 引数で指定されたセッションIDのジョブ詳細一覧情報を取得し、
	 * 共通テーブルビューアーにセットします。
	 * <p>
	 * <ol>
	 * <li>引数で指定されたセッションIDのジョブ詳細一覧情報を取得します。</li>
	 * <li>共通テーブルビューアーにジョブ詳細一覧情報をセットします。</li>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * 
	 * @see com.clustercontrol.jobmanagement.action.GetJobDetail#getJobDetail(String, String)
	 * @see #setJobId(String)
	 */
	public void update(String sessionId, String jobunitId) {
		//ジョブ詳細情報取得
		JobTreeItem item = null;
		if (sessionId != null && sessionId.length() > 0) {
			try {
				item = JobEndpointWrapper.getJobDetailList(sessionId);
			} catch (InvalidRole_Exception e) {
				if(LocalSession.isDialogFlag()){
					LocalSession.setDialogFlag(false);
					MessageDialog.openInformation(null, Messages.getString("message"),
							Messages.getString("message.accesscontrol.16"));
					LocalSession.setDialogFlag(true);
				}
			} catch (Exception e) {
				m_log.warn("update() getJobDetailList, " + e.getMessage(), e);
				if(LocalSession.isDialogFlag()){
					LocalSession.setDialogFlag(false);
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
					LocalSession.setDialogFlag(true);
				}
			}
		}
		if (item == null) {
			return;
		}
		m_viewer.setInput(item);
		m_viewer.expandAll();

		if (m_sessionId != null && m_sessionId.length() > 0
				&& sessionId != null && sessionId.length() > 0
				&& m_sessionId.compareTo(sessionId) == 0) {
			selectDetail(item.getChildren().get(0));
		} else {
			setJobId(null);
		}
		m_sessionId = sessionId;
		m_jobunitId = jobunitId;

		//セッションIDを表示
		if (m_sessionId != null) {
			m_sessionIdLabel.setText(Messages.getString("session.id") + " : "
					+ m_sessionId);
		} else {
			m_sessionIdLabel.setText(Messages.getString("session.id") + " : ");
		}
	}

	/**
	 * ジョブ詳細の行を選択します。<BR>
	 * 前回選択したジョブIDと同じジョブIDの行を選択します。
	 * 
	 * @param item テーブルツリーアイテム
	 */
	public void selectDetail(JobTreeItem item) {
		if (getJobId() != null && getJobId().length() > 0) {
			if (m_viewer.getSelection().isEmpty()) {
				boolean select = false;
				JobInfo info = item.getData();
				if (info == null) {
					m_log.info("selectDetail info is null");
					return;
				}
				String jobId = info.getId();
				if (getJobId().compareTo(jobId) == 0) {
					select = true;
				}

				if (select) {
					m_viewer.setSelection(new StructuredSelection(item), true);
				} else {
					for (int i = 0; i < item.getChildren().size(); i++) {
						JobTreeItem children = item.getChildren().get(i);
						selectDetail(children);
					}
				}
			}
		}
	}

	/**
	 * このコンポジットが利用するテーブルツリービューアを返します。
	 * 
	 * @return テーブルツリービューア
	 */
	public TableTreeViewer getTableTreeViewer() {
		return m_viewer;
	}

	/**
	 * このコンポジットが利用するテーブルを返します。
	 * 
	 * @return テーブル
	 */
	public Table getTable() {
		return m_viewer.getTableTree().getTable();
	}

	/**
	 * このコンポジットが利用するテーブルツリーを返します。
	 * 
	 * @return テーブルツリー
	 */
	public TableTree getTableTree() {
		return m_viewer.getTableTree();
	}

	/**
	 * セッションIDを返します。
	 * 
	 * @return セッションID
	 */
	public String getSessionId() {
		return m_sessionId;
	}

	/**
	 * セッションIDを設定します。
	 * 
	 * @param sessionId セッションID
	 */
	public void setSessionId(String sessionId) {
		m_sessionId = sessionId;
	}

	/**
	 * ジョブIDを返します。
	 * 
	 * @return ジョブID
	 */
	public String getJobId() {
		return m_jobId;
	}

	/**
	 * ジョブIDを設定します。
	 * 
	 * @param jobId ジョブID
	 */
	public void setJobId(String jobId) {
		m_jobId = jobId;
	}

	/**
	 * 所属ジョブユニットのジョブIDを返します。
	 * 
	 * @return 所属ジョブユニットのジョブID
	 */
	public String getJobunitId() {
		return m_jobunitId;
	}

	/**
	 * 所属ジョブユニットのジョブIDを設定します。
	 * 
	 * @param jobunitId 所属ジョブユニットのジョブID
	 */
	public void setJobunitId(String jobunitId) {
		m_jobunitId = jobunitId;
	}



}