/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.composite;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.jobmanagement.action.GetJobKickTableDefine;
import com.clustercontrol.jobmanagement.bean.JobTriggerTypeConstant;
import com.clustercontrol.jobmanagement.composite.action.JobKickDoubleClickListener;
import com.clustercontrol.jobmanagement.composite.action.JobKickSelectionChangedListener;
import com.clustercontrol.jobmanagement.util.JobEndpointWrapper;
import com.clustercontrol.util.Messages;
import com.clustercontrol.viewer.CommonTableViewer;
import com.clustercontrol.ws.jobmanagement.InvalidRole_Exception;
import com.clustercontrol.ws.jobmanagement.JobKick;

/**
 * ジョブ[実行契機]ビュー用のコンポジットクラスです。
 * 
 * @version 4.1.0
 * @since 1.0.0
 */
public class JobKickListComposite extends Composite {

	// ログ
	private static Log m_log = LogFactory.getLog( JobKickListComposite.class );

	/** テーブルビューア */
	private CommonTableViewer m_viewer = null;
	/** 選択アイテム */
	private ArrayList m_selectItem = null;

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public JobKickListComposite(Composite parent, int style) {
		super(parent, style);
		initialize();
	}

	/**
	 * コンポジットを構築します。
	 */
	private void initialize() {
		GridLayout layout = new GridLayout(1, true);
		this.setLayout(layout);
		layout.marginHeight = 0;
		layout.marginWidth = 0;

		Table table = new Table(this, SWT.H_SCROLL | SWT.V_SCROLL
				| SWT.FULL_SELECTION | SWT.MULTI);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 1;
		table.setLayoutData(gridData);

		m_viewer = new CommonTableViewer(table);
		m_viewer.createTableColumn(GetJobKickTableDefine.get(),
				GetJobKickTableDefine.SORT_COLUMN_INDEX,
				GetJobKickTableDefine.SORT_ORDER);

		m_viewer
		.addSelectionChangedListener(new JobKickSelectionChangedListener(
				this));
		// ダブルクリックリスナの追加
		m_viewer.addDoubleClickListener(new JobKickDoubleClickListener(this));
	}

	/**
	 * テーブルビューアーを更新します。<BR>
	 * ジョブ[実行契機]一覧情報を取得し、共通テーブルビューアーにセットします。
	 * <p>
	 * <ol>
	 * <li>スケジュール一覧情報を取得します。</li>
	 * <li>共通テーブルビューアーにジョブ[実行契機]一覧情報をセットします。</li>
	 * </ol>
	 * 
	 * @see #getJobKick()
	 */
	@Override
	public void update() {
		ArrayList<Object> listInput = new ArrayList<Object>();
		for (JobKick jobKick : getJobKick()) {
			ArrayList<Object> a = new ArrayList<Object>();
			a.add(JobTriggerTypeConstant.typeToString(jobKick.getType()));
			a.add(jobKick.getId());
			a.add(jobKick.getName());
			a.add(jobKick.getJobId());
			a.add(jobKick.getJobName());
			a.add(jobKick.getJobunitId());
			a.add(jobKick.getCalendarId());
			a.add(jobKick.getValid());
			a.add(jobKick.getOwnerRoleId());
			a.add(jobKick.getCreateUser());
			if(jobKick.getCreateTime() == null){
				a.add(null);
			}
			else{
				a.add(new Date(jobKick.getCreateTime()));
			}
			a.add(jobKick.getUpdateUser());
			if(jobKick.getUpdateTime() == null){
				a.add(null);
			}
			else{
				a.add(new Date(jobKick.getUpdateTime()));
			}
			listInput.add(a);
		}
		m_viewer.setInput(listInput);
	}

	/**
	 * 実行契機一覧情報を取得します。
	 * 
	 * @return 実行契機一覧情報
	 * 
	 * @see com.clustercontrol.jobmanagement.action.GetSchedule#getSchedule()
	 */
	private List<JobKick> getJobKick() {
		List<JobKick> jobKick = null;

		//実行契機情報取得
		try {
			jobKick = JobEndpointWrapper.getJobKickList();
		} catch (InvalidRole_Exception e) {
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("getJobSchedule(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}
		if(jobKick == null){
			jobKick = new ArrayList<JobKick>();
		}
		return jobKick;
	}

	/**
	 * このコンポジットが利用するテーブルビューアを返します。
	 * 
	 * @return テーブルビューア
	 */
	public TableViewer getTableViewer() {
		return m_viewer;
	}

	/**
	 * このコンポジットが利用するテーブルを返します。
	 * 
	 * @return テーブル
	 */
	public Table getTable() {
		return m_viewer.getTable();
	}

	/**
	 * 選択アイテムを返します。
	 * 
	 * @return 選択アイテム
	 */
	public ArrayList getSelectItem() {
		return m_selectItem;
	}

	/**
	 * 選択アイテムを設定します。
	 * 
	 * @param selectItem 選択アイテム
	 */
	public void setSelectItem(ArrayList selectItem) {
		this.m_selectItem = selectItem;
	}
}