/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.port.dialog;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.monitor.run.dialog.CommonMonitorNumericDialog;
import com.clustercontrol.monitor.util.MonitorSettingEndpointWrapper;
import com.clustercontrol.notify.util.NotifyGroupIdGenerator;
import com.clustercontrol.port.bean.PortRunCountConstant;
import com.clustercontrol.port.bean.PortRunIntervalConstant;
import com.clustercontrol.port.bean.ProtocolConstant;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.monitor.InvalidRole_Exception;
import com.clustercontrol.ws.monitor.MonitorDuplicate_Exception;
import com.clustercontrol.ws.monitor.MonitorInfo;
import com.clustercontrol.ws.monitor.PortCheckInfo;

/**
 * サービス･ポート監視作成・変更ダイアログクラスです。
 * 
 * @version 4.0.0
 * @since 2.4.0
 */
public class PortCreateDialog extends CommonMonitorNumericDialog {

	// ログ
	private static Log m_log = LogFactory.getLog( PortCreateDialog.class );

	// ----- instance フィールド ----- //
	/** タイムアウト用テキストボックス */
	private Text m_textTimeout = null;

	/** TCP接続のみ */
	private Button m_radioTCP = null;

	/** サービス */
	private Button m_radioService = null;

	/** サービスプロトコル */
	private Combo m_comboService = null;

	/** ポート番号 */
	private Text m_textPortNo = null;

	/** 実行回数 */
	private Combo m_comboRunCount = null;

	/** 実行間隔（秒） */
	private Combo m_comboRunInterval = null;

	// ----- コンストラクタ ----- //

	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent
	 *            親のシェルオブジェクト
	 */
	public PortCreateDialog(Shell parent) {
		super(parent);
	}

	/**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent
	 *            親のシェルオブジェクト
	 * @param notifyId
	 *            変更する通知ID
	 * @param updateFlg
	 *            更新するか否か（true:変更、false:新規登録）
	 * 
	 */
	public PortCreateDialog(Shell parent, String monitorId, boolean updateFlg) {
		super(parent);

		this.monitorId = monitorId;
		this.updateFlg = updateFlg;
	}

	// ----- instance メソッド ----- //

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent
	 *            親のインスタンス
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		// 項目名に「応答時間（ミリ秒）」を設定
		item1 = Messages.getString("response.time.milli.sec");
		item2 = Messages.getString("response.time.milli.sec");

		super.customizeDialog(parent);
		m_numericValueInfo.setInfoWarnText("0", "1000", "1000", "3000");

		// タイトル
		shell.setText(Messages.getString("dialog.port.create.modify"));

		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;

		/*
		 * チェック設定グループ（条件グループの子グループ）
		 */
		Group groupCheckRule = new Group(groupRule, SWT.NONE);
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 15;
		groupCheckRule.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupCheckRule.setLayoutData(gridData);
		groupCheckRule.setText(Messages.getString("check.rule"));

		/*
		 * 監視プロトコル
		 */
		// TCP接続のみ
		this.m_radioTCP = new Button(groupCheckRule, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_radioTCP.setLayoutData(gridData);
		this.m_radioTCP.setText(Messages.getString("tcp.connect.only"));
		this.m_radioTCP.setSelection(true);

		// サービス
		this.m_radioService = new Button(groupCheckRule, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_radioService.setLayoutData(gridData);
		this.m_radioService.setText(Messages.getString("service.protocol"));
		this.m_radioService.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				setEnabledComboService(m_radioService.getSelection());
			}
		});

		//コンボボックス
		this.m_comboService = new Combo(groupCheckRule, SWT.DROP_DOWN | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_comboService.setLayoutData(gridData);
		this.m_comboService.add(ProtocolConstant.STRING_PROTOCOL_FTP);
		this.m_comboService.add(ProtocolConstant.STRING_PROTOCOL_SMTP);
		this.m_comboService.add(ProtocolConstant.STRING_PROTOCOL_SMTPS);
		this.m_comboService.add(ProtocolConstant.STRING_PROTOCOL_POP3);
		this.m_comboService.add(ProtocolConstant.STRING_PROTOCOL_POP3S);
		this.m_comboService.add(ProtocolConstant.STRING_PROTOCOL_IMAP);
		this.m_comboService.add(ProtocolConstant.STRING_PROTOCOL_IMAPS);
		this.m_comboService.add(ProtocolConstant.STRING_PROTOCOL_NTP);
		this.m_comboService.add(ProtocolConstant.STRING_PROTOCOL_DNS);
		//this.m_comboService.setEnabled(false);

		//空白
		label = new Label(groupCheckRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 4;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * ポート番号
		 */
		// ラベル
		label = new Label(groupCheckRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("port.number") + " : ");
		// テキスト
		this.m_textPortNo = new Text(groupCheckRule, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textPortNo.setLayoutData(gridData);
		this.m_textPortNo.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		// 空白
		label = new Label(groupCheckRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 9;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * 実行回数
		 */
		// ラベル
		label = new Label(groupCheckRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("trial.run.count") + " : ");
		// コンボボックス
		this.m_comboRunCount = new Combo(groupCheckRule, SWT.DROP_DOWN | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_comboRunCount.setLayoutData(gridData);
		this.m_comboRunCount.add(PortRunCountConstant.STRING_COUNT_01);
		this.m_comboRunCount.add(PortRunCountConstant.STRING_COUNT_02);
		this.m_comboRunCount.add(PortRunCountConstant.STRING_COUNT_03);

		// 空白
		label = new Label(groupCheckRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 9;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * 実行間隔
		 */
		// ラベル
		label = new Label(groupCheckRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("trial.run.interval") + " : ");
		// コンボボックス
		this.m_comboRunInterval = new Combo(groupCheckRule, SWT.DROP_DOWN | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_comboRunInterval.setLayoutData(gridData);
		this.m_comboRunInterval.add(PortRunIntervalConstant.STRING_SEC_01);
		this.m_comboRunInterval.add(PortRunIntervalConstant.STRING_SEC_02);
		this.m_comboRunInterval.add(PortRunIntervalConstant.STRING_SEC_05);

		// 空白
		label = new Label(groupCheckRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 9;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * タイムアウト
		 */
		// ラベル
		label = new Label(groupCheckRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("time.out") + " : ");

		// テキスト
		this.m_textTimeout = new Text(groupCheckRule, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textTimeout.setLayoutData(gridData);
		this.m_textTimeout.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		// ラベル（単位）
		label = new Label(groupCheckRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("milli.sec"));

		// 空白
		label = new Label(groupCheckRule, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 7;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		// 収集値表示名のデフォルト値を設定
		this.itemName.setText(Messages.getString("response.time"));

		// 収集値単位のデフォルト値を設定
		this.measure.setText(Messages.getString("time.msec"));

		// ダイアログを調整
		this.adjustDialog();

		// 初期表示
		MonitorInfo info = null;
		if(this.monitorId == null){
			// 作成の場合
			info = new MonitorInfo();
			this.setInfoInitialValue(info);
		} else {
			// 変更の場合、情報取得
			try {
				info = MonitorSettingEndpointWrapper.getMonitor(monitorId, HinemosModuleConstant.MONITOR_PORT);
			} catch (InvalidRole_Exception e) {
				// アクセス権なしの場合、エラーダイアログを表示する
				MessageDialog.openInformation(
						null,
						Messages.getString("message"),
						Messages.getString("message.accesscontrol.16"));

			} catch (Exception e) {
				// Managerとの通信で予期せぬ内部エラーが発生したことを通知する
				m_log.warn("customizeDialog(), " + e.getMessage(), e);
				MessageDialog.openInformation(
						null,
						Messages.getString("message"),
						Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());

			}
		}
		this.setInputData(info);

	}

	protected void setEnabledComboService(boolean enable) {
		this.m_comboService.setEnabled(enable);
	}

	/**
	 * 更新処理
	 * 
	 */
	@Override
	public void update(){
		super.update();

		// 必須項目を明示
		if(this.m_textPortNo.getEnabled() && "".equals(this.m_textPortNo.getText())){
			this.m_textPortNo.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_textPortNo.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		if(this.m_textTimeout.getEnabled() && "".equals(this.m_textTimeout.getText())){
			this.m_textTimeout.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_textTimeout.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}

	/**
	 * 各項目に入力値を設定します。
	 * 
	 * @param monitor
	 *            設定値として用いる通知情報
	 */
	@Override
	protected void setInputData(MonitorInfo monitor) {
		super.setInputData(monitor);

		this.inputData = monitor;

		// 監視条件 port監視情報
		PortCheckInfo portInfo = monitor.getPortCheckInfo();
		if (portInfo.getServiceId() == null || portInfo.getServiceId().equals(ProtocolConstant.TYPE_PROTOCOL_TCP)) {
			this.m_radioTCP.setSelection(true);
			this.m_radioService.setSelection(false);
			setEnabledComboService(m_radioService.getSelection());
		} else {
			this.m_radioTCP.setSelection(false);
			this.m_radioService.setSelection(true);
			setEnabledComboService(m_radioService.getSelection());
			this.m_comboService.setText(ProtocolConstant.typeToString(portInfo.getServiceId()));
		}
		if (portInfo.getPortNo() != null) {
			this.m_textPortNo.setText(portInfo.getPortNo().toString());
		}
		if (portInfo.getRunCount() != null) {
			this.m_comboRunCount.setText(PortRunCountConstant.typeToString(portInfo.getRunCount().intValue()));
		}
		if (portInfo.getRunInterval() != null) {
			this.m_comboRunInterval.setText(PortRunIntervalConstant.typeToString(portInfo.getRunInterval().intValue()));
		}
		// タイムアウト
		if (portInfo.getTimeout() != null) {
			this.m_textTimeout.setText(portInfo.getTimeout().toString());
		}

		m_numericValueInfo.setInputData(monitor);

	}

	/**
	 * 入力値を用いて通知情報を生成します。
	 * 
	 * @return 入力値を保持した通知情報
	 */
	@Override
	protected MonitorInfo createInputData() {
		super.createInputData();
		if(validateResult != null){
			return null;
		}

		// port監視固有情報を設定
		monitorInfo.setMonitorTypeId(HinemosModuleConstant.MONITOR_PORT);

		// 監視条件 port監視情報
		PortCheckInfo portInfo = new PortCheckInfo();

		/** 監視サービスの選択 */
		if (this.m_radioTCP.getSelection()) {
			portInfo.setServiceId(ProtocolConstant.TYPE_PROTOCOL_TCP);
		} else if (this.m_radioService.getSelection()) {/* サービスプロトコル選択時の処理 */
			if (this.m_comboService.getText() != null && !"".equals((this.m_comboService.getText()).trim())) {
				portInfo.setServiceId(ProtocolConstant.stringToType(this.m_comboService.getText()));
			}
		}
		/** ポート番号 */
		if (this.m_textPortNo.getText() != null && !"".equals((this.m_textPortNo.getText()).trim())) {
			try{
				Integer portNo = new Integer(this.m_textPortNo.getText().trim());
				portInfo.setPortNo(portNo);
			}
			catch(NumberFormatException e){
				this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.port.8"));
				return null;
			}
		}
		/** 試行回数 */
		if (this.m_comboRunCount.getText() != null
				&& !"".equals((this.m_comboRunCount.getText()).trim())) {
			portInfo.setRunCount(PortRunCountConstant.stringToType(this.m_comboRunCount.getText()));
		}
		/** 試行間隔 */
		if (this.m_comboRunInterval.getText() != null
				&& !"".equals((this.m_comboRunInterval.getText()).trim())) {
			portInfo.setRunInterval(PortRunIntervalConstant.stringToType(this.m_comboRunInterval.getText()));
		}
		/** タイムアウト */
		if (!"".equals((this.m_textTimeout.getText()).trim())) {

			try{
				Integer timeout = new Integer(this.m_textTimeout.getText().trim());
				portInfo.setTimeout(timeout);
			}
			catch(NumberFormatException e){
				this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.port.3"));
				return null;
			}
		}
		monitorInfo.setPortCheckInfo(portInfo);


		// 結果判定の定義
		validateResult = m_numericValueInfo.createInputData(monitorInfo);
		if(validateResult != null){
			return null;
		}

		// 通知関連情報とアプリケーションの設定
		// 通知グループIDの設定
		monitorInfo.setNotifyGroupId(NotifyGroupIdGenerator.createNotifyGroupId(HinemosModuleConstant.MONITOR_PORT, monitorInfo.getMonitorId()));
		validateResult = m_notifyInfo.createInputData(monitorInfo);
		if (validateResult != null) {
			if(validateResult.getID() == null){	// 通知ID警告用出力
				if(!displayQuestion(validateResult)){
					validateResult = null;
					return null;
				}
			}
			else{	// アプリケーション未入力チェック
				return null;
			}
		}

		return monitorInfo;
	}

	/**
	 * 入力値をマネージャに登録します。
	 * 
	 * @return true：正常、false：異常
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		boolean result = false;

		MonitorInfo info = this.inputData;
		if(info != null){
			String[] args = { info.getMonitorId() };
			if(!this.updateFlg){
				// 作成の場合
				try {
					result = MonitorSettingEndpointWrapper.addMonitor(info);

					if(result){
						MessageDialog.openInformation(
								null,
								Messages.getString("successful"),
								Messages.getString("message.monitor.33", args));
					} else {
						MessageDialog.openError(
								null,
								Messages.getString("failed"),
								Messages.getString("message.monitor.34", args));
					}
				} catch (MonitorDuplicate_Exception e) {
					// 監視項目IDが重複している場合、エラーダイアログを表示する
					MessageDialog.openInformation(
							null,
							Messages.getString("message"),
							Messages.getString("message.monitor.53", args));
				} catch (Exception e) {
					String errMessage = "";
					if (e instanceof InvalidRole_Exception) {
						// アクセス権なしの場合、エラーダイアログを表示する
						MessageDialog.openInformation(
								null,
								Messages.getString("message"),
								Messages.getString("message.accesscontrol.16"));
					} else {
						errMessage = ", " + e.getMessage();
					}

					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.monitor.34", args) + errMessage);
				}
			} else {
				// 変更の場合
				String errMessage = "";
				try {
					result = MonitorSettingEndpointWrapper.modifyMonitor(info);
				} catch (InvalidRole_Exception e) {
					// アクセス権なしの場合、エラーダイアログを表示する
					MessageDialog.openInformation(
							null,
							Messages.getString("message"),
							Messages.getString("message.accesscontrol.16"));
				} catch (Exception e) {
					errMessage = ", " + e.getMessage();
				}

				if(result){
					MessageDialog.openInformation(
							null,
							Messages.getString("successful"),
							Messages.getString("message.monitor.35", args));
				} else {
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.monitor.36", args) + errMessage);
				}
			}
		}

		return result;
	}

	/**
	 * MonitorInfoに初期値を設定します
	 * 
	 * @see com.clustercontrol.dialog.CommonMonitorDialog#setInfoInitialValue()
	 */
	@Override
	protected void setInfoInitialValue(MonitorInfo monitor) {

		super.setInfoInitialValue(monitor);

		PortCheckInfo portCheckInfo = new PortCheckInfo();
		// ポート番号
		portCheckInfo.setPortNo(ProtocolConstant.DEFAULT_PORT_NUM_PROTOCOL_TCP);
		// リトライ回数（回）
		portCheckInfo.setRunCount(PortRunCountConstant.TYPE_COUNT_01);
		// リトライ間隔（ミリ秒）
		portCheckInfo.setRunInterval(PortRunIntervalConstant.TYPE_SEC_01);
		// タイムアウト（ミリ秒）
		portCheckInfo.setTimeout(TIMEOUT_SEC);
		monitor.setPortCheckInfo(portCheckInfo);
	}

}