/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.repository.dialog;

import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.custom.TableTreeItem;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.accesscontrol.util.LocalSession;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyDefineConstant;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.bean.SnmpVersionConstant;
import com.clustercontrol.composite.RoleIdListComposite;
import com.clustercontrol.composite.RoleIdListComposite.Mode;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.repository.action.GetNodeProperty;
import com.clustercontrol.repository.bean.NodeConstant;
import com.clustercontrol.repository.util.NodePropertyUtil;
import com.clustercontrol.repository.util.RepositoryEndpointWrapper;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.viewer.PropertySheet;
import com.clustercontrol.ws.repository.FacilityDuplicate_Exception;
import com.clustercontrol.ws.repository.InvalidRole_Exception;
import com.clustercontrol.ws.repository.NodeInfo;

/**
 * ノードの作成・変更ダイアログクラス<BR>
 * 
 * @version 4.0.0
 * @since 1.0.0
 */
public class NodeCreateDialog extends CommonDialog {

	// ログ
	private static Log m_log = LogFactory.getLog( NodeCreateDialog.class );


	// 後でpackするためsizeXはダミーの値。
	private final int sizeX = 500;
	private final int sizeY = 800;

	// ----- instance フィールド ----- //

	/** 初期表示ノードのファシリティID */
	private String facilityId = "";

	/** 一括変更対象スコープのファシリティID */
	private String scopeId = "";

	/** 変更前のプロパティ */
	private Property propertyOld = null;

	/** 変更前のノード情報 */
	private NodeInfo nodeInfoOld = null;

	/** 変更用ダイアログ判別フラグ */
	private boolean isModifyDialog = false;

	/** ノード属性プロパティシート */
	private PropertySheet propertySheet = null;

	/** クリア実行フラグ */
	private boolean isClearDone = false;

	/** クリアボタン */
	private Button clearButtonScope = null;

	/** オーナーロールID用テキスト */
	private RoleIdListComposite m_ownerRoleId = null;

	//SNMPで検出する機能のための追加コンポーネント
	private Button buttonAuto = null;

	private Label    ipAddressText = null; //"IP Address";
	private Text     ipAddressBox  = null;
	private Label    communityText = null; //"community";
	private Text     communityBox  = null;
	private Label    portText      = null; //"port";
	private Text     portBox       = null;
	private Label    versionText      = null; //"version";
	private Combo    versionBox       = null;
	private TableTree table = null;

	private NodeInfo nodeInfo = null;

	// ----- コンストラクタ ----- //

	/**
	 * 指定した形式のダイアログのインスタンスを返します。
	 * 
	 * @param parent
	 *            親のシェルオブジェクト
	 * @param facilityId
	 *            初期表示するノードのファシリティID
	 * @param isModifyDialog
	 *            変更用ダイアログとして利用する場合は、true
	 */
	public NodeCreateDialog(Shell parent, String facilityId,
			boolean isModifyDialog) {
		super(parent);

		this.facilityId = facilityId;
		this.isModifyDialog = isModifyDialog;
	}

	// ----- instance メソッド ----- //

	/**
	 * ダイアログの初期サイズを返します。
	 * 
	 * @return 初期サイズ
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(500, 600);
	}

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent
	 *            親のインスタンス
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();
		// タイトル
		shell.setText(Messages
				.getString("dialog.repository.node.create.modify"));

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		parent.setLayout(layout);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;

		//SNMPによる検出のグループ(SNMP)
		Group groupAuto = new Group(parent, SWT.RIGHT);
		layout = new GridLayout(5, false);
		groupAuto.setLayout(layout);
		if (isModifyDialog) {
			groupAuto.setText("Find By SNMP " +
					Messages.getString("repository.find.by.snmp.modify"));
		} else {
			groupAuto.setText("Find By SNMP");
		}

		//IPアドレス
		this.ipAddressText = new Label(groupAuto, SWT.NONE);
		this.ipAddressText.setText(Messages.getString("ip.address") + " : ");

		this.ipAddressBox = new Text(groupAuto, SWT.BORDER | SWT.SINGLE);
		GridData grid = new GridData(GridData.FILL_HORIZONTAL);
		grid.minimumWidth = 120;
		this.ipAddressBox.setLayoutData(grid);

		// プロパティ取得
		if (this.isModifyDialog) {
			GetNodeProperty getNodeProperty = new GetNodeProperty(this.facilityId,
					PropertyDefineConstant.MODE_MODIFY);
			propertyOld = getNodeProperty.getProperty();
			nodeInfoOld = getNodeProperty.getNodeInfo();
		} else {
			GetNodeProperty getNodeProperty = new GetNodeProperty(this.facilityId,
					PropertyDefineConstant.MODE_ADD);
			propertyOld = getNodeProperty.getProperty();
			nodeInfoOld = getNodeProperty.getNodeInfo();
		}
		Property propertyChild = null;
		propertyChild = PropertyUtil.getProperty(propertyOld, NodeConstant.IP_ADDRESS_VERSION).get(0);
		String ipAddressVersion = propertyChild.getValue().toString();
		String ipAddress = null;
		if ("6".equals(ipAddressVersion)) {
			propertyChild = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.IP_ADDRESS_V6).get(0);
			ipAddress = propertyChild.getValue().toString();
		} else {
			propertyChild = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.IP_ADDRESS_V4).get(0);
			ipAddress = propertyChild.getValue().toString();
		}
		if ("".equals(ipAddress)) {
			this.ipAddressBox.setText("192.168.0.1");
		} else {
			this.ipAddressBox.setText(ipAddress);
		}

		//ポート名
		this.portText = new Label(groupAuto, SWT.NONE);
		this.portText.setText(Messages.getString("port.number") + " : ");

		this.portBox = new Text(groupAuto, SWT.BORDER | SWT.SINGLE);
		grid = new GridData(GridData.FILL_HORIZONTAL);
		grid.minimumWidth = 50;
		this.portBox.setLayoutData(grid);
		propertyChild = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.SNMP_PORT).get(0);
		String snmpPort = propertyChild.getValue().toString();
		if ("".equals(snmpPort)) {
			this.portBox.setText("161");
		} else {
			this.portBox.setText(snmpPort);
		}

		// 改行のため、ダミーのラベルを挿入。
		Label dummy2 = new Label(groupAuto, SWT.NONE);

		//コミュニティ名
		this.communityText = new Label(groupAuto, SWT.NONE);
		this.communityText.setText(Messages.getString("community.name") + " : ");

		this.communityBox = new Text(groupAuto, SWT.BORDER | SWT.SINGLE);
		this.communityBox.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		propertyChild = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.SNMP_COMMUNITY).get(0);
		String snmpCommunity = propertyChild.getValue().toString();
		if ("".equals(snmpCommunity)) {
			this.communityBox.setText("public");
		} else {
			this.communityBox.setText(snmpCommunity);
		}

		//バージョン
		this.versionText = new Label(groupAuto, SWT.NONE);
		this.versionText.setText(Messages.getString("snmp.version") + " : ");

		this.versionBox = new Combo(groupAuto, SWT.DROP_DOWN | SWT.READ_ONLY);
		this.versionBox.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		this.versionBox.add("1",0);
		this.versionBox.add("2c",1);
		// デフォルトをv2cとする
		propertyChild = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.SNMP_VERSION).get(0);
		String snmpVersion = propertyChild.getValue().toString();
		if ("1".equals(snmpVersion)) {
			this.versionBox.select(0);
		} else {
			this.versionBox.select(1);
		}

		// 変更時はFind By SNMPの項目はclearボタンが押下されない限り変更不可
		if (isModifyDialog) {
			this.ipAddressBox.setEditable(false);
			this.portBox.setEditable(false);
			this.communityBox.setEditable(false);
			this.versionBox.setTouchEnabled(false);
		}

		
		this.buttonAuto = new Button(groupAuto, SWT.PUSH | SWT.RIGHT);
		this.buttonAuto.setText(" Find ");
		gridData = new GridData();
		gridData.horizontalIndent= 30;
		this.buttonAuto.setLayoutData(gridData);
		this.buttonAuto.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				try{
					String ipAddressString = ipAddressBox.getText();

					// IPアドレスチェック
					InetAddress address = InetAddress.getByName(ipAddressString);

					if (address instanceof Inet4Address){

						//IPv4の場合はさらにStringをチェック
						if (!ipAddressString.matches(".{1,3}?\\..{1,3}?\\..{1,3}?\\..{1,3}?")){
							MessageDialog.openWarning(getShell(), "Warning",  Messages.getString("message.repository.37"));
							return;
						}

					} else if (address instanceof Inet6Address){
						//IPv6の場合は特にStringチェックは無し
					} else {
						MessageDialog.openWarning(getShell(), "Warning",  Messages.getString("message.repository.37"));
						return;
					}

					String ipAddress = ipAddressBox.getText();
					Integer port = Integer.parseInt(portBox.getText());
					String community = communityBox.getText();
					String version = SnmpVersionConstant.typeToString(new Integer(versionBox.getSelectionIndex()));


					Property propertySNMP = null;
					NodeInfo nodeInfo = null;

					if (isModifyDialog) {
						nodeInfo = GetNodeInfoBySNMP(ipAddress, port, community, version,
								PropertyDefineConstant.MODE_MODIFY);
						propertySNMP = NodePropertyUtil.node2property(nodeInfo, PropertyDefineConstant.MODE_MODIFY, Locale.getDefault());
						/*
						 * ノード変更時のfind by snmpはデバイスのみ修正する。
						 */
						// 古い値を消して(device)
						for(Property propertyObj : PropertyUtil.getProperty(propertyOld, NodeConstant.DEVICE)){
							Property p = propertyObj;
							p.getParent().removeChildren(p);
						}
						// 新しい値を入れる(device)
						for(Property propertyObj : PropertyUtil.getProperty(propertySNMP, NodeConstant.DEVICE)){
							Property p = propertyObj;
							propertyOld.addChildren(p);
						}

						/*
						 * 上記の処理だけでは、順番がおかしくなってしまうため、
						 * ユーザ等を再登録して、後ろにまわす。
						 */
						{
							Property p = null;
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.SERVER_VIRTUALIZATION).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.NETWORK_VIRTUALIZATION).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.NODE_VARIABLE).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.MAINTENANCE).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.CREATE_TIME).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.CREATOR_NAME).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.MODIFY_TIME).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);
							p = (Property)PropertyUtil.getProperty(propertyOld, NodeConstant.MODIFIER_NAME).get(0);
							propertyOld.removeChildren(p);
							propertyOld.addChildren(p);

							// 備考は複数含まれることがある。
							ArrayList<Property> oArray = PropertyUtil.getProperty(propertyOld, NodeConstant.NOTE);
							for (Property o : oArray) {
								propertyOld.removeChildren(o);
								propertyOld.addChildren(o);
							}
						}
						propertySheet.setInput(propertyOld);
					} else {
						nodeInfo = GetNodeInfoBySNMP(ipAddress, port, community, version,
								PropertyDefineConstant.MODE_ADD);
						propertySNMP = NodePropertyUtil.node2property(nodeInfo, PropertyDefineConstant.MODE_ADD, Locale.getDefault());
						propertySheet.setInput(propertySNMP);
					}

					// シートの展開と色表示
					update();
					expand();
					
				} catch (NumberFormatException e1){
					// port番号が不正な場合
					MessageDialog.openWarning(getShell(), "Warning", Messages.getString("message.repository.38"));

				} catch (UnknownHostException e2) {
					// IPアドレスが不正な場合
					MessageDialog.openWarning(getShell(), "Warning",  Messages.getString("message.repository.37"));
				}
			}
		});

		/////////////////////////////////////////////////////////////

		/*
		 * オーナーロールID
		 */
		Label labelRoleId = new Label(parent, SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelRoleId.setLayoutData(gridData);
		labelRoleId.setText(Messages.getString("owner.role.id") + " : ");
		if(!this.isModifyDialog()){
			this.m_ownerRoleId = new RoleIdListComposite(parent, SWT.NONE, true, Mode.OWNER_ROLE);
		} else {
			this.m_ownerRoleId = new RoleIdListComposite(parent, SWT.NONE, false, Mode.OWNER_ROLE);
		}
		if (nodeInfoOld != null && nodeInfoOld.getOwnerRoleId() != null) {
			m_ownerRoleId.setText(nodeInfoOld.getOwnerRoleId());
		}
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_ownerRoleId.setLayoutData(gridData);
		
		
		/*
		 * 属性プロパティシート
		 */

		// ラベル
		Label label = new Label(parent, SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("attribute") + " : ");

		// プロパティシート
		table = new TableTree(parent, SWT.H_SCROLL | SWT.V_SCROLL
				| SWT.FULL_SELECTION | SWT.MULTI | SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.heightHint = SWT.MIN;
		gridData.horizontalSpan = 1;
		table.setLayoutData(gridData);
		table.addSelectionListener(new SelectionAdapter(){
			@Override
			public void widgetSelected(SelectionEvent e) {
				// シートの展開と色表示
				update();
			}
		});

		this.propertySheet = new PropertySheet(table);
		this.propertySheet.setSize(230, 300);

		// プロパティ設定
		this.propertySheet.setInput(propertyOld);

		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		line.setLayoutData(gridData);

		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);

		//ダイアログのサイズ調整（pack:resize to be its preferred size）
		shell.pack();
		shell.setSize(new Point(shell.getSize().x, sizeY ));

		this.update();
		this.expand();
	}

	/**
	 * プロパティシートの展開
	 * 
	 */
	private void expand(){
		
		m_log.debug("expand");
		/*プロパティシートの展開を指定します。*/
		//レベル1までの展開
		this.propertySheet.expandToLevel(1);
		//サーバ基本情報
		Object element =  this.propertySheet.getElementAt(4);
		this.propertySheet.expandToLevel(element, 2);
		//サービス
		element =  this.propertySheet.getElementAt(6);
		this.propertySheet.expandToLevel(element, 1);
		//デバイス
		element =  this.propertySheet.getElementAt(7);
		this.propertySheet.expandToLevel(element, 1);
	}
	
	/**
	 * 更新処理
	 * 
	 */
	public void update(){

		m_log.debug("update");

		/*必須項目を色別表示します。*/

		// ファシリティIDのインデックス：0
		setColor(table.getItem(0));

		// ファシリティ名のインデックス：1
		setColor(table.getItem(1));

		// プラットフォームのインデックス：4-0-1
		if(table.getItem(4).getItemCount() > 0 && table.getItem(4).getItem(0).getItemCount() > 0) {
			setColor(table.getItem(4).getItem(0).getItem(0));
		}

		// ネットワークのインデックス：4-1
		if (table.getItem(4).getItemCount() > 1 && table.getItem(4).getItem(1).getItemCount() > 0) {
			if("4".equals((table.getItem(4).getItem(1).getItem(0).getText(1)))){
				table.getItem(4).getItem(1).getItem(0).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				setColor(table.getItem(4).getItem(1).getItem(1));
				table.getItem(4).getItem(1).getItem(2).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}else if("6".equals((table.getItem(4).getItem(1).getItem(0).getText(1)))){
				table.getItem(4).getItem(1).getItem(0).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				table.getItem(4).getItem(1).getItem(1).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				setColor(table.getItem(4).getItem(1).getItem(2));
			}else {
				table.getItem(4).getItem(1).getItem(0).setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
				table.getItem(4).getItem(1).getItem(1).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				table.getItem(4).getItem(1).getItem(2).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}
		}

		// OSのインデックス:4
		// -ノード名のインデックス:2
		if (table.getItem(4).getItemCount() > 2 && table.getItem(4).getItem(2).getItemCount() > 0) {
			setColor(table.getItem(4).getItem(2).getItem(0));
		}
		
		// 仮想化のインデックス:8
		// -ノード種別のインデックス:0
		if (table.getItem(8).getItemCount() > 6) {
			if("controller".equals(table.getItem(8).getItem(0).getText(1)) || "host".equals(table.getItem(8).getItem(0).getText(1))){
				table.getItem(8).getItem(1).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				table.getItem(8).getItem(2).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				table.getItem(8).getItem(3).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				setColor(table.getItem(8).getItem(4));
				setColor(table.getItem(8).getItem(5));
				setColor(table.getItem(8).getItem(6));
			}else if("guest".equals(table.getItem(8).getItem(0).getText(1))){
				setColor(table.getItem(8).getItem(1));
				setColor(table.getItem(8).getItem(2));
				setColor(table.getItem(8).getItem(3));
				table.getItem(8).getItem(4).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				table.getItem(8).getItem(5).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				table.getItem(8).getItem(6).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}else {
				// 仮想化のツリーを開いており、ノード種別が空欄の場合
				table.getItem(8).getItem(1).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				table.getItem(8).getItem(3).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				table.getItem(8).getItem(2).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				table.getItem(8).getItem(4).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				table.getItem(8).getItem(5).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
				table.getItem(8).getItem(6).setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
			}
		}
	}
	
	/**
	 * 空文字だったらピンク色にする。
	 * @param item
	 */
	private void setColor(TableTreeItem item) {
		if ("".equals(item.getText(1))) {
			item.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		} else {
			item.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}

	/**
	 * 既存のボタンに加え、クリアボタンを追加します。
	 * 
	 * @param parent
	 *            ボタンバーコンポジット
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		// クリアボタン
		clearButtonScope =
				this.createButton(parent, IDialogConstants.OPEN_ID, Messages
						.getString("clear"), false);
		this.getButton(IDialogConstants.OPEN_ID).addSelectionListener(
				new SelectionAdapter() {
					@Override
					public void widgetSelected(SelectionEvent e) {
						// プロパティ取得及び設定
						Property property = null;
						NodeInfo nodeInfo = null;
						if (isModifyDialog()) {
							GetNodeProperty getNodeProperty = new GetNodeProperty(null,
									PropertyDefineConstant.MODE_MODIFY);
							property = getNodeProperty.getProperty();
							nodeInfo = getNodeProperty.getNodeInfo();

							//ファシリティID設定
							ArrayList<Property> list = PropertyUtil.getProperty(property,
									NodeConstant.FACILITY_ID);
							list.get(0).setValue(getFacilityId());
						} else {
							GetNodeProperty getNodeProperty = new GetNodeProperty(null,
									PropertyDefineConstant.MODE_ADD);
							property = getNodeProperty.getProperty();
							nodeInfo = getNodeProperty.getNodeInfo();
						}

						//Booleanエディタのプロパティにnullを設定する
						PropertyUtil.setPropertyValue(property,
								PropertyDefineConstant.EDITOR_BOOL, null);

						setInputData(property);

						// 変更モードの場合にclearボタン契機でFind By SNMPの変更を可能にする
						ipAddressBox.setEditable(true);
						portBox.setEditable(true);
						communityBox.setEditable(true);
						versionBox.setTouchEnabled(true);
						
						// クリア実行フラグ
						isClearDone = true;
					}
				});

		super.createButtonsForButtonBar(parent);
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	@Override
	protected ValidateResult validate() {
		ValidateResult result = null;

		return result;
	}

	/**
	 * 入力値をマネージャに登録します。
	 * 
	 * @return true：正常、false：異常
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		boolean result = false;

		Property property = this.getInputData();
		if(property != null){
			String errMessage = "";
			if(!this.isModifyDialog()){
				// 作成の場合
				Property copy = PropertyUtil.copy(property);
				PropertyUtil.deletePropertyDefine(copy);
				try {
					nodeInfo = NodePropertyUtil.property2node(copy);
					if (m_ownerRoleId.getText().length() > 0) {
						nodeInfo.setOwnerRoleId(m_ownerRoleId.getText());
					}
					RepositoryEndpointWrapper.addNode(nodeInfo);

					// リポジトリキャッシュの更新
					LocalSession.doCheck();

					MessageDialog.openInformation(
							null,
							Messages.getString("successful"),
							Messages.getString("message.repository.4"));

					result = true;

				} catch (FacilityDuplicate_Exception e) {
					// ファシリティIDが重複している場合、エラーダイアログを表示する
					//ファシリティID取得
					ArrayList values = PropertyUtil.getPropertyValue(copy, NodeConstant.FACILITY_ID);
					String args[] = { (String)values.get(0) };

					MessageDialog.openInformation(
							null,
							Messages.getString("message"),
							Messages.getString("message.repository.26", args));

				} catch (Exception e) {
					if (e instanceof InvalidRole_Exception) {
						// アクセス権なしの場合、エラーダイアログを表示する
						MessageDialog.openInformation(
								null,
								Messages.getString("message"),
								Messages.getString("message.accesscontrol.16"));
					} else {
						errMessage = ", " + e.getMessage();
						m_log.warn("action()", e);
					}
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.repository.5") + errMessage);
				}
			} else {
				// 変更の場合
				Property copy = PropertyUtil.copy(property);
				PropertyUtil.deletePropertyDefine(copy);
				try {
					nodeInfo = NodePropertyUtil.property2node(copy);
					if (m_ownerRoleId.getText().length() > 0) {
						nodeInfo.setOwnerRoleId(m_ownerRoleId.getText());
					}
					RepositoryEndpointWrapper.modifyNode(nodeInfo);

					// リポジトリキャッシュの更新
					LocalSession.doCheck();

					MessageDialog.openInformation(
							null,
							Messages.getString("successful"),
							Messages.getString("message.repository.10"));

					result = true;

				} catch (Exception e) {
					if (e instanceof InvalidRole_Exception) {
						// アクセス権なしの場合、エラーダイアログを表示する
						MessageDialog.openInformation(
								null,
								Messages.getString("message"),
								Messages.getString("message.accesscontrol.16"));
					} else {
						errMessage = ", " + e.getMessage();
					}

					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.repository.11") + errMessage);
				}
			}
		}

		return result;
	}

	/**
	 * 変更用ダイアログなのかを返します。
	 * 
	 * @return 変更用ダイアログの場合、true
	 */
	public boolean isModifyDialog() {
		return this.isModifyDialog;
	}

	/**
	 * 一括変更対象スコープのファシリティIDを返します。
	 * 
	 * @return 一括変更対象スコープのファシリティID
	 */
	public String getScopeId() {
		return this.scopeId;
	}

	/**
	 * クリアボタンがクリックされているかを返します。
	 * 
	 * @return クリアボタンがクリックされている場合、true
	 */
	public boolean isClearDone() {
		return this.isClearDone;
	}

	/**
	 * 入力値を保持したデータモデルを生成します。
	 * 
	 * @return データモデル
	 */
	public Property getInputData() {
		return (Property) this.propertySheet.getInput();
	}

	/**
	 * 入力値を保持したデータモデルを設定します。
	 * 
	 * @param property
	 */
	public void setInputData(Property property) {
		propertySheet.setInput(property);
		this.update();
	}

	/**
	 * @return Returns the facilityId.
	 */
	public String getFacilityId() {
		return facilityId;
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		if (isModifyDialog()) {
			return Messages.getString("modify");
		} else {
			return Messages.getString("register");
		}
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * NodeInfoを返します。
	 * 
	 * @return NodeInfo
	 */
	public NodeInfo getNodeInfo() {
		return this.nodeInfo;
	}

	/**
	 *SNMPを利用してノード情報を生成します。<BR>
	 *
	 * @param pollingData SNMPポーリングのためのIPアドレスなど
	 * @param mode 編集可否モード
	 * @return ノード情報のプロパティ
	 */
	private static NodeInfo GetNodeInfoBySNMP(String ipAddress, int port, String community, String version,int mode) {

		try {
			NodeInfo nodeInfo = RepositoryEndpointWrapper.getNodePropertyBySNMP(ipAddress, port, community, version);
			m_log.info("snmp2 " + nodeInfo.getNodeFilesystemInfo().size());
			NodePropertyUtil.setDefaultNode(nodeInfo);
			return nodeInfo;
		} catch (InvalidRole_Exception e) {
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("GetNodePropertyBySNMP(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}
		return null;
	}

}