/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.StringTokenizer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.fault.HinemosException;

public class CommandCreator {

	private static Log log = LogFactory.getLog(CommandCreator.class);

	public static enum PlatformType { AUTO, WINDOWS, UNIX, REGACY };

	public static final String sysUser;

	public static final String osName;
	public static final PlatformType sysPlatform;

	static {
		sysUser = System.getProperty("user.name");
		log.info("java process user detected : " + sysUser);

		osName = System.getProperty("os.name");
		if (osName != null && (osName.startsWith("Windows") || osName.startsWith("windows"))) {
			sysPlatform = PlatformType.WINDOWS;
		} else {
			sysPlatform = PlatformType.UNIX;
		}
		log.info("os detected : " + osName);
		log.info("platform detected : " + sysPlatform);
	}

	/**
	 * convert platform string to type
	 * @param str platform string
	 * @return platform type
	 */
	public static PlatformType convertPlatform(String str) {
		// Local Variable
		PlatformType platform = null;

		// MAIN
		if ("windows".equals(str)) {
			platform = PlatformType.WINDOWS;
		} else if ("unix".equals(str)) {
			platform = PlatformType.UNIX;
		} else if ("compatible".equals(str) || "regacy".equals(str)) {
			platform = PlatformType.REGACY;
		} else {
			platform = PlatformType.AUTO;
		}
		if (log.isDebugEnabled())
			log.debug("string converted. (str = " + str + ", type = " + platform + ")");
		return platform;
	}

	public static String[] createCommand(String execUser, String execCommand, PlatformType platform) throws HinemosException {
		return createCommand(execUser, execCommand, platform, YesNoConstant.TYPE_YES);
	}

	/**
	 * create command for platform
	 * @param execUser execution user
	 * @param execCommand command string
	 * @param platform target platform
	 * @return command
	 * @throws HinemosException
	 */
	public static String[] createCommand(String execUser, String execCommand, PlatformType platform, Integer specifyUser) throws HinemosException {
		// Local Variables
		String[] command = null;

		// Main
		if (specifyUser == YesNoConstant.TYPE_YES && execUser == null) {
			throw new NullPointerException("execUser is not defined. (execUser = " + execUser + ")");
		}
		if (execCommand == null) {
			throw new NullPointerException("execCommand is not defined. (execCommand = " + execCommand + ")");
		}
		if (platform == null) {
			throw new NullPointerException("platform is not defined. (platform = " + platform + ")");
		}

		String user;
		if (specifyUser == YesNoConstant.TYPE_YES) {
			// ジョブを実行するユーザを指定する場合はexecUserをそのまま使用する
			user = execUser;
		} else {
			// エージェントの実行ユーザで実行するの場合はsysuserに置き換える
			user = sysUser;
		}

		switch (platform) {
		case WINDOWS :
			command = createWindowsCommand(user, execCommand);
			break;
		case UNIX :
			command = createUnixCommand(user, execCommand);
			break;
		case REGACY :
			command = createRegacyCommand(user, execCommand);
			break;
		case AUTO :
		default :
			command = createCommand(user, execCommand, sysPlatform, specifyUser);
			break;
		}

		return command;
	}

	/**
	 * create command for Windows like CMD /C [COMMAND]
	 * @param execUser execution user
	 * @param execCommand command string parsed by CMD
	 * @return command and arguments
	 * @throws HinemosException
	 */
	private static String[] createWindowsCommand(String execUser, String execCommand) throws HinemosException {
		// Local Variables
		String[] command = null;
		ArrayList<String> commandList = new ArrayList<String>();
		
		// MAIN
		if (execUser.equals(sysUser)) {
			String[] quoteSplitedCommand= execCommand.split("\"");
			for (int i = 0; i < quoteSplitedCommand.length; ++i) {
				if (i % 2 == 0) {
					// "で囲まれた部分の外側
					StringTokenizer st = new StringTokenizer(quoteSplitedCommand[i]);
					while (st.hasMoreTokens()) {
						commandList.add(st.nextToken());
					}
				} else {
					// "で囲まれた部分の内側
					commandList.add(quoteSplitedCommand[i]);
				}
			}
			command = commandList.toArray(new String[commandList.size()]);
			log.debug("created command for windows. (cmd = " + Arrays.toString(command) + ")");
		} else {
			throw new HinemosException("execution user and jvm user must be same on Windows. (execUser = " + execUser + ", sysUser = " + sysUser + ")");
		}
		return command;
	}

	/**
	 * create command for UNIX line (sudo -u [USER]) sh -c [COMMAND]
	 * @param execUser execution user
	 * @param execCommand command string parsed by sh
	 * @return command and arguments
	 * @throws HinemosException
	 */
	private static String[] createUnixCommand(String execUser, String execCommand) throws HinemosException {
		// Local Variables
		String[] command = null;

		// MAIN
		if (execUser.equals(sysUser)) {
			command = new String[]{ "sh", "-c", execCommand };
			if (log.isDebugEnabled()) log.debug("created command for unix. (cmd = " + Arrays.toString(command));
		} else {
			command = new String[]{ "sudo", "-u", execUser, "sh", "-c", execCommand };
			if (log.isDebugEnabled()) log.debug("created command for unix. (cmd = " + Arrays.toString(command));
		}
		return command;
	}

	/**
	 * create regacy (before 3.1) command like (su [USER] -c) execCommand
	 * @param execUser execution user
	 * @param execCommand command string
	 * @return command and arguments
	 * @throws HinemosException
	 */
	private static String[] createRegacyCommand(String execUser, String execCommand) throws HinemosException {
		// Local Variables
		String[] command = null;

		// MAIN
		if (execUser.equals(sysUser)) {
			// split by half space
			command = execCommand.split(" ");
		} else {
			if ("root".equals(sysUser)) {
				command = new String[]{ "su", execUser, "-c", execCommand };
				if (log.isDebugEnabled()) log.debug("created command for regacy. (cmd = " + Arrays.toString(command));
			} else {
				throw new HinemosException("execution user and jvm user must be same. (execUser = " + execUser + ", sysUser = " + sysUser + ")");
			}
		}
		return command;
	}
	/**
	 * プロダクト実行ユーザを取得します。
	 * @return プロダクト実行ユーザ
	 */
	public static String getSysUser() {
		return sysUser;
	}


	public static void main(String[] args) {

		String command = "hogehoge \"a b\" c , d";
		String[] commandArr = null;


		System.out.println("SpecifyUser = YES");

		// Windows環境用
		System.setProperty("user.name", "Administrator");
		try {
			commandArr = createCommand("Administrator", command, PlatformType.WINDOWS, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			commandArr = createCommand("hinemos", command, PlatformType.WINDOWS, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		// UNIX/Linux環境用
		System.setProperty("user.name", "root");
		try {
			commandArr = createCommand("root", command, PlatformType.UNIX, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			commandArr = createCommand("hinemos", command, PlatformType.UNIX, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		// 下位互換性環境用
		System.setProperty("user.name", "root");
		try {
			commandArr = createCommand("hinemos", command, PlatformType.REGACY, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		System.setProperty("user.name", "Administrator");
		try {
			commandArr = createCommand("hinemos", command, PlatformType.REGACY, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			commandArr = createCommand("Administrator", command, PlatformType.REGACY, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		// 環境自動識別用
		System.setProperty("user.name", "Administrator");
		System.setProperty("os.name", "Windows Server 2008");
		try {
			commandArr = createCommand("Administrator", command, PlatformType.AUTO, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		System.setProperty("user.name", "root");
		System.setProperty("os.name", "Linux");
		try {
			commandArr = createCommand("root", command, PlatformType.AUTO, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		System.out.println("SpecifyUser = NO");

		// Windows環境用
		System.setProperty("user.name", "Administrator");
		try {
			commandArr = createCommand("Administrator", command, PlatformType.WINDOWS, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			commandArr = createCommand("hinemos", command, PlatformType.WINDOWS, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		// UNIX/Linux環境用
		System.setProperty("user.name", "root");
		try {
			commandArr = createCommand("root", command, PlatformType.UNIX, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			commandArr = createCommand("hinemos", command, PlatformType.UNIX, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		// 下位互換性環境用
		System.setProperty("user.name", "root");
		try {
			commandArr = createCommand("hinemos", command, PlatformType.REGACY, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		System.setProperty("user.name", "Administrator");
		try {
			commandArr = createCommand("hinemos", command, PlatformType.REGACY, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			commandArr = createCommand("Administrator", command, PlatformType.REGACY, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		// 環境自動識別用
		System.setProperty("user.name", "Administrator");
		System.setProperty("os.name", "Windows Server 2008");
		try {
			commandArr = createCommand("Administrator", command, PlatformType.AUTO, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		System.setProperty("user.name", "root");
		System.setProperty("os.name", "Linux");
		try {
			commandArr = createCommand("root", command, PlatformType.AUTO, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
	}
}
