/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.accesscontrol.util;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.metamodel.EntityType;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.annotation.HinemosObjectPrivilege;
import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.ObjectPrivilegeMode;
import com.clustercontrol.accesscontrol.model.ObjectPrivilegeEntity;
import com.clustercontrol.accesscontrol.model.ObjectPrivilegeTargetEntity;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.commons.util.HinemosEntityManager;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.fault.JobMasterNotFound;
import com.clustercontrol.fault.ObjectPrivilege_InvalidRole;
import com.clustercontrol.jobmanagement.model.JobMstEntityPK;

/**
 * オブジェクト権限チェックのUtilクラス
 *
 */
public class ObjectPrivilegeUtil {

	private static Log m_log = LogFactory.getLog( ObjectPrivilegeUtil.class );

	/** オブジェクト権限マップ（オブジェクト種別、Entityクラス） */
	private static Map<String, Class<?>> m_objectPrivilegeMap
	= new HashMap<String, Class<?>>();

	/** オブジェクト種別に対応したクラスの取得 */
	private static Class<?> getObjectPrivilegeClass(String objectType) {
		// オブジェクト権限マップの作成
		createObjectPrivilegeMap();
		return m_objectPrivilegeMap.get(objectType);
	}

	/** オブジェクト権限マップの作成 */
	private static void createObjectPrivilegeMap() {
		if (m_objectPrivilegeMap == null || m_objectPrivilegeMap.size() == 0) {
			EntityManagerFactory emf
			= new JpaTransactionManager().getEntityManager().getEntityManagerFactory();
			Set<EntityType<?>> entityTypes = emf.getMetamodel().getEntities();
			for (EntityType<?> entityType : entityTypes) {
				Class<?> clazz = entityType.getBindableJavaType();
				if (ObjectPrivilegeTargetEntity.class.isAssignableFrom(clazz)) {
					try {
						HinemosObjectPrivilege hinemosObjectPrivilege
						= clazz.getAnnotation(HinemosObjectPrivilege.class);
						String objectType = hinemosObjectPrivilege.objectType();
						if (hinemosObjectPrivilege.isModifyCheck()) {
							m_objectPrivilegeMap.put(objectType, clazz);
						}
					} catch (Exception e) {
						continue;
					}
				}
			}
		}
	}

	/** オブジェクト権限の更新可否チェック */
	public static Object getObjectPrivilegeObject(String objectType, String objectId, ObjectPrivilegeMode mode)
			throws JobMasterNotFound, ObjectPrivilege_InvalidRole  {
		EntityManager em = new JpaTransactionManager().getEntityManager();
		Class<?> objectPrivilegeClass = getObjectPrivilegeClass(objectType);
		if (HinemosModuleConstant.JOB.equals(objectType)) {
			// JobMstEntityの場合は、objectId　!= PK
			return ((HinemosEntityManager)em).find(objectPrivilegeClass, new JobMstEntityPK(objectId, objectId), mode);
		} else {
			return ((HinemosEntityManager)em).find(objectPrivilegeClass, objectId, mode);
		}
	}

	/** オブジェクトに紐づくオブジェクト権限を取得する。 */
	public static List<ObjectPrivilegeEntity> getObjectPrivilegeEntities(Class<?> objectPrivilegeClass, String objectId, String roleId) {
		EntityManager em = new JpaTransactionManager().getEntityManager();
		HinemosObjectPrivilege hinemosObjectPrivilege = objectPrivilegeClass.getAnnotation(HinemosObjectPrivilege.class);
		if (hinemosObjectPrivilege == null) {
			// HinemosObjectPrivilegeアノテーションが設定されていない場合はnullを返す
			return null;
		}
		String objectType = hinemosObjectPrivilege.objectType();
		return em.createNamedQuery("ObjectPrivilegeEntity.findByObjectIdTypeRoleId", ObjectPrivilegeEntity.class)
				.setParameter("objectType", objectType)
				.setParameter("objectId", objectId)
				.setParameter("roleId", roleId)
				.getResultList();
	}

	/** 指定されたオブジェクトに対応したオブジェクト権限を削除する。 */
	public static void deleteObjectPrivilege(String objectType, String objectId) {
		EntityManager em = new JpaTransactionManager().getEntityManager();
		em.createNamedQuery("ObjectPrivilegeEntity.deleteByObjectTypeObjectId")
		.setParameter("objectType", objectType)
		.setParameter("objectId", objectId)
		.executeUpdate();
	}
}
