package com.clustercontrol.performance.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.performance.factory.OperateCollectCategoryCollectMaster;
import com.clustercontrol.performance.factory.OperateCollectItemCalcMethodMaster;
import com.clustercontrol.performance.factory.OperateCollectItemCodeMaster;
import com.clustercontrol.performance.factory.OperateCollectPollingMaster;
import com.clustercontrol.performance.monitor.entity.CollectorCategoryCollectMstData;
import com.clustercontrol.performance.monitor.entity.CollectorItemCalcMethodMstData;
import com.clustercontrol.performance.monitor.entity.CollectorItemCodeMstData;
import com.clustercontrol.performance.monitor.entity.CollectorPollingMstData;

/**
 * 性能収集時のマスターデータのキャッシュ
 *
 * 
 * @version 4.0.0
 * @since 4.0.0
 */
public class CollectorMasterCache {
	private static Log m_log = LogFactory.getLog( CollectorMasterCache.class );

	// ポーリングデータのキャッシュ
	private static Collection<CollectorCategoryCollectMstData> categoryCollectMstDataListCache = null;
	private static Collection<CollectorItemCodeMstData> itemCodeMstDataListCache = null;
	private static Collection<CollectorItemCalcMethodMstData> itemCalcMethodMstDataListCache = null;
	private static Collection<CollectorPollingMstData> pollingMstDataListCache = null;

	/*
	 * 
	 * Utility
	 * 
	 */

	/**
	 * 指定した収集項目IDのマスタ情報を返却する
	 * 
	 * @param itemCode
	 * @return
	 */
	public static CollectorItemCodeMstData getCategoryCodeMst(String itemCode) {
		for(CollectorItemCodeMstData data : getItemCodeMstDataListCache()){
			if(data.getItemCode().equals(itemCode)){
				return data;
			}
		}
		return null;
	}


	/**
	 * ポーリング対象の文字列(OIDなど)のリストを取得する
	 * 
	 * @param itemCodeListe
	 * @param collectMethod
	 * @param platformId
	 * @param subPlatformId
	 * @return
	 */
	protected static ArrayList<String> getPollingTarget(ArrayList<String> itemCodeListe, String collectMethod, String platformId, String subPlatformId) {
		Set<String> pollingTargetSet = new HashSet<String>();

		while (true) {
			for(String itemCode : itemCodeListe){
				for(CollectorPollingMstData data : getPollingMstDataListCache()){
					if(data.getItemCode().equals(itemCode) &&
							data.getCollectMethod().equals(collectMethod) &&
							data.getPlatformId().equals(platformId) &&
							data.getSubPlatformId().equals(subPlatformId)){
						pollingTargetSet.add(data.getPollingTarget());
					}
				}
			}
			// VM管理やクラウド管理などではsubPlatformIdに「VMWARE」や「AWS」などが存在している収集項目に加え、
			// subPlatformIdが空の場合の（つまり物理と同じ）収集項目が収集できる必要があるため、
			// subPlatformIdが空でない場合にはsubPlatformIdを空にした収集項目も検索する
			if (subPlatformId.isEmpty()) break;
			subPlatformId = "";
		}
		ArrayList<String> pollingTarget = new ArrayList<String>(pollingTargetSet);
		if(m_log.isDebugEnabled()){
			for(String target : pollingTargetSet){
				m_log.debug("getPollingTarget() target = " + target);
			}
		}
		return pollingTarget;
	}

	/**
	 * 指定した収集項目IDの内訳の収集項目IDのリストを返却する
	 * 
	 * @param itemCode
	 * @param collectMethod
	 * @param platformId
	 * @param subPlatformId
	 * @return
	 */
	protected static ArrayList<String> getBreakdownItemCodeList(String itemCode, String collectMethod, String platformId, String subPlatformId) {
		ArrayList<String> breakdownTmpItemCodeList = new ArrayList<String>();

		for(CollectorItemCodeMstData itemCodeMstData : getItemCodeMstDataListCache()){
			if(itemCodeMstData.getParentItemCode() != null &&
					itemCodeMstData.getParentItemCode().equals(itemCode)){
				breakdownTmpItemCodeList.add(itemCodeMstData.getItemCode());
			}
		}
		if(m_log.isDebugEnabled()){
			for(String code : breakdownTmpItemCodeList){
				m_log.debug("getBreakdownItemCodeList() itemCode(tmp) = " + code);
			}
		}

		Set<String> breakdownItemCodeSet = new HashSet<String>();
		while (true) {
			for(String tmpItemCode : breakdownTmpItemCodeList){
				for(CollectorItemCalcMethodMstData calcMethod : getItemCalcMethodMstDataListCache()){
					if(calcMethod.getCollectMethod().equals(collectMethod) &&
							calcMethod.getPlatformId().equals(platformId) &&
							calcMethod.getSubPlatformId().equals(subPlatformId) &&
							calcMethod.getItemCode().equals(tmpItemCode)){
						breakdownItemCodeSet.add(tmpItemCode);
						break;
					}
				}
			}
			// VM管理やクラウド管理などではsubPlatformIdに「VMWARE」や「AWS」などが存在している収集項目に加え、
			// subPlatformIdが空の場合の（つまり物理と同じ）収集項目が収集できる必要があるため、
			// subPlatformIdが空でない場合にはsubPlatformIdを空にした収集項目も検索する
			if (subPlatformId.isEmpty()) break;
			subPlatformId = "";
		}
		ArrayList<String> breakdownItemCodeList = new ArrayList<String>(breakdownItemCodeSet);
		if(m_log.isDebugEnabled()){
			for(String code : breakdownItemCodeList){
				m_log.debug("getBreakdownItemCodeList() itemCode = " + code);
			}
		}
		return breakdownItemCodeList;
	}

	/**
	 * 指定したカテゴリの収集方法(SNMP/WBEMなど)を取得する
	 * 
	 * @param platformId
	 * @param subPlatformId
	 * @param categoryCode
	 * @return
	 */
	public static String getCollectMethod(String platformId, String subPlatformId, String categoryCode) {
		while (true) {
			for(CollectorCategoryCollectMstData data : getCategoryCollectMstDataListCache()){
				if(data.getPlatformId().equals(platformId) &&
						data.getSubPlatformId().equals(subPlatformId) &&
						data.getCategoryCode().equals(categoryCode)){
					return data.getCollectMethod();
				}
			}
			// VM管理やクラウド管理などではsubPlatformIdに「VMWARE」や「AWS」などが存在している収集項目に加え、
			// subPlatformIdが空の場合の（つまり物理と同じ）収集項目が収集できる必要があるため、
			// subPlatformIdが空でない状態で収集カテゴリが取れなかった場合、subPlatformIdを空にして
			// 再度収集カテゴリを見つける
			if (subPlatformId.isEmpty()) return "";
			subPlatformId = "";
		}
	}

	/**
	 * itemCodeに対応するデバイスタイプを返却
	 * 
	 * @param itemCode
	 * @return
	 */
	public static String getDeviceType(String itemCode) {
		String devicetype = null;
		for(CollectorItemCodeMstData itemCodeMstData : getItemCodeMstDataListCache()){
			if(itemCodeMstData.getItemCode() != null && itemCodeMstData.getItemCode().equals(itemCode)){
				devicetype = itemCodeMstData.getDeviceType();
				break;
			}
		}

		return devicetype;
	}

	/*
	 * 
	 * キャッシュの取得
	 * 
	 */

	/**
	 * CollectorItemCodeMstのキャッシュ
	 * 
	 * @return
	 */
	private static synchronized Collection<CollectorItemCodeMstData> getItemCodeMstDataListCache() {
		if(itemCodeMstDataListCache == null){
			itemCodeMstDataListCache = new ArrayList<CollectorItemCodeMstData>();

			Collection<CollectorItemCodeMstData> dataList =
					new OperateCollectItemCodeMaster().findAll();
			for(CollectorItemCodeMstData data : dataList){
				itemCodeMstDataListCache.add(data);
			}
		}

		return itemCodeMstDataListCache;
	}

	/**
	 * CollectorItemCalcMethodMstのキャッシュ
	 * @return
	 */
	private static synchronized Collection<CollectorItemCalcMethodMstData> getItemCalcMethodMstDataListCache() {
		if(itemCalcMethodMstDataListCache == null){
			itemCalcMethodMstDataListCache = new ArrayList<CollectorItemCalcMethodMstData>();

			Collection<CollectorItemCalcMethodMstData> dataList =
					new OperateCollectItemCalcMethodMaster().findAll();
			for(CollectorItemCalcMethodMstData data : dataList){
				itemCalcMethodMstDataListCache.add(data);
			}
		}

		return itemCalcMethodMstDataListCache;
	}

	/**
	 * CollectorPollingMstのキャッシュ
	 * @return
	 */
	private static synchronized Collection<CollectorPollingMstData> getPollingMstDataListCache() {
		if(pollingMstDataListCache == null){
			pollingMstDataListCache = new ArrayList<CollectorPollingMstData>();

			Collection<CollectorPollingMstData> pollingMstLocalList =
					new OperateCollectPollingMaster().findAll();
			for(CollectorPollingMstData data : pollingMstLocalList){
				pollingMstDataListCache.add(data);
			}
		}
		return pollingMstDataListCache;
	}

	/**
	 * CollectorCategoryCollectMstのキャッシュ
	 * @return
	 */
	private static synchronized Collection<CollectorCategoryCollectMstData> getCategoryCollectMstDataListCache() {
		if(categoryCollectMstDataListCache == null){
			categoryCollectMstDataListCache = new ArrayList<CollectorCategoryCollectMstData>();

			Collection<CollectorCategoryCollectMstData> collects =
					new OperateCollectCategoryCollectMaster().findAll();
			for(CollectorCategoryCollectMstData data : collects){
				categoryCollectMstDataListCache.add(data);
			}
		}
		return categoryCollectMstDataListCache;
	}

}
